/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.common;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.Element;
import org.dom4j.Node;

import ch.docuteam.darc.util.XMLUtil;
import ch.docuteam.tools.exception.Exception;


/**
 * Abstract superclass of all elements that are contained in a <a href="../mets/Document.html">Document</a>.
 * All these elements have these four properties in common:
 * <ul>
 * <li>document: The METS document</li>
 * <li>parent: The node containing me</li>
 * <li>element: My dom4j xpath element</li>
 * <li>id: (optional) An identification string</li>
 * </ul>
 * This class inserts Exceptions into the <a href="../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when a mandatory element is missing</li>
 * </ul>
 * @author denis
 *
 */
public abstract class NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected DocumentAbstract			document;
	protected NodeAbstract				parent;
	protected Element					element;
	protected String					id;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public DocumentAbstract getDocument()
	{
		return this.document;
	}

	public NodeAbstract getParent()
	{
		return this.parent;
	}

	public Element getElement()
	{
		return this.element;
	}

	public String getId()
	{
		return this.id;
	}


	public String getElementText(String xPath, Boolean isMandatory)
	{
		Node subNode = this.element.selectSingleNode(xPath);
		if (isMandatory && (subNode == null))
			Exception.remember("Mandatory element missing in File: '" + this.document.getName() + "':\n\t'" + xPath + "'\nin location: '" + this.element.getPath() + "'\nID: " + this.id);

		return ((subNode == null)? "": subNode.getText());
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	//	Set the id. Protected because only subclasses may set the id.
	protected void setId(String id)
	{
		this.id = id;
		this.element.addAttribute("ID", this.id);
	}

	//	--------		Accessing Dynamic	-------------------------------------------------------

	/**
	 *
	 * @param xPath
	 * @return
	 */
	protected List<String> getDynamicElementTexts(String xPath)
	{
		@SuppressWarnings("unchecked")
		List<Node> nodes = this.element.selectNodes(xPath);
		if (nodes.isEmpty())		return null;

		List<String> results = new ArrayList<String>(nodes.size());
		for (Node node: nodes)		results.add(node.getText());

		return results;
	}


	protected void setDynamicElementTexts(List<String> values, String xPath)
	{
		XMLUtil.set(this.element, xPath, values);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
