/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.dc;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.Document;
import org.dom4j.Element;

import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.mets.dmdsec.DMDSectionAbstract;
import ch.docuteam.darc.mets.dmdsec.DMDSectionWithOAI_DC;


/**
 * This class, used by the class <a href="../mets/dmdsec/DMDSectionWithOAI_DC.html">DMDSection</a>, represents an OAI_DC element.
 * It contains lists of DC elements.
 * <p>
 * When reading a METS file, it comes in one of two different flavors: with or without a surrounding OAI_DC element.
 * If the surrounding OAI_DC element is <b>missing</b>, create it and place all DC elements into it.
 *
 * @author denis
 *
 */
public class OAI_DC extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String		OAI_DCNamespacePrefix = "OAI_DC";
	static private final String		OAI_DCNamespaceURI = "http://www.openarchives.org/OAI/2.0/oai_dc/";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private List<DC>				contributors = new ArrayList<DC>();
	private List<DC>				coverages = new ArrayList<DC>();
	private List<DC>				creators = new ArrayList<DC>();
	private List<DC>				dates = new ArrayList<DC>();
	private List<DC>				descriptions = new ArrayList<DC>();
	private List<DC>				formats = new ArrayList<DC>();
	private List<DC>				identifiers = new ArrayList<DC>();
	private List<DC>				languages = new ArrayList<DC>();
	private List<DC>				publishers = new ArrayList<DC>();
	private List<DC>				relations = new ArrayList<DC>();
	private List<DC>				rights = new ArrayList<DC>();
	private List<DC>				sources = new ArrayList<DC>();
	private List<DC>				subjects = new ArrayList<DC>();
	private List<DC>				titles = new ArrayList<DC>();
	private List<DC>				types = new ArrayList<DC>();

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	/**
	 * This method is used only when a METS-File is being read.
	 * There are 2 modes: one WITH and one WITHOUT a surrounding OAI_DC element.
	 * If NO surrounding OAI_DC Element exists, create one and fill it with the existing DC elements.
	 * If a surrounding OAI_DC Element exists, just read it.
	 */
	private OAI_DC(DMDSectionAbstract parent, Element element)
	{
		this.document = parent.getDocument();
		this.parent = parent;

		//	There are 2 modes: one WITH and one WITHOUT a surrounding OAI_DC element:
		@SuppressWarnings("rawtypes")
		List oaiDcs = element.selectNodes("./METS:mdWrap[@MDTYPE='" + DC.getNamespacePrefix() + "']/METS:xmlData/" + OAI_DCNamespacePrefix + ":dc");

		if (oaiDcs.size() == 0)
		{
			//	NO surrounding OAI_DC Element exists - so create one and fill it with the existing DC elements:

			//	Generate OAI_DC element and add it to my parent element:
			this.element = ((Element)element.selectSingleNode("./METS:mdWrap[@MDTYPE='" + DC.getNamespacePrefix() + "']/METS:xmlData"))
											.addElement(OAI_DCNamespacePrefix + ":dc");

			//	Add the required namespace to the root element:
			this.document.getRootElement().addNamespace(OAI_DCNamespacePrefix, OAI_DCNamespaceURI);

			//	Now move the DC Elements from their existing positions in the DMDSection to the new OAI_DC Element:
			DC.parse(this, element, true);
		}
		else
		{
			//	Surrounding OAI_DC Element exists; so just read it:

			if (oaiDcs.size() >= 2)		ch.docuteam.tools.exception.Exception.remember("Bad number of OAI_DC Elements in the METS file. Expected: 0 or 1, found: " + oaiDcs.size());

			this.element = (Element)oaiDcs.get(0);

			//	Now read the DC Elements:
			DC.parse(this, element, false);
		}
	}


	/**
	 * This method is used only when a OAI_DC is created programmatically out of a Document.
	 */
	private OAI_DC(DMDSectionWithOAI_DC parent, Document oaiDcDocument)
	{
		this.document = parent.getDocument();
		this.parent = parent;

		//	Generate OAI_DC element and add it to the DMDSection's element:
		Element mdWrapElement = this.parent.getElement().addElement("METS:mdWrap").addAttribute("MDTYPE", "DC");
		this.element = mdWrapElement.addElement("METS:xmlData").addElement(OAI_DCNamespacePrefix + ":dc");
		parent.setOaiDc(this);

		DC.create(this, oaiDcDocument);
	}


	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read.
	 * If there are no DC Elements around, just return null. Otherwise create an OAI_DC Element out of the given document.
	 */
	static public OAI_DC parse(DMDSectionWithOAI_DC parent, Element element)
	{
		//	Are there any DC Elements? If no, return null:
		if (element.selectNodes("./METS:mdWrap[@MDTYPE='" + DC.getNamespacePrefix() + "']//" + DC.getNamespacePrefix() + ":*").size() == 0)		return null;

		//	Create an OAI_DC Element:
		return new OAI_DC(parent, element);
	}


	/**
	 * This method is used only when a OAI_DC is created programmatically out of a DC-Document.
	 * If there are no DC Elements around, just return null. Otherwise create an OAI_DC Element out of the given DC-document.
	 */
	static public OAI_DC create(ch.docuteam.darc.mets.Document metsDocument, Document oaiDcDocument)
	{
		//	Get the namespace-prefix from the oaiDcDocument (we are flexible!):
		String dcNamespacePrefixUsedInFile = oaiDcDocument.getRootElement().getNamespaceForURI("http://purl.org/dc/elements/1.1/").getPrefix();
		if (dcNamespacePrefixUsedInFile == null || dcNamespacePrefixUsedInFile.isEmpty())		return null;

		//	Are there some DC Elements? If no, return null:
//		if (oaiDcDocument.selectNodes(".//" + DCNamespacePrefix + ":*").size() == 0)		return null;
		if (oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":*").size() == 0)		return null;

		//	Create a new dmdSection in the METS document:
		DMDSectionWithOAI_DC dmdSection = new DMDSectionWithOAI_DC(metsDocument);

		//	Create an OAI_DC Element in this newly created DMDSection:
		return new OAI_DC(dmdSection, oaiDcDocument);
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * @return the contributors
	 */
	public List<DC> getContributors()
	{
		return this.contributors;
	}

	/**
	 * @return the coverages
	 */
	public List<DC> getCoverages()
	{
		return this.coverages;
	}

	/**
	 * @return the creators
	 */
	public List<DC> getCreators()
	{
		return this.creators;
	}

	/**
	 * @return the date
	 */
	public List<DC> getDates()
	{
		return this.dates;
	}

	/**
	 * @return the descriptions
	 */
	public List<DC> getDescriptions()
	{
		return this.descriptions;
	}

	/**
	 * @return the formats
	 */
	public List<DC> getFormats()
	{
		return this.formats;
	}

	/**
	 * @return the identifiers
	 */
	public List<DC> getIdentifiers()
	{
		return this.identifiers;
	}

	/**
	 * @return the languages
	 */
	public List<DC> getLanguages()
	{
		return this.languages;
	}

	/**
	 * @return the publishers
	 */
	public List<DC> getPublishers()
	{
		return this.publishers;
	}

	/**
	 * @return the relations
	 */
	public List<DC> getRelations()
	{
		return this.relations;
	}

	/**
	 * @return the rights
	 */
	public List<DC> getRights()
	{
		return this.rights;
	}

	/**
	 * @return the sources
	 */
	public List<DC> getSources()
	{
		return this.sources;
	}

	/**
	 * @return the subjects
	 */
	public List<DC> getSubjects()
	{
		return this.subjects;
	}

	/**
	 * @return the titles
	 */
	public List<DC> getTitles()
	{
		return this.titles;
	}

	/**
	 * @return the types
	 */
	public List<DC> getTypes()
	{
		return this.types;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString()
	{
		final int maxLen = 2;
		return "[OAI_DC:identifiers=" + (this.identifiers != null ? this.identifiers.subList(0, Math.min(this.identifiers.size(), maxLen)) : null) +
				", titles=" + (this.titles != null ? this.titles.subList(0, Math.min(this.titles.size(), maxLen)) : null) + "]";
	}

	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
