/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import bsh.EvalError;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.util.KeyAndValue;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;
import ch.docuteam.tools.string.ToolTipText;
import ch.docuteam.tools.translations.I18N;
import ch.docuteam.tools.util.JavaInterpreter;

/**
 * A MetadataElement specifies an abstract template for a data value belonging to a SIP node.
 * The metadata elements are initialized through the file "./config/levels.xml".
 * A metadata element has an id (= accessor name) and, calculated from the accessor name, a setter and getter method for setting and retrieving its value.
 * <p>
 * It is <b>NOT</b> a MetadataElement that gets attached to a LevelOfDescription, but a <a href="LevelMetadataElement.html">LevelMetadataElement</a>
 * which adds some level-specific properties to the MetadataElement.
 * <p>
 * Optionally, a validator class can be supplied. This class must implement the interface <a href="MetadataElementValidator.html">MetadataElementValidator</a>;
 * an instance is used to validate manual changes of this metadata element value.
 * <p>
 * Optionally, a default java expression can be supplied, with which an instance of this metadata element is initialized.
 * <p>
 * Optionally, a List&lt;String&gt; of allowed values can be supplied. This list is then shown as a drop-down list in the GUI.
 * <p>
 * Optionally, a tooltip text can be supplied, which is shown in the GUI when the mouse hovers over this element.
 *
 * @author denis
 *
 */
public class MetadataElement implements ToolTipText
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	The values separator. Default is ';', it can be changed using the method setAllowedValuesSeparator().
	static private String							AllowedValuesSeparator = ";";

	//	A Map containing all MetadataElements. Access key is the accessorName.
	static private Map<String, MetadataElement>		All = new HashMap<String, MetadataElement>(60);

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String									accessorName;

	//	Getting and setting values:
	private Method									setterMethod;
	private Method									getterMethod;

	//	Validating the value (optional):
	private MetadataElementValidator				validatorInstance;
	private Method									validatorMethod;

	//	Default value (optional):
	private String									defaultExpression;

	//	Allowed values (optional):
//	private List<String>							allowedValues;
	private List<KeyAndValue>						allowedValues;

	private String									toolTipText;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	MetadataElement(Class<?> targetClass, String accessorName, String validatorClassName, String defaultExpression, String allowedValues) throws ClassNotFoundException, SecurityException, NoSuchMethodException, InstantiationException, IllegalAccessException
	{
		this.accessorName = accessorName;

		String accessorNameCapitalized = StringUtil.first(accessorName, 1).toUpperCase() + accessorName.substring(1);
		this.getterMethod = targetClass.getMethod("get" + accessorNameCapitalized);
		this.setterMethod = targetClass.getMethod("set" + accessorNameCapitalized, List.class);

		if (validatorClassName != null)
		{
			Class<?> validatorClass = Class.forName(validatorClassName);
			this.validatorInstance = (MetadataElementValidator)validatorClass.newInstance();
			this.validatorMethod = validatorClass.getMethod("check", String.class, NodeAbstract.class, this.getClass());
		}

		this.defaultExpression = defaultExpression;

//		if (allowedValues != null)
//		{
//			this.allowedValues = new ArrayList<String>(StringUtil.occurrencesOf(AllowedValuesSeparator, allowedValues) + 1);
//			for (String s: allowedValues.split(AllowedValuesSeparator))		this.allowedValues.add(s.trim());
//		}
//		else
//		{
//			this.allowedValues = new ArrayList<String>();
//		}

		if (allowedValues != null)
		{
			this.allowedValues = new ArrayList<KeyAndValue>(StringUtil.occurrencesOf(AllowedValuesSeparator, allowedValues) + 1);
			for (String s: allowedValues.split(AllowedValuesSeparator))		this.allowedValues.add(new KeyAndValue(s.trim()));
		}
		else
		{
			this.allowedValues = new ArrayList<KeyAndValue>();
		}

		try
		{
			this.toolTipText = I18N.translate(this.accessorName + "ToolTip");
			if (this.toolTipText.isEmpty())		this.toolTipText = I18N.translate(this.accessorName);
		}
		catch (Exception ex)
		{
			this.toolTipText = this.accessorName;
		}

		All.put(accessorName, this);

		Logger.debug("Created: " + this);
	}

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static public void clear()
	{
		All.clear();
	}

	//	--------		Accessing			-------------------------------------------------------

	static void setAllowedValuesSeparator(String sep)
	{
		AllowedValuesSeparator = sep;
	}

	/**
	 * Return a MetadataElement instance with this accessorName, or null if it doesn't exist.
	 */
	static public MetadataElement get(String accessorName)
	{
		initializeIfNecessary();
		return All.get(accessorName);
	}

	/**
	 * Return all MetadataElements as a Map (key is the accessorName).
	 */
	static public Map<String, MetadataElement> getAll()
	{
		initializeIfNecessary();
		return All;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------

	static private void initializeIfNecessary()
	{
		if (All.isEmpty())		LevelOfDescription.initializeIfNecessary();
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	public String getId()
	{
		return this.accessorName;
	}

	public String getAccessorName()
	{
		return this.accessorName;
	}

//	public List<String> getAllowedValues()
//	{
//		return this.allowedValues;
//	}

	public List<KeyAndValue> getAllowedValues()
	{
		return this.allowedValues;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------

	@Override
	public String getToolTipText()
	{
		return this.toolTipText;
	}

	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * Send the getter method to the passed NodeAbstract's c element and return the return value of this method.
	 */
	@SuppressWarnings("unchecked")
	public List<String> getValueFromNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		List<String> value = (List<String>)this.getterMethod.invoke(node.getMyDMDSectionWithEAD().getC());

		return value;
	}

	/**
	 * Send the setter method to the passed NodeAbstract's c element with the parameter 'value'.
	 */
	public void setValueInNode(List<String> value, NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		Logger.debug("Setting in:" + node + " metadata element:<" + this.accessorName + "> to:'" + value + "'");

		this.setterMethod.invoke(node.getMyDMDSectionWithEAD().getC(), value);
	}


	/**
	 * Evaluate the String 'value' in the context of the object 'o'. Always allow null value.
	 * @param value The value to be tested
	 * @param node This is the context of the value (i.e. the container object). It may be an arbitraty object or null, the validator must handle it.
	 * @return Null if the value is OK, an error message otherwise.
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public String validateValueAgainstNode(String value, NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		if (this.validatorInstance == null)		return null;
		if (value == null)						return null;		//	Always allow null
		if (value.isEmpty())					return null;		//	Always allow empty value

		Logger.debug("Validating in:" + node + " metadata element:<" + this.accessorName + "> value:'" + value + "'");

		return (String)this.validatorMethod.invoke(this.validatorInstance, value, node, this);
	}


	/**
	 * Return the default value, which comes from a String in the levels config file.
	 * This string can be undefined, then the default value is null.
	 * This string can be a java expression, which will be executed. The current AbstractNode is accessable via the variable "object".
	 * @param o
	 * @return
	 * @throws EvalError
	 */
	public String getDefaultValueFromNode(NodeAbstract node) throws EvalError
	{
		if (this.defaultExpression == null)		return null;

		Logger.debug("Getting default value in:" + node + " metadata element:<" + this.accessorName + "> expression:'" + this.defaultExpression + "'");

		Object result = JavaInterpreter.execute(node, this.defaultExpression);

		return result == null? null: result.toString();
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return "[MetadataElement:" + this.accessorName + "]";
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
