/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.dmdsec;

import org.dom4j.*;

import ch.docuteam.darc.ead.C;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.tools.file.FileUtil;


/**
 * This class, used by the class <a href="../Document.html">Document</a>, represents the METS Descriptive Metadata Section
 * that contains an <a href="../../ead/C.html">C</a> element within an ead element.
 * <p>
 * This class inserts Exceptions into the <a href="../../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when the number of C elements found in this DMD Section is not exactly 1</li>
 * </ul>
 * @author denis
 */
public class DMDSectionWithEAD extends DMDSectionAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String		EADNamespacePrefix = "EAD";
	static private final String		EADNamespaceURI = "urn:isbn:1-931666-22-9";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private C						c;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	/**
	 * This method is used only when a DMDSectionWithEAD is created programmatically
	 * 	OR if no DMDSectionWithEAD already exists for a node when a METS-file is being read.
	 */
	public DMDSectionWithEAD(NodeAbstract node)
	{
		super(node.getDocument());

		Element mdWrap = this.element.addElement("METS:mdWrap").addAttribute("MDTYPE", "EAD");
		Element xmlData = mdWrap.addElement("METS:xmlData");

		//	Add the required namespace specification "EAD" to the document's root element. If it is already there, do nothing.
		//	NOTE: I have to do this already HERE because when the C element gets created, the namespace must already be defined.
		this.document.getRootElement().add(new Namespace(EADNamespacePrefix, EADNamespaceURI));

		//	Create the intermediary ead element:
		Element dscElement = this.createIntermediaryArchdescElement(xmlData);

		//	Create the c element. If the node is a file, cut away the suffix from the node's label when setting the unitTitle:
		this.c = new C(this, dscElement, node.isFile()? FileUtil.asFileNameWithoutExtension(node.getLabel()): node.getLabel());
	}

	//	========	Constructors Private	=======================================================

	/**
	 * This constructor is used only when a METS-File is being read.
	 * Create the DMDSection out of the given document.
	 */
	protected DMDSectionWithEAD(Document document, Element element)
	{
		super(document, element);

		//	Get the C-Element (there should be exactly one):
		//		NOTE: The C-Element can be in two different places:
		//			either:	./METS:mdWrap/METS:xmlData
		//			or:		./METS:mdWrap/METS:xmlData/EAD:ead/EAD:archdesc/EAD:dsc
		//			The latter is the correct one - if I don't find an EAD:ead element, create it and move the c element into there:

		Element xmlDataElement = (Element)this.element.selectSingleNode("./METS:mdWrap/METS:xmlData");
		if (xmlDataElement.selectSingleNode("./EAD:ead") == null)
		{
			//	So I have to create the intermediary ead element:
			Element cElement = (Element)xmlDataElement.selectSingleNode("./EAD:c");
			if (cElement == null)
			{
				ch.docuteam.tools.exception.Exception.remember("Mandatory EAD:c element missing in dmdSec in path: '" + xmlDataElement.getPath() + "'");
				return;
			}

			Element dscElement = this.createIntermediaryArchdescElement((Element)this.element.selectSingleNode("./METS:mdWrap/METS:xmlData"));

			//	Move the c element from its original position to the just created dsc element:
			cElement.detach();
			dscElement.add(cElement);
		}

		Element parentOfCElement = (Element)this.element.selectSingleNode("./METS:mdWrap/METS:xmlData/EAD:ead/EAD:archdesc/EAD:dsc");
		if (parentOfCElement.selectSingleNode("./EAD:c") == null)
		{
			ch.docuteam.tools.exception.Exception.remember("Mandatory EAD:c element missing in dmdSec in path: '" + parentOfCElement.getPath() + "'");
			return;
		}

		this.c = (C.parse(this, parentOfCElement, 0)).get(0);
	}

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * @return the EAD C:
	 */
	public C getC()
	{
		return this.c;
	}


	/**
	 * Return my dom4j-Element "EAD:ead" which lies inside "METS:mdWrap/METS:xmlData/".
	 * @return
	 */
	public Node getEADElement()
	{
		return this.getElement().selectSingleNode("METS:mdWrap/METS:xmlData/EAD:ead");
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder("\n[DMDSection:" + this.id);
		buf.append("\n\t" + this.c);
		buf.append("\n]");

		return buf.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	/**
	 * Create the intermediary Archdesc element and return the dsc element.
	 */
	private Element createIntermediaryArchdescElement(Element xmlDataElement)
	{
		Element eadElement = xmlDataElement.addElement("EAD:ead");
		Element eadHeader = eadElement.addElement("EAD:eadheader");
		eadHeader.addElement("EAD:eadid");
		eadHeader.addElement("EAD:filedesc").addElement("EAD:titlestmt").addElement("EAD:titleproper");

		Element archdescElement = eadElement.addElement("EAD:archdesc").addAttribute("level", "otherlevel");
		archdescElement.addElement("EAD:did").addElement("EAD:unittitle");

		return archdescElement.addElement("EAD:dsc");
	}

	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
