/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.structmap;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.dom4j.Element;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.exceptions.FileAlreadyExistsException;
import ch.docuteam.darc.exceptions.FileOperationNotAllowedException;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;


/**
 * This class, used by the class <a href="../Document.html">Document</a>, represents the METS StructMap.
 * <p>
 * The structure map is a tree structure, so here I have only the root element of this tree as a property.
 * Temporarily, root may be null (e.g. directly after reading the METS template), but is usually supposed to be an instance of <a href="./NodeAbstract.html">NodeAbstract</a>.
 * <p>
 * This class inserts Exceptions into the <a href="../../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when the number of root elements in the StructMap is greater than 1</li>
 * </ul>
 * @author denis
 */
public class StructureMap extends ch.docuteam.darc.common.NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	/**
	 * The structure map is a tree structure, so here I have only the root element of this tree.
	 * The tree elements are either instances of <a href="./StructureMapNodeFile.html" title="class StructureMapNodeFile">StructureMapNodeFile</a>
	 * or <a href="./StructureMapNodeFolder.html" title="class StructureMapNodeFolder">StructureMapNodeFolder</a>.
	 * Their common abstract superclass is <a href="./StructureMapNodeAbstract.html" title="class StructureMapNodeAbstract">StructureMapNodeAbstract</a>.
	 *
	 * Temporarily, root may be null (e.g. directly after reading the METS template)
	 */
	protected NodeAbstract			root;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	protected StructureMap() {}

	//	========	Constructors Private	=======================================================

	/**
	 * 	This constructor is used only when a METS-File is being read.
	 */
	private StructureMap(Document document)
	{
		this.document = document;
		this.element = (Element)document.selectSingleNode("./METS:mets/METS:structMap");

		//	Consistency check (break if the structure map contains a bad number of root elements):
		if (this.element.elements().size() != 1)
		{
			Exception.remember("Bad number of root elements in the METS file's Structure Map. Expected: 1, found: " + this.element.elements().size());
			this.document = null;
			this.element = null;
			return;
		}

		this.root = NodeAbstract.parse(this, (Element)(this.element.elements().get(0)));
	}

	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read.
	 */
	static public StructureMap parse(Document document)
	{
		return new StructureMap(document);
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public NodeAbstract getRoot()
	{
		return this.root;
	}

	//	--------		Inquiring			-------------------------------------------------------

	/**
	 * Return true if the node being searched for exists in the tree, or null otherwise.
	 */
	public boolean containsId(String id)
	{
		return this.root.searchId(id) != null;
	}


	public void checkIfFileOperationNotAllowed(String fileName, String fileOp) throws FileOperationNotAllowedException
	{
		if (this.document.getDocumentType() == DocumentAbstract.Type.METS
				&& !((Document)this.document).areFileOperationsAllowed())
					throw new FileOperationNotAllowedException(this.document, fileName, fileOp);
	}


	/**
	 * This method name is blazing, right?
	 * @return
	 */
	public boolean hasNodesWithDynamicMetadataElementInstancesWhichAreMandatoryButNotSet()
	{
		for (NodeAbstract n: this.root.getWithDescendants())	if (n.hasDynamicMetadataElementInstancesWhichAreMandatoryButNotSet())	return true;
		return false;
	}


	public boolean checkFixity()
	{
		return this.root.checkFixity();
	}


	public boolean checkAgainstSubmissionAgreement()
	{
		return this.root.checkAgainstSubmissionAgreement();
	}

	public List<NodeFile> filesNotAllowedBySubmissionAgreement()
	{
		return this.root.filesNotAllowedBySubmissionAgreement();
	}


	public boolean checkAgainstSubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		return this.root.checkAgainstSubmissionAgreement(sa, dssId);
	}

	public List<NodeFile> filesNotAllowedBySubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		return this.root.filesNotAllowedBySubmissionAgreement(sa, dssId);
	}

	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Submitting			-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	/**
	 * Create a new root node out of the sourceFilePath, which can point to a file or a folder. Create recursively the tree structure.
	 * @throws FileAlreadyExistsException
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public NodeAbstract createRootNode(String sourceFilePath) throws IOException, FileAlreadyExistsException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(sourceFilePath, "CreateRootNode");

		//	Remove any existing root element including its DMD section (if any):
		if (this.root != null)
		{
			this.root.delete();
			this.root = null;
		}

		//	Clear any possible isAtLeastOneFileNotReadable and isAtLeastOneFileNotWritable flag in document before inserting new nodes:
		this.document.setIsAtLeastOneFileNotReadable(false);
		this.document.setIsAtLeastOneFileNotWritable(false);

		//	Insert new root node recursively:
		return this.root = NodeAbstract.createRootNode(this, new File(sourceFilePath));
	}


	/**
	 * Create a new root folder with the given name. Create corresponding empty folder in the file system.
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public NodeFolder createNewRootFolder(String newRootFolderName) throws IOException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(newRootFolderName, "CreateNewRootFolder");

		//	Remove any existing root element including its DMD section (if any):
		if (this.root != null)
		{
			this.root.delete();
			this.root = null;
		}

		//	Clear any possible isAtLeastOneFileNotReadable and isAtLeastOneFileNotWritable flag in document before inserting new nodes:
		this.document.setIsAtLeastOneFileNotReadable(false);
		this.document.setIsAtLeastOneFileNotWritable(false);

		//	Insert new root node:
		this.root = new NodeFolder(this, newRootFolderName);

		return (NodeFolder)this.root;
	}


	/**
	 * This method moves the root file into a newly created root folder called "Root".
	 * This method is needed only by NodeFolder.migrateToFileKeepOriginal(), therefore the package visibility.
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 * @throws FileAlreadyExistsException
	 */
	NodeFolder moveRootFileToNewRootFolder(NodeFile oldRootFile) throws FileOperationNotAllowedException, IOException, FileUtilExceptionListException, FileAlreadyExistsException
	{
		this.checkIfFileOperationNotAllowed(oldRootFile.getLabel(), "CreateNewRootFolder");

		//	Clear any possible isAtLeastOneFileNotReadable and isAtLeastOneFileNotWritable flag in document before inserting new nodes:
		this.document.setIsAtLeastOneFileNotReadable(false);
		this.document.setIsAtLeastOneFileNotWritable(false);

		//	Insert new root node:
		this.root = new NodeFolder(this, "Root");

		//	Move this node into the new root folder and set its type to "file" (before it was "rootfile"):
		oldRootFile.moveTo((NodeFolder)this.root);
		oldRootFile.setType(NodeFile.Type);

		return (NodeFolder)this.root;
	}

	//	--------		Utilities			-------------------------------------------------------

	/**
	 * Return the node with this amdId, if it exists in the tree, or null.
	 */
	public NodeAbstract searchId(String id)
	{
		return this.root.searchId(id);
	}


	/**
	 * Return the (possibly empty) list of nodes with this label.
	 */
	public List<NodeAbstract> searchLabel(String label)
	{
		return this.root.searchLabel(label);
	}


	/**
	 * Return the node being searched for, if it exists in the tree, or null.
	 */
	public NodeAbstract searchFileId(String fileId)
	{
		return this.root.searchFileId(fileId);
	}


	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder("\n[StructureMap:");
		buf.append((this.root != null)? this.root.treeString(1): "(empty)");
		buf.append("\n]");

		return buf.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
