/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.util;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.util.*;

import org.dom4j.*;
import org.dom4j.io.SAXReader;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.mdconfig.LevelOfDescription;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.mets.structmap.NodeFolder;
import ch.docuteam.tools.exception.ExceptionCollector;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.Zipper;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.out.Tracer;
import ch.docuteam.tools.string.DateFormatter;

import com.google.gson.Gson;

/**
 * Convert a BAR-SIP to a Matterhorn-SIP.
 * @author denis
 *
 */
public abstract class BARSIPReader
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * @param args
	 */
	static public void main(String... args) throws Exception
	{
		String workspace = OperatingSystem.userHome() + "Desktop/Workspace/";
		String sipPath = workspace + "SIPs/";
		String barSipPath = workspace + "BAR-SIPs/";

		//	This example contains 4 files that can't be identified by DROID:
		String sipName = "SIP_20130307_Test1_Lem";
//		String sipName = "SIP_20130326_Test2";
//		String sipName = "SIP_20130402_M. Hess_1";
//		String sipName = "SIP_20070923_RIS_3Dossier";
//		String sipName = "SIP_20070923_RIS_3Dossier_plus";
//		String sipName = "SIP_20130402_Urs_1";
//		String sipName = "SIP_20130424_UVA_2";

//		String sipName = "SIP_20130307_Test1_Lem.zip";
//		String sipName = "SIP_20130326_Test2.zip";
//		String sipName = "SIP_20130402_M. Hess_1.zip";
//		String sipName = "SIP_20070923_RIS_3Dossier.zip";
//		String sipName = "SIP_20070923_RIS_3Dossier_plus.zip";
//		String sipName = "SIP_20130402_Urs_1.zip";
//		String sipName = "SIP_20130424_UVA_2.zip";

		//	I have to use the levels file that defines the BAR-levels:
		LevelOfDescription.setInitializationFilePath("config/levels_BAR.xml");

		Document doc = null;
		try
		{
			//	----------	Import the BAR-SIP:
			FileUtil.delete(sipPath + sipName);
			convertTo(barSipPath + sipName, sipPath + sipName, "BARSIPConverter.main");		//	Create a new Document from an existing BAR-SIP

			//	----------	Create the EAD file:
//			if (sipName.toLowerCase().endsWith(".zip"))
//				doc = Document.openReadOnly(OperatingSystem.userHome() + "/Desktop/SIPs/" + sipName, "BARSIPConverter.main");
//			else
//				doc = Document.openReadOnly(OperatingSystem.userHome() + "/Desktop/SIPs/" + sipName + "/mets.xml", "BARSIPConverter.main");
//			Tracer.trace(doc.toString());
//
//			doc.createEADFile(OperatingSystem.userHome() + "/Desktop/EADs/" + sipName + "_EAD.xml");
//			if (!ExceptionCollector.isEmpty())	Tracer.trace(ExceptionCollector.toStringAll());
		}
		catch (Throwable x)
		{
			x.printStackTrace();
		}
		finally
		{
			if (doc != null)					doc.cleanupWorkingCopy();
		}

		if (!ExceptionCollector.isEmpty())		Tracer.trace(ExceptionCollector.toStringAll());
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	static public void convertTo(String barSIPPath, String newZIPOrFolderFilePath) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, DocumentException, FileUtilExceptionListException, DocumentIsReadOnlyException, java.lang.Exception
	{
		convertTo(barSIPPath, newZIPOrFolderFilePath, "");
	}

	static public void convertTo(String barSIPPath, String newZIPOrFolderFilePath, String operatorName) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, DocumentException, FileUtilExceptionListException, DocumentIsReadOnlyException, java.lang.Exception
	{
		convertTo(barSIPPath, newZIPOrFolderFilePath, operatorName, null);
	}

	static public void convertTo(String barSIPPath, String newZIPOrFolderFilePath, String operatorName, Observer observer) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, DocumentException, FileUtilExceptionListException, DocumentIsReadOnlyException, java.lang.Exception
	{
		BARSIPDocument.convertTo(barSIPPath, newZIPOrFolderFilePath, operatorName, observer);
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	static private class BARSIPDocument extends DocumentAbstract
	{
		//	===========================================================================================
		//	========	Structure				=======================================================
		//	===========================================================================================

		//	========	Static Final Public		=======================================================

		//	========	Static Final Private	=======================================================

		static private final String				BARNamespace = "http://bar.admin.ch/arelda/v4";
		static private final String				RelativePathToMetadata_xml = "/header/metadata.xml";

		//	========	Static Public			=======================================================

		//	========	Static Private			=======================================================

		//	========	Instance Public			=======================================================

		//	========	Instance Private		=======================================================

		//	===========================================================================================
		//	========	Main					=======================================================
		//	===========================================================================================

		//	===========================================================================================
		//	========	Methods					=======================================================
		//	===========================================================================================

		//	========	Static Initializer		=======================================================

		//	========	Constructors Public		=======================================================

		//	========	Constructors Private	=======================================================

		static private BARSIPDocument read(String barSIPPath) throws DocumentException
		{
			SAXReader reader = new SAXReader();
			reader.setDocumentFactory(new BARSIPFactory());
			BARSIPDocument barSIP = (BARSIPDocument)reader.read(new File(barSIPPath + RelativePathToMetadata_xml));
			barSIP.filePath = barSIPPath;

			return barSIP;
		}

		//	========	Static Public			=======================================================

		//	--------		Initializing		-------------------------------------------------------
		//	--------		Accessing			-------------------------------------------------------
		//	--------		Inquiring			-------------------------------------------------------
		//	--------		Business Ops		-------------------------------------------------------

		static private void convertTo(String barSIPPath, String newZIPOrFolderFilePath, String operatorName, Observer observer) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, DocumentException, FileUtilExceptionListException, DocumentIsReadOnlyException, java.lang.Exception
		{
			if (!new File(barSIPPath).exists())		throw new FileNotFoundException(barSIPPath);

			Document document = null;
			BARSIPDocument barSIP = null;
			boolean isZIP = false;

			try
			{
				if (barSIPPath.toLowerCase().endsWith(".zip"))
				{
					isZIP = true;

					//	Unzip into temporary folder:
					String barSIPName = (new File(barSIPPath).getName());
					barSIPName = barSIPName.substring(0, barSIPName.length() - 4);
					String workingFolder = FileUtil.getTempFolder() + "/" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);

					Zipper.unzip(barSIPPath, workingFolder);

					//	Set the (method-local) barSIPPath to the temporary working path:
					//	Does this zip contain a folder with the BARSIP or the two folders "content" and "header"?
					List<String> zipContents = Arrays.asList(new File(workingFolder).list());
					if (zipContents.contains("content") && zipContents.contains("header"))
						barSIPPath = workingFolder;
					else
						barSIPPath = workingFolder + "/" + barSIPName;
				}

				document = Document.createNewWithRootFolderName(newZIPOrFolderFilePath, "Paket", "sa_all-formats-01", "dss-01", operatorName, observer);

				barSIP = read(barSIPPath);
				barSIP.insertPaketInto((NodeFolder)document.getStructureMap().getRoot());

				//	Save SIP to original folder:
				document.saveWithoutBackup();
			}
			finally
			{
				if (isZIP)
				{
					//	Cleanup working folder:
					FileUtil.deleteOnExit(new File(barSIPPath).getParentFile());
				}

				if (document != null)
				{
					//	Cleanup and remove lock-file:
					document.unlockIfNecessary();
					document.cleanupWorkingCopy();
				}
			}
		}

		//	--------		Persistence			-------------------------------------------------------
		//	--------		Support				-------------------------------------------------------
		//	--------		Utilities			-------------------------------------------------------
		//	---------		Misc				-------------------------------------------------------
		//	--------		Debugging			-------------------------------------------------------
		//	---------		Temporary			-------------------------------------------------------

		//	========	Static Private			=======================================================

		//	--------		Initializing		-------------------------------------------------------
		//	--------		Accessing			-------------------------------------------------------
		//	--------		Inquiring			-------------------------------------------------------
		//	--------		Business Ops		-------------------------------------------------------
		//	--------		Persistence			-------------------------------------------------------
		//	--------		Support				-------------------------------------------------------

		/**
		 * Return the text of the node denoted by xPath relative to node.
		 * If node or the resulting xPath node is null, return null; otherwise return the node's text.
		 */
		static private String getText(Node node, String xPath)
		{
			if (node == null)		return null;

			Node subNode = node.selectSingleNode(xPath);
			if (subNode == null)	return null;

			return subNode.getText();
		}


		static private String getAsFileName(Node node, String xPath)
		{
			return FileUtil.asSafeFileName(getText(node, xPath));
		}


		static private String getZusatzDaten(Node node, String xPath)
		{
			if (node == null)		return null;

			Node subNode = node.selectSingleNode(xPath);
			if (subNode == null)	return null;

			Map<String, String> map = new TreeMap<String, String>();
			for (Object o: subNode.selectNodes("BAR:merkmal"))
			{
				Element itemElement = (Element)o;
				map.put(itemElement.attributeValue("name"), itemElement.getText());
			}

			return new Gson().toJson(map);
		}

		//	--------		Utilities			-------------------------------------------------------
		//	---------		Misc				-------------------------------------------------------
		//	--------		Debugging			-------------------------------------------------------
		//	---------		Temporary			-------------------------------------------------------

		//	========	Instance Public			=======================================================

		//	--------		Initializing		-------------------------------------------------------
		//	--------		Accessing			-------------------------------------------------------
		//	--------		Inquiring			-------------------------------------------------------
		//	--------		Interface			-------------------------------------------------------

		/**
		 * Return the type of this document: METS, EAD, BAR, ...
		 */
		@Override
		public Type getDocumentType()
		{
			return Type.BAR;
		}

		//	--------		Actions				-------------------------------------------------------
		//	--------		Business Ops		-------------------------------------------------------
		//	--------		Persistence			-------------------------------------------------------
		//	--------		Support				-------------------------------------------------------
		//	--------		Utilities			-------------------------------------------------------
		//	---------		Misc				-------------------------------------------------------
		//	--------		Debugging			-------------------------------------------------------
		//	---------		Temporary			-------------------------------------------------------

		//	========	Instance Private		=======================================================

		//	--------		Initializing		-------------------------------------------------------
		//	--------		Accessing			-------------------------------------------------------
		//	--------		Inquiring			-------------------------------------------------------
		//	--------		Interface			-------------------------------------------------------
		//	--------		Actions				-------------------------------------------------------
		//	--------		Business Ops		-------------------------------------------------------

		private void insertPaketInto(NodeFolder rootFolder) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException
		{
			Node paketNode = this.getRootElement();

			Logger.debug("Inserting Paket");

			rootFolder.setLevel(this.levels.get("Paket"));

			//	Fill in data into rootFolder from rootNode:
			rootFolder.setDynamicMetadataValueForName("usage", getZusatzDaten(paketNode, "BAR:zusatzDaten"));		//	Key-Value Structure, converted to JSON-Format.

			this.insertAblieferungInto(rootFolder, paketNode);
		}


		private void insertAblieferungInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException
		{
			Node ablieferungNode = parentNode.selectSingleNode("BAR:ablieferung");

			Logger.debug("Inserting Ablieferung");

			NodeFolder ablieferungFolder = parentFolder.createNewFolder("Ablieferung");
			ablieferungFolder.setLevel(this.levels.get("Ablieferung"));

			//	Fill in data into ablieferungFolder from ablieferungNode:
			ablieferungFolder.setDynamicMetadataValueForName("accessNr", getText(ablieferungNode, "BAR:ablieferndeStelle"));
			ablieferungFolder.setDynamicMetadataValueForName("refCode", getText(ablieferungNode, "BAR:ablieferungsnummer"));
			ablieferungFolder.setDynamicMetadataValueForName("objectType", getText(ablieferungNode, "BAR:ablieferungstyp"));
			ablieferungFolder.setDynamicMetadataValueForName("material", getText(ablieferungNode, "BAR:ablieferungsteile"));
			ablieferungFolder.setDynamicMetadataValueForName("refCodeAdmin", getText(ablieferungNode, "BAR:angebotsnummer"));
			ablieferungFolder.setDynamicMetadataValueForName("fromYear", getText(ablieferungNode, "BAR:entstehungszeitraum/BAR:von/BAR:datum"));
			ablieferungFolder.setDynamicMetadataValueForName("toYear", getText(ablieferungNode, "BAR:entstehungszeitraum/BAR:bis/BAR:datum"));
			ablieferungFolder.setDynamicMetadataValueForName("comment", getText(ablieferungNode, "BAR:bemerkung"));
			ablieferungFolder.setDynamicMetadataValueForName("appraisalAndDestruction", getText(ablieferungNode, "BAR:referenzBewertungsentscheid"));
			ablieferungFolder.setDynamicMetadataValueForName("accessRestriction", getText(ablieferungNode, "BAR:referenzSchutzfristenFormular"));
			ablieferungFolder.setDynamicMetadataValueForName("accessPolicy", getText(ablieferungNode, "BAR:schutzfristenkategorie"));
			ablieferungFolder.setDynamicMetadataValueForName("accessRestrictionPeriod", getText(ablieferungNode, "BAR:schutzfrist"));
			ablieferungFolder.setDynamicMetadataValueForName("origination", getText(ablieferungNode, "BAR:provenienz/BAR:aktenbildnerName"));

			//	Dive in:
			this.insertOrdnungssystemInto(ablieferungFolder, ablieferungNode);
		}


		private void insertOrdnungssystemInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException
		{
			Node osNode = parentNode.selectSingleNode("BAR:ordnungssystem");
			String label = getAsFileName(osNode, "BAR:name");

			Logger.debug("Inserting Ordnungssystem " + label);

			NodeFolder osFolder = parentFolder.createNewFolder(label, true);
			osFolder.setLevel(this.levels.get("Ordnungssystem"));

			//	Fill in data into osFolder from osNode:
			osFolder.setDynamicMetadataValueForName("refCode", getText(osNode, "BAR:generation"));
			osFolder.setDynamicMetadataValueForName("fromYear", getText(osNode, "BAR:anwendungszeitraum/BAR:von/BAR:datum"));
			osFolder.setDynamicMetadataValueForName("toYear", getText(osNode, "BAR:anwendungszeitraum/BAR:bis/BAR:datum"));
			osFolder.setDynamicMetadataValueForName("involved", getText(osNode, "BAR:mitbenutzung"));
			osFolder.setDynamicMetadataValueForName("comment", getText(osNode, "BAR:bemerkung"));

			//	Dive in:
			this.insertOrdnungssystempositionenInto(osFolder, osNode);
		}


		private void insertOrdnungssystempositionenInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException
		{
			List<?> osPositions = parentNode.selectNodes("BAR:ordnungssystemposition");
			for (Object o: osPositions)
			{
				Node osPosNode = (Node)o;
				String label = getAsFileName(osPosNode, "BAR:titel");

				Logger.debug("Inserting Ordnungssystemposition " + label);

				NodeFolder osPosFolder = parentFolder.createNewFolder(label, true);
				osPosFolder.setLevel(this.levels.get("Ordnungssystemposition"));

				//	Fill in data into osPosFolder from osPosNode:
				osPosFolder.setDynamicMetadataValueForName("refCode", getText(osPosNode, "BAR:nummer"));
				osPosFolder.setDynamicMetadataValueForName("refCodeAdmin", ((Element)osPosNode).attributeValue("id"));
				osPosFolder.setDynamicMetadataValueForName("origination", getText(osPosNode, "BAR:federfuehrendeOrganisationseinheit"));
				osPosFolder.setDynamicMetadataValueForName("accessPolicy", getText(osPosNode, "BAR:schutzfristenkategorie"));
				osPosFolder.setDynamicMetadataValueForName("accessRestrictionPeriod", getText(osPosNode, "BAR:schutzfrist"));
				osPosFolder.setDynamicMetadataValueForName("accessRestrictionExplanation", getText(osPosNode, "BAR:schutzfristenBegruendung"));
				osPosFolder.setDynamicMetadataValueForName("accessRestrictionClassification", getText(osPosNode, "BAR:klassifizierungskategorie"));
				osPosFolder.setDynamicMetadataValueForName("accessRestrictionPrivacy", getText(osPosNode, "BAR:datenschutz"));
				osPosFolder.setDynamicMetadataValueForName("accessRestrictionStatus", getText(osPosNode, "BAR:oeffentlichkeitsstatus"));
				osPosFolder.setDynamicMetadataValueForName("accessRestrictionStatusExplanation", getText(osPosNode, "BAR:oeffentlichkeitsstatusBegruendung"));
				osPosFolder.setDynamicMetadataValueForName("accessRestriction", getText(osPosNode, "BAR:sonstigeBestimmungen"));

				//	Dive in:
				this.insertOrdnungssystempositionenInto(osPosFolder, osPosNode);		//	Recursion
				this.insertDossiersInto(osPosFolder, osPosNode);
			}
		}


		private void insertDossiersInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException
		{
			List<?> dossiers = parentNode.selectNodes("BAR:dossier");
			for (Object o: dossiers)
			{
				Node dossierNode = (Node)o;
				String label = getAsFileName(dossierNode, "BAR:titel");

				Logger.debug("Inserting Dossier " + label);

				NodeFolder dossierFolder = parentFolder.createNewFolder(label, true);
				dossierFolder.setLevel(this.levels.get("Dossier"));

				//	Fill in data into dossierFolder from dossierNode:
				dossierFolder.setDynamicMetadataValueForName("refCode", getText(dossierNode, "BAR:aktenzeichen"));
				dossierFolder.setDynamicMetadataValueForName("archivalHistory", getText(dossierNode, "BAR:zusatzmerkmal"));
				dossierFolder.setDynamicMetadataValueForName("abstract", getText(dossierNode, "BAR:inhalt"));
				dossierFolder.setDynamicMetadataValueForName("refCodeAdmin", ((Element)dossierNode).attributeValue("id"));
				dossierFolder.setDynamicMetadataValueForName("characteristics", getText(dossierNode, "BAR:erscheinungsform"));
				dossierFolder.setDynamicMetadataValueForName("scopeContent", getText(dossierNode, "BAR:formInhalt"));
				dossierFolder.setDynamicMetadataValueForName("material", getText(dossierNode, "BAR:umfang"));
				dossierFolder.setDynamicMetadataValueForName("origination", getText(dossierNode, "BAR:federfuehrendeOrganisationseinheit"));
				dossierFolder.setDynamicMetadataValueForName("fromYear", getText(dossierNode, "BAR:entstehungszeitraum/BAR:von/BAR:datum"));
				dossierFolder.setDynamicMetadataValueForName("toYear", getText(dossierNode, "BAR:entstehungszeitraum/BAR:bis/BAR:datum"));
				dossierFolder.setDynamicMetadataValueForName("creationPeriodNotes", getText(dossierNode, "BAR:entstehungszeitraumAnmerkung"));
				dossierFolder.setDynamicMetadataValueForName("comment", getText(dossierNode, "BAR:bemerkung"));
				dossierFolder.setDynamicMetadataValueForName("accessPolicy", getText(dossierNode, "BAR:schutzfristenkategorie"));
				dossierFolder.setDynamicMetadataValueForName("accessRestrictionPeriod", getText(dossierNode, "BAR:schutzfrist"));
				dossierFolder.setDynamicMetadataValueForName("accessRestrictionExplanation", getText(dossierNode, "BAR:schutzfristenBegruendung"));
				dossierFolder.setDynamicMetadataValueForName("accessRestrictionClassification", getText(dossierNode, "BAR:klassifizierungskategorie"));
				dossierFolder.setDynamicMetadataValueForName("accessRestrictionPrivacy", getText(dossierNode, "BAR:datenschutz"));
				dossierFolder.setDynamicMetadataValueForName("accessRestrictionStatus", getText(dossierNode, "BAR:oeffentlichkeitsstatus"));
				dossierFolder.setDynamicMetadataValueForName("accessRestrictionStatusExplanation", getText(dossierNode, "BAR:oeffentlichkeitsstatusBegruendung"));
				dossierFolder.setDynamicMetadataValueForName("accessRestriction", getText(dossierNode, "BAR:sonstigeBestimmungen"));

				//	Dive in:
				this.insertDossiersInto(dossierFolder, dossierNode);					//	Recursion
				this.insertDocumentsInto(dossierFolder, dossierNode);
				this.insertDateiRefsInto(dossierFolder, dossierNode);
			}
		}


		private void insertDocumentsInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException
		{
			List<?> documents = parentNode.selectNodes("BAR:dokument");
			for (Object o: documents)
			{
				Node documentNode = (Node)o;
				String label = getAsFileName(documentNode, "BAR:titel");

				Logger.debug("Inserting Document " + label);

				NodeFolder documentFolder = parentFolder.createNewFolder(label, true);
				documentFolder.setLevel(this.levels.get("Dokument"));

				//	Fill in data into documentFolder from documentNode:
				documentFolder.setDynamicMetadataValueForName("refCodeAdmin", ((Element)documentNode).attributeValue("id"));
				documentFolder.setDynamicMetadataValueForName("characteristics", getText(documentNode, "BAR:erscheinungsform"));
				documentFolder.setDynamicMetadataValueForName("scopeContent", getText(documentNode, "BAR:dokumenttyp"));
				documentFolder.setDynamicMetadataValueForName("origination", getText(documentNode, "BAR:autor"));
				documentFolder.setDynamicMetadataValueForName("fromYear", getText(documentNode, "BAR:entstehungszeitraum/BAR:von/BAR:datum"));
				documentFolder.setDynamicMetadataValueForName("toYear", getText(documentNode, "BAR:entstehungszeitraum/BAR:bis/BAR:datum"));
				documentFolder.setDynamicMetadataValueForName("creationPeriod", getText(documentNode, "BAR:registrierdatum/BAR:datum"));
				documentFolder.setDynamicMetadataValueForName("comment", getText(documentNode, "BAR:bemerkung"));
				documentFolder.setDynamicMetadataValueForName("accessRestrictionClassification", getText(documentNode, "BAR:klassifizierungskategorie"));
				documentFolder.setDynamicMetadataValueForName("accessRestrictionPrivacy", getText(documentNode, "BAR:datenschutz"));
				documentFolder.setDynamicMetadataValueForName("accessRestrictionStatus", getText(documentNode, "BAR:oeffentlichkeitsstatus"));
				documentFolder.setDynamicMetadataValueForName("accessRestrictionStatusExplanation", getText(documentNode, "BAR:oeffentlichkeitsstatusBegruendung"));
				documentFolder.setDynamicMetadataValueForName("accessRestriction", getText(documentNode, "BAR:sonstigeBestimmungen"));

				//	Dive in:
				this.insertDateiRefsInto(documentFolder, documentNode);
			}
		}


		private void insertDateiRefsInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, FileUtilExceptionListException
		{
			List<?> fileRefs = parentNode.selectNodes("BAR:dateiRef");
			for (Object o: fileRefs)
			{
				Node fileRefNode = (Node)o;
				String relativeFilePath = this.getFilePath(fileRefNode);

				Logger.debug("Inserting FileRef " + relativeFilePath);

				//	Insert file or folder, rename if necessary:
				NodeAbstract fileRefFile = parentFolder.insertFileOrFolder(this.filePath + relativeFilePath, true);

				fileRefFile.setLevel(this.levels.get("Datei"));

				//	Fill in data into fileRefFile from fileRefNode:
				//		(none)
			}
		}

		//	--------		Persistence			-------------------------------------------------------
		//	--------		Support				-------------------------------------------------------

		private String getFilePath(Node fileRefNode)
		{
			String fileRefId = fileRefNode.getText();
			Node fileNode = this.getRootElement().selectSingleNode("BAR:inhaltsverzeichnis//BAR:datei [@id='" + fileRefId + "']");

			Node parent = fileNode;
			String relativeFilePath = "";
			do
			{
				relativeFilePath = "/" + parent.selectSingleNode("BAR:name").getText() + relativeFilePath;
				parent = parent.getParent();
			}
			while (!(parent.selectSingleNode("BAR:name") == null));

			return relativeFilePath;
		}

		//	--------		Utilities			-------------------------------------------------------
		//	---------		Misc				-------------------------------------------------------
		//	--------		Debugging			-------------------------------------------------------
		//	---------		Temporary			-------------------------------------------------------

		//	===========================================================================================
		//	========	Inner Classes			=======================================================
		//	===========================================================================================

		/**
		 * This Factory sets the default namespace for the BAR-SIP Document.
		 */
		static private class BARSIPFactory extends DocumentFactory
		{
			@Override
			public org.dom4j.Document createDocument()
			{
				Map<String, String> namespaces = new TreeMap<String, String>();
				namespaces.put("BAR", BARNamespace);
				this.setXPathNamespaceURIs(namespaces);

				BARSIPDocument barSIP = new BARSIPDocument();
				barSIP.setDocumentFactory(this);

				return barSIP;
			}
		}
	}

}
