/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.util;

import java.io.File;
import java.io.IOException;
import java.util.*;

import org.dom4j.*;
import org.dom4j.io.*;
import org.dom4j.tree.DefaultDocument;
import org.jdesktop.swingx.treetable.MutableTreeTableNode;

import ch.docuteam.darc.mdconfig.LevelOfDescription;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.structmap.*;
import ch.docuteam.tools.exception.ExceptionCollector;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.out.Tracer;
import ch.docuteam.tools.string.DateFormatter;
import ch.docuteam.tools.string.StringUtil;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

/**
 * @author denis
 *
 */
public abstract class BARSIPWriter
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String				RelativePathToMetadata_xml = "/header/metadata.xml";
	static private final String				RelativePathToContent = "/content";

	static private final String				BARFileNamePrefix = "SIP_" + DateFormatter.getCurrentDateTimeString("yyyyMMdd") + "_";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * @param args
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 */
	public static void main(String[] args) throws IOException, FileUtilExceptionListException
	{
		String workspace = OperatingSystem.userHome() + "Desktop/Workspace/";
		String sipPath = workspace + "SIPs/";
		String barSipPath = workspace + "BAR-SIPs/";

		//	I have to use the levels file that defines the BAR-levels:
		LevelOfDescription.setInitializationFilePath("config/levels_BAR.xml");

		Document doc = null;
		try
		{
			String sipName = "SIP_20130307_Test1_Lem";

			//	ToDo:	----------	Create the BAR-SIP:
			convertTo(sipPath + sipName, barSipPath + BARFileNamePrefix + sipName, "BARSIPWriter.main");		//	Create a new Document from an existing BAR-SIP
		}
		catch (Throwable x)
		{
			x.printStackTrace();
		}
		finally
		{
			if (doc != null)		doc.cleanupWorkingCopy();
		}

		if (!ExceptionCollector.isEmpty())		Tracer.trace(ExceptionCollector.toStringAll());
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	static public void convertTo(String documentPath, String barSIPPath, String operatorName) throws Exception
	{
		if (new File(barSIPPath).exists())		FileUtil.delete(barSIPPath);

		Logger.info("Creating BAR-SIP: '" + barSIPPath + "'...");

		createFolderStructure(barSIPPath);

		SAXReader reader = new SAXReader();
		reader.setDocumentFactory(new BARSIPFactory());
		org.dom4j.Document barSIPDocument = reader.read(new File(barSIPPath + RelativePathToMetadata_xml));

		Document doc = Document.openReadOnly(documentPath, operatorName);

		Node insertPointAblieferungNode = barSIPDocument.selectSingleNode("/BAR:paket");
		Node insertPointInhaltsverzeichnisNode = barSIPDocument.selectSingleNode("/BAR:paket/BAR:inhaltsverzeichnis/BAR:ordner/BAR:name[text()='content']/..");
		File insertPointFolder = new File(barSIPPath + RelativePathToContent);
		insertRecursively(doc.getStructureMap().getRoot(), barSIPDocument, (Element)insertPointAblieferungNode, (Element)insertPointInhaltsverzeichnisNode, insertPointFolder);

		java.io.Writer oswriter = new java.io.OutputStreamWriter(new java.io.FileOutputStream(barSIPPath + RelativePathToMetadata_xml), "utf-8");
		XMLWriter writer = new XMLWriter(oswriter, new OutputFormat("	", false, "utf-8"));
		writer.write(barSIPDocument);
		writer.close();

		Logger.info("Creating BAR-SIP: '" + barSIPPath + "'... done!");
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	static private void createFolderStructure(String path) throws IOException, FileUtilExceptionListException
	{
		FileUtil.copyToOverwriting("resources/templates/BARSIP", path);
	}


	static private void insertRecursively(NodeAbstract node, org.dom4j.Document barSIPDocument, Element insertPointAblieferungElement, Element insertPointInhaltsverzeichnisElement, File insertPointFolder) throws IOException, FileUtilExceptionListException
	{
		String level = node.getLevel().getName().toLowerCase();
		String fileName = node.getFile().getName();
		Element newAblieferungElement = null;

		Logger.debug("Inserting node: '" + node.getLabel() + "' with level: '" + level + "'");

		if (level.equals("paket"))							newAblieferungElement = fillPaket(insertPointAblieferungElement, node);
		else if (level.equals("ablieferung"))				newAblieferungElement = createAblieferung(insertPointAblieferungElement, node);
		else if (level.equals("ordnungssystem"))			newAblieferungElement = createOrdnungssystem(insertPointAblieferungElement, node);
		else if (level.equals("ordnungssystemposition"))	newAblieferungElement = createOrdnungssystemposition(insertPointAblieferungElement, node);
		else if (level.equals("dossier"))					newAblieferungElement = createDossier(insertPointAblieferungElement, node);
		else if (level.equals("dokument"))					newAblieferungElement = createDokument(insertPointAblieferungElement, node);
		else if (level.equals("datei"))						newAblieferungElement = insertPointAblieferungElement;		//	Dont create any new ablieferungselement, just continue
		else if (level.equals("undefiniert"))				;			//	ToDo
		else												;			//	ToDo

		if (newAblieferungElement == null)					return;		//	ToDo

		//	Distinguish between file and folder:
		if (node.isFile())
		{
			NodeFile fileNode = (NodeFile)node;

			//	Insert new inhaltsverzeichnisElement for a file:
			String id = getNextDateiId();
			createInhaltsverzeichnisElement(insertPointInhaltsverzeichnisElement, fileNode, id);

			//	Create a dateiRef to this inhaltsverzeichnisElement in the ablieferungElement:
			newAblieferungElement.addElement("BAR:dateiRef").addText(id);

			//	Copy file:
			try
			{
				FileUtil.copyToFolderOverwriting(node.getFile(), insertPointFolder, false);
			}
			catch (IOException ex)
			{
				ex.printStackTrace();
			}
			catch (FileUtilExceptionListException ex)
			{
				ex.printStackTrace();
			}
		}
		else //	Node is folder:
		{
			NodeFolder folderNode = (NodeFolder)node;

			//	Insert new inhaltsverzeichnisElement for a folder:
			Element newInhaltsverzeichnisElement = createInhaltsverzeichnisElement(insertPointInhaltsverzeichnisElement, folderNode);

			//	Create new folder:
			File newFolder = FileUtil.createFolderOverwriting(fileName, insertPointFolder);

			//	Dive in (recursion!):
			for (MutableTreeTableNode child: folderNode.getChildren())
				insertRecursively((NodeAbstract)child, barSIPDocument, newAblieferungElement, newInhaltsverzeichnisElement, newFolder);
		}
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	static private Element fillPaket(Element paketElement, NodeAbstract node)
	{
		Logger.debug("Filling paket from: '" + node.getLabel() + "'");

		createZusatzDatenFromDynamicMetadata(paketElement, node, "usage");

		return paketElement;
	}


	static private Element createInhaltsverzeichnisElement(Element insertPointInhaltsverzeichnisElement, NodeFile fileNode, String id)
	{
		String fileName = fileNode.getFile().getName();

		Element newInhaltsverzeichnisElement = insertPointInhaltsverzeichnisElement.addElement("BAR:datei");

		setAttribute(newInhaltsverzeichnisElement, "id", id);
		addElementAndSetText(newInhaltsverzeichnisElement, "BAR:name", fileName);
		addElementAndSetText(newInhaltsverzeichnisElement, "BAR:originalName", fileName);
		addElementAndSetText(newInhaltsverzeichnisElement, "BAR:pruefalgorithmus", fileNode.getChecksumType());
		addElementAndSetText(newInhaltsverzeichnisElement, "BAR:pruefsumme", fileNode.getChecksum());

		addEigenschaft(newInhaltsverzeichnisElement, "test", "Sonstiges");		//	ToDo: Loop through other dynamic metadata?

		return newInhaltsverzeichnisElement;
	}


	static private Element createInhaltsverzeichnisElement(Element insertPointInhaltsverzeichnisElement, NodeFolder folderNode)
	{
		String fileName = folderNode.getFile().getName();

		Element newInhaltsverzeichnisElement = insertPointInhaltsverzeichnisElement.addElement("BAR:ordner");

		addElementAndSetText(newInhaltsverzeichnisElement, "BAR:name", fileName);
		addElementAndSetText(newInhaltsverzeichnisElement, "BAR:originalName", fileName);

		return newInhaltsverzeichnisElement;
	}


	static private Element createAblieferung(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.debug("Creating ablieferung from: '" + node.getLabel() + "'");

		Element ablieferungElement = insertPointAblieferungElement.addElement("BAR:ablieferung");

		setAttribute(ablieferungElement, "xsi:type", "BAR:ablieferungFilesSIP");
//		addElementAndSetText(ablieferungElement, "BAR:ablieferungstyp", "FILES");		//	ToDo: This or node.objectType (see below)???
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferungstyp", node, "objectType");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferndeStelle", node, "accessNr");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:entstehungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:entstehungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferungsteile", node, "material");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:bemerkung", node, "comment");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferungsnummer", node, "refCode");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:angebotsnummer", node, "refCodeAdmin");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:referenzBewertungsentscheid", node, "appraisalAndDestruction");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:referenzSchutzfristenFormular", node, "accessRestriction");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:schutzfristenkategorie", node, "accessPolicy");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:schutzfrist", node, "accessRestrictionPeriod");
		addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:provenienz/BAR:aktenbildnerName", node, "origination");

		return ablieferungElement;
	}


	static private Element createOrdnungssystem(Element insertPointInhaltsverzeichnisElement, NodeAbstract node)
	{
		Logger.debug("Creating ordnungssystem from: '" + node.getLabel() + "'");

		Element ordnungssystemElement = insertPointInhaltsverzeichnisElement.addElement("BAR:ordnungssystem");

		addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:generation", node, "refCode");
		addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:anwendungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:anwendungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:mitbenutzung", node, "involved");
		addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:bemerkung", node, "comment");

		return ordnungssystemElement;
	}


	static private Element createOrdnungssystemposition(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.debug("Creating ordnungssystemposition from: '" + node.getLabel() + "'");

		Element newOrdnungssystempositionElement = insertPointAblieferungElement.addElement("BAR:ordnungssystemposition");

		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:nummer", node, "refCode");
		setAttributeFromDynamicMetadata(newOrdnungssystempositionElement, "id", node, "refCodeAdmin");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:federfuehrendeOrganisationseinheit", node, "origination");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:schutzfristenkategorie", node, "accessPolicy");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:schutzfrist", node, "accessRestrictionPeriod");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:schutzfristenBegruendung", node, "accessRestrictionExplanation");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:klassifizierungskategorie", node, "accessRestrictionClassification");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:datenschutz", node, "accessRestrictionPrivacy");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:oeffentlichkeitsstatus", node, "accessRestrictionStatus");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:oeffentlichkeitsstatusBegruendung", node, "accessRestrictionStatusExplanation");
		addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:sonstigeBestimmungen", node, "accessRestriction");

		return newOrdnungssystempositionElement;
	}


	static private Element createDossier(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.debug("Creating dossier from: '" + node.getLabel() + "'");

		Element newDossierElement = insertPointAblieferungElement.addElement("BAR:dossier");

		setAttributeFromDynamicMetadata(newDossierElement, "id", node, "refCodeAdmin");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:zusatzmaterial", node, "archivalHistory");
		addElementAndSetText(newDossierElement, "BAR:titel", node.getUnitTitle());
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:inhalt", node, "abstract");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:formInhalt", node, "scopeContent");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:erscheinungsform", node, "characteristics");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:federführendeOrganisationseinheit", node, "origination");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:entstehungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:entstehungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:entstehungszeitraumAnmerkung", node, "creationPeriodNotes");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:klassifizierungskategorie", node, "accessRestrictionClassification");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:datenschutz", node, "accessRestrictionPrivacy");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:oeffentlichkeitsstatus", node, "accessRestrictionStatus");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:oeffentlichkeitsstatusBegruendung", node, "accessRestrictionStatusExplanation");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:sonstigeBestimmungen", node, "accessRestriction");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:bemerkung", node, "comment");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:aktenzeichen", node, "refCode");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:schutzfristenkategorie", node, "accessPolicy");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:schutzfrist", node, "accessRestrictionPeriod");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:schutzfristenBegruendung", node, "accessRestrictionExplanation");
		addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:umfang", node, "material");

		return newDossierElement;
	}


	static private Element createDokument(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.debug("Creating dokument from: '" + node.getLabel() + "'");

		Element newDokumentElement = insertPointAblieferungElement.addElement("BAR:dokument");

		addElementAndSetText(newDokumentElement, "BAR:titel", node.getLabel());
		setAttributeFromDynamicMetadata(newDokumentElement, "id", node, "refCodeAdmin");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:erscheinungsform", node, "characteristics");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:dokumenttyp", node, "scopeContent");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:autor", node, "origination");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:entstehungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:entstehungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:registrierdatum/BAR:datum", node, "creationPeriod");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:bemerkung", node, "comment");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:klassifizierungskategorie", node, "accessRestrictionClassification");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:datenschutz", node, "accessRestrictionPrivacy");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:oeffentlichkeitsstatus", node, "accessRestrictionStatus");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:oeffentlichkeitsstatusBegruendung", node, "accessRestrictionStatusExplanation");
		addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:sonstigeBestimmungen", node, "accessRestriction");

		return newDokumentElement;
	}

	//	--------		Utilities - Basic	-------------------------------------------------------

	static private int		DateiIdCounter = 0;

	static private String getNextDateiId()
	{
		return "d" + StringUtil.last("000000" + DateiIdCounter++, 6);
	}


	static private void addElementAndSetTextFromDynamicMetadata(Element insertPoint, String newElementName, NodeAbstract node, String metadataName)
	{
		try
		{
			String text = node.getDynamicMetadataValueForName(metadataName);
			addElementAndSetText(insertPoint, newElementName, text);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	static private void addElementAndSetText(Element insertPoint, String newElementName, String text)
	{
		if (text == null)		return;

		XMLUtil.add(insertPoint, newElementName, text);
	}


	static private void addEigenschaft(Element insertPoint, String eigenschaftName, String text)
	{
		Element newElement = XMLUtil.add(insertPoint, "BAR:eigenschaft", text);
		setAttribute(newElement, "name", eigenschaftName);
	}


	static private void setAttributeFromDynamicMetadata(Element insertPoint, String attributeName, NodeAbstract node, String metadataName)
	{
		try
		{
			String text = node.getDynamicMetadataValueForName(metadataName);
			setAttribute(insertPoint, attributeName, text);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	static private void setAttribute(Element insertPoint, String attributeName, String text)
	{
		if (text == null)		return;

		insertPoint.addAttribute(attributeName, text);
	}


	@SuppressWarnings("unchecked")
	static private void createZusatzDatenFromDynamicMetadata(Element insertPoint, NodeAbstract node, String metadataName)
	{
		String jsonString = "";
		try
		{
			jsonString = node.getDynamicMetadataValueForName(metadataName);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			return;
		}

		Map<String, String> map;
		try
		{
			map = new Gson().fromJson(jsonString, Map.class);
		}
		catch (JsonSyntaxException ex)
		{
			//	In the case the string is not in JSON-format, put the complete string as the content:
			map = new HashMap<String, String>();
			map.put(metadataName, jsonString);		//	ToDo: Name of the attribute, if content is not in JSON-Format? For now: metadataName.
		}

		if (map.isEmpty())				return;

		//	Add the element "zusatzDaten" behind the element "paketTyp" within the insertPoint element:
		Element zusatzDatenElement = DocumentHelper.createElement("BAR:zusatzDaten");
		int paketTypPosition = 0;
		for (int i = 0; i < insertPoint.content().size(); i++)
		{
			if ("paketTyp".equals(((Node)insertPoint.content().get(i)).getName()))
			{
				paketTypPosition = i;
				break;
			}
		}
		insertPoint.content().add(paketTypPosition + 1, zusatzDatenElement);

		//	Loop through the map and insert "merkmal" elements into the "zusatzDaten" element. Attribute "name" contains the map key:
		for (Map.Entry<String, String> entry: map.entrySet())
		{
			String key = entry.getKey();
			String value = entry.getValue();
			if (value.isEmpty())		continue;

			Element newElement = zusatzDatenElement.addElement("BAR:merkmal");
			newElement.addAttribute("name", key);
			newElement.setText(value);
		}
	}

	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	/**
	 * This Factory sets the default namespace for the BAR-SIP Document.
	 */
	static private class BARSIPFactory extends DocumentFactory
	{
		@Override
		public org.dom4j.Document createDocument()
		{
			Map<String, String> namespaces = new TreeMap<String, String>();
			namespaces.put("BAR", "http://bar.admin.ch/arelda/v4");
			this.setXPathNamespaceURIs(namespaces);

			DefaultDocument barSIP = new DefaultDocument();
			barSIP.setDocumentFactory(this);

			return barSIP;
		}
	}

}
