package ch.docuteam.darc.util;

import java.util.Arrays;
import java.util.List;
import java.util.Scanner;
import java.util.regex.MatchResult;

import org.dom4j.Element;
import org.dom4j.Node;

import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;

public abstract class XMLUtil
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	/**
	 * Add to element all elements specified by xPath.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary.
	 * @param element
	 * @param xPath
	 * @param doDeleteExistingValues
	 */
	static public Element add(Element parentElement, String xPath)
	{
		return addElements(parentElement, xPath, false);
	}


	/**
	 * Add to element all elements specified by xPath and set the text value of the last element.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary.
	 * @param element
	 * @param xPath
	 * @param values
	 * @param doDeleteExistingElements
	 * @return
	 */
	static public Element add(Element parentElement, String xPath, String textValue)
	{
		return add(parentElement, xPath, Arrays.asList(new String[] {textValue}));
	}


	/**
	 * Add to element all elements specified by xPath and set the text value of the last element. If there are several text values,
	 * create that many last elements.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary.
	 * @param element
	 * @param xPath
	 * @param values
	 * @param doDeleteExistingElements
	 * @return
	 */
	static public Element add(Element parentElement, String xPath, List<String> textValues)
	{
		return addElementsWithTexts(parentElement, xPath, textValues, false);
	}



	/**
	 * Add to element all elements specified by xPath.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary. Delete existing elements beforehand.
	 * @param element
	 * @param xPath
	 * @param doDeleteExistingValues
	 */
	static public Element set(Element parentElement, String xPath)
	{
		return addElements(parentElement, xPath, true);
	}


	/**
	 * Add to element all elements specified by xPath and set the text value of the last element.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary. Delete existing elements beforehand.
	 * @param element
	 * @param xPath
	 * @param values
	 * @param doDeleteExistingElements
	 * @return
	 */
	static public Element set(Element parentElement, String xPath, String textValue)
	{
		return set(parentElement, xPath, Arrays.asList(new String[] {textValue}));
	}


	/**
	 * Add to element all elements specified by xPath and set the text value of the last element. If there are several text values,
	 * create that many last elements.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary. Delete existing elements beforehand.
	 * @param element
	 * @param xPath
	 * @param values
	 * @param doDeleteExistingElements
	 * @return
	 */
	static public Element set(Element parentElement, String xPath, List<String> textValues)
	{
		return addElementsWithTexts(parentElement, xPath, textValues, true);
	}

	//	========	Static Private			=======================================================

	/**
	 * Add to element all elements specified by xPath.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary. Delete existing elements if asked for.
	 * @param element
	 * @param xPath
	 * @param doDeleteExistingValues
	 */
	static private Element addElements(Element element, String xPath, boolean doDeleteExistingElements)
	{
		if (doDeleteExistingElements)			for (Object node: element.selectNodes(xPath))		((Node)node).detach();

		//	Create all intermediary elements if necessary:
		List<String> xPathElements = StringUtil.split(xPath, "/");
		Element lastIntermediaryElement = createIntermediaryElements(element, xPathElements);

		//	Add last element - EVEN IF ONE ORE MORE ALREADY EXIST!:
		return addElement(lastIntermediaryElement, xPathElements.get(xPathElements.size() - 1));
	}


	/**
	 * Add to element all elements specified by xPath and set the text value of the last element. If there are several text values,
	 * create that many last elements.
	 * The xPath may include attribute specs for each path element (but only one pro path element for now).
	 * Create all intermediary elements if necessary. Delete existing elements if asked for.
	 * @param element
	 * @param xPath
	 * @param values
	 * @param doDeleteExistingElements
	 * @return
	 */
	static private Element addElementsWithTexts(Element element, String xPath, List<String> textValues, boolean doDeleteExistingElements)
	{
		if (doDeleteExistingElements)			for (Object node: element.selectNodes(xPath))		((Node)node).detach();
		if (textValues == null)					return element;

		//	Create all intermediary elements if necessary:
		List<String> xPathElements = StringUtil.split(xPath, "/");
		Element lastIntermediaryElement = createIntermediaryElements(element, xPathElements);

		//	Add last element(s) with texts - EVEN IF ONE ORE MORE ALREADY EXIST!:
		Element lastAddedElement = lastIntermediaryElement;
		for (String s: textValues)
			(lastAddedElement = addElement(lastIntermediaryElement, xPathElements.get(xPathElements.size() - 1))).setText(s == null? "": s);

		return lastAddedElement;
	}


	static private Element createIntermediaryElements(Element element, List<String> xPathElements)
	{
		Element parentElement = element;
		for (int i= 0; i < xPathElements.size() - 1; i++)
		{
			String xPathElement = xPathElements.get(i);
			if (xPathElement.equals("."))		continue;

			//	Conditional: If an intermediary element already exists, don't create it:
			Element newElement = (Element)parentElement.selectSingleNode(xPathElement);
			if (newElement == null)				newElement = addElement(parentElement, xPathElement);

			parentElement = newElement;
		}

		return parentElement;
	}


	/**
	 * Add to the parent element a new element according to the xPath string.
	 * If the xPath string contains an attribute spec (but no more than one for now!), add this attribute to the newly created element.
	 */
	static private Element addElement(Element parentElement, String xPathElement)
	{
		if (!xPathElement.contains("[@"))		return parentElement.addElement(xPathElement);

		//	xPathElement contains an attribute specification:
		Element newElement = null;
		Scanner s = new Scanner(xPathElement);
		try
		{
			s.findInLine("(.+)\\[@(\\w*) *= *'(\\w*)'\\]");
			MatchResult result = s.match();

			//	e.g.: "element[@attribute = 'value']"
			//	group1 is the element name
			//	group2 is the attribute name
			//	group3 is the attribute value
			newElement = parentElement.addElement(result.group(1));
			if (result.group(2) != null && result.group(3) != null)		newElement.addAttribute(result.group(2), result.group(3));
		}
		catch (IllegalStateException ex)
		{
			Logger.error("Bad syntax in " + xPathElement);
		}
		finally
		{
			s.close();
		}

		return newElement;
	}

}
