/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.packer.gui;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.List;

import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreePath;

import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.decorator.*;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.treetable.DefaultTreeTableModel;
import org.jdesktop.swingx.treetable.TreeTableNode;

import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.ingest.AIPCreatorProxy;
import ch.docuteam.darc.mdconfig.*;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.structmap.*;
import ch.docuteam.darc.mets.structmap.NodeAbstract.SubmitStatus;
import ch.docuteam.darc.premis.Event;
import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.darc.sa.SubmissionAgreement.Overview;
import ch.docuteam.darc.util.KeyAndValue;
import ch.docuteam.tools.exception.ExceptionCollector;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.PropertyFile;
import ch.docuteam.tools.gui.*;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.out.Tracer;
import ch.docuteam.tools.translations.I18N;


/**
 * @author denis
 *
 */
@SuppressWarnings("serial")
public class SIPView extends JFrame
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Protected	=======================================================

	static protected final String			METSFileName = Document.DefaultMETSFileName;

	static protected final ImageIcon		IconLevelUnknown = new ImageIcon("./resources/images/LevelUnknown.png");
	static protected final ImageIcon		IconLevelNotAllowed = new ImageIcon("./resources/images/LevelNotAllowed.png");
	static protected final ImageIcon		IconNotAllowedBySA = new ImageIcon("./resources/images/Mark.png");

	static protected final Color			ColorFGForNonReadableFile = Color.WHITE;
	static protected final Color			ColorBGForNonReadableFile = Color.LIGHT_GRAY;

	static protected final Color			ColorFGForNonWritableFile = null;							//	OS default
	static protected final Color			ColorBGForNonWritableFile = new Color(253, 152, 154);		//	Pale red
//	static protected final Color			ColorFGForNonWritableFile = Color.WHITE;
//	static protected final Color			ColorBGForNonWritableFile = new Color(255, 255, 155);		//	Pale yellow

	//	========	Static Public			=======================================================

	static public String					IngestSubmitDirectory = null;

	//	========	Static Protected		=======================================================

	//	The following static fields contain default values - they might be overwritten when reading the property file.
	//	In addition, some of these fields can be overwritten using command line parameters:

	static protected String					DefaultFrameTitle = "";
	static protected Boolean				OpenFullScreen = false;
	static protected Boolean				SaveWithBackups = true;
	static protected Integer				ScreenPosX = null;			//	If not overridden by the property file, center on screen
	static protected Integer				ScreenPosY = null;			//	If not overridden by the property file, center on screen
	static protected Integer				ScreenSizeX = 1000;			//	If not overridden by the property file, use this window width
	static protected Integer				ScreenSizeY = 700;			//	If not overridden by the property file, use this window height

	//	========	Instance Public			=======================================================

	//	========	Instance Protected		=======================================================

	//	--------		Visuals				-------------------------------------------------------

	protected LauncherView					launcherView;

	protected JXTreeTable					treeTable;
	protected JTableWithDynamicToolTipTexts	dataTable;
	protected JTable						eventTable;
	protected JTableWithDynamicToolTipTexts	eventDetailTable;
	protected MetadataTable					metadataTable;

	protected JLabel						infoLabel;
	protected JLabel						fileDataPropertiesLabel;
	protected JLabel						fileDateEventsLabel;
	protected JLabel						fileDataEventsDetailsLabel;

	protected JTextField					metaTitleTextField;
	protected JTextField					metaLevelTextField;

	protected JButton						logoButton;

	protected JTextField					footerTextField;

	protected JSplitPane					splitPane;
	protected JTabbedPane					tabbedPane;

	protected FilePreviewer					previewPanel;

	protected Action						insertAction;
	protected Action						saveAction;
	protected Action						closeAction;
	protected Action						saveAsAction;
	protected Action						createFolderAction;
	protected Action						renameItemAction;
	protected Action						replaceFileAction;
	protected Action						deleteItemAction;
	protected Action						deleteItemDontAskAction;
	protected Action						openSAExternallyAction;
	protected Action						testOrAssignSAAction;
	protected Action						openAssignLevelsByLayerViewAction;
	protected Action						openAssignLevelsByLabelViewAction;
	protected Action						exportAsEADFileAction;
	protected Action						openDocuteamHomepageAction;
	protected Action						expandAllAction;
	protected Action						collapseAllAction;
	protected Action						removeMetadataElementAction;
	protected Action						insertMetadataElementAction;
	protected Action						redisplayNodeAction;
	protected Action						systemOutDocumentAction;
	protected Action						exportAction;
	protected Action						saveAsTemplateAction;
	protected Action						submitRequestAction;
	protected Action						submitRetractAction;
	protected Action						submitCheckAction;
	protected Action						submitAction;

	protected JComboBox						selectMetadataElementComboBox;

	protected List<Action>					setLevelActions;

	protected JMenu							fileMenu;
	protected JMenu							searchMenu;
	protected JMenu							viewMenu;
	protected JMenu							itemMenu;
	protected JMenu							itemLevelsSubMenu;
	protected JMenu							saMenu;
	protected JPopupMenu					popupMenu;
	protected int							popupMenuStartOfLevelsSubMenu;

	//	--------		Logic				-------------------------------------------------------

	/**
	 * This sipPath is used as an ID, to prevent several views on the same SIP.
	 */
	protected String						sipPath;
	protected Document						document;

	protected NodeAbstract					selectedNode;
	protected int							selectedIndex = -1;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	protected SIPView(LauncherView launcherView)
	{
		super(DefaultFrameTitle);

		this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		this.addWindowListener(
				new WindowAdapter() { @Override public void windowClosing(WindowEvent e) { SIPView.this.closeButtonClicked(); }});
		this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));

		this.launcherView = launcherView;

		//	Tables:

		this.treeTable = new JXTreeTable(new TreeTableModel(null));

		//	HighlightPredicate for non-writable elements:
		HighlightPredicate nonWritablePredicate = new HighlightPredicate()
			{
				@Override
				public boolean isHighlighted(Component renderer, ComponentAdapter adapter)
				{
					try
					{
						return !((NodeAbstract)SIPView.this.treeTable.getPathForRow(adapter.row).getLastPathComponent()).canWrite();
					}
					catch (NullPointerException ex)
					{
						return false;
					}
				}
			};
		this.treeTable.addHighlighter(new ColorHighlighter(nonWritablePredicate, ColorBGForNonWritableFile, ColorFGForNonWritableFile));

		//	HighlightPredicate for non-readable elements:
		HighlightPredicate nonReadablePredicate = new HighlightPredicate()
			{
				@Override
				public boolean isHighlighted(Component renderer, ComponentAdapter adapter)
				{
					try
					{
						NodeAbstract node = (NodeAbstract)SIPView.this.treeTable.getPathForRow(adapter.row).getLastPathComponent();
						return !node.fileExists() || !node.canRead();
					}
					catch (NullPointerException ex)
					{
						return false;
					}
				}
			};
		this.treeTable.addHighlighter(new ColorHighlighter(nonReadablePredicate, ColorBGForNonReadableFile, ColorFGForNonReadableFile));

		this.treeTable.setEnabled(true);
		this.treeTable.setAutoCreateColumnsFromModel(false);
		this.treeTable.setRootVisible(true);
		this.treeTable.setShowsRootHandles(true);
		this.treeTable.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		this.treeTable.getSelectionModel().addListSelectionListener(
				new ListSelectionListener() { @Override public void valueChanged(ListSelectionEvent e) { SIPView.this.treeViewSelectionChanged(e); }});
		this.treeTable.addMouseListener(
				new MouseAdapter() { @Override public void mousePressed(MouseEvent e) { SIPView.this.treeViewSelectionWasClicked(e); }});
		this.treeTable.setDragEnabled(true);
		this.treeTable.setTransferHandler(new TreeTableTransferHandler());
		this.treeTable.setTreeCellRenderer(new MyTreeCellRenderer());
		this.treeTable.getColumn(0).setPreferredWidth(300);
		this.treeTable.getColumn(1).setPreferredWidth(30);
		this.treeTable.getColumn(2).setPreferredWidth(10);
		this.treeTable.getColumn(3).setPreferredWidth(100);
		this.treeTable.getColumn(3).setCellRenderer(new RelativeSizeBarTableCellRenderer(this.treeTable.getColumn(3)));
		this.treeTable.getColumn(4).setMaxWidth(10);
		this.treeTable.getColumn(4).setCellRenderer(new HasMandatoryMetadataFieldsNotSetCellRenderer());
		this.treeTable.getColumn(5).setMaxWidth(12);
		this.treeTable.getColumn(5).setCellRenderer(new SubmitStatusTableCellRenderer());

		//	Add key bindings for expanding and collapsing nodes using the cursor-left and cursor-right keys:
		this.treeTable.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, 0), "Collapse");
		this.treeTable.getActionMap().put("Collapse",
			new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { SIPView.this.collapseCurrentNode(); }});
		this.treeTable.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, 0), "Expand");
		this.treeTable.getActionMap().put("Expand",
			new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { SIPView.this.expandCurrentNode(); }});

		this.dataTable = new JTableWithDynamicToolTipTexts(new FileDataViewTableModel(), 1);
		this.dataTable.setEnabled(false);
		this.dataTable.getColumnModel().getColumn(0).setMaxWidth(100);
		this.dataTable.getColumnModel().getColumn(0).setMinWidth(100);
		this.dataTable.setGridColor(Color.LIGHT_GRAY);

		this.eventTable = new JTable(new EventListTableModel());
		this.eventTable.setEnabled(true);
		this.eventTable.getSelectionModel().addListSelectionListener(
				new ListSelectionListener() { @Override public void valueChanged(ListSelectionEvent e) { SIPView.this.eventTableSelectionChanged(e); }});
		this.eventTable.setGridColor(Color.LIGHT_GRAY);

		this.eventDetailTable = new JTableWithDynamicToolTipTexts(new EventDetailTableModel(), 1);
		this.eventDetailTable.setEnabled(false);
		this.eventDetailTable.getColumnModel().getColumn(0).setMaxWidth(100);
		this.eventDetailTable.getColumnModel().getColumn(0).setMinWidth(100);
		this.eventDetailTable.setGridColor(Color.LIGHT_GRAY);

		this.metadataTable = new MetadataTable(new MetadataTableModel(), 2);
		this.metadataTable.setEnabled(true);
		this.metadataTable.getSelectionModel().addListSelectionListener(
				new ListSelectionListener() { @Override public void valueChanged(ListSelectionEvent e) { SIPView.this.metadataTableSelectionChanged(e); }});
		this.metadataTable.getColumnModel().getColumn(0).setPreferredWidth(30);
		this.metadataTable.getColumnModel().getColumn(0).setMaxWidth(30);
		this.metadataTable.getColumnModel().getColumn(1).setPreferredWidth(200);
		this.metadataTable.getColumnModel().getColumn(1).setMaxWidth(300);
		this.metadataTable.getColumnModel().getColumn(2).setPreferredWidth(200);
		this.metadataTable.setGridColor(Color.LIGHT_GRAY);


		//	TextFields:

		this.metaTitleTextField = new JTextField();
		this.metaTitleTextField.addFocusListener(
				new FocusAdapter() { @Override public void focusLost(FocusEvent e) { SIPView.this.metaTitleTextFieldWasChanged(); }});
		this.metaTitleTextField.addActionListener(
				new ActionListener() { @Override public void actionPerformed(ActionEvent e) { SIPView.this.metaTitleTextFieldWasChanged(); }});

		this.metaLevelTextField = new JTextField();
		this.metaLevelTextField.setEditable(false);		//	This is always read-only

		//	ComboBoxes:

		this.selectMetadataElementComboBox = new JComboBox();
		this.selectMetadataElementComboBox.setEnabled(false);
		this.selectMetadataElementComboBox.addActionListener(
				new ActionListener() { @Override public void actionPerformed(ActionEvent e) { SIPView.this.pickMetadataElement(); }});
		this.selectMetadataElementComboBox.setToolTipText(I18N.translate("ToolTipSelectMetadataElement"));


		//	Actions:

		this.saveAction = new AbstractAction(I18N.translate("ButtonSave"), new ImageIcon("./resources/images/Save.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.saveButtonClicked(e); }};
		this.saveAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_DOWN_MASK));
		this.saveAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSave"));
		this.saveAction.setEnabled(true);

		this.saveAsAction = new AbstractAction(I18N.translate("ButtonSaveAs"), new ImageIcon("./resources/images/Save.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.saveAsButtonClicked(); }};
		this.saveAsAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK));
		this.saveAsAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSaveAs"));
		this.saveAsAction.setEnabled(true);

		this.closeAction = new AbstractAction(I18N.translate("ButtonClose"), new ImageIcon("./resources/images/Close.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.closeButtonClicked(); }};
		this.closeAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_W, InputEvent.CTRL_DOWN_MASK));
		this.closeAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipClose"));
		this.closeAction.setEnabled(true);


		this.insertAction = new AbstractAction(I18N.translate("ButtonInsert"), new ImageIcon("./resources/images/Insert.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.insertFileOrFolderButtonClicked(); }};
		this.insertAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipInsert"));
		this.insertAction.setEnabled(false);

		this.createFolderAction = new AbstractAction(I18N.translate("ButtonCreateFolder"), new ImageIcon("./resources/images/AddFolder.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.createFolderButtonClicked(); }};
		this.createFolderAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipCreateFolder"));
		this.createFolderAction.setEnabled(false);

		this.renameItemAction = new AbstractAction(I18N.translate("ButtonRenameItem"), new ImageIcon("./resources/images/Rename.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.renameItemButtonClicked(); }};
		this.renameItemAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_F2, InputEvent.CTRL_DOWN_MASK));
		this.renameItemAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipRenameItem"));
		this.renameItemAction.setEnabled(false);

		this.deleteItemAction = new AbstractAction(I18N.translate("ButtonDeleteItem"), new ImageIcon("./resources/images/Delete.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.deleteItemButtonClicked(e); }};
		this.deleteItemAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, InputEvent.CTRL_DOWN_MASK));
		this.deleteItemAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipDeleteItem"));
		this.deleteItemAction.setEnabled(false);

		this.deleteItemDontAskAction = new AbstractAction(I18N.translate("ButtonDeleteItemDontAsk"), new ImageIcon("./resources/images/Delete.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.deleteItemDontAskButtonClicked(); }};
		this.deleteItemDontAskAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK));
		this.deleteItemDontAskAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipDeleteItemDontAsk"));
		this.deleteItemDontAskAction.setEnabled(false);

		this.replaceFileAction = new AbstractAction(I18N.translate("ButtonReplaceFile"), new ImageIcon("./resources/images/Replace.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.replaceFileButtonClicked(); }};
		this.replaceFileAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipReplaceFile"));
		this.replaceFileAction.setEnabled(false);


		this.openSAExternallyAction = new AbstractAction(I18N.translate("ButtonOpenSAExternally"), new ImageIcon("./resources/images/View.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.openSAExternallyButtonClicked(); }};
		this.openSAExternallyAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenSAExternally"));
		this.openSAExternallyAction.setEnabled(false);

		this.testOrAssignSAAction = new AbstractAction(I18N.translate("ButtonTestOrAssignSA"), new ImageIcon("./resources/images/CheckSA.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.testOrAssignSAButtonClicked(); }};
		this.testOrAssignSAAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipTestOrAssignSA"));
		this.testOrAssignSAAction.setEnabled(false);


		this.openAssignLevelsByLayerViewAction = new AbstractAction(I18N.translate("ButtonOpenAssignLevelsByLayerView"), new ImageIcon("./resources/images/Structure.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.openAssignLevelsByLayerViewButtonClicked(); }};
		this.openAssignLevelsByLayerViewAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenAssignLevelsByLayerView"));
		this.openAssignLevelsByLayerViewAction.setEnabled(true);

		this.openAssignLevelsByLabelViewAction = new AbstractAction(I18N.translate("ButtonOpenAssignLevelsByLabelView"), new ImageIcon("./resources/images/Structure.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.openAssignLevelsByLabelViewButtonClicked(); }};
		this.openAssignLevelsByLabelViewAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenAssignLevelsByLabelView"));
		this.openAssignLevelsByLabelViewAction.setEnabled(true);


		this.exportAsEADFileAction = new AbstractAction(I18N.translate("ButtonExportAsEADFile"), new ImageIcon("./resources/images/ExportAsEADFile.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.exportAsEADFileButtonClicked(); }};
		this.exportAsEADFileAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipExportAsEADFile"));
		this.exportAsEADFileAction.setEnabled(true);

		this.exportAction = new AbstractAction(I18N.translate("ButtonExport"), new ImageIcon("./resources/images/Export.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.exportButtonClicked(); }};
		this.exportAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipExport"));
		this.exportAction.setEnabled(false);


		this.openDocuteamHomepageAction = new AbstractAction(I18N.translate("ButtonOpenDocuteamHomepage"), new ImageIcon("./resources/images/Home.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.openDocuteamHomepage(); }};
		this.openDocuteamHomepageAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenDocuteamHomepage"));
		this.openDocuteamHomepageAction.setEnabled(true);

		this.systemOutDocumentAction = new AbstractAction("SystemOut Document")
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.systemOutDocument(); }};

		this.saveAsTemplateAction = new AbstractAction(I18N.translate("ButtonSaveAsTemplate"), new ImageIcon("./resources/images/Save.png"))
				{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.saveAsTemplateButtonClicked(); }};
		this.saveAsTemplateAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_DOWN_MASK | InputEvent.ALT_DOWN_MASK));
		this.saveAsTemplateAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSaveAsTemplate"));
		this.saveAsTemplateAction.setEnabled(true);


		this.expandAllAction = new AbstractAction(I18N.translate("ButtonExpandAll"), new ImageIcon("./resources/images/ExpandAll.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.expandAll(); }};
		this.expandAllAction.setEnabled(false);

		this.collapseAllAction = new AbstractAction(I18N.translate("ButtonCollapseAll"), new ImageIcon("./resources/images/CollapseAll.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.collapseAll(); }};
		this.collapseAllAction.setEnabled(false);


		this.removeMetadataElementAction = new AbstractAction(I18N.translate("ButtonRemoveMetadataElement"), new ImageIcon("./resources/images/Delete.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.removeMetadataElement(); }};
		this.removeMetadataElementAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipRemoveMetadataElement"));
		this.removeMetadataElementAction.setEnabled(false);

		this.insertMetadataElementAction = new AbstractAction(I18N.translate("ButtonInsertMetadataElement"), new ImageIcon("./resources/images/Insert.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.insertMetadataElement(); }};
		this.insertMetadataElementAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipInsertMetadataElement"));
		this.insertMetadataElementAction.setEnabled(false);


		this.redisplayNodeAction = new AbstractAction(I18N.translate("ButtonRedisplayNode"), new ImageIcon("./resources/images/Redisplay.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.redisplayNode(); }};
		this.redisplayNodeAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipRedisplayNode"));


		this.submitRequestAction = new AbstractAction(I18N.translate("ActionSubmitRequest"), new ImageIcon("./resources/images/SubmitRequest.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.submitRequestDocument(); }};
		this.submitRequestAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSubmitRequest"));

		this.submitRetractAction = new AbstractAction(I18N.translate("ActionSubmitRetract"), new ImageIcon("./resources/images/SubmitRetract.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.submitRetractDocument(); }};
		this.submitRetractAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSubmitRetract"));

		this.submitCheckAction = new AbstractAction(I18N.translate("ActionSubmitCheck"), new ImageIcon("./resources/images/SubmitCheck.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.checkSubmission(); }};
		this.submitCheckAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSubmitCheck"));

		this.submitAction = new AbstractAction(I18N.translate("ActionSubmit"), new ImageIcon("./resources/images/Submit.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.submit(); }};
		this.submitAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSubmit"));


		//	Buttons:

		JButton saveButton = new JButton(this.saveAction);
		saveButton.setHideActionText(true);


		JButton insertFileOrFolderButton = new JButton(this.insertAction);
		insertFileOrFolderButton.setHideActionText(true);

		JButton createFolderButton = new JButton(this.createFolderAction);
		createFolderButton.setHideActionText(true);

		JButton renameItemButton = new JButton(this.renameItemAction);
		renameItemButton.setHideActionText(true);

		JButton deleteItemButton = new JButton(this.deleteItemAction);
		deleteItemButton.setHideActionText(true);


		JButton removeMetadataElementButton = new JButton(this.removeMetadataElementAction);
		removeMetadataElementButton.setHideActionText(true);

		JButton insertMetadataElementButton = new JButton(this.insertMetadataElementAction);
		insertMetadataElementButton.setHideActionText(true);


		this.logoButton = new JButton(new ImageIcon("./resources/images/Logo_docuteam_packer.png"));
		this.logoButton.setEnabled(true);
		this.logoButton.setHideActionText(true);
		this.logoButton.setContentAreaFilled(false);
		this.logoButton.setBorderPainted(false);
		this.logoButton.setToolTipText(I18N.translate("ToolTipOpenDocuteamHomepage"));
		this.logoButton.addActionListener(
				new ActionListener() { @Override public void actionPerformed(ActionEvent e) { SIPView.this.openDocuteamHomepage(); }});

		//	Text Fields:

		this.footerTextField = new JTextField();
		this.footerTextField.setEditable(false);

		//	Labels:

		this.infoLabel = new JLabel();

		this.fileDataPropertiesLabel = new JLabel(I18N.translate("LabelFileDataProperties"));
		this.fileDateEventsLabel = new JLabel(I18N.translate("LabelFileDataEvents"));
		this.fileDataEventsDetailsLabel = new JLabel(I18N.translate("LabelFileDataEventDetails"));


		//	Complex components:

		SearchPanel searchPanel = new SearchPanel();


		//	Construct the View:

		//	Menus:

		this.fileMenu = new JMenu(I18N.translate("MenuFile"));
		this.fileMenu.setIcon(new ImageIcon("./resources/images/MenuFile.png"));
		this.fileMenu.add(new JMenuItem(this.saveAction));
		this.fileMenu.add(new JMenuItem(this.saveAsAction));
		this.fileMenu.add(new JMenuItem(this.saveAsTemplateAction));
		this.fileMenu.addSeparator();
		this.fileMenu.add(new JMenuItem(this.closeAction));
		this.fileMenu.addSeparator();
		this.fileMenu.add(new JMenuItem(this.exportAsEADFileAction));
		this.fileMenu.addSeparator();
		this.fileMenu.add(new JMenuItem(this.submitCheckAction));
		this.fileMenu.add(new JMenuItem(this.submitAction));
		if (LauncherView.IsDevelopingMode)
		{
			this.fileMenu.addSeparator();
			this.fileMenu.add(new JMenuItem(this.systemOutDocumentAction));
		}

		this.searchMenu = new JMenu(I18N.translate("MenuSearch"));
		this.searchMenu.setIcon(new ImageIcon("./resources/images/MenuSearch.png"));
		this.searchMenu.add(new JMenuItem(searchPanel.searchAction));
		this.searchMenu.add(new JMenuItem(searchPanel.clearSearchTextFieldAction));
		this.searchMenu.addSeparator();
		this.searchMenu.add(new JMenuItem(searchPanel.selectNextHitAction));
		this.searchMenu.add(new JMenuItem(searchPanel.selectPreviousHitAction));

		this.itemMenu = new JMenu(I18N.translate("MenuItem"));
		this.itemMenu.setIcon(new ImageIcon("./resources/images/MenuItem.png"));
		this.itemMenu.add(new JMenuItem(this.insertAction));
		this.itemMenu.add(new JMenuItem(this.createFolderAction));
		this.itemMenu.add(new JMenuItem(this.renameItemAction));
		this.itemMenu.add(new JMenuItem(this.replaceFileAction));
		this.itemMenu.add(new JMenuItem(this.deleteItemAction));
		this.itemMenu.add(new JMenuItem(this.deleteItemDontAskAction));
		this.itemMenu.addSeparator();
		this.itemMenu.add(new JMenuItem(this.openAssignLevelsByLayerViewAction));
		this.itemMenu.add(new JMenuItem(this.openAssignLevelsByLabelViewAction));
//		this.itemMenu.addSeparator();
//		this.itemMenu.add(new JMenuItem(this.submitRequestAction));
//		this.itemMenu.add(new JMenuItem(this.submitRetractAction));
		this.itemMenu.addSeparator();
		this.itemMenu.add(new JMenuItem(this.exportAction));

		this.popupMenu = new JPopupMenu();
		this.popupMenu.add(new JMenuItem(this.insertAction));
		this.popupMenu.add(new JMenuItem(this.createFolderAction));
		this.popupMenu.add(new JMenuItem(this.renameItemAction));
		this.popupMenu.add(new JMenuItem(this.replaceFileAction));
		this.popupMenu.add(new JMenuItem(this.deleteItemAction));
		this.popupMenu.add(new JMenuItem(this.deleteItemDontAskAction));
		this.popupMenu.addSeparator();

		this.popupMenuStartOfLevelsSubMenu = this.popupMenu.getComponentCount();
		this.initializeLevelsSubMenu();

		this.saMenu = new JMenu(I18N.translate("MenuSA"));
		this.saMenu.setIcon(new ImageIcon("./resources/images/MenuSA.png"));
		this.saMenu.add(new JMenuItem(this.openSAExternallyAction));
		this.saMenu.add(new JMenuItem(this.testOrAssignSAAction));

		this.viewMenu = new JMenu(I18N.translate("MenuView"));
		this.viewMenu.setIcon(new ImageIcon("./resources/images/MenuView.png"));
		this.viewMenu.add(new JMenuItem(this.expandAllAction));
		this.viewMenu.add(new JMenuItem(this.collapseAllAction));
		this.viewMenu.addSeparator();
		this.viewMenu.add(new JMenuItem(this.redisplayNodeAction));

		JMenuBar menuBar = new JMenuBar();
		menuBar.add(this.fileMenu);
		menuBar.add(this.searchMenu);
		menuBar.add(this.itemMenu);
		menuBar.add(this.saMenu);
		menuBar.add(this.viewMenu);
		menuBar.setVisible(true);
		this.setJMenuBar(menuBar);

		//	----------	Header Panel:

		JPanel headerPanel = new JPanel(new BorderLayout());
		headerPanel.setBorder(BorderFactory.createEmptyBorder(0, 10, 0, 10));

		Box buttonsPanel = new Box(BoxLayout.X_AXIS);
		buttonsPanel.add(saveButton);
		buttonsPanel.add(Box.createHorizontalStrut(20));
		buttonsPanel.add(insertFileOrFolderButton);
		buttonsPanel.add(createFolderButton);
		buttonsPanel.add(renameItemButton);
		buttonsPanel.add(deleteItemButton);
		buttonsPanel.add(Box.createHorizontalStrut(20));
		buttonsPanel.add(this.infoLabel);
		buttonsPanel.add(Box.createHorizontalGlue());
		buttonsPanel.add(this.logoButton);

		headerPanel.add(buttonsPanel, BorderLayout.CENTER);

		//	----------	Main View left: the Tree view:

		JPanel mainLeftPanel = new JPanel(new BorderLayout());
		mainLeftPanel.add(searchPanel, BorderLayout.NORTH);
		mainLeftPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 5));
		mainLeftPanel.add(new JScrollPane(this.treeTable), BorderLayout.CENTER);

		//	----------	Main View right: Tabbed Pane:

		//	----------	DataTab: File Data, Event List, Event Details:

		JPanel dataNorthPanel = new JPanel();
		dataNorthPanel.setLayout(new BoxLayout(dataNorthPanel, BoxLayout.Y_AXIS));			//	Top to bottom layout
		dataNorthPanel.add(this.fileDataPropertiesLabel);
		dataNorthPanel.add(Box.createVerticalStrut(10));
		dataNorthPanel.add(this.dataTable);
		dataNorthPanel.add(Box.createVerticalStrut(10));
		dataNorthPanel.add(this.fileDateEventsLabel);
		dataNorthPanel.add(Box.createVerticalStrut(10));

		JPanel dataSouthPanel = new JPanel();
		dataSouthPanel.setLayout(new BoxLayout(dataSouthPanel, BoxLayout.Y_AXIS));			//	Top to bottom layout
		dataSouthPanel.add(Box.createVerticalStrut(10));
		dataSouthPanel.add(this.fileDataEventsDetailsLabel);
		dataSouthPanel.add(Box.createVerticalStrut(10));
		dataSouthPanel.add(this.eventDetailTable);

		JPanel dataView = new JPanel(new BorderLayout());
		dataView.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
		dataView.add(dataNorthPanel, BorderLayout.NORTH);
		dataView.add(new JScrollPane(this.eventTable), BorderLayout.CENTER);
		dataView.add(dataSouthPanel, BorderLayout.SOUTH);


		//	----------	MetadataTab:

		GridBagPanel metadataNorthPanel = new GridBagPanel(new EmptyBorder(10, 0, 10, 0), new Insets(0, 5, 0, 0));
		metadataNorthPanel.add(new JLabel(I18N.translate("LabelTitle")),		1,    0,   	GridBagConstraints.WEST);
		metadataNorthPanel.add(this.metaTitleTextField,							1,    1,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
		metadataNorthPanel.add(new JLabel(I18N.translate("LabelLevel")),		2,    0,   	GridBagConstraints.WEST);
		metadataNorthPanel.add(this.metaLevelTextField,							2,    1,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);

		GridBagPanel metadataSouthPanel = new GridBagPanel(new EmptyBorder(0, 0, 0, 0), new Insets(5, 0, 0, 0));
		metadataSouthPanel.add(new JLabel(I18N.translate("LabelSelectMetadataElement")),
																				1, 1, 0, 3,	GridBagConstraints.SOUTHWEST);
		metadataSouthPanel.add(this.selectMetadataElementComboBox,				2,    0,   	GridBagConstraints.SOUTHWEST);
		metadataSouthPanel.add(insertMetadataElementButton,						2,    1,   	GridBagConstraints.SOUTHWEST);
		metadataSouthPanel.add(new JLabel(),									2,    2,   	GridBagConstraints.SOUTHWEST,		GridBagConstraints.HORIZONTAL,	1, 0);
		metadataSouthPanel.add(removeMetadataElementButton,						2,    3,   	GridBagConstraints.SOUTHEAST);

		JPanel metadataView = new JPanel(new BorderLayout());
		metadataView.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
		metadataView.add(metadataNorthPanel, BorderLayout.NORTH);
		metadataView.add(new JScrollPane(this.metadataTable), BorderLayout.CENTER);
		metadataView.add(metadataSouthPanel, BorderLayout.SOUTH);

		//	----------	PreviewTab: File Preview:

		this.previewPanel = new FilePreviewer();


		//	Combine the three tabs into a tabbedPane:

		this.tabbedPane = new JTabbedPane();
		this.tabbedPane.addChangeListener(
				new ChangeListener() { @Override public void stateChanged(ChangeEvent e) { SIPView.this.tabChanged(); }});
		this.tabbedPane.addTab(I18N.translate("TitleFileData"), null, dataView);
		this.tabbedPane.addTab(I18N.translate("TitleMetadata"), null, metadataView);
		this.tabbedPane.addTab(I18N.translate("TitleFilePreview"), null, this.previewPanel);

		//	Put the tabbedPane into the main right panel:

		JPanel mainRightPanel = new JPanel(new BorderLayout());
		mainRightPanel.setBorder(BorderFactory.createEmptyBorder(10, 5, 10, 10));		//	This looks better under Windows but worse under Mac OS
		mainRightPanel.add(this.tabbedPane);

		//	Combine main left and main right panels in the main center view:

		this.splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true, mainLeftPanel, mainRightPanel);

		//	----------	Footer:

		JLabel versionLabel = new JLabel(LauncherView.VersionNumber);
		versionLabel.addMouseListener(
				new MouseAdapter(){ @Override public void mouseClicked(MouseEvent e){ SIPView.this.versionLabelClicked(e); }});

		GridBagPanel footerPanel = new GridBagPanel(new EmptyBorder(5, 10, 5, 10), new Insets(0, 0, 0, 5));
		footerPanel.add(this.footerTextField,			1,	1,   	GridBagConstraints.EAST,		GridBagConstraints.HORIZONTAL,	1, 0);
		footerPanel.add(versionLabel,					1,	2,   	GridBagConstraints.EAST);


		//	----------	Now arrange all in the main panel:

		this.add(headerPanel, BorderLayout.NORTH);
		this.add(this.splitPane, BorderLayout.CENTER);
		this.add(footerPanel, BorderLayout.SOUTH);
	}

	//	========	Constructors protected	=======================================================

	//	========	Static Public			=======================================================

	/**
	 * Open a new SIPView with the SIP passed. If admIdToSelect is not null, select the node with this admId.
	 * @param zipOrMETSFilePath
	 * @param readOnly
	 * @return
	 */
	static public SIPView open(LauncherView launcherView, String sipPath, boolean readOnly, String admIdToSelect)
	{
		SIPView sipView = new SIPView(launcherView);
		sipView.sipPath = sipPath;

		//	Display the window in the desired size:
		if (OpenFullScreen)
		{
			sipView.setLocation(0, 0);
			sipView.setSize(Toolkit.getDefaultToolkit().getScreenSize());
		}
		else
		{
			sipView.setPreferredSize(new Dimension(ScreenSizeX, ScreenSizeY));
			sipView.pack();

			if (ScreenPosX != null && ScreenPosY != null)
				sipView.setLocation(ScreenPosX, ScreenPosY);
			else
				sipView.setLocationRelativeTo(null);				//	Centered on screen
		}

		sipView.setDividerLocation(sipView.getWidth() / 2);
		sipView.treeTable.requestFocusInWindow();

		sipView.setVisible(true);

		//	Fill the view with contents:
		if (sipPath.toLowerCase().endsWith(".zip"))		sipView.read(sipPath, readOnly, admIdToSelect);
		else											sipView.read(sipPath + "/" + METSFileName, readOnly, admIdToSelect);

		return sipView;
	}

	//	========	Static Protected			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * 	This method is called only once on packer startup. Assume that PropertyFile was already initialized with the correct file path.
	 * @param propertyFileName
	 */
	static protected void initialize()
	{
		SaveWithBackups = "true".equalsIgnoreCase(PropertyFile.get("docuteamPacker.versioning", "true"));
		Logger.info("    saveWithBackups: " + SaveWithBackups);
		try
		{
			Document.setKeepBackupsCount(new Integer(PropertyFile.get("docuteamPacker.versioning.keepBackupsCount", null)));
		}
		catch (NumberFormatException e)
		{
			Document.setKeepBackupsCount(null);
		}
		Logger.info("    keepBackupsCount: " + Document.getKeepBackupsCount());

		OpenFullScreen = "true".equalsIgnoreCase(PropertyFile.get("docuteamPacker.openFullScreen", OpenFullScreen.toString()));
		try
		{
			ScreenSizeX = new Integer(PropertyFile.get("docuteamPacker.screenSize.x", ScreenSizeX.toString()));
		}
		catch (NumberFormatException e){}	//	Ignore it and leave the default values
		try
		{
			ScreenSizeY = new Integer(PropertyFile.get("docuteamPacker.screenSize.y", ScreenSizeY.toString()));
		}
		catch (NumberFormatException e){}	//	Ignore it and leave the default values
		try
		{
			ScreenPosX = new Integer(PropertyFile.get("docuteamPacker.screenPos.x", null));
		}
		catch (NumberFormatException e){}	//	Ignore it and leave the default values
		try
		{
			ScreenPosY = new Integer(PropertyFile.get("docuteamPacker.screenPos.y", null));
		}
		catch (NumberFormatException e){}	//	Ignore it and leave the default values
		Logger.info("    openFullScreen: " + OpenFullScreen);
		Logger.info("    ScreenSizeX: " + ScreenSizeX);
		Logger.info("    ScreenSizeY: " + ScreenSizeY);
		Logger.info("    ScreenPosX: " + ScreenPosX);
		Logger.info("    ScreenPosY: " + ScreenPosY);

		IngestSubmitDirectory = PropertyFile.get("docuteamPacker.ingestSubmitDir" + LauncherView.PropertyFilePathOSSuffix, null);
		if (IngestSubmitDirectory != null)	IngestSubmitDirectory = FileUtil.asCanonicalFileName(IngestSubmitDirectory);
		Logger.info("    ingestSubmitDirectory: " + IngestSubmitDirectory);

		try
		{
			AIPCreatorProxy.initializeImpl(PropertyFile.get("docuteamPacker.AIPCreator.className", null));
		}
		catch (Exception e){}				//	Ignore it and leave the AIPCreatorProxy uninitialized
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public Document getDocument()
	{
		return this.document;
	}

	public String getSIPPath()
	{
		return this.sipPath;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance protected		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initialize the (dynamic) levels submenu. This is done on startup and every time after a SIP was read.
	 */
	protected void initializeLevelsSubMenu()
	{
		if (this.itemLevelsSubMenu != null)		this.itemMenu.remove(this.itemLevelsSubMenu);

		//	Remove all menu items from the popup menu after the "static" part of the popup menu (= all "level" items):
		for (int i = this.popupMenu.getComponentCount(); i > this.popupMenuStartOfLevelsSubMenu; i--)		this.popupMenu.remove(i-1);

		this.setLevelActions = new Vector<Action>(10);
		this.itemLevelsSubMenu = new JMenu(I18N.translate("MenuLevels"));
		this.itemLevelsSubMenu.setIcon(new ImageIcon("./resources/images/AssignLevel.png"));
		if (this.document != null)
			for (LevelOfDescription level: this.document.getLevels().getAll())
			{
				ImageIcon icon = level.getIcon() != null? level.getIcon(): IconLevelUnknown;
				AbstractAction action = new AbstractAction(level.getName(), icon)
				{	@Override public void actionPerformed(ActionEvent e){ SIPView.this.setLevelButtonClicked(e); }};

				this.setLevelActions.add(action);
				this.itemLevelsSubMenu.add(new JMenuItem(action));
				this.popupMenu.add(new JMenuItem(action));
			}

		this.itemMenu.add(this.itemLevelsSubMenu, this.popupMenuStartOfLevelsSubMenu);

//		this.popupMenu.addSeparator();
//		this.popupMenu.add(new JMenuItem(this.submitRequestAction));
//		this.popupMenu.add(new JMenuItem(this.submitRetractAction));
	}

	//	--------		Accessing			-------------------------------------------------------

	private TreeTableModel getTreeTableModel()
	{
		return (TreeTableModel)this.treeTable.getTreeTableModel();
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * Read the SIP. If admIdToSelect is not null, select the node with this admId.
	 * @param zipOrMETSFilePath
	 * @param readOnly
	 */
	protected void read(final String zipOrMETSFilePath, final boolean readOnly, final String admIdToSelect)
	{
		new SwingWorker<Integer, Object>()
		{
			@Override public Integer doInBackground()
			{
				String canonicalZIPOrMETSFileName = FileUtil.asCanonicalFileName(zipOrMETSFilePath);

				SIPView.this.footerTextField.setText(I18N.translate("MessageFooterOpeningFile") + canonicalZIPOrMETSFileName + "...");
				SmallPeskyMessageWindow waitWindow = SmallPeskyMessageWindow.openBlocking(SIPView.this, I18N.translate("MessageTempReadingSIP"));

				try
				{
					SIPView.this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

					ExceptionCollector.clear();
					SIPView.this.document =
						readOnly
							? Document.openReadOnly(canonicalZIPOrMETSFileName, LauncherView.Operator, waitWindow)
							: Document.openReadWrite(canonicalZIPOrMETSFileName, LauncherView.Operator, waitWindow);

					SIPView.this.footerTextField.setText(canonicalZIPOrMETSFileName);
					SIPView.this.setTitle(canonicalZIPOrMETSFileName);		//	Put the ORIGINAL metsFileName into the title, NOT the working copy fileName!

					if (!ExceptionCollector.isEmpty())
					{
						//	Close the waitWindow here otherwise it interferes with the MessageDialog:
						waitWindow.close();
						new ScrollableMessageDialog(SIPView.this, I18N.translate("TitleWarningsOccurred"), ExceptionCollector.getMessageAll(), new ImageIcon("./resources/images/DocuteamPacker.png"));
					}
				}
				catch (java.lang.Exception e)
				{
					e.printStackTrace();
					SIPView.this.document = null;

					SIPView.this.footerTextField.setText(I18N.translate("MessageFooterCantCreateFile") + canonicalZIPOrMETSFileName);
					SIPView.this.setTitle(DefaultFrameTitle);

					//	Close the waitWindow here otherwise it interferes with the MessageDialog:
					waitWindow.close();
					JOptionPane.showMessageDialog(SIPView.this, e.toString(), I18N.translate("TitleCantReadSIP"), JOptionPane.ERROR_MESSAGE);
				}
				finally
				{
					//	At this point, document may or may not be null:
					if (SIPView.this.document == null)		SIPView.this.closeButtonClicked();
					else									SIPView.this.populateView(SIPView.this.document);

					waitWindow.close();		//	In case it was not closed yet...
					SIPView.this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

					SIPView.this.toFront();
					SIPView.this.requestFocus();

					SIPView.this.selectNode(admIdToSelect);
				}

				return 0;
			}
		}.execute();
	}


	protected void save()
	{
		this.saveCopyAs(null, false);
	}


	protected void save(boolean doUnlockAndCleanupAfterwards)
	{
		this.saveCopyAs(null, doUnlockAndCleanupAfterwards);
	}


	protected void saveCopyAs(String filePath)
	{
		this.saveCopyAs(filePath, false);
	}


	protected void saveCopyAs(final String filePath, final boolean doUnlockAndCleanupAfterwards)
	{
		if (this.document == null)		return;

		new SwingWorker<Integer, Object>()
		{
			@Override public Integer doInBackground()
			{
				SmallPeskyMessageWindow waitWindow = SmallPeskyMessageWindow.openBlocking(SIPView.this, I18N.translate("MessageTempSavingSIP"));
				try
				{
					SIPView.this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

					if (filePath == null || filePath.isEmpty())
					{
						//	This is the regular save:

						if (!SIPView.this.document.canWrite())
						{
							JOptionPane.showMessageDialog(SIPView.this, I18N.translate("MessageIsReadOnlyCantSave"), I18N.translate("TitleCantSaveSIP"), JOptionPane.ERROR_MESSAGE);
							return 0;
						}

						Logger.debug("Saving document");

						//	Save the SIP:
						if (SaveWithBackups)		SIPView.this.document.saveWithBackup();
						else						SIPView.this.document.saveWithoutBackup();
					}
					else
					{
						//	This is save-as:

						Logger.debug("Saving document as: " + filePath);

						//	Save SIP or ZIP to a different location:
						SIPView.this.document.saveTo(filePath);

						//	I have to reread the SIPTable in the launcher view because I might have locked or unlocked a SIP in
						//	the workspace when saving the SIP to a different place, or I might have added a new SIP:
						SIPView.this.launcherView.rereadSIPTable();
					}

					//	Unlock and cleanup document in necessary:
					if (doUnlockAndCleanupAfterwards)
					{
						try
						{
							SIPView.this.document.unlockIfNecessary();
							SIPView.this.document.cleanupWorkingCopy();
						}
						catch (Exception e)
						{
							e.printStackTrace();
							JOptionPane.showMessageDialog(SIPView.this, e.toString(), "Unable to cleanup working folder due to errors", JOptionPane.ERROR_MESSAGE);
						}
					}

					SIPView.this.footerTextField.setText(I18N.translate("MessageFooterSaved") + SIPView.this.document.getFilePath());
					SIPView.this.setTitle(SIPView.this.document.getOriginalSIPFolder());		//	Put the ORIGINAL metsFileName into the title, NOT the working copy fileName!
				}
				catch (FileOrFolderIsInUseException e)
				{
					waitWindow.close();
					JOptionPane.showMessageDialog(SIPView.this, I18N.translate("MessageFileOrFolderIsInUseException", e.getOriginalSIPName(), e.getSecurityCopySIPName(), FileUtil.asFileName(e.getOriginalSIPName()), FileUtil.asFileName(e.getSecurityCopySIPName())), I18N.translate("TitleCantSaveSIP"), JOptionPane.ERROR_MESSAGE);
				}
				catch (OriginalSIPIsMissingException e)
				{
					waitWindow.close();
					JOptionPane.showMessageDialog(SIPView.this, I18N.translate("MessageOriginalSIPIsMissingException", e.getOriginalSIPFolderPath()), I18N.translate("TitleCantCopySIP"), JOptionPane.WARNING_MESSAGE);
				}
				catch (Exception e)
				{
					e.printStackTrace();

					waitWindow.close();
					JOptionPane.showMessageDialog(SIPView.this, e.toString(), I18N.translate("TitleCantSaveSIP"), JOptionPane.ERROR_MESSAGE);
					SIPView.this.footerTextField.setText(I18N.translate("MessageFooterCantSave") + SIPView.this.document.getFilePath());
				}
				finally
				{
					waitWindow.close();
					SIPView.this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

					SIPView.this.enableOrDisableActions();

					SIPView.this.toFront();
					SIPView.this.requestFocus();
				}

				return 0;
			}
		}.execute();
	}


	protected void previewSelectedItemIfNecessary()
	{
		//	If the preview tab is not visible AND the separate preview window is not open, do nothing:
		if (!(this.tabbedPane.getSelectedIndex() == 2 || this.previewPanel.isPreviewInSeparateWindow()))		return;

		try
		{
			this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

			this.previewPanel.setNode(this.selectedNode);
		}
		finally
		{
			this.previewPanel.validate();
			this.previewPanel.repaint();

			this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		}
	}


	protected void importFilesAndFolders(final List<File> files)
	{
		//	Import all files and/or folders recursively:
		new SwingWorker<Integer, Object>()
		{
			@Override public Integer doInBackground()
			{
				SmallPeskyMessageWindow waitWindow = SmallPeskyMessageWindow.openBlocking(SIPView.this, I18N.translate("MessageTempInsertingFolder"));

				try
				{
					SIPView.this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

					NodeAbstract lastInsertedNode = null;
					try
					{
						for (File f: files)
						{
							Logger.debug("Inserting: " + f);

							String filePath = f.getPath();
							lastInsertedNode = ((NodeFolder)SIPView.this.selectedNode).insertFileOrFolder(filePath);
						}
					}
					catch (java.lang.Exception e)
					{
						if (LauncherView.IsDevelopingMode)	e.printStackTrace();

						waitWindow.close();		//	In case it was not closed yet...
						JOptionPane.showMessageDialog(SIPView.this, e.toString(), I18N.translate("TitleCantInsertFileOrFolder"), JOptionPane.ERROR_MESSAGE);
					}

					SIPView.this.getTreeTableModel().refreshTreeStructure(SIPView.this.treeTable.getPathForRow(SIPView.this.selectedIndex));
					SIPView.this.selectNode(lastInsertedNode);

					SIPView.this.toFront();
					SIPView.this.requestFocus();

					if (LauncherView.IsDevelopingMode)	SIPView.this.traceTree();
				}
				finally
				{
					waitWindow.close();		//	In case it was not closed yet...
					SIPView.this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				}

				return 0;
			}
		}.execute();
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	protected void populateView(Document documentOrNull)
	{
		this.initializeLevelsSubMenu();

		this.treeTable.setTreeTableModel(new TreeTableModel(documentOrNull));
		this.treeTable.getSelectionModel().addSelectionInterval(0, 0);
	}

	//	--------		Interface			-------------------------------------------------------

	protected void collapseCurrentNode()
	{
		this.treeTable.collapseRow(this.treeTable.getSelectedRow());
	}


	protected void expandCurrentNode()
	{
		this.treeTable.expandRow(this.treeTable.getSelectedRow());
	}


	/**
	 * Listen to row selections in the tree view:
	 * @param e: The event transporting all necessary info
	 */
	protected void treeViewSelectionChanged(ListSelectionEvent e)
	{
		if (e.getValueIsAdjusting())		return;

		//	When changing the tree selection, commit (or cancel if commit fails) any open cell editor in the metadataTable:
		this.metadataTable.commitOrCancelCurrentCellEditor();

		this.selectedIndex = this.treeTable.getSelectedRow();
		this.selectedNode = null;

		//	Nothing selected in the tree view:
		if (this.treeTable.getSelectedRowCount() == 0)
		{
			Logger.debug("No Selection: " + this.selectedIndex);

			this.footerTextField.setText("");
			this.updateView();
			this.previewSelectedItemIfNecessary();
			return;
		}

		//	Multiple selection in the tree view:
		if (this.treeTable.getSelectedRowCount() >= 2)
		{
			//	NOTE: On multiple selection, this.selectedNode is NULL!!! That is to display nothing in the tabs.
			StringBuilder footerText = new StringBuilder();
			for (int i: this.treeTable.getSelectedRows())		footerText.append(", ").append(((NodeAbstract)this.treeTable.getPathForRow(i).getLastPathComponent()).getLabel());
			this.footerTextField.setText(footerText.substring(2));		//	Cut off the leading ", "

			Logger.debug("Multiple selection: " + this.selectedIndex + ": " + this.footerTextField.getText());

			this.clearView();
			this.previewSelectedItemIfNecessary();
			return;
		}

		//	From here on: exactly one item is selected:

		TreePath pathForSelectedRow = this.treeTable.getPathForRow(this.selectedIndex);
		//	pathForSelectedRow is null when the tree part where the selection was in, gets collapsed.
		if (pathForSelectedRow == null)
		{
			Logger.debug("PathForSelectedRow is null: " + this.selectedIndex);

			this.footerTextField.setText("");
			this.clearView();
			this.previewSelectedItemIfNecessary();
			return;
		}

		this.selectedNode = (NodeAbstract)pathForSelectedRow.getLastPathComponent();

		Logger.debug("Selected Node: " + this.selectedIndex + ": " + this.selectedNode);

		this.footerTextField.setText(this.selectedNode.getPathString() + ((this.selectedNode.isFolder())? "/": ""));

		this.updateView();
		this.fillInsertMetadataElementComboBox();
		this.previewSelectedItemIfNecessary();

		//	Select the oldest event:
		int eventsSize = this.selectedNode.getMyEvents().size();
		if (eventsSize > 0)		this.eventTable.getSelectionModel().setSelectionInterval(eventsSize - 1, eventsSize - 1);
	}


	/**
	 * Special mouse clicks (like double-click or right-click) are handled here.
	 * @param e
	 */
	protected void treeViewSelectionWasClicked(MouseEvent e)
	{
		if (e.getClickCount() == 2)
		{
			//	Double-Click:

			//	Open external preview:
			this.previewPanel.openFileExternallyButtonClicked(this.selectedNode);
		}
		else if (e.getButton() == MouseEvent.BUTTON3)
		{
			//	Right-Click:

			//	Find out which row was right-clicked:
			int rowNumber = this.treeTable.rowAtPoint(e.getPoint());

			//	Select this row:
			this.treeTable.getSelectionModel().setSelectionInterval(rowNumber, rowNumber);

			//	Show popup menu:
			this.popupMenu.show(this.treeTable, e.getX(), e.getY());
		}
		//	else: ignore. Single clicks are handled in treeViewSelectionChanged().
	}


	/**
	 * Listen to row selections in the event table:
	 * @param e: The event transporting all necessary infos
	 */
	protected void eventTableSelectionChanged(ListSelectionEvent e)
	{
		Integer selectedEventIndex = ((ListSelectionModel)e.getSource()).getLeadSelectionIndex();
		NodeAbstract listTableModel = ((EventListTableModel)this.eventTable.getModel()).getFileStructureNode();

		if ((listTableModel == null) || (selectedEventIndex < 0) || (selectedEventIndex >= listTableModel.getMyEvents().size()))
		{
			((EventDetailTableModel)this.eventDetailTable.getModel()).setEvent(null);
		}
		else
		{
			Event selectedEvent = listTableModel.getMyEvent(selectedEventIndex);
			((EventDetailTableModel)this.eventDetailTable.getModel()).setEvent(selectedEvent);
		}
	}


	protected void metadataTableSelectionChanged(ListSelectionEvent e)
	{
		if (e.getValueIsAdjusting())		return;

		this.enableOrDisableActions();
	}


	/**
	 * The [Close...] button was clicked.
	 * Returning false means that saving was cancelled.
	 */
	protected boolean closeButtonClicked()
	{
		//	If this view is disabled, reject the attempt to close it:
		if (!this.isEnabled())
		{
			JOptionPane.showMessageDialog(this, I18N.translate("MessageCantCloseSIP"), I18N.translate("TitleCloseSIP"), JOptionPane.YES_OPTION);
			return false;
		}

		if (this.document != null)
		{
			//	This is because the save() method DOES already unlock and cleanup:
			boolean doUnlockAndCleanupAfterwards = true;

			if (this.document.isModified())
			{
				//	Ask whether to save the loaded document:
				int answer = JOptionPane.showConfirmDialog(this, "'" + this.document.getSIPName() + "':\n" + I18N.translate("QuestionSaveModified"), I18N.translate("TitleCloseSIP"), JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
				switch (answer)
				{
					case JOptionPane.YES_OPTION:
					{
						//	Save current document, unlock and cleanup after saving, then continue closing:
						doUnlockAndCleanupAfterwards = false;
						this.save(true);
						break;
					}
					case JOptionPane.NO_OPTION:
					{
						//	Don't save current document but continue closing:
						break;
					}
					case JOptionPane.CANCEL_OPTION:
					{
						//	Cancel, don't continue:
						return false;
					}
					default:
					{
						//	Cancel, don't continue:
						return false;
					}
				}
			}

			if (doUnlockAndCleanupAfterwards)
			{
				try
				{
					this.document.unlockIfNecessary();
					this.document.cleanupWorkingCopy();
				}
				catch (Exception e)
				{
					e.printStackTrace();
					JOptionPane.showMessageDialog(SIPView.this, e.toString(), "Unable to cleanup working folder due to errors", JOptionPane.ERROR_MESSAGE);
				}
			}
		}

		this.launcherView.unregister(this);

		this.setVisible(false);
		this.dispose();

		return true;
	}


	/**
	 * The [Save] button was clicked:
	 */
	protected void saveButtonClicked(ActionEvent actionEvent)
	{
		//	If the shift-key is held while clicking the save-button: ask for a file name to save a copy to:
		if ((actionEvent.getModifiers() & ActionEvent.SHIFT_MASK) != 0)
		{
			this.saveAsButtonClicked();
		}
		else
		{
			this.save();
		}
	}


	/**
	 * The [Save as...] button was clicked:
	 */
	protected void saveAsButtonClicked()
	{
		FileDialog fileDialog = new FileDialog(this, I18N.translate("TitleSaveSIPAs"), FileDialog.SAVE);
		fileDialog.setDirectory(new File(LauncherView.LastUsedOpenOrSaveDirectory).getAbsolutePath());
		fileDialog.setFile(this.document.getSIPName());
		fileDialog.setLocationRelativeTo(this);

		fileDialog.setVisible(true);
		if (fileDialog.getFile() == null)		return;

		this.saveCopyAs(fileDialog.getDirectory() + fileDialog.getFile());
	}


	/**
	 * The [Insert] button was clicked:
	 */
	protected void insertFileOrFolderButtonClicked()
	{
		final JFileChooser fileChooser = new JFileChooser(LauncherView.DataDirectory);
		fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
		fileChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		fileChooser.setMultiSelectionEnabled(true);

		if (fileChooser.showOpenDialog(this) != JFileChooser.APPROVE_OPTION)		return;		//	Cancel-button clicked

		//	Remember the insert directory:
		LauncherView.DataDirectory = FileUtil.asCanonicalFileName(fileChooser.getSelectedFile().getParent());

		//	The term getSelectedFile().getParent() CAN return an empty String under Windows (e.g. when "C:/" is selected). Hence:
		if (LauncherView.DataDirectory.isEmpty())		LauncherView.DataDirectory = FileUtil.asCanonicalFileName(ch.docuteam.tools.os.OperatingSystem.userHome());

		this.importFilesAndFolders(Arrays.asList(fileChooser.getSelectedFiles()));
	}


	/**
	 * The [Create Folder] button was clicked:
	 */
	protected void createFolderButtonClicked()
	{
		String title = I18N.translate("TitleNewFolder");
		String message = I18N.translate("MessageEnterNewFolderName");
		String textFieldContent = "";
		int messageType = JOptionPane.QUESTION_MESSAGE;

		NodeFolder newFolder = null;
		String newFolderName;
		Boolean ok;
		do
		{
			newFolderName = (String)JOptionPane.showInputDialog(this, message, title, messageType, null, null, textFieldContent);
			if ((newFolderName == null) || newFolderName.length() == 0)		return;

			ok = false;

			if (!FileUtil.isFileNameAllowed(newFolderName))
			{
				title = I18N.translate("TitleCantCreateNewFolder");
				message = I18N.translate("MessageBadLettersInFilename") + "\n" + I18N.translate("MessageEnterNewFolderName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = newFolderName;
				continue;
			}

			try
			{
				newFolder = ((NodeFolder)this.selectedNode).createNewFolder(newFolderName);
			}
			catch (FileAlreadyExistsException ex)
			{
				title = I18N.translate("TitleCantCreateNewFolder");
				message = I18N.translate("MessageNameAlreadyExists", newFolderName, this.selectedNode.getLabel()) + "\n" + I18N.translate("MessageEnterNewFolderName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = newFolderName;
				continue;
			}
			catch (java.lang.Exception ex)
			{
				title = I18N.translate("TitleCantCreateNewFolder");
				message = ex.getMessage() + "\n" + I18N.translate("MessageEnterNewFolderName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = newFolderName;
				continue;
			}

			ok = true;
		} while (!ok);

		this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(this.selectedIndex));

		SIPView.this.selectNode(newFolder);

		if (LauncherView.IsDevelopingMode)		this.traceTree();
	}


	/**
	 * The [Rename Item] button was clicked:
	 */
	protected void renameItemButtonClicked()
	{
		String title = I18N.translate("TitleRenameItem");
		String message = I18N.translate("MessageEnterNewItemName");
		String textFieldContent = this.selectedNode.getLabel();
		int messageType = JOptionPane.QUESTION_MESSAGE;

		String newItemName;
		Boolean ok;
		do
		{
			newItemName = (String)JOptionPane.showInputDialog(this, message, title, messageType, null, null, textFieldContent);
			if ((newItemName == null) || newItemName.length() == 0)		return;

			ok = false;

			if (!FileUtil.isFileNameAllowed(newItemName))
			{
				title = I18N.translate("TitleCantRenameItem");
				message = I18N.translate("MessageBadLettersInFilename") + "\n" + I18N.translate("MessageEnterNewItemName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = newItemName;
				continue;
			}

			try
			{
				this.selectedNode.rename(newItemName);
			}
			catch (java.lang.Exception ex)
			{
				title = I18N.translate("TitleCantRenameItem");
				message = ex.getMessage() + "\n" + I18N.translate("MessageEnterNewItemName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = this.selectedNode.getLabel();
				continue;
			}

			ok = true;
		} while (!ok);

		this.getTreeTableModel().refreshNode(this.treeTable.getPathForRow(this.selectedIndex));
		this.enableOrDisableActions();

		this.updateView();
		int lastEventIndex = this.selectedNode.getMyEvents().size() - 1;
		this.eventTable.getSelectionModel().setSelectionInterval(lastEventIndex, lastEventIndex);

		if (LauncherView.IsDevelopingMode)		this.traceTree();
	}


	/**
	 * The [Delete Item] button was clicked (with or without shift key):
	 */
	protected void deleteItemButtonClicked(ActionEvent actionEvent)
	{
		//	If the shift-key is held while clicking the delete-button: DON'T ask.

		if ((actionEvent.getModifiers() & ActionEvent.SHIFT_MASK) == 0)
		{
			//	The shift-key was pressed:

			int selectedRowCount = this.treeTable.getSelectedRowCount();
			if (selectedRowCount == 0)			return;
			else if (selectedRowCount == 1)
			{
				//	Single selection:

				if (this.selectedNode.isFolder() && (this.selectedNode.getChildCount() != 0))
				{
					if (JOptionPane.showConfirmDialog(this, I18N.translate("QuestionDeleteWithAllSubElements"), I18N.translate("TitleDeleteFolder"), JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)		return;
				}
				else
				{
					if (JOptionPane.showConfirmDialog(this, I18N.translate("QuestionDeleteItem"), I18N.translate("TitleDeleteItem"), JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)		return;
				}
			}
			else
			{
				//	Multiple selection:

				if (JOptionPane.showConfirmDialog(this, I18N.translate("QuestionDeleteMultipleItems", selectedRowCount), I18N.translate("TitleDeleteMultipleItems"), JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)		return;
			}
		}

		this.deleteItemDontAskButtonClicked();
	}


	/**
	 * The [Delete Item Dont Ask] button was clicked:
	 */
	protected void deleteItemDontAskButtonClicked()
	{
		int selectedRowCount = this.treeTable.getSelectedRowCount();
		if (selectedRowCount == 0)			return;
		else if (selectedRowCount == 1)
		{
			//	Single selection:

			if (this.selectedNode.isRoot())
			{
				JOptionPane.showMessageDialog(this, I18N.translate("MessageCantDeleteRootItem"), I18N.translate("TitleCantDeleteItem"), JOptionPane.WARNING_MESSAGE);
				return;
			}

			try
			{
				this.selectedNode.delete();
			}
			catch (FileOperationNotAllowedException ex)
			{
				//	This can actually never happen: this ex gets thrown only when a SIP was opened in read-only mode.
				ex.printStackTrace();
			}
			catch (Exception e)
			{
				JOptionPane.showMessageDialog(this, e.toString(), I18N.translate("TitleCantDeleteItem"), JOptionPane.ERROR_MESSAGE);
			}

			this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(this.selectedIndex).getParentPath());
		}
		else
		{
			//	Multiple selection: check if all selected items are files. If at least one of them is a folder, reject:

			for (int i: this.treeTable.getSelectedRows())
			{
				if (((NodeAbstract)this.treeTable.getPathForRow(i).getLastPathComponent()).isFolder())
				{
					JOptionPane.showMessageDialog(this, I18N.translate("MessageCantDeleteMultipleFolders"), I18N.translate("TitleCantDeleteMultipleFolders"), JOptionPane.WARNING_MESSAGE);
					return;
				}
			}

			//	I have to make a copy of the selection list because after a "refreshTreeStructure()", the selection has changed:
			int[] selectedRows = this.treeTable.getSelectedRows();

			//	I have to go backwards through the list because the index number of the selected items changes when deleting going forward:
			for (int i = selectedRows.length - 1; i >= 0; i--)
			{
				try
				{
					((NodeAbstract)this.treeTable.getPathForRow(selectedRows[i]).getLastPathComponent()).delete();
				}
				catch (FileOperationNotAllowedException ex)
				{
					//	This can actually never happen: this ex gets thrown only when a SIP was opened in read-only mode.
					ex.printStackTrace();
				}
				catch (Exception e)
				{
					JOptionPane.showMessageDialog(this, e.toString(), I18N.translate("TitleCantDeleteItem"), JOptionPane.ERROR_MESSAGE);
					break;
				}

				this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(selectedRows[i]).getParentPath());
			}
		}

		this.enableOrDisableActions();

		if (LauncherView.IsDevelopingMode)		this.traceTree();
	}


	protected void replaceFileButtonClicked()
	{
		if (this.selectedNode == null)			return;
		if (this.selectedNode.isFolder())		return;

		final JFileChooser fileChooser = new JFileChooser(LauncherView.DataDirectory);
		fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
//		fileChooser.setApproveButtonText("Select");
		fileChooser.setDialogTitle(I18N.translate("TitleReplaceFile"));
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setMultiSelectionEnabled(false);

		if (fileChooser.showOpenDialog(this) != JFileChooser.APPROVE_OPTION)		return;		//	Cancel-button clicked

		//	Remember the insert directory:
		LauncherView.DataDirectory = FileUtil.asCanonicalFileName(fileChooser.getSelectedFile().getParent());

		//	The term getSelectedFile().getParent() CAN return an empty String under Windows (e.g. when "C:/" is selected). Hence:
		if (LauncherView.DataDirectory.isEmpty())		LauncherView.DataDirectory = FileUtil.asCanonicalFileName(ch.docuteam.tools.os.OperatingSystem.userHome());

		try
		{
			((NodeFile)this.selectedNode).replaceByFile(fileChooser.getSelectedFile().getCanonicalPath());
		}
		catch (FileOperationNotAllowedException ex)
		{
			//	This can actually never happen: this ex gets thrown only when a SIP was opened in read-only mode.
			ex.printStackTrace();
		}
		catch (Exception e)
		{
			JOptionPane.showMessageDialog(this, e.toString(), I18N.translate("TitleCantReplaceFile"), JOptionPane.ERROR_MESSAGE);
		}

		int rememberSelectedIndex = this.selectedIndex;
		this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(this.selectedIndex).getParentPath());
		this.treeTable.getSelectionModel().setSelectionInterval(rememberSelectedIndex, rememberSelectedIndex);
	}



	protected void tabChanged()
	{
		this.previewSelectedItemIfNecessary();
	}


	protected void openSAExternallyButtonClicked()
	{
		try
		{
			this.document.getSubmissionAgreement().openAsHTMLExternally();
		}
		catch (Exception e)
		{
			JOptionPane.showMessageDialog(this, I18N.translate("MessageCantOpenSAExternally", e.toString()), I18N.translate("TitleCantOpenSAExternally"), JOptionPane.ERROR_MESSAGE);
		}
	}


	protected void testOrAssignSAButtonClicked()
	{
		TestOrAssignSADialog dialog = TestOrAssignSADialog.open(this);
		if (!dialog.setSAButtonWasClicked)		return;

		this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(0));
		this.enableOrDisableActions();
	}


	protected void openAssignLevelsByLayerViewButtonClicked()
	{
		//	If an item is selected AND this item is a folder, then open the AssignLevelOfDescriptionsToTreeLevelsDialog for this item.
		if (this.selectedNode == null)							return;
		if (this.selectedNode.isFile())							return;

		if (!new AssignLevelsByLayerDialog(this, this.selectedNode).goButtonWasClicked)
																return;

		//	Refresh the selected node:
		this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(this.selectedIndex));
		this.enableOrDisableActions();
	}


	protected void openAssignLevelsByLabelViewButtonClicked()
	{
		//	If an item is selected AND this item is a folder, then open the AssignLevelOfDescriptionsToTreeLevelsDialog for this item.
		if (this.selectedNode == null)							return;
		if (this.selectedNode.isFile())							return;

		if (!new AssignLevelsByLabelDialog(this, this.selectedNode).goButtonWasClicked)
																return;

		//	Refresh the selected node:
		this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(this.selectedIndex));
		this.enableOrDisableActions();
	}


	protected void setLevelButtonClicked(ActionEvent e)
	{
		ExceptionCollector.clear();

		LevelOfDescription level = this.document.getLevels().get(e.getActionCommand());

		//	Multiple selection in the tree view: change the levelOfDescription of all selected nodes:
		if (this.treeTable.getSelectedRowCount() >= 2)
		{
			for (int i: this.treeTable.getSelectedRows())
			{
				try
				{
					((NodeAbstract)this.treeTable.getPathForRow(i).getLastPathComponent()).setLevel(level);
				}
				catch (Exception ex)
				{
					ch.docuteam.tools.exception.Exception.remember(ex);
				}

				this.getTreeTableModel().refreshNode(this.treeTable.getPathForRow(i));
			}

			return;
		}

		//	Single selection: set the level of the selected node:
		try
		{
			this.selectedNode.setLevel(level);
		}
		catch (Exception ex)
		{
			ch.docuteam.tools.exception.Exception.remember(ex);
		}

		//	Refresh the selected node:
		this.getTreeTableModel().refreshNode(this.treeTable.getPathForRow(this.selectedIndex));
		this.fillInsertMetadataElementComboBox();

		//	Update the view and select the last event in the event list:
		this.updateView();
		int lastEventIndex = this.selectedNode.getMyEvents().size() - 1;
		this.eventTable.getSelectionModel().setSelectionInterval(lastEventIndex, lastEventIndex);

		if (!ExceptionCollector.isEmpty())		ExceptionCollector.systemOut();
	}


	protected void exportAsEADFileButtonClicked()
	{
		FileDialog fileDialog = new FileDialog(this, I18N.translate("TitleExportAsEADFile"), FileDialog.SAVE);
		fileDialog.setDirectory(new File(LauncherView.LastUsedOpenOrSaveDirectory).getAbsolutePath());
		fileDialog.setFile(this.document.getSIPName() + ".xml");
		fileDialog.setLocationRelativeTo(this);

		fileDialog.setVisible(true);
		if (fileDialog.getFile() == null)		return;

		try
		{
			this.document.createEADFile(fileDialog.getDirectory() + fileDialog.getFile());
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	/**
	 * For now, only one item at a time can be exported.
	 * Later, we may allow multiple items to be exported, provided that they all are within the same folder.
	 */
	protected void exportButtonClicked()
	{
		if (this.selectedNode == null)						return;
		if (this.treeTable.getSelectedRowCount() >= 2)		return;

		JFileChooser fileChooser = new JFileChooser(LauncherView.DataDirectory);
		fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		fileChooser.setDialogTitle(I18N.translate("TitleSelectExportDestinationFolder"));
		fileChooser.setMultiSelectionEnabled(false);
		int result = fileChooser.showSaveDialog(this);
		if (result == JFileChooser.CANCEL_OPTION)			return;

		String destinationFolder = fileChooser.getSelectedFile().getPath();

		//	Check if this file or folder already exists in the destination folder - if yes, ask if to overwrite:
		if (new File(destinationFolder + "/" + this.selectedNode.getFile().getName()).exists())
		{
			if (JOptionPane.showConfirmDialog(this, I18N.translate("QuestionExportOverwriteFile"), I18N.translate("TitleExportItem"), JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)		return;
		}

		try
		{
			FileUtil.copyToFolderMerging(this.selectedNode.getFile(), new File(destinationFolder));
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	/**
	 * The [Save as Template...] button was clicked:
	 */
	protected void saveAsTemplateButtonClicked()
	{
		if (this.document == null)		return;

		String title = I18N.translate("TitleCreateTemplate");
		String message = I18N.translate("MessageEnterNewTemplateName");
		int messageType = JOptionPane.QUESTION_MESSAGE;
		String textFieldContent = new File(this.document.getSIPFolder()).getName();

		Boolean ok;
		do
		{
			String newItemName = (String)JOptionPane.showInputDialog(this, message, title, messageType, null, null, textFieldContent);
			if ((newItemName == null) || newItemName.length() == 0)		return;

			ok = true;

			if (!FileUtil.isFileNameAllowed(newItemName))
			{
				ok = false;
				title = I18N.translate("TitleCantCreateTemplate");
				message = I18N.translate("MessageBadLettersInFilename") + "\n" + I18N.translate("MessageEnterNewTemplateName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = newItemName;
				continue;
			}

			if (new File(LauncherView.TemplateDirectory + "/" + newItemName).exists())
			{
				int confirm = JOptionPane.showConfirmDialog(this, I18N.translate("MessageOverwriteExistingTemplate"));

				switch (confirm)
				{
					case JOptionPane.YES_OPTION:		break;				//	Save/overwrite template: leave this loop and continue
					case JOptionPane.NO_OPTION:								//	Enter another template name
					{
						ok = false;
						title = I18N.translate("TitleCreateTemplate");
						message = I18N.translate("MessageEnterNewTemplateName");
						messageType = JOptionPane.QUESTION_MESSAGE;
						textFieldContent = new File(this.document.getSIPFolder()).getName();
						continue;
					}
					default:							return;				//	Exit this method
				}
			}

			try
			{
				this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

				this.document.saveAsTemplate(LauncherView.TemplateDirectory + "/" + newItemName);

				this.enableOrDisableActions();
				this.footerTextField.setText(I18N.translate("MessageFooterTemplateSaved") + newItemName);
			}
			catch (CantCreateTemplateWithRootFileException ex)
			{
				JOptionPane.showMessageDialog(this, I18N.translate("MessageCantCreateTemplateWithRootFile"));
				return;
			}
			catch (java.lang.Exception ex)
			{
				ok = false;
				title = I18N.translate("TitleCantCreateTemplate");
				message = ex.getMessage() + "\n" + I18N.translate("MessageEnterNewTemplateName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = this.document.getName();
			}
			finally
			{
				this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			}

		} while (!ok);
	}


	protected void openDocuteamHomepage()
	{
		if(!java.awt.Desktop.isDesktopSupported())
		{
			System.err.println("Desktop is not supported");
			return;
		}

		java.awt.Desktop desktop = java.awt.Desktop.getDesktop();

		if(!desktop.isSupported(java.awt.Desktop.Action.BROWSE))
		{
			System.err.println("Desktop doesn't support the browse action");
			return;
		}

		try
		{
			desktop.browse(new java.net.URI("http://www.docuteam.ch"));
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
		}
	}


	protected void openHelpPage()
	{
		if(!java.awt.Desktop.isDesktopSupported())
		{
			System.err.println("Desktop is not supported");
			return;
		}

		java.awt.Desktop desktop = java.awt.Desktop.getDesktop();

		if(!desktop.isSupported(java.awt.Desktop.Action.BROWSE))
		{
			System.err.println("Desktop doesn't support the browse action");
			return;
		}

		try
		{
			desktop.browse(new java.net.URI("http://wiki.docuteam.ch/doku.php?id=oais:docupack"));
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
		}
	}


	protected void versionLabelClicked(MouseEvent e)
	{
		if ((e.getModifiers() & MouseEvent.META_MASK) != 0)
		{
			this.systemOutDocument();
		}
		else if ((e.getModifiers() & MouseEvent.ALT_MASK) != 0)
		{
			JDialog d = new JDialog(this, true);
			d.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			d.setTitle("I did it!");
			d.add(new JLabel(new ImageIcon("./resources/images/I.jpg")));
			d.setPreferredSize(new Dimension(100, 150));
			d.pack();
			d.setLocationRelativeTo(this);
			d.setVisible(true);
		}
	}

	protected void expandAll()
	{
		this.treeTable.expandAll();
	}

	protected void collapseAll()
	{
		this.treeTable.collapseAll();
	}


	protected void pickMetadataElement()
	{
		this.enableOrDisableActions();
	}

	protected void insertMetadataElement()
	{
		LevelMetadataElement lme = (LevelMetadataElement)this.selectMetadataElementComboBox.getSelectedItem();

		try
		{
			this.selectedNode.addDynamicMetadataElementInstanceWithName(lme.getId());
		}
		catch (MetadataElementCantAddException ex)
		{
			JOptionPane.showMessageDialog(this, I18N.translate("MessageCantAddMetadataElement"), lme.getId(), JOptionPane.ERROR_MESSAGE);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		this.fillInsertMetadataElementComboBox();
		this.updateView();
	}

	protected void removeMetadataElement()
	{
		MetadataElementInstance me = ((MetadataTableModel)this.metadataTable.getModel()).get(this.metadataTable.getSelectedRow());

		try
		{
			this.selectedNode.deleteDynamicMetadataElementInstanceWithName(me.getName(), me.getIndex());
		}
		catch (MetadataElementCantDeleteException ex)
		{
			JOptionPane.showMessageDialog(this, I18N.translate("MessageCantDeleteMetadataElement"), me.getName(), JOptionPane.ERROR_MESSAGE);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		this.fillInsertMetadataElementComboBox();
		this.updateView();
	}


	protected void metaTitleTextFieldWasChanged()
	{
		if (this.metaTitleTextField.getText().equals(this.selectedNode.getUnitTitle()))		return;

		this.selectedNode.setUnitTitle(this.metaTitleTextField.getText());
		this.updateView();
	}


	protected void redisplayNode()
	{
		if (this.selectedIndex == -1)		return;

		this.getTreeTableModel().refreshTreeStructure(this.treeTable.getPathForRow(this.selectedIndex).getParentPath());
		this.treeTable.expandRow(this.selectedIndex);
		this.treeTable.getSelectionModel().setSelectionInterval(this.selectedIndex, this.selectedIndex);
	}


	/**
	 * Select in the treeTable the node with the given admId. If this node doesn't exist, ignore it.
	 * @param admId
	 */
	protected void selectNode(String admId)
	{
		Logger.debug("Select admId: " + admId);

		NodeAbstract node = this.document.getStructureMap().getRoot().searchId(admId);
		if (node == null)			return;

		this.selectNode(node);
	}


	/**
	 * Select in the treeTable the node node. If node is null, ignore it.
	 * @param treePath
	 */
	private void selectNode(NodeAbstract node)
	{
		Logger.debug("Select Node: " + node);

		if (node == null)			return;

		this.selectNode(node.getTreePath());
	}


	/**
	 * Select in the treeTable the node at treePath. If treePath is null, ignore it.
	 * @param treePath
	 */
	private void selectNode(TreePath treePath)
	{
		Logger.debug("Select Path: " + treePath);

		if (treePath == null)		return;

		this.treeTable.getTreeSelectionModel().setSelectionPath(treePath);
		this.treeTable.scrollPathToVisible(treePath);
	}



	@SuppressWarnings("unused")
	private void submitRequestNodeWithSubtree()
	{
		try
		{
			this.selectedNode.setSubmitStatusRecursivelyAllOrNone(SubmitStatus.SubmitRequested);
		}
		catch (CantSetSubmitStatusRecursiveException ex)
		{
			new ScrollableMessageDialog(this, I18N.translate("HeaderSettingSubmitStatus"), this.translateAndFormat(ex));
		}

		this.updateView();
	}


	@SuppressWarnings("unused")
	private void submitRetractNodeWithSubtree()
	{
		try
		{
			this.selectedNode.setSubmitStatusRecursivelyAllOrNone(SubmitStatus.SubmitUndefined);
		}
		catch (CantSetSubmitStatusRecursiveException ex)
		{
			new ScrollableMessageDialog(this, I18N.translate("HeaderSettingSubmitStatus"), this.translateAndFormat(ex));
		}

		this.updateView();
	}


	private void submitRequestDocument()
	{
		try
		{
			this.document.getStructureMap().getRoot().setSubmitStatusRecursivelyAllOrNone(SubmitStatus.SubmitRequested);
		}
		catch (CantSetSubmitStatusRecursiveException ex)
		{
			new ScrollableMessageDialog(this, I18N.translate("HeaderSettingSubmitStatus"), this.translateAndFormat(ex));
		}
	}


	private void submitRetractDocument()
	{
		try
		{
			this.document.getStructureMap().getRoot().setSubmitStatusRecursivelyAllOrNone(SubmitStatus.SubmitUndefined);
		}
		catch (CantSetSubmitStatusRecursiveException ex)
		{
			new ScrollableMessageDialog(this, I18N.translate("HeaderSettingSubmitStatus"), this.translateAndFormat(ex));
		}
	}


	private void checkSubmission()
	{
		List<String> submitCheckMessages = this.document.checkSubmission(IngestSubmitDirectory);
		if (submitCheckMessages.isEmpty())
			JOptionPane.showMessageDialog(this, I18N.translate("MessageSubmitCheckSuccessful"), I18N.translate("HeaderSubmitCheck"), JOptionPane.INFORMATION_MESSAGE);
		else
			new ScrollableMessageDialog(this, I18N.translate("HeaderSubmitCheck"), this.translateAndFormat(I18N.translate("MessageSubmitCheckFailed"), submitCheckMessages));
	}


	private void submit()
	{
		if (JOptionPane.showConfirmDialog(this, I18N.translate("MessageSubmitAsk", IngestSubmitDirectory), I18N.translate("HeaderSubmit"), JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE) != JOptionPane.OK_OPTION)		return;

		Logger.debug("Submitting SIP: '" + this.document.getSIPName() + "' to folder: '" + IngestSubmitDirectory + "'");

		new SwingWorker<Integer, Object>()
		{
			@Override public Integer doInBackground()
			{
				SmallPeskyMessageWindow w = SmallPeskyMessageWindow.openBlocking(SIPView.this, I18N.translate("MessageTempSubmitting"));
				try
				{
					//	Check then submit:
					List<String> submitCheckMessages = SIPView.this.document.submit(IngestSubmitDirectory);

					if (submitCheckMessages.isEmpty())
						JOptionPane.showMessageDialog(SIPView.this, I18N.translate("MessageSubmitDone", IngestSubmitDirectory));
					else
						new ScrollableMessageDialog(SIPView.this, I18N.translate("HeaderSubmit"), SIPView.this.translateAndFormat(I18N.translate("MessageSubmitCheckFailed"), submitCheckMessages));
				}
				catch (Exception ex)
				{
					new ScrollableMessageDialog(SIPView.this, I18N.translate("HeaderSubmit"), ex.getMessage());
				}
				finally
				{
					w.close();
				}

				SIPView.this.updateView();

				return 0;
			}
		}.execute();

		return;
	}


	private String translateAndFormat(CantSetSubmitStatusRecursiveException ex)
	{
		return this.translateAndFormat(I18N.translate("MessageSettingSubmitStatusFailed", I18N.translate(ex.getSubmitStatus().name())), ex.getRejectMessages());
	}

	private String translateAndFormat(String firstRow, List<String> messageList)
	{
		//	The message strings are expected to look like this: "MessageToTranslate<Space>AdditionalInfos".
		//	The AdditionalInfos may contain more phrases to translate, which are expected to be in the format "I18N{PhraseToTranslate}".
		StringBuilder messageString = new StringBuilder(firstRow);
		for (String s: messageList)
		{
			int separatorIndex = s.indexOf(" ");

			String message = I18N.translate(s.substring(0, separatorIndex));
			String additionalInfos = s.substring(separatorIndex);

			//	Replace every occurrence of "I18N{something}" within additionalInfos by the translation of "something":
			if (additionalInfos.contains("I18N{") && additionalInfos.contains("}"))
			{
				int startIndex = 0;
				do
				{
					int keywordStartIndex = additionalInfos.indexOf("I18N{", startIndex);
					if (keywordStartIndex == -1)	break;
					int keywordEndIndex = additionalInfos.indexOf("}", keywordStartIndex);
					if (keywordEndIndex == -1)		break;

					String keyword = additionalInfos.substring(keywordStartIndex + "I18N{".length(), keywordEndIndex);
					String keywordTranslated = I18N.translate(keyword);
					additionalInfos = additionalInfos.replace(additionalInfos.substring(keywordStartIndex, keywordEndIndex + "}".length()), keywordTranslated);

					startIndex = keywordStartIndex + keywordTranslated.length();
				}
				while (true);
			}

			messageString
					//	This is "AdditionalInfos":
					.append(additionalInfos)
					.append(": ")
					//	This is the "MessageToTranslate":
					.append(message)
					.append("\n");
		}

		return messageString.toString();
	}

	//	--------		Utilities			-------------------------------------------------------

	protected void setDividerLocation(int width)
	{
		this.splitPane.setDividerLocation(width);
	}


	protected void enableOrDisableActions()
	{
		//	First disable all actions, later enable specific actions if appropriate:

		//	Document actions:
		this.saveAction.setEnabled(false);
		this.saveAsAction.setEnabled(false);
		this.closeAction.setEnabled(false);
		this.openSAExternallyAction.setEnabled(false);
		this.testOrAssignSAAction.setEnabled(false);
		this.exportAsEADFileAction.setEnabled(false);
		this.exportAction.setEnabled(false);
		this.expandAllAction.setEnabled(false);
		this.collapseAllAction.setEnabled(false);
		this.saveAsTemplateAction.setEnabled(false);
		this.submitCheckAction.setEnabled(false);
		this.submitAction.setEnabled(false);

		//	Node actions:
		this.deleteItemAction.setEnabled(false);
		this.deleteItemDontAskAction.setEnabled(false);
		this.renameItemAction.setEnabled(false);
		this.replaceFileAction.setEnabled(false);
		this.createFolderAction.setEnabled(false);
		this.insertAction.setEnabled(false);
		this.openAssignLevelsByLayerViewAction.setEnabled(false);
		this.openAssignLevelsByLabelViewAction.setEnabled(false);
		this.redisplayNodeAction.setEnabled(false);
		this.exportAction.setEnabled(false);
		this.submitRequestAction.setEnabled(false);
		this.submitRetractAction.setEnabled(false);

		//	Disable actions for setting the level, later enable allowed levels if appropriate:
		this.itemLevelsSubMenu.setEnabled(false);
		for (Action setLevelAction: this.setLevelActions)		setLevelAction.setEnabled(false);

		//	Disable actions and fields for dynamic metadata, later enable them if appropriate:
		this.metaTitleTextField.setEnabled(false);
		this.selectMetadataElementComboBox.setEnabled(false);
		this.insertMetadataElementAction.setEnabled(false);
		this.removeMetadataElementAction.setEnabled(false);


		if (this.document == null)
		{
			Logger.warn("Document is null???");

			this.showInInfoLabel("");
			return;
		}

		this.saveAsAction.setEnabled(!this.document.isAtLeastOneFileNotReadable());
		this.closeAction.setEnabled(true);
		this.openSAExternallyAction.setEnabled(true);
		this.exportAsEADFileAction.setEnabled(true);
		this.expandAllAction.setEnabled(true);
		this.collapseAllAction.setEnabled(true);
		this.saveAsTemplateAction.setEnabled(true);

		boolean documentIsWritable = this.document.canWrite();
		if (!documentIsWritable)
		{
			//	If document is readOnly or locked, don't show the message "MessageMandatoryFieldsNotSet",
			//	but only one of "LabelIsLocked" or "LabelIsReadOnly":
			if (this.document.isLocked())			this.showInInfoLabel(I18N.translate("LabelIsLocked") + this.document.getLockedBy(), true);
			else if (this.document.isReadOnly())	this.showInInfoLabel(I18N.translate("LabelIsReadOnly"), true);
			else									this.showInInfoLabel("");
		}
		else
		{
			this.saveAction.setEnabled(true);
			this.testOrAssignSAAction.setEnabled(true);

			//	The submit actions are only possible if the AIPCretor is initialized AND the ingest submit folder is defined:
			if ((!(IngestSubmitDirectory == null || IngestSubmitDirectory.isEmpty()))
				&& AIPCreatorProxy.isUsable())
			{
				this.submitCheckAction.setEnabled(true);
				this.submitAction.setEnabled(true);
			}

			//	If document is NOT readOnly, show one of "LabelIsModified", "MessageMandatoryFieldsNotSet", or nothing:
			if (this.document.isModified())			this.showInInfoLabel(I18N.translate("LabelIsModified"));
			else if (this.document.hasNodesWithDynamicMetadataElementInstancesWhichAreMandatoryButNotSet())
													this.showInInfoLabel(I18N.translate("MessageMandatoryFieldsNotSet"), true);
			else									this.showInInfoLabel("");
		}

		//	Check out selected item(s):

		//	No selection:
		if (this.treeTable.getSelectedRowCount() == 0)		return;

		//	Multiple selection:
		if (this.treeTable.getSelectedRowCount() >= 2)
		{
			if (!documentIsWritable)						return;

			//	Don't allow delete if any of the selected nodes has descendants or predecessors not writable by current user:
			boolean canDelete = true;
			for (int i: this.treeTable.getSelectedRows())
			{
				NodeAbstract n = (NodeAbstract)this.treeTable.getPathForRow(i).getLastPathComponent();
				if (	(n.isRoot() || !n.fileExists() || !n.canRead() || !n.canWrite() || n.hasPredecessorNotWritableByCurrentUser())
					||	(n.isFolder() && ((NodeFolder)n).hasDescendantNotWritableByCurrentUser()))
				{
					canDelete = false;
					break;
				}
			}

			this.deleteItemAction.setEnabled(canDelete);
			this.deleteItemDontAskAction.setEnabled(canDelete);

			//	Enable setting any level:
			this.itemLevelsSubMenu.setEnabled(true);
			for (Action setLevelAction: this.setLevelActions)		setLevelAction.setEnabled(true);

			return;
		}

		//	From here on: exactly one node is selected:

		//	pathForSelectedRow is null when the tree part where the selection was in, gets collapsed.
		TreePath pathForSelectedRow = this.treeTable.getPathForRow(this.selectedIndex);
		if (pathForSelectedRow == null)			return;
		if (this.selectedNode == null)			return;

		if (!documentIsWritable)
		{
			//	Document is not writable:
			this.redisplayNodeAction.setEnabled(this.selectedNode.isFolder());
			this.exportAction.setEnabled(true);

			return;
		}

		if (this.selectedNode.isFolder())
		{
			//	Node is folder:

			this.redisplayNodeAction.setEnabled(true);

			if (!this.selectedNode.fileExists() || !this.selectedNode.canRead())	;
				//	File doesn't exist or is not readable: leave all actions disabled
			else if (!this.selectedNode.canWrite())
			{
				//	Folder is not writable:
				this.exportAction.setEnabled(true);
			}
			else if (this.selectedNode.hasPredecessorNotWritableByCurrentUser())
			{
				//	A predecessor folder is not writable:
				this.openAssignLevelsByLayerViewAction.setEnabled(true);
				this.openAssignLevelsByLabelViewAction.setEnabled(true);
				this.exportAction.setEnabled(true);

				this.enableDynamicMetadataActions();
				this.enableOnlyAllowedSetLevelActions();
			}
			else if (((NodeFolder)this.selectedNode).hasDescendantNotWritableByCurrentUser())
			{
				//	Folder has read-only children:
				this.renameItemAction.setEnabled(true);
				this.createFolderAction.setEnabled(true);
				this.insertAction.setEnabled(true);
				this.openAssignLevelsByLayerViewAction.setEnabled(true);
				this.openAssignLevelsByLabelViewAction.setEnabled(true);
				this.exportAction.setEnabled(true);

				this.enableDynamicMetadataActions();
				this.enableOnlyAllowedSetLevelActions();
			}
			else if (!this.selectedNode.doesSubmitStatusAllowEditing())
			{
				//	Submit status doesn't allow editing:
				this.exportAction.setEnabled(true);
			}
			else
			{
				//	Folder is readable and writable:
				if (!this.selectedNode.isRoot())
				{
					this.deleteItemAction.setEnabled(true);
					this.deleteItemDontAskAction.setEnabled(true);
				}

				this.renameItemAction.setEnabled(true);
				this.createFolderAction.setEnabled(true);
				this.insertAction.setEnabled(true);
				this.openAssignLevelsByLayerViewAction.setEnabled(true);
				this.openAssignLevelsByLabelViewAction.setEnabled(true);
				this.exportAction.setEnabled(true);

				this.enableDynamicMetadataActions();
				this.enableOnlyAllowedSetLevelActions();
			}
		}
		else
		{
			//	Node is file:

			if (!this.selectedNode.fileExists() || !this.selectedNode.canRead())	;
				//	File doesn't exist or is not readable: leave all actions disabled
			else if (!this.selectedNode.canWrite())
			{
				//	File is not writable:
				this.exportAction.setEnabled(true);
			}
			else if (this.selectedNode.hasPredecessorNotWritableByCurrentUser())
			{
				//	A predecessor folder is not writable:
				this.exportAction.setEnabled(true);

				this.enableDynamicMetadataActions();
				this.enableOnlyAllowedSetLevelActions();
			}
			else if (!this.selectedNode.doesSubmitStatusAllowEditing())
			{
				//	Submit status doesn't allow editing:
				this.exportAction.setEnabled(true);
			}
			else
			{
				//	File is readable and writable:
				if (!this.selectedNode.isRoot())
				{
					this.deleteItemAction.setEnabled(true);
					this.deleteItemDontAskAction.setEnabled(true);
				}

				this.renameItemAction.setEnabled(true);
				this.exportAction.setEnabled(true);
				this.replaceFileAction.setEnabled(true);

				this.enableDynamicMetadataActions();
				this.enableOnlyAllowedSetLevelActions();
			}
		}
	}

	/**
	 * Enable setting only allowed levels
	 */
	protected void enableOnlyAllowedSetLevelActions()
	{
		this.itemLevelsSubMenu.setEnabled(true);
		for (Action setLevelAction: this.setLevelActions)
			setLevelAction.setEnabled(this.selectedNode.doesParentAllowSubLevel((String)setLevelAction.getValue(Action.NAME)));
	}

	/**
	 * Enable dynamic metadata buttons and fields
	 */
	protected void enableDynamicMetadataActions()
	{
		this.metaTitleTextField.setEnabled(true);
		this.selectMetadataElementComboBox.setEnabled(true);
		this.insertMetadataElementAction.setEnabled(this.selectMetadataElementComboBox.getSelectedItem() != null);
		int selectedMetadataElementIndex = this.metadataTable.getSelectedRow();
		if (selectedMetadataElementIndex == -1)
			this.removeMetadataElementAction.setEnabled(false);
		else
			try
			{
				this.removeMetadataElementAction.setEnabled(((MetadataTableModel)this.metadataTable.getModel()).get(selectedMetadataElementIndex).canBeDeleted());
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
	}


	protected void updateView()
	{
		((FileDataViewTableModel)this.dataTable.getModel()).setFileStructureNode(this.selectedNode);
		((EventListTableModel)this.eventTable.getModel()).setFileStructureNode(this.selectedNode);
		((MetadataTableModel)this.metadataTable.getModel()).setFileStructureNode(this.selectedNode);

		if (this.selectedNode == null)
		{
			this.metaTitleTextField.setText(null);
			this.metaLevelTextField.setText(null);
		}
		else
		{
			this.metaTitleTextField.setText(this.selectedNode.getUnitTitle());
			this.metaTitleTextField.setCaretPosition(0);
			this.metaLevelTextField.setText(this.selectedNode.getLevel().getName());
		}

		this.enableOrDisableActions();
	}


	protected void clearView()
	{
		((FileDataViewTableModel)this.dataTable.getModel()).setFileStructureNode(null);
		((EventListTableModel)this.eventTable.getModel()).setFileStructureNode(null);
		((MetadataTableModel)this.metadataTable.getModel()).setFileStructureNode(null);
		this.metaTitleTextField.setText(null);
		this.metaLevelTextField.setText(null);

		this.enableOrDisableActions();
	}


	protected void fillInsertMetadataElementComboBox()
	{
		this.selectMetadataElementComboBox.removeAllItems();
		this.selectMetadataElementComboBox.addItem(null);
		for (LevelMetadataElement lme: this.selectedNode.getDynamicMetadataElementsWhichCanBeAdded())	this.selectMetadataElementComboBox.addItem(lme);
	}


	protected void showInInfoLabel(String message)
	{
		this.showInInfoLabel(message, false);
	}

	protected void showInInfoLabel(String message, boolean emphasized)
	{
		this.infoLabel.setForeground(emphasized? Color.RED: Color.BLACK);
		this.infoLabel.setText(message);
	}

	//	--------		Debugging			-------------------------------------------------------

	protected void systemOutDocument()
	{
		System.out.println(this.document);
	}


	protected void traceTree()
	{
		Tracer.trace(((NodeAbstract)this.getTreeTableModel().getRoot()).treeString(0));
	}

	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	/**
	 * This inner class contains the relevant data (the list of dragged Nodes, the dragged Node's parent path, and the source sip path)
	 * when dragging Nodes within the tree.
	 * @author denis
	 *
	 */
	private static class TreeNodeListTransferable implements Transferable
	{
		//	This is our own data flavor:
		static private final DataFlavor			DocuteamPackerTreeNodeListDataFlavor = new DataFlavor(Object.class, "DocuteamPackerTreeNodeListDataFlavor");
		static private final DataFlavor[]		SupportedDataFlavors = { DocuteamPackerTreeNodeListDataFlavor };


		/**
		 * The sipPath is to distinguish the drag source from the drag target.
		 */
		private String							sipPath;

		/**
		 * These are the dragged nodes.
		 */
		private List<NodeAbstract>				draggedNodes;

		/**
		 * This is the dragged node's parent path. All dragged nodes must have the same parent - if not, the drag is rejected.
		 * I need the parentPath for refreshing and expanding the path after the drag.
		 */
		private TreePath						parentPath;


		private TreeNodeListTransferable(JXTreeTable treeTable)
		{
			this.draggedNodes = new Vector<NodeAbstract>();
			for (int i: treeTable.getSelectedRows())	this.draggedNodes.add((NodeAbstract)(treeTable.getPathForRow(i).getLastPathComponent()));

			//	I can safely assume that this.draggedNodes is not empty because - how can one drag 0 nodes?:
			this.sipPath = this.draggedNodes.get(0).getDocument().getSIPFolder();
			this.parentPath = treeTable.getPathForRow(treeTable.getSelectedRows()[0]).getParentPath();
		}


		/* (non-Javadoc)
		 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
		 */
		@Override
		public DataFlavor[] getTransferDataFlavors()
		{
			return SupportedDataFlavors;
		}

		/* (non-Javadoc)
		 * @see java.awt.datatransfer.Transferable#isDataFlavorSupported(java.awt.datatransfer.DataFlavor)
		 */
		@Override
		public boolean isDataFlavorSupported(DataFlavor flavor)
		{
			//	This method seems never to be called???
			return DocuteamPackerTreeNodeListDataFlavor.equals(flavor);
		}

		/* (non-Javadoc)
		 * @see java.awt.datatransfer.Transferable#getTransferData(java.awt.datatransfer.DataFlavor)
		 */
		@Override
		public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException
		{
			if (DocuteamPackerTreeNodeListDataFlavor.equals(flavor))		return this;

			throw new UnsupportedFlavorException(flavor);
		}
	}


	/**
	 * This is my Transfer Handler for D&D:
	 */
	protected class TreeTableTransferHandler extends TransferHandler
	{
		/**
		 * 	Allow move only
		 */
		@Override
		public int getSourceActions(JComponent c)
		{
			return MOVE;	//	MOVE/COPY/COPY_OR_MOVE/LINK/NONE
		}


		/**
		 * Create an instance of our own Transferable out of the drag source component.
		 */
		@Override
		protected Transferable createTransferable(JComponent c)
		{
			return new TreeNodeListTransferable((JXTreeTable)c);
		}


		/**
		 * 	Only on drop, not on paste. Allow only our own data flavor and file lists. Allow dragging multiple items. Check source nodes and target node.
		 */
		@Override
		public boolean canImport(TransferSupport transferSupport)
		{
			//	If this document doesn't allow file operations, disable D&D:
			if (!SIPView.this.getDocument().areFileOperationsAllowed())						return false;

			//	Accept only drop:
			if (!transferSupport.isDrop())													return false;

			//	Don't allow insert, only "over":
			if (((JTable.DropLocation)transferSupport.getDropLocation()).isInsertRow())		return false;

			//	Accept only file lists or our own data flavor:
			if (!	(	transferSupport.isDataFlavorSupported(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor)
					||	transferSupport.isDataFlavorSupported(DataFlavor.javaFileListFlavor)))
																							return false;

			//	Check target to allow DnD:
			int targetIndex = ((JTable.DropLocation)transferSupport.getDropLocation()).getRow();

			//	This happens when the drag target is outside of the table:
			if (targetIndex == -1)															return false;

			//	Target node:
			NodeAbstract targetNode = (NodeAbstract)((JXTreeTable)transferSupport.getComponent()).getPathForRow(targetIndex).getLastPathComponent();

			//	Don't allow drop on files:
			if (targetNode.isFile())														return false;

			//	Don't allow drop if target or any of its predecessors is not readable or not writable:
			if (!targetNode.fileExists() || !targetNode.canRead() || !targetNode.canWrite()
				|| targetNode.hasPredecessorNotReadableOrWritableByCurrentUser())			return false;


			//	Now distinguish the different data flavors:
			if (transferSupport.isDataFlavorSupported(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor))
			{
				//	Moving nodes within packer:

				String targetSIP = SIPView.this.document.getSIPFolder();
				TreeNodeListTransferable transferData = null;
				String sourceSIP = "";
				try
				{
					transferData = (TreeNodeListTransferable)transferSupport.getTransferable().getTransferData(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor);
					sourceSIP = transferData.sipPath;
				}
				catch (UnsupportedFlavorException ex)
				{
					//	Ignore silently:
					if (LauncherView.IsDevelopingMode)		ex.printStackTrace();
					return false;
				}
				catch (IOException ex)
				{
					//	Ignore silently:
					if (LauncherView.IsDevelopingMode)		ex.printStackTrace();
					return false;
				}

				//	Don't allow dragging between different SIPs:
				if (!targetSIP.equals(sourceSIP))											return false;

				//	Source parent node:
				TreeTableNode sourceParent = transferData.draggedNodes.get(0).getParent();

				//	Don't allow dragging onto own parent:
				if (sourceParent == targetNode)												return false;

				//	Loop through all source nodes and test each of them:
				for (NodeAbstract node: transferData.draggedNodes)
				{
					//	Don't allow root as drag source (root can not be moved):
					if (node.isRoot())														return false;

					//	Make sure that all selected source nodes have the same parent. if not, reject the drag:
					if (node.getParent() != sourceParent)									return false;

					//	Don't allow dragging onto one of the source nodes:
					if (node == targetNode)													return false;

					//	Don't allow dragging onto a descendant of one of the source nodes:
					if (node.isFolder() && ((NodeFolder)node).isPredecessorOf(targetNode))	return false;

					//	Don't allow dragging if the node itself or any of its predecessors is not readable or not writable:
					if (!node.fileExists() || !node.canRead() || !node.canWrite()
						|| node.hasPredecessorNotReadableOrWritableByCurrentUser())			return false;

					//	Don't allow dragging if the node itself or any of its descendants is not readable or not writable:
					if (node.isFolder() && (((NodeFolder)node).hasDescendantNotReadableOrWritableByCurrentUser()))
																							return false;
				}

				return true;
			}
			else if (transferSupport.isDataFlavorSupported(DataFlavor.javaFileListFlavor))
			{
				//	Inserting files and folders from the file system or any other fileList provider:
				//	Always allow file lists:
				return true;
			}

			//	Reject any other data flavor:
			return false;
		}


		/**
		 * Actually import or move the data. Distinguish file lists (importing) and our own data flavor (moving nodes within the tree).
		 */
		@Override
		public boolean importData(TransferSupport transferSupport)
		{
			if (!this.canImport(transferSupport))	return false;

			//	Import files or move nodes within the tree?
			boolean isOK = false;
			if (transferSupport.isDataFlavorSupported(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor))
			{
				//	It is our own data flavor:
				isOK = this.moveNodesWithinSIP(transferSupport);
			}
			else if (transferSupport.isDataFlavorSupported(DataFlavor.javaFileListFlavor))
			{
				//	It is a file list:
				isOK = this.importFileList(transferSupport);
			}
			else
				return false;	//	Should actually never happen: flavor is neither javaFileListFlavor nor DocuteamPackerTreeNodeListDataFlavor.

			return isOK;
		}


		/**
		 * For cleanup after successful move. Not needed here for now.
		 */
		@Override
		public void exportDone(JComponent component, Transferable t, int action)
		{
		}


		/**
		 * Move nodes within the tree. This method is only called when the DataFlavor is our own (TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor).
		 * @param transferSupport
		 * @return
		 */
		private boolean moveNodesWithinSIP(TransferSupport transferSupport)
		{
			//	Target component:
			JXTreeTable treeTable = (JXTreeTable)transferSupport.getComponent();

			//	Target tree path:
			TreePath targetPath = treeTable.getPathForRow(((JTable.DropLocation)transferSupport.getDropLocation()).getRow());
			Logger.debug("D&D TargetPath       = " + targetPath);

			//	Target node:
			NodeFolder targetNode = (NodeFolder)targetPath.getLastPathComponent();
			Logger.debug("D&D TargetNode       = " + targetNode);

			//	Source node's parent path:
			TreePath sourceParentPath;

			//	Do the move:
			try
			{
				TreeNodeListTransferable transferData = (TreeNodeListTransferable)transferSupport.getTransferable().getTransferData(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor);

				//	Source parent path (I know that all source nodes have the same parent):
				sourceParentPath = transferData.parentPath;
				Logger.debug("D&D SourceParentPath = " + sourceParentPath);

				for (NodeAbstract n: transferData.draggedNodes)		n.moveTo(targetNode);
			}
			catch (java.lang.Exception e)
			{
				JOptionPane.showMessageDialog(SIPView.this, e.toString(), I18N.translate("TitleCantMoveItem"), JOptionPane.WARNING_MESSAGE);
				return false;
			}

			//	Refresh the view:
			TreeTableModel model = (TreeTableModel)treeTable.getTreeTableModel();

			model.refreshTreeStructure(sourceParentPath);
			model.refreshTreeStructure(targetPath);

			treeTable.expandPath(sourceParentPath);
			treeTable.expandPath(targetPath);

			return true;
		}


		/**
		 * Import files. This method is only called when the DataFlavor is DataFlavor.javaFileListFlavor.
		 * @param transferSupport
		 * @return
		 */
		@SuppressWarnings("unchecked")
		private boolean importFileList(TransferSupport transferSupport)
		{
			try
			{
				List<File> droppedFiles = (List<File>)transferSupport.getTransferable().getTransferData(DataFlavor.javaFileListFlavor);
				Logger.debug("D&D files = " + droppedFiles);

				//	Do the import:
				SIPView.this.importFilesAndFolders(droppedFiles);
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
				return false;
			}

			return true;
		}
	}



	/**
	 * @author denis
	 *
	 * This class is needed to translate between the View and the FileStructure tree.
	 */
	protected class TreeTableModel extends DefaultTreeTableModel
	{
		protected TreeTableModel(Document document)
		{
			super((document == null)
					? null
					: document.getStructureMap().getRoot(),
						Arrays.asList(I18N.translate("HeaderName"),
									I18N.translate("HeaderTreeViewSizeKB"),
									I18N.translate("HeaderTreeViewSize%"),
									I18N.translate("HeaderTreeViewSize%"),
									I18N.translate("HeaderTreeViewMandatoryStatus"),
									I18N.translate("HeaderTreeViewSubmitStatus")));
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public int getColumnCount()
		{
			return 6;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public int getHierarchicalColumn()
		{
			return 0;
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public Class<?> getColumnClass(int column)
		{
			switch (column)
			{
				case 0:		return String.class;
				case 1:		return Long.class;
				case 2:		return Integer.class;
				case 3:		return Object.class;		//	Must be Object so that the size gets rendered correctly as a bar
				case 4:		return Boolean.class;
				case 5:		return SubmitStatus.class;
			}

			return null;
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public Object getValueAt(Object node, int column)
		{
			NodeAbstract f = (NodeAbstract)node;

			switch (column)
			{
				case 0:		return f.getLabel();
				case 1:		return f.getSize() / 1024;
				case 2:		return f.getRelativeSize();
				case 3:		return f.getRelativeSize();
				case 4:		return f.hasDynamicMetadataElementInstancesWhichAreMandatoryButNotSet();
				case 5:		return f.getSubmitStatus();
			}

			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean isCellEditable(Object node, int column)
		{
			return false;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Object getChild(Object parent, int index)
		{
			return ((NodeAbstract)parent).getChildAt(index);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public int getChildCount(Object parent)
		{
			return ((NodeAbstract)parent).getChildCount();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public int getIndexOfChild(Object parent, Object child)
		{
			return ((NodeAbstract)parent).getIndex((NodeAbstract)child);
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean isLeaf(Object node)
		{
			return ((NodeAbstract)node).isFile();
		}


		/**
		 * Refresh this tree section in the view tree.
		 * @param path
		 */
		protected void refreshTreeStructure(TreePath path)
		{
			this.modelSupport.fireTreeStructureChanged(path);
		}


		/**
		 * Refresh this node in the view tree.
		 * @param path
		 */
		protected void refreshNode(TreePath path)
		{
			this.modelSupport.firePathChanged(path);
		}
	}


	static protected class RelativeSizeBarTableCellRenderer extends DefaultTableCellRenderer
	{
		//	100 vertical bars:
		static protected final String Bars = "||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||";
		static protected final int BarCharWidth = 5;			//	Magic Number: approximate width of the "|" character in pixels!

		protected TableColumn column;

		protected RelativeSizeBarTableCellRenderer(TableColumn column)
		{
			this.column = column;
		}

		@Override
		public void setValue(Object object)
		{
			if (object == null)		return;

			int columnWidth = this.column.getWidth();
			float relativeFileSize = new Integer(object.toString()).floatValue() / 100;

			int nrOfBars100 = columnWidth / BarCharWidth;
			int nrOfBars = (int)(relativeFileSize * nrOfBars100);
			String bars = Bars.substring(0, Math.min(Bars.length(), nrOfBars));

			this.setToolTipText("" + relativeFileSize);
			super.setValue(bars);
		}
	}


	static protected class HasMandatoryMetadataFieldsNotSetCellRenderer extends DefaultTableCellRenderer
	{
		@Override
		public void setValue(Object object)
		{
			if (object == null)		return;

			if ((Boolean)object)
			{
				super.setValue("!");
				this.setToolTipText(I18N.translate("MessageMandatoryFieldsNotSet"));
			}
			else
			{
				super.setValue("");
				this.setToolTipText(I18N.translate("MessageMandatoryFieldsAllSet"));
			}
		}
	}


	static protected class SubmitStatusTableCellRenderer extends DefaultTableCellRenderer
	{
		@Override
		public void setValue(Object object)
		{
			if (object == null)		return;

			SubmitStatus submitStatus = (SubmitStatus)object;
			switch (submitStatus)
			{
				case SubmitUndefined:
					this.setIcon(null);
					break;
				case SubmitRequested:
					this.setIcon(new ImageIcon("./resources/images/SubmitRequestedFlag.png"));
					break;
				case SubmitRequestPending:
					this.setIcon(new ImageIcon("./resources/images/SubmitRequestPendingFlag.png"));
					break;
				case SubmitFailed:
					this.setIcon(new ImageIcon("./resources/images/SubmitFailedFlag.png"));
					break;
				case Submitted:
					this.setIcon(new ImageIcon("./resources/images/SubmittedFlag.png"));
					break;
				default:
					this.setIcon(null);
					break;
			}

			this.setToolTipText(I18N.translate("HeaderPropertiesSubmitStatus") + " " + I18N.translate(submitStatus.toString()));
		}
	}



	/**
	 * I use this TreeCellRenderer to specify the icon, label, and tooltip text of each row.
	 * To set the foreground and background color of each row, I use Highlighters applied to the treeTable.
	 * @author denis
	 *
	 */
	static protected class MyTreeCellRenderer extends DefaultTreeCellRenderer
	{
		@Override
		public Component getTreeCellRendererComponent
		(
				JTree		tree,
				Object		value,
				boolean		sel,
				boolean		expanded,
				boolean		leaf,
				int			row,
				boolean		hasFocus
		)
		{
			MyTreeCellRenderer component = (MyTreeCellRenderer)super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);

			if (value == null)	return component;

			NodeAbstract node = (NodeAbstract)value;

			component.setText(node.getLabel());
			StringBuilder toolTipBuilder = new StringBuilder("<html>");
			toolTipBuilder
				.append(I18N.translate("HeaderName"))
				.append(" ")
				.append(node.getLabel())
				.append("<br>")
				.append(I18N.translate("LabelTitle"))
				.append(" ")
				.append(node.getUnitTitle())
				.append("<br>")
				.append(I18N.translate("LabelLevel"))
				.append(" ")
				.append(node.getLevel() != null? node.getLevel().getName(): "?")
				.append("<br>");

			try
			{
				if (!node.isAllowedBySA())
				{
					component.setIcon(IconNotAllowedBySA);
					toolTipBuilder
						.append("(")
						.append(I18N.translate("ToolTipIconNotAllowedBySA"))
						.append(")");
				}
				else if (!node.doesParentAllowMyLevel())
				{
					component.setIcon(IconLevelNotAllowed);
					toolTipBuilder
						.append("(")
						.append(I18N.translate("ToolTipLevelIconParentDoesntAllow", ((NodeFolder)node.getParent()).getLevel().getName(), node.getLevel().getName()))
						.append(")");
				}
				else if (node.getLevel() == null)
				{
					component.setIcon(IconLevelUnknown);
				}
				else
				{
					ImageIcon icon = node.getLevel().getIcon();
					if (icon == null)
					{
						component.setIcon(IconLevelUnknown);
						toolTipBuilder
							.append("(")
							.append(I18N.translate("ToolTipLevelIconCouldNotBeFound"))
							.append(")");
					}
					else
					{
						component.setIcon(icon);
					}
				}
			}
			catch (Exception x)
			{
				component.setIcon(IconLevelUnknown);
				toolTipBuilder
					.append("(")
					.append(I18N.translate("ToolTipLevelIconCouldNotBeLoaded"))
					.append(")");
			}

			if (!node.fileExists())
			{
				toolTipBuilder
					.append("<br>*****&nbsp;&nbsp;&nbsp;")
					.append(I18N.translate("ToolTipFileIsMissing"));
			}
			else if (!node.canRead())
			{
				toolTipBuilder
					.append("<br>*****&nbsp;&nbsp;&nbsp;")
					.append(I18N.translate("ToolTipFileIsNotReadable"));
			}
			else if (!node.canWrite())
			{
				toolTipBuilder
					.append("<br>*****&nbsp;&nbsp;&nbsp;")
					.append(I18N.translate("ToolTipFileIsReadOnly"));
			}

			component.setToolTipText(toolTipBuilder.toString());

			return component;
		}
	}



	@SuppressWarnings("serial")
	protected class FileDataViewTableModel extends AbstractTableModel
	{
		protected NodeAbstract		fileStructureNode;


		protected void setFileStructureNode(NodeAbstract fileStructureNode)
		{
			this.fileStructureNode = fileStructureNode;
			this.fireTableDataChanged();
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		@Override
		public int getRowCount()
		{
			return 12;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		@Override
		public int getColumnCount()
		{
			return 2;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		@Override
		public Object getValueAt(int rowIndex, int columnIndex)
		{
			if (columnIndex == 0)
			{
				switch (rowIndex)
				{
					case 0:		return I18N.translate("HeaderName");
					case 1:		return I18N.translate("HeaderPropertiesPath");
					case 2:		return I18N.translate("HeaderType");
					case 3:		return I18N.translate("HeaderPropertiesMIMEType");
					case 4:		return I18N.translate("HeaderPropertiesFormat");
					case 5:		return I18N.translate("HeaderPropertiesFormatKey");
					case 6:		return I18N.translate("HeaderPropertiesSizeKB");
					case 7:		return I18N.translate("HeaderPropertiesSize%");
					case 8:		return I18N.translate("HeaderPropertiesChildren");
					case 9:		return I18N.translate("HeaderPropertiesDescendants");
					case 10:	return I18N.translate("HeaderPropertiesEvents");
					case 11:	return I18N.translate("HeaderPropertiesSubmitStatus");
				}
			}
			else if (columnIndex == 1)
			{
				if (this.fileStructureNode == null)	return null;

				switch (rowIndex)
				{
					case 0:		return this.fileStructureNode.getLabel();
					case 1:		return this.fileStructureNode.getPathString();
					case 2:		return this.fileStructureNode.getType();
					case 3:		return this.fileStructureNode.getMimeType();
					case 4:		return this.fileStructureNode.getFormatName();
					case 5:		return this.fileStructureNode.getFormatKey();
					case 6:		return ((Long)(this.fileStructureNode.getSize() / 1024)).toString();
					case 7:		return this.fileStructureNode.getRelativeSize().toString();
					case 8:		return this.fileStructureNode.isFile()? "-": this.fileStructureNode.getChildCount();
					case 9:		return this.fileStructureNode.isFile()? "-": this.fileStructureNode.getDescendantCount();
					case 10:	return ((Integer)this.fileStructureNode.getMyEvents().size()).toString();
					case 11:	return I18N.translate(this.fileStructureNode.getSubmitStatus().toString());
				}
			}

			return null;
		}
	}


	@SuppressWarnings("serial")
	protected class EventListTableModel extends AbstractTableModel
	{
		protected NodeAbstract		fileStructureNode;


		protected void setFileStructureNode(NodeAbstract fileStructureNode)
		{
			this.fileStructureNode = fileStructureNode;
			this.fireTableDataChanged();
		}


		protected NodeAbstract getFileStructureNode()
		{
			return this.fileStructureNode;
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		@Override
		public int getRowCount()
		{
			if (this.fileStructureNode == null)		return 0;

			return this.fileStructureNode.getMyEvents().size();
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		@Override
		public int getColumnCount()
		{
			return 3;
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getColumnName(int column)
		{
			switch (column)
			{
				case 0:		return I18N.translate("HeaderTimestamp");
				case 1:		return I18N.translate("HeaderType");
				case 2:		return I18N.translate("HeaderEventOutcome");
			}

			return null;
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		@Override
		public Object getValueAt(int rowIndex, int columnIndex)
		{
			if (this.fileStructureNode == null)		return 0;

			Event event = this.fileStructureNode.getMyEvent(rowIndex);

			switch (columnIndex)
			{
				case 0:		return event.getDateTime();
				case 1:		return event.getType();
				case 2:		return event.getOutcome();
			}

			return null;
		}
	}



	@SuppressWarnings("serial")
	protected class EventDetailTableModel extends AbstractTableModel
	{
		protected Event		event;


		protected void setEvent(Event event)
		{
			this.event = event;
			this.fireTableDataChanged();
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		@Override
		public int getRowCount()
		{
			return 5;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		@Override
		public int getColumnCount()
		{
			return 2;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		@Override
		public Object getValueAt(int rowIndex, int columnIndex)
		{
			if (columnIndex == 0)
			{
				switch (rowIndex)
				{
					case 0:		return I18N.translate("HeaderTimestamp");
					case 1:		return I18N.translate("HeaderType");
					case 2:		return I18N.translate("HeaderEventDetail");
					case 3:		return I18N.translate("HeaderEventOutcome");
					case 4:		return I18N.translate("HeaderEventOutcomeDetail");
				}
			}
			else if (columnIndex == 1)
			{
				if (this.event == null)	return null;

				switch (rowIndex)
				{
					case 0:		return this.event.getDateTime();
					case 1:		return this.event.getType();
					case 2:		return this.event.getDetail();
					case 3:		return this.event.getOutcome();
					case 4:		return this.event.getOutcomeDetail();
				}
			}

			return null;
		}
	}


	/**
	 * This is a subclass of JTableWithSpecificCellEditorPerRow and overrides the method "public String getToolTipText(MouseEvent e)".
	 *
	 * @author Docuteam
	 *
	 */
	protected class MetadataTable extends JTableWithSpecificCellEditorPerRow
	{
		public MetadataTable(MetadataTableModel tableModel, int... toolTipTextColumnIndexes)
		{
			super(tableModel, toolTipTextColumnIndexes);

			//	Disallow rearranging of columns:
			this.getTableHeader().setReorderingAllowed(false);
		}

		//	--------		Interface			-------------------------------------------------------

		@Override
		public String getToolTipText(MouseEvent e)
		{
			switch (this.columnAtPoint(e.getPoint()))
			{
				case 0:		return I18N.translate("ToolTipDynamicMetadataTypes");
				case 1:		return ((MetadataTableModel)this.dataModel).getToolTipText(this.rowAtPoint(e.getPoint()));
			}

			//	Otherwise: show the content of the table cell as tooltip text:
			return super.getToolTipText(e);
		}
	}


	/**
	 * Metadata table model containing the method "getToolTipText(int i)" to return the specific toolip text of the corresponding metadataElementInstance.
	 * @author Docuteam
	 *
	 */
	@SuppressWarnings("serial")
	protected class MetadataTableModel extends AbstractTableModel implements TableModelWithSpecificCellEditorPerRow
	{
		static private final int			EditableColumn = 2;
		static private final int			ClickCountToStart = 2;

		private TextAreaTableCellEditor		TextAreaRenderer = new TextAreaTableCellEditor();
		private TextAreaTableCellEditor		TextAreaEditor = new TextAreaTableCellEditor();
		private DefaultCellEditor			ComboBoxEditor = new DefaultCellEditor(new JComboBox());


		protected MetadataTableModel()
		{
			//	Initialize the ComboBoxEditor:
			this.ComboBoxEditor.setClickCountToStart(ClickCountToStart);
		}


		protected NodeAbstract						fileStructureNode;
		protected List<MetadataElementInstance>		metadataElementInstances;


		/**
		 * Return the specific tooltip text of this MetadataElementInstance
		 * @param i
		 * @return
		 */
		protected String getToolTipText(int i)
		{
			return this.metadataElementInstances.get(i).getToolTipText();
		}


		/**
		 * Return the MetadataElement at the list position i.
		 * @param i
		 * @return
		 */
		protected MetadataElementInstance get(int i)
		{
			return this.metadataElementInstances.get(i);
		}


		protected void setFileStructureNode(NodeAbstract fileStructureNode)
		{
			if (fileStructureNode == null)
			{
				this.fileStructureNode = null;
				this.metadataElementInstances = null;
			}
			else
			{
				this.fileStructureNode = fileStructureNode;
				this.metadataElementInstances = fileStructureNode.getDynamicMetadataElementInstancesToBeDisplayed();
			}

			this.fireTableDataChanged();
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		@Override
		public int getRowCount()
		{
			if (this.metadataElementInstances == null)		return 0;

			return this.metadataElementInstances.size();
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		@Override
		public int getColumnCount()
		{
			return 3;
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getColumnName(int column)
		{
			switch (column)
			{
				case 0:		return I18N.translate("HeaderMetadataAttributes");
				case 1:		return I18N.translate("HeaderMetadataName");
				case 2:		return I18N.translate("HeaderMetadataValue");
			}

			return null;
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		@Override
		public Object getValueAt(int rowIndex, int columnIndex)
		{
			if (this.metadataElementInstances == null)		return null;

			MetadataElementInstance mdei = this.metadataElementInstances.get(rowIndex);

			switch (columnIndex)
			{
				case 0:		return mdei.getAttributesString();
				case 1:		return I18N.translate(mdei.getName());
				case 2:		return mdei.getDisplayValue();
			}

			return null;
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
		 */
		@Override
		public boolean isCellEditable(int rowIndex, int columnIndex)
		{
			if (columnIndex != EditableColumn)				return false;

			if (!SIPView.this.document.canWrite())			return false;

			if (!this.fileStructureNode.fileExists()
			||  !this.fileStructureNode.canRead()
			||  !this.fileStructureNode.canWrite())			return false;

			if (!this.fileStructureNode.doesSubmitStatusAllowEditing())
															return false;

			if (this.metadataElementInstances == null)		return false;

			if (!this.metadataElementInstances.get(rowIndex).isReadOnly())
															return true;

			return false;
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object, int, int)
		 */
		@Override
		public void setValueAt(Object value, int rowIndex, int columnIndex)
		{
			if (this.metadataElementInstances == null)		return;
			if (columnIndex != EditableColumn)				return;

			try
			{
				//	There are only two different types possible for value: String or KeyAndValue:
				if (value.getClass() == KeyAndValue.class)
					this.metadataElementInstances.get(rowIndex).setValue(((KeyAndValue)value).getOriginalString());
				else
					this.metadataElementInstances.get(rowIndex).setValue(value.toString());
			}
			catch (MetadataElementValidatorException ex)
			{
				JOptionPane.showMessageDialog(SIPView.this, ex.getMessage());
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}

			this.fireTableDataChanged();
		}


		/**
		 * This method returns a specific cell editor, depending on the row and column:
		 * If the row contains a MDElement whose allowedValues attribute is not null, return a ComboBox containing the supplied values as the cell editor.
		 * Otherwise return null, meaning the default cell editor.
		 * @param rowIndex
		 * @return
		 */
		@Override
		public TableCellEditor getCellEditor(int rowIndex, int columnIndex)
		{
			LevelMetadataElement lme = this.metadataElementInstances.get(rowIndex).getLevelMetadataElement();

			//	If the levelMetadataElement has several allowedValues, the editor is the ComboBoxEditor
			//		(takes precedence over the TextAreaEditor):
			List<KeyAndValue> allowedValues = lme.getMetadataElement().getAllowedValues();
			if (allowedValues != null && !allowedValues.isEmpty())
			{
				JComboBox comboBox = (JComboBox)this.ComboBoxEditor.getComponent();

				comboBox.removeAllItems();
				for (KeyAndValue item: allowedValues)		comboBox.addItem(item);

				//	If the first item of the allowedValues list is a "*", make this ComboBox editable:
				if (allowedValues.get(0).getOriginalString().equals("*"))
				{
					comboBox.removeItemAt(0);
					comboBox.setEditable(true);
				}
				else
				{
					comboBox.setEditable(false);
				}

				return this.ComboBoxEditor;
			}


			//	If the levelMetadataElement has a displayRows value >= 2, the editor is a TextAreaTableCellEditor:
			int displayRows = lme.getDisplayRows();
			if (displayRows >= 2)		return this.TextAreaEditor;

			//	Otherwise return null - this causes JTable to return the default cell editor.
			return null;
		}


		/**
		 * This method returns a specific cell renderer, depending on the row and column:
		 * If the row contains a MDElement whose allowedValues attribute is not null, return a ComboBox containing the supplied values as the cell editor.
		 * Otherwise return null, meaning the default cell editor.
		 * @param rowIndex
		 * @return
		 */
		@Override
		public TableCellRenderer getCellRenderer(int rowIndex, int columnIndex)
		{
			//	If the levelMetadataElement has a displayRows value >= 2, the renderer is a TextAreaTableCellEditor:
			int displayRows = this.metadataElementInstances.get(rowIndex).getLevelMetadataElement().getDisplayRows();
			if (displayRows >= 2)		return this.TextAreaRenderer;

			return null;
		}
	}



	static protected class TextAreaTableCellEditor extends AbstractCellEditor implements TableCellEditor, TableCellRenderer, KeyListener
	{
		static private final int		FontSizeMagicNumber = 5;
		static private final Font		Font = (Font)UIManager.get("Table.font");
		static private final int		FontSize = Font.getSize() + FontSizeMagicNumber;
		static private final Color		ForegroundColor = (Color)UIManager.get("Table.foreground");
		static private final Color		BackgroundColor = (Color)UIManager.get("Table.background");
		static private final Color		SelectionForegroundColor = (Color)UIManager.get("Table.selectionForeground");
		static private final Color		SelectionBackgroundColor = (Color)UIManager.get("Table.selectionBackground");


		private JTextArea				textArea;
		private JScrollPane				scrollPane;
		private JTable					table;


		private TextAreaTableCellEditor()
		{
			this.textArea = new JTextArea();
			this.textArea.setLineWrap(true);
			this.textArea.setWrapStyleWord(true);
			this.textArea.setFont(Font);
			this.textArea.addKeyListener(this);

			this.scrollPane = new JScrollPane(this.textArea);
			this.scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
			this.scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);
		}


		@Override
		public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column)
		{
			this.textArea.setText(value == null? "": value.toString());

			if (isSelected)
			{
				this.textArea.setForeground(SelectionForegroundColor);
				this.textArea.setBackground(SelectionBackgroundColor);
			}
			else
			{
				this.textArea.setForeground(ForegroundColor);
				this.textArea.setBackground(BackgroundColor);
			}

			int requiredRowHeight = FontSize * ((MetadataTableModel)table.getModel()).get(row).getLevelMetadataElement().getDisplayRows();
			if (requiredRowHeight != table.getRowHeight(row))		table.setRowHeight(row, requiredRowHeight);

			//	Hide the scrollbars in rendering mode, so return this.textArea instead of this.scrollPane:
			return this.textArea;
		}


		@Override
		public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column)
		{
			//	Remember the table I am in. I need this for setting the selection when loosing focus.
			if (this.table == null)		this.table = table;

			this.textArea.setText(value == null? "": value.toString());

			int requiredRowHeight = FontSize * ((MetadataTableModel)table.getModel()).get(row).getLevelMetadataElement().getDisplayRows();
			if (requiredRowHeight != table.getRowHeight(row))		table.setRowHeight(row, requiredRowHeight);

			//	Show the scrollbars in editing mode, so return this.scrolPane instead of this.textArea:
			return this.scrollPane;
		}


		@Override
		public Object getCellEditorValue()
		{
			return this.textArea.getText();
		}


		@Override
		public boolean isCellEditable(EventObject anEvent)
		{
			if (anEvent instanceof MouseEvent)		return ((MouseEvent)anEvent).getClickCount() >= MetadataTableModel.ClickCountToStart;

			return true;
		}

		//	-----	Key Listener methods to trap the TAB key:

		/* (non-Javadoc)
		 * @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
		 */
		@Override
		public void keyPressed(KeyEvent e)
		{
			//	Trap the TAB and Alt-Enter keys:
			if ((e.getKeyCode() == KeyEvent.VK_TAB)
				|| (e.getKeyCode() == KeyEvent.VK_ENTER && e.isAltDown()))
			{
				int editingRow = this.table.getEditingRow();

				e.consume();
				this.fireEditingStopped();

				//	On TAB key, move focus to next row or, if I am editing the last row, to the 1st row:
				if (e.getKeyCode() == KeyEvent.VK_TAB)
				{
					int nextEditingRow = editingRow + 1;
					if (nextEditingRow >= this.table.getRowCount())		nextEditingRow = 0;
					this.table.getSelectionModel().setSelectionInterval(nextEditingRow, nextEditingRow);
				}
			}
		}

		@Override
		public void keyTyped(KeyEvent e){}

		@Override
		public void keyReleased(KeyEvent e){}
	}


	protected class AssignLevelsByLayerDialog extends JDialog
	{
		protected boolean				goButtonWasClicked = false;

		protected final List<JComboBox>	comboBoxes;
		protected final JButton			goButton;

		//	===========================================================================================
		//	========	Methods					=======================================================
		//	===========================================================================================

		//	========	Static Initializer		=======================================================

		//	========	Constructors Public		=======================================================

		protected AssignLevelsByLayerDialog(JFrame owner, final NodeAbstract node)
		{
			super(owner, I18N.translate("TitleAssignLevelsByLayer"), true);

			this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			this.getRootPane().registerKeyboardAction(
					new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLayerDialog.this.close(); }},
					KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
					JComponent.WHEN_IN_FOCUSED_WINDOW);

			this.goButton = new JButton(new ImageIcon("./resources/images/Go.png"));
			this.goButton.setToolTipText(I18N.translate("ToolTipAssignLevelsByLayer"));
			this.goButton.setEnabled(false);
			this.goButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLayerDialog.this.goButtonClicked(node); }});

			//	The treeDepth is defined as 0 for a leaf and max(children.treeDepth) + 1 otherwise.
			//	Here I create one row for each level, so the number of rows is the treeDepth + 1.
			int treeDepth = node.getTreeDepth() + 1;
			this.comboBoxes = new Vector<JComboBox>(treeDepth);

			GridBagPanel gridBagPanel = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(2, 5, 0, 5));
			gridBagPanel.add(new JLabel(I18N.translate("LabelTreeLevel")),		0,    0,    GridBagConstraints.CENTER);
			gridBagPanel.add(new JLabel(I18N.translate("LabelAssignedLevel")),	0,    1,    GridBagConstraints.CENTER);

			int i;
			for (i = 1; i <= treeDepth; i++)
			{
				Vector<LevelOfDescription> levels = new Vector<LevelOfDescription>(10);
				levels.add(null);
				levels.addAll(node.getDocument().getLevels().getAll());
				JComboBox comboBox = new JComboBox(levels);
				comboBox.addActionListener(
						new ActionListener(){ @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLayerDialog.this.enableOrDisableGoButton(); }});
				this.comboBoxes.add(comboBox);

				gridBagPanel.add(new JLabel("" + i),			i+1,    0,    GridBagConstraints.CENTER);
				gridBagPanel.add(comboBox,						i+1,    1,    GridBagConstraints.WEST);
			}

			gridBagPanel.add(this.goButton,						i+1,    1,    GridBagConstraints.EAST);
			this.add(gridBagPanel);

			this.setResizable(false);
			this.pack();
			this.setLocationRelativeTo(owner);
			this.setVisible(true);
		}


		protected void goButtonClicked(NodeAbstract topNode)
		{
			ExceptionCollector.clear();

			int topNodeDepth = topNode.getDepth();

			for (NodeAbstract node: topNode.getWithDescendants())
			{
				int relativeNodeDepth = node.getDepth() - topNodeDepth;
				LevelOfDescription selectedLevel = (LevelOfDescription)this.comboBoxes.get(relativeNodeDepth).getSelectedItem();
				if (selectedLevel == null)		continue;

				try
				{
					node.setLevel(selectedLevel);
				}
				catch (Exception ex)
				{
					ch.docuteam.tools.exception.Exception.remember(ex);
				}
			}

			this.goButtonWasClicked = true;
			this.close();

			if (!ExceptionCollector.isEmpty())		ExceptionCollector.systemOut();
		}


		protected void close()
		{
			this.setVisible(false);
			this.dispose();
		}


		protected void enableOrDisableGoButton()
		{
			this.goButton.setEnabled(!this.areAllComboBoxesEmpty());
		}


		protected boolean areAllComboBoxesEmpty()
		{
			for (JComboBox comboBox: this.comboBoxes)		if (comboBox.getSelectedItem() != null)		return false;

			return true;
		}
	}


	protected class AssignLevelsByLabelDialog extends JDialog
	{
		protected boolean				goButtonWasClicked = false;

		protected final JComboBox		comboBoxSeries;
		protected final JComboBox		comboBoxDossier;
		protected final JComboBox		comboBoxItem;
		protected final JButton			goButton;

		//	===========================================================================================
		//	========	Methods					=======================================================
		//	===========================================================================================

		//	========	Static Initializer		=======================================================

		//	========	Constructors Public		=======================================================

		protected AssignLevelsByLabelDialog(JFrame owner, final NodeAbstract node)
		{
			super(owner, I18N.translate("TitleAssignLevelsByLabel"), true);

			this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			this.getRootPane().registerKeyboardAction(
					new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLabelDialog.this.close(); }},
					KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
					JComponent.WHEN_IN_FOCUSED_WINDOW);

			Vector<LevelOfDescription> levels = new Vector<LevelOfDescription>(10);
			levels.add(null);
			levels.addAll(node.getDocument().getLevels().getAll());
			this.comboBoxSeries = new JComboBox(levels);
			this.comboBoxSeries.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLabelDialog.this.enableOrDisableGoButton(); }});

			levels = new Vector<LevelOfDescription>(10);
			levels.add(null);
			levels.addAll(node.getDocument().getLevels().getAll());
			this.comboBoxDossier = new JComboBox(levels);
			this.comboBoxDossier.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLabelDialog.this.enableOrDisableGoButton(); }});

			levels = new Vector<LevelOfDescription>(10);
			levels.add(null);
			levels.addAll(node.getDocument().getLevels().getAll());
			this.comboBoxItem = new JComboBox(levels);
			this.comboBoxItem.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLabelDialog.this.enableOrDisableGoButton(); }});

			this.goButton = new JButton(new ImageIcon("./resources/images/Go.png"));
			this.goButton.setToolTipText(I18N.translate("ToolTipAssignLevelsByLabel"));
			this.goButton.setEnabled(false);
			this.goButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e) { AssignLevelsByLabelDialog.this.goButtonClicked(node); }});

			GridBagPanel gridBagPanel = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(2, 5, 0, 5));
			gridBagPanel.add(new JLabel(I18N.translate("LabelFunctionalLevel")),	0,    0,    GridBagConstraints.CENTER);
			gridBagPanel.add(new JLabel(I18N.translate("LabelAssignedLevel")),		0,    1,    GridBagConstraints.CENTER);
			gridBagPanel.add(new JLabel(I18N.translate("LabelSeries")),				2,    0,    GridBagConstraints.CENTER);
			gridBagPanel.add(this.comboBoxSeries,									2,    1,    GridBagConstraints.WEST);
			gridBagPanel.add(new JLabel(I18N.translate("LabelDossier")),			3,    0,    GridBagConstraints.CENTER);
			gridBagPanel.add(this.comboBoxDossier,									3,    1,    GridBagConstraints.WEST);
			gridBagPanel.add(new JLabel(I18N.translate("LabelItem")),				4,    0,    GridBagConstraints.CENTER);
			gridBagPanel.add(this.comboBoxItem,										4,    1,    GridBagConstraints.WEST);
			gridBagPanel.add(this.goButton,											5,    1,    GridBagConstraints.EAST);
			this.add(gridBagPanel);

			this.setResizable(false);
			this.pack();
			this.setLocationRelativeTo(owner);
			this.setVisible(true);
		}


		protected void goButtonClicked(NodeAbstract topNode)
		{
			if (!this.areAllComboBoxesSet())		return;

			ExceptionCollector.clear();

			for (NodeAbstract node: topNode.getWithDescendants())
			{
				try
				{
					//	set the node's level according to it's type (file or folder) or label:
					if (node.isFile())											node.setLevel((LevelOfDescription)this.comboBoxItem.getSelectedItem());
					else if (((NodeFolder)node).doesLabelHaveNumericPrefix())	node.setLevel((LevelOfDescription)this.comboBoxSeries.getSelectedItem());
					else														node.setLevel((LevelOfDescription)this.comboBoxDossier.getSelectedItem());
				}
				catch (Exception ex)
				{
					ch.docuteam.tools.exception.Exception.remember(ex);
				}
			}

			this.goButtonWasClicked = true;
			this.close();

			if (!ExceptionCollector.isEmpty())		ExceptionCollector.systemOut();
		}


		protected void close()
		{
			this.setVisible(false);
			this.dispose();
		}


		protected void enableOrDisableGoButton()
		{
			this.goButton.setEnabled(this.areAllComboBoxesSet());
		}


		protected boolean areAllComboBoxesSet()
		{
			return (this.comboBoxSeries.getSelectedItem() != null
				&&  this.comboBoxDossier.getSelectedItem() != null
				&&  this.comboBoxItem.getSelectedItem() != null);
		}
	}



	static protected class TestOrAssignSADialog extends JDialog
	{
		protected JComboBox							saComboBox;
		protected JButton							setSAButton;
		protected JButton							deleteAllButton;
		protected JButton							markAllButton;
		protected JLabel							badFilesCountLabel;

		protected JTable							badFilesTable;

		protected Overview							selectedSAOverview;
		protected boolean							setSAButtonWasClicked = false;

		protected SIPView							sipView;


		static protected TestOrAssignSADialog open(SIPView sipView)
		{
			TestOrAssignSADialog dialog = new TestOrAssignSADialog(sipView);

			dialog.setSAButtonWasClicked = false;
			dialog.saComboBox.setSelectedIndex(0);

			dialog.setVisible(true);

			//	Return the dialog after it was closed:
			return dialog;
		}


		private TestOrAssignSADialog(SIPView sipView)
		{
			super(sipView, I18N.translate("TitleTestOrAssignSADialog"), true);
			this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));

			this.sipView = sipView;

			this.saComboBox = new JComboBox(this.getAllFinalSAOverviews().toArray());
			this.saComboBox.setToolTipText(I18N.translate("ToolTipSelectSA"));
			this.saComboBox.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)	{ TestOrAssignSADialog.this.saSelected(); }});

			this.setSAButton = new JButton(new ImageIcon("./resources/images/Go.png"));
			this.setSAButton.setToolTipText(I18N.translate("ToolTipSetSA"));
			this.setSAButton.addActionListener(
				new ActionListener(){ @Override public void actionPerformed(ActionEvent e)	{ TestOrAssignSADialog.this.setSAButtonClicked(); }});

			this.deleteAllButton = new JButton(new ImageIcon("./resources/images/Delete.png"));
			this.deleteAllButton.setToolTipText(I18N.translate("ToolTipDeleteAllBadFiles"));
			this.deleteAllButton.addActionListener(
				new ActionListener(){ @Override public void actionPerformed(ActionEvent e)	{ TestOrAssignSADialog.this.deleteAllButtonClicked(); }});

			this.markAllButton = new JButton(new ImageIcon("./resources/images/Mark.png"));
			this.markAllButton.setToolTipText(I18N.translate("ToolTipMarkAllBadFiles"));
			this.markAllButton.addActionListener(
				new ActionListener(){ @Override public void actionPerformed(ActionEvent e)	{ TestOrAssignSADialog.this.markAllButtonClicked(); }});

			this.badFilesTable = new JTable(new BadFilesTableModel());
			this.badFilesTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			this.badFilesTable.getColumnModel().getColumn(1).setMaxWidth(30);

			this.badFilesCountLabel = new JLabel();

			Box buttonBox = new Box(BoxLayout.X_AXIS);
			buttonBox.add(this.setSAButton);
			buttonBox.add(Box.createHorizontalStrut(10));
			buttonBox.add(this.markAllButton);
			buttonBox.add(this.deleteAllButton);

			GridBagPanel gridBag = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(5, 5, 5, 5));
			gridBag.add(new JLabel(I18N.translate("LabelSelectSA")),				1, 1, 1, 2,	GridBagConstraints.WEST);
			gridBag.add(this.saComboBox,											2, 2, 1, 2,	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(new JLabel(I18N.translate("LabelBadFilesTable")),			3, 3, 1, 2,	GridBagConstraints.WEST);
			gridBag.add(new JScrollPane(this.badFilesTable),						4, 4, 1, 2,	GridBagConstraints.CENTER,		GridBagConstraints.BOTH,		1, 1);
			gridBag.add(this.badFilesCountLabel,									5,    1,   	GridBagConstraints.WEST);
			gridBag.add(buttonBox,													5,    2,   	GridBagConstraints.EAST);

			this.add(gridBag);

			//	The ESC key closes this View:
			gridBag.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), "Close");
			gridBag.getActionMap().put("Close",
				new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { TestOrAssignSADialog.this.close(); }});

			this.setPreferredSize(new Dimension(600, 400));
			this.pack();

			this.setLocationRelativeTo(sipView);

			//	Select the 1st SAOverview in the comboBox, which is the one used in my document (or null):
			this.saComboBox.setSelectedIndex(0);
		}


		protected void saSelected()
		{
			this.selectedSAOverview = (Overview)this.saComboBox.getSelectedItem();
			if (this.selectedSAOverview == null)
			{
				((BadFilesTableModel)this.badFilesTable.getModel()).getList().clear();
				this.badFilesCountLabel.setText("0");
				this.enableOrDisableButtonsAndFields();
				return;
			}

			SubmissionAgreement selectedSA = null;
			try
			{
				selectedSA = this.selectedSAOverview.getSubmissionAgreement();
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}

			if (selectedSA == null)
			{
				((BadFilesTableModel)this.badFilesTable.getModel()).getList().clear();
				this.badFilesCountLabel.setText("0");
				this.enableOrDisableButtonsAndFields();
				return;
			}

			List<NodeFile> badFiles = this.sipView.getDocument().filesNotAllowedBySubmissionAgreement(selectedSA, this.selectedSAOverview.dssId);
			((BadFilesTableModel)this.badFilesTable.getModel()).setList(badFiles);
			this.badFilesCountLabel.setText("" + badFiles.size());

			this.enableOrDisableButtonsAndFields();
		}


		protected void setSAButtonClicked()
		{
			if (this.selectedSAOverview == null)		return;

			List<NodeFile> badFiles = null;
			try
			{
				badFiles = this.sipView.getDocument().setSubmissionAgreement(this.selectedSAOverview.saId, this.selectedSAOverview.dssId);
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}

			if (badFiles == null)
			{
				((BadFilesTableModel)this.badFilesTable.getModel()).getList().clear();
				this.badFilesCountLabel.setText("0");
				this.enableOrDisableButtonsAndFields();
				return;
			}

			((BadFilesTableModel)this.badFilesTable.getModel()).setList(badFiles);
			this.badFilesCountLabel.setText("" + badFiles.size());
			this.setSAButtonWasClicked = true;

			this.enableOrDisableButtonsAndFields();
		}


		protected void deleteAllButtonClicked()
		{
			for (NodeFile nodeFile: ((BadFilesTableModel)this.badFilesTable.getModel()).getList())
				try
				{
					nodeFile.delete();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}

			((BadFilesTableModel)this.badFilesTable.getModel()).clearList();
			this.badFilesCountLabel.setText("0");

			this.enableOrDisableButtonsAndFields();
		}


		protected void markAllButtonClicked()
		{
			for (NodeFile nodeFile: ((BadFilesTableModel)this.badFilesTable.getModel()).getList())		nodeFile.setIsNotAllowedBySA();

			((BadFilesTableModel)this.badFilesTable.getModel()).fireTableDataChanged();
		}


		protected void close()
		{
			this.setVisible(false);
			this.dispose();
		}


		protected List<Overview> getAllFinalSAOverviews()
		{
			List<Overview> saOverviewList = SubmissionAgreement.getAllFinalOverviews();
			saOverviewList.add(0, null);

			//	Search the SAOverview that matches the SA used in my document. If found, move it to the begin of the list:
			for (Overview overview: saOverviewList)
				if (overview != null
					&& overview.saId.equals(this.sipView.getDocument().getSAId()) && overview.dssId.equals(this.sipView.getDocument().getDSSId()))
				{
					saOverviewList.remove(overview);
					saOverviewList.add(0, overview);
					break;
				}

			return saOverviewList;
		}


		protected void enableOrDisableButtonsAndFields()
		{
			if (this.selectedSAOverview == null)
			{
				this.setSAButton.setEnabled(false);
				this.markAllButton.setEnabled(false);
				this.deleteAllButton.setEnabled(false);
				return;
			}

			this.setSAButton.setEnabled(true);

			if (this.setSAButtonWasClicked)
			{
				this.markAllButton.setEnabled(true);
				this.deleteAllButton.setEnabled(true);
			}
			else
			{
				this.markAllButton.setEnabled(false);
				this.deleteAllButton.setEnabled(false);
			}
		}



		@SuppressWarnings("serial")
		static protected class BadFilesTableModel extends AbstractTableModel
		{
			List<NodeFile> badFiles = new ArrayList<NodeFile>();

			public void clearList()
			{
				this.badFiles.clear();
				this.fireTableDataChanged();
			}

			public void setList(List<NodeFile> badFiles)
			{
				this.badFiles = badFiles;
				this.fireTableDataChanged();
			}

			public List<NodeFile> getList()
			{
				return this.badFiles;
			}

			public NodeFile get(int i)
			{
				return this.badFiles.get(i);
			}


			@Override
			public int getRowCount()
			{
				return this.badFiles.size();
			}

			@Override
			public int getColumnCount()
			{
				return 2;
			}

			@Override
			public String getColumnName(int columnIndex)
			{
				switch (columnIndex)
				{
					case 0:		return I18N.translate("HeaderNodePath");
					case 1:		return I18N.translate("HeaderNodeIsAllowedBySA");
				}

				return null;
			}

			@Override
			public Class<?> getColumnClass(int columnIndex)
			{
				switch (columnIndex)
				{
					case 0:		return String.class;
					case 1:		return Boolean.class;
				}

				return null;
			}

			@Override
			public Object getValueAt(int rowIndex, int columnIndex)
			{
				NodeFile nodeFile = this.badFiles.get(rowIndex);
				if (nodeFile == null)		return null;

				switch (columnIndex)
				{
					case 0:		return nodeFile.getPathString();
					case 1:		return !nodeFile.isAllowedBySA();
				}

				return null;
			}
		}
	}


	protected class SearchPanel extends JPanel
	{
		protected JTextField			searchTextField;
		protected JTextField			hitCountTextField;

		protected Action				searchAction;
		protected Action				clearSearchTextFieldAction;
		protected Action				selectNextHitAction;
		protected Action				selectPreviousHitAction;

		protected List<NodeAbstract>	hits = new Vector<NodeAbstract>();
		protected int					currentHitSelectionIndex = -1;


		protected SearchPanel()
		{
			super(new BorderLayout());

			this.searchTextField = new JTextField();
			this.searchTextField.addActionListener(
					new ActionListener() { @Override public void actionPerformed(ActionEvent e) { SearchPanel.this.searchButtonClicked(); }});
			this.searchTextField.addKeyListener(
					new KeyAdapter() { @Override public void keyTyped(KeyEvent e) { SearchPanel.this.enableOrDisableActions(); }});
			this.searchTextField.setToolTipText(I18N.translate("ToolTipSearchTextField"));

			this.searchTextField.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, InputEvent.CTRL_DOWN_MASK), "SearchNext");
			this.searchTextField.getActionMap().put("SearchNext",
				new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { SearchPanel.this.selectNextButtonClicked(); }});

			this.searchTextField.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_UP, InputEvent.CTRL_DOWN_MASK), "SearchPrevious");
			this.searchTextField.getActionMap().put("SearchPrevious",
				new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { SearchPanel.this.selectPreviousButtonClicked(); }});

			this.searchAction = new AbstractAction(I18N.translate("ActionSearch"), new ImageIcon("./resources/images/Search.png"))
			{	@Override public void actionPerformed(ActionEvent e){ SearchPanel.this.searchButtonClicked(); }};
			this.searchAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearch"));

			this.clearSearchTextFieldAction = new AbstractAction(I18N.translate("ActionSearchClearTextField"), new ImageIcon("./resources/images/Clear.png"))
			{	@Override public void actionPerformed(ActionEvent e){ SearchPanel.this.clearSearchTextFieldButtonClicked(); }};
			this.clearSearchTextFieldAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearchClearTextField"));

			this.selectNextHitAction = new AbstractAction(I18N.translate("ActionSearchSelectNext"), new ImageIcon("./resources/images/SearchNext.png"))
			{	@Override public void actionPerformed(ActionEvent e){ SearchPanel.this.selectNextButtonClicked(); }};
			this.selectNextHitAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearchSelectNext"));

			this.selectPreviousHitAction = new AbstractAction(I18N.translate("ActionSearchSelectPrevious"), new ImageIcon("./resources/images/SearchPrevious.png"))
			{	@Override public void actionPerformed(ActionEvent e){ SearchPanel.this.selectPreviousButtonClicked(); }};
			this.selectPreviousHitAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearchSelectPrevious"));

			this.hitCountTextField = new JTextField();
			this.hitCountTextField.setEnabled(false);
			this.hitCountTextField.setColumns(4);

			JButton searchButton = new JButton(this.searchAction);
			searchButton.setHideActionText(true);
			JButton clearSearchTextFieldButton = new JButton(this.clearSearchTextFieldAction);
			clearSearchTextFieldButton.setHideActionText(true);
			JButton selectNextHitButton = new JButton(this.selectNextHitAction);
			selectNextHitButton.setHideActionText(true);
			JButton selectPreviousHitButton = new JButton(this.selectPreviousHitAction);
			selectPreviousHitButton.setHideActionText(true);

			Box buttonBox = new Box(BoxLayout.X_AXIS);
			buttonBox.add(searchButton);
			buttonBox.add(clearSearchTextFieldButton);
			buttonBox.add(selectNextHitButton);
			buttonBox.add(selectPreviousHitButton);
			buttonBox.add(this.hitCountTextField);

			this.add(this.searchTextField, BorderLayout.CENTER);
			this.add(buttonBox, BorderLayout.EAST);

			this.enableOrDisableActions();
		}


		private void searchButtonClicked()
		{
			if (this.searchTextField.getText().trim().isEmpty())
			{
				this.hits.clear();
				this.currentHitSelectionIndex = -1;
			}
			else
			{
				this.hits = SIPView.this.document.searchFor(this.searchTextField.getText());

				if (this.hits.isEmpty())
				{
					this.currentHitSelectionIndex = -1;
				}
				else
				{
					this.currentHitSelectionIndex = 0;
					this.selectNode(this.hits.get(this.currentHitSelectionIndex));
				}
			}

			this.enableOrDisableActions();
		}


		private void clearSearchTextFieldButtonClicked()
		{
			this.searchTextField.setText("");
			this.searchButtonClicked();
		}


		private void selectNextButtonClicked()
		{
			if (this.hits.isEmpty())		return;

			if (this.currentHitSelectionIndex < this.hits.size() - 1)		++this.currentHitSelectionIndex;
			this.selectNode(this.hits.get(this.currentHitSelectionIndex));

			this.enableOrDisableActions();
		}


		private void selectPreviousButtonClicked()
		{
			if (this.hits.isEmpty())		return;

			if (this.currentHitSelectionIndex > 0)							--this.currentHitSelectionIndex;
			this.selectNode(this.hits.get(this.currentHitSelectionIndex));

			this.enableOrDisableActions();
		}


		private void selectNode(NodeAbstract node)
		{
			if (node == null)		return;

			SIPView.this.selectNode(node);
		}


		private void enableOrDisableActions()
		{
			if (this.searchTextField.getText().isEmpty())
			{
				this.clearSearchTextFieldAction.setEnabled(false);
				this.searchAction.setEnabled(false);
			}
			else
			{
				this.clearSearchTextFieldAction.setEnabled(true);
				this.searchAction.setEnabled(true);
			}

			if (this.hits.isEmpty())
			{
				this.selectNextHitAction.setEnabled(false);
				this.selectPreviousHitAction.setEnabled(false);
			}
			else
			{
				if (this.currentHitSelectionIndex > 0)						this.selectPreviousHitAction.setEnabled(true);
				else														this.selectPreviousHitAction.setEnabled(false);

				if (this.currentHitSelectionIndex < this.hits.size() - 1)	this.selectNextHitAction.setEnabled(true);
				else														this.selectNextHitAction.setEnabled(false);
			}

			this.updateHitCountTextField();
		}


		private void updateHitCountTextField()
		{
			this.hitCountTextField.setText((this.currentHitSelectionIndex + 1) + "/" + this.hits.size());
		}
	}

}
