/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

import bsh.EvalError;
import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.util.KeyAndValue;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;
import ch.docuteam.tools.string.ToolTipText;
import ch.docuteam.tools.translations.I18N;
import ch.docuteam.tools.util.JavaInterpreter;

/**
 * A MetadataElement specifies an abstract template for a data value belonging to a SIP node.
 * The metadata elements are initialized through the file "./config/levels.xml".
 * A metadata element has an id (= accessor name) and, calculated from the accessor name, a setter and getter method for setting and retrieving its value.
 * <p>
 * It is <b>NOT</b> a MetadataElement that gets attached to a LevelOfDescription, but a <a href="LevelMetadataElement.html">LevelMetadataElement</a>
 * which adds some level-specific properties to the MetadataElement.
 * <p>
 * Optionally, several validator classes can be supplied. This classes must implement the interface <a href="MetadataElementValidator.html">MetadataElementValidator</a>;
 * instances are used to validate manual changes of this metadata element value.
 * <p>
 * Optionally, several postaction classes can be supplied. This classes must implement the interface <a href="MetadataElementSetterPostAction.html">MetadataElementSetterPostAction</a>;
 * instances are used to execute after manual changes of this metadata element value.
 * <p>
 * Optionally, a default java expression can be supplied, with which an instance of this metadata element is initialized.
 * <p>
 * Optionally, a List&lt;String&gt; of allowed values can be supplied. This list is then shown as a drop-down list in the GUI.
 * <p>
 * Optionally, a tooltip text can be supplied, which is shown in the GUI when the mouse hovers over this element.
 *
 * @author denis
 *
 */
public class MetadataElement implements ToolTipText
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	A Map containing all MetadataElements. Access key is the accessorName.
	static private Map<String, MetadataElement>		All = new LinkedHashMap<String, MetadataElement>(71);

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String									accessorName;

	//	Getting and setting values:
	private Method									setterMethod;
	private Method									getterMethod;

	//	Validate the value (optional, separated by AllowedValuesSeparator):
	private List<ValidatorInstance>					validatorInstances = new ArrayList<ValidatorInstance>();

	//	Specify the post-actions (optional, separated by AllowedValuesSeparator):
	private List<PostActionInstance>				postActionInstances = new ArrayList<PostActionInstance>();

	//	Default value (optional):
	private String									defaultExpression;

	//	Allowed values (optional):
	private List<KeyAndValue>						allowedValues;

	private String									toolTipText;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	MetadataElement(Class<?> targetClass, String accessorName, String validatorAttribute, String postActionAttribute, String defaultExpression, String allowedValues) throws SecurityException, NoSuchMethodException, InstantiationException, IllegalAccessException
	{
		this.accessorName = accessorName;

		String accessorNameCapitalized = StringUtil.first(accessorName, 1).toUpperCase() + accessorName.substring(1);
		this.getterMethod = targetClass.getMethod("get" + accessorNameCapitalized);
		this.setterMethod = targetClass.getMethod("set" + accessorNameCapitalized, List.class);

		if (validatorAttribute!= null)
		{
			String[] validatorClassNames = validatorAttribute.trim().split(LevelOfDescription.getAllowedValuesSeparator());
			for (String validatorClassName : validatorClassNames)
			{
				try
				{
					this.validatorInstances.add(new ValidatorInstance(validatorClassName));
				}
				catch (ClassNotFoundException ex)
				{
					//	In case of a bad validator class, remember the exception and continue.
					//	So keep this Metadata Element and just ignore its bad validator.
					ch.docuteam.tools.exception.Exception.remember(new MetadataElementValidatorClassNotFoundException(this.accessorName, ex));
				}
			}
		}

		if (postActionAttribute != null)
		{
			String[] postActionClassNames = postActionAttribute.trim().split(LevelOfDescription.getAllowedValuesSeparator());
			for (String postActionClassName : postActionClassNames)
			{
				try
				{
					this.postActionInstances.add(new PostActionInstance(postActionClassName));
				}
				catch (ClassNotFoundException ex)
				{
					//	In case of a bad post-action class, remember the exception and continue.
					//	So keep this Metadata Element and just ignore its bad post-action.
					ch.docuteam.tools.exception.Exception.remember(new MetadataElementSetterPostActionClassNotFoundException(this.accessorName, ex));
				}
			}
		}

		this.defaultExpression = defaultExpression;

		if (allowedValues != null)
		{
			this.allowedValues = new ArrayList<KeyAndValue>(StringUtil.occurrencesOf(LevelOfDescription.getAllowedValuesSeparator(), allowedValues) + 1);
			for (String s: allowedValues.split(LevelOfDescription.getAllowedValuesSeparator()))		this.allowedValues.add(new KeyAndValue(s.trim()));
		}
		else
		{
			this.allowedValues = new ArrayList<KeyAndValue>();
		}

		try
		{
			this.toolTipText = I18N.translate(this.accessorName + "ToolTip");
			if (this.toolTipText.isEmpty())		this.toolTipText = I18N.translate(this.accessorName);
		}
		catch (Exception ex)
		{
			this.toolTipText = this.accessorName;
		}

		All.put(accessorName, this);

		Logger.getLogger().debug("Created: " + this);
	}

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static public void clear()
	{
		All.clear();
	}

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * Return a MetadataElement instance with this accessorName, or null if it doesn't exist.
	 */
	static public MetadataElement get(String accessorName)
	{
		initializeIfNecessary();
		return All.get(accessorName);
	}

	/**
	 * Return all MetadataElements as a Map (key is the accessorName).
	 */
	static public Map<String, MetadataElement> getAll()
	{
		initializeIfNecessary();
		return All;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------

	static private void initializeIfNecessary()
	{
		if (All.isEmpty())		LevelOfDescription.initializeIfNecessary();
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	public String getId()
	{
		return this.accessorName;
	}

	public String getAccessorName()
	{
		return this.accessorName;
	}

	public List<KeyAndValue> getAllowedValues()
	{
		return this.allowedValues;
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean isPostActionDefined()
	{
		return this.postActionInstances.size() > 0;
	}

	//	--------		Interface			-------------------------------------------------------

	@Override
	public String getToolTipText()
	{
		return this.toolTipText;
	}

	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * Send the getter method to the passed NodeAbstract's c element and return the return value of this method.
	 */
	@SuppressWarnings("unchecked")
	public List<String> getValueFromNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		List<String> value = (List<String>)this.getterMethod.invoke(node.getMyDMDSectionWithEAD().getC());
		return value;
	}

	/**
	 * Send the setter method to the passed NodeAbstract's c element with the parameter 'value'.
	 */
	public void setValueInNode(List<String> value, NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		Logger.getLogger().debug("Setting in:" + node + " metadata element:<" + this.accessorName + "> to:'" + value + "'");

		this.setterMethod.invoke(node.getMyDMDSectionWithEAD().getC(), value);
	}


	/**
	 * Evaluate the String 'value' in the context of the object 'o'. Always allow null value and empty string.
	 * @param value The value to be tested
	 * @param node This is the context of the value (i.e. the container object). It may be an arbitraty object or null, the validator must be able to handle it.
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementValidatorException
	 * @throws MetadataElementAllowedValuesException
	 */
	public void validateValueAgainstNode(String value, NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, MetadataElementAllowedValuesException
	{
		Logger.getLogger().debug("Validating in:" + node + " metadata element:<" + this.accessorName + "> value:'" + value + "'");

		//	NOTE: If the allowedValues don't allow an empty string, then raise an exception:

		//	Check against allowed values:
		if (!this.allowedValues.isEmpty()
		&& !this.allowedValues.get(0).getOriginalString().equals("*"))
		{
			boolean found = false;
			for (KeyAndValue knv: this.allowedValues)
			{
				if (knv.getOriginalString().equals(value))
				{
					found = true;
					break;
				}
			}

			if (!found)		throw new MetadataElementAllowedValuesException(value, this);
		}

		//	Check against validators:
		if (value == null)						return;		//	Always allow null
		if (value.isEmpty())					return;		//	Always allow empty value
		if (this.validatorInstances.size() == 0)		return;

		try
		{
			//	The exception "MetadataElementValidatorException" that might be thrown, is wrapped within an InvocationTargetException.
			//	If this is the case, unwrap and rethrow that exception.
			//	If this is not the case, just rethrow the original exception.
			for (ValidatorInstance validatorInstance : this.validatorInstances)
			{
				validatorInstance.check(value, node, this);
			}
		}
		catch (InvocationTargetException ex)
		{
			if (MetadataElementValidatorException.class == ex.getTargetException().getClass())
				throw (MetadataElementValidatorException)ex.getTargetException();
			else
				throw ex;
		}
	}


	/**
	 * Execute the post-action in the context of the node 'node'.
	 * @param oldValue The old value
	 * @param newValue The new value
	 * @param node This is the context of the value (i.e. the container object). It may be an arbitraty object or null, the validator must be able to handle it.
	 * @return Null if all went well, an error message otherwise.
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementSetterPostActionException
	 */
	public void executePostActionInNode(String oldValue, String newValue, NodeAbstract node, int index) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementSetterPostActionException
	{
		if (this.postActionInstances.size() == 0)	return;

		Logger.getLogger().debug("Executing post-action in:" + node + " metadata element:<" + this.accessorName + "> old value:'" + oldValue + "' newValue: '" + newValue + "'");

		try
		{
			for (PostActionInstance postActionInstance : this.postActionInstances)
			{
				postActionInstance.execute(oldValue, newValue, node, this, index);
			}
		}
		catch (InvocationTargetException ex)
		{
			if (MetadataElementSetterPostActionException.class == ex.getTargetException().getClass())
				throw (MetadataElementSetterPostActionException)ex.getTargetException();
			else
				throw ex;
		}

	}


	/**
	 * Return the default value, which comes from a String in the levels config
	 * file.
	 * 
	 * This string can be undefined, then the default value is null. The string
	 * will be interpreted as a java expression. The current AbstractNode is
	 * accessible via the variable "object1", "object2" will be replaced with
	 * the current MetadataElement.
	 * 
	 * @param node
	 *            the context node for this default Expression
	 * @return null or the string result of the evaluated expression.
	 * @throws EvalError
	 */
	public String getDefaultValueFromNode(NodeAbstract node) throws EvalError
	{
		if (this.defaultExpression == null)		return null;

		Logger.getLogger().debug("Getting default value in:" + node + " metadata element:<" + this.accessorName + "> expression:'" + this.defaultExpression + "'");

		// Send JavaInterpreter as first parameter a string that must be a valid java expression defined in levels.xml and
		// as second parameter the current abstract node and as third parameter this instance of metadata element
		Object result = JavaInterpreter.execute(this.defaultExpression, node, this);


		return result == null? null: result.toString();
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return "[MetadataElement:" + this.accessorName + "]";
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================


	private class ValidatorInstance
	{
		private MetadataElementValidator validatorInstance;
		
		private Method method;
		
		public ValidatorInstance(String validatorClassName) throws ClassNotFoundException, NoSuchMethodException, SecurityException, InstantiationException, IllegalAccessException
		{
			Class<?> validatorClass = Class.forName(validatorClassName.trim());
			this.validatorInstance = (MetadataElementValidator) validatorClass.newInstance();
			this.method = validatorClass.getMethod("check", String.class, NodeAbstract.class, MetadataElement.class);
		}
		
		public void check(String value, NodeAbstract node, MetadataElement element) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException
		{
			this.method.invoke(this.validatorInstance, value, node, element);
		}
	}

	/**
	 * Wraps MetadataElementSetterPostAction with
	 *
	 * @author christian
	 *
	 */
	private class PostActionInstance
	{
		private MetadataElementSetterPostAction postActionInstance;
		
		private Method method;
		
		public PostActionInstance(String postActionClassName) throws ClassNotFoundException, InstantiationException, IllegalAccessException, NoSuchMethodException, SecurityException
		{
			Class<?> postActionClass = Class.forName(postActionClassName);
			this.postActionInstance = (MetadataElementSetterPostAction) postActionClass.newInstance();
			this.method = postActionClass.getMethod("execute", String.class, String.class, NodeAbstract.class, MetadataElement.class, int.class);
		}
		
		public void execute(String oldValue, String newValue, NodeAbstract node, MetadataElement element, int index) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException
		{
			this.method.invoke(this.postActionInstance, oldValue, newValue, node, element, index);
		}
	}
}
