/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.premis;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.Element;

import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.mets.amdsec.DigiprovWithPremis;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.id.UniqueID;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.string.DateFormatter;


/**
 * This class represents a METS digiprovMD PREMIS Event.
 * <p>
 * This class inserts Exceptions into the <a href="../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Creating an Event, when the linkObjectId is null</li>
 * <li>Creating an Event, when the type is null</li>
 * <li>Creating an Event, when the detail is null</li>
 * <li>Creating an Event, when the outcome is null</li>
 * </ul>
 * @author denis
 */
public class Event extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	private static final String			IDType = "Docuteam";
	private static final String			PerformedBy = "Performed by: '" + OperatingSystem.userName() + "'";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected String					idType = IDType;

	protected EventType					type = null;
	protected String					dateTime = "";
	protected String					detail = "";
	protected EventOutcome				outcome = null;
	protected String					outcomeDetail = "";
	protected String					linkingObjectIdType = IDType;
	protected String					linkingObjectIdValue = "";


	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	/**
	 * 	This constructor is used only when a new digiprov event is created programmatically
	 */
	public Event(DigiprovWithPremis parent, Element parentElement, String linkObjectId, String type, String detail, String outcome, String outcomeDetail)
	{
		if (linkObjectId == null)		Exception.remember("new Event: LinkObjectId must not be null");
		if (type == null)				Exception.remember("new Event: type must not be null");
		if (detail == null)				Exception.remember("new Event: detail must not be null");
		if (outcome == null)			Exception.remember("new Event: outcome must not be null");

		this.parent = parent;
		this.document = parent.getDocument();

		this.id = UniqueID.getXML();
		this.type = EventType.get(type);
		this.dateTime = DateFormatter.getCurrentDateTimeString();
		this.detail = (detail == null || detail.trim().isEmpty())? PerformedBy: detail + ". " + PerformedBy;
		this.outcome = EventOutcome.get(outcome);
		this.outcomeDetail = outcomeDetail;
		this.linkingObjectIdValue = linkObjectId;

		this.element = parentElement.addElement("PREMIS:event");
		Element eventIdentifier = this.element.addElement("PREMIS:eventIdentifier");
		eventIdentifier.addElement("PREMIS:eventIdentifierType").addText(this.idType);
		eventIdentifier.addElement("PREMIS:eventIdentifierValue").addText(this.id);
		this.element.addElement("PREMIS:eventType").addText(this.getType());
		this.element.addElement("PREMIS:eventDateTime").addText(this.dateTime);
		this.element.addElement("PREMIS:eventDetail").addText(this.detail);
		Element eventOutcomeInformation = this.element.addElement("PREMIS:eventOutcomeInformation");
		eventOutcomeInformation.addElement("PREMIS:eventOutcome").addText(this.outcome.getValue());
		if (!this.outcomeDetail.isEmpty())		eventOutcomeInformation.addElement("PREMIS:eventOutcomeDetail").addElement("PREMIS:eventOutcomeDetailNote").addText(this.outcomeDetail);
		Element linkingObjectIdentifier = this.element.addElement("PREMIS:linkingObjectIdentifier")
			.addAttribute("xmlns:xlink", "http://www.w3.org/1999/xlink")
			.addAttribute("xlink:type", "simple");
		linkingObjectIdentifier.addElement("PREMIS:linkingObjectIdentifierType").addText(this.linkingObjectIdType);
		linkingObjectIdentifier.addElement("PREMIS:linkingObjectIdentifierValue").addText(this.linkingObjectIdValue);
	}


	//	========	Constructors Private	=======================================================

	/**
	 * 	This constructor is used only when a METS-File is being read
	 */
	private Event(DigiprovWithPremis parent, Element eventElement)
	{
		this.document = parent.getDocument();
		this.parent = parent;
		this.element = eventElement;

		this.idType = this.getElementText("./PREMIS:eventIdentifier/PREMIS:eventIdentifierType", true);
		this.id = this.getElementText("./PREMIS:eventIdentifier/PREMIS:eventIdentifierValue", true);

		this.type = EventType.get(this.getElementText("./PREMIS:eventType", true));
		this.dateTime = this.getElementText("./PREMIS:eventDateTime", true);
		this.detail = this.getElementText("./PREMIS:eventDetail", false);		//	OPTIONAL

		this.outcome = EventOutcome.get(this.getElementText("./PREMIS:eventOutcomeInformation/PREMIS:eventOutcome", true));
		this.outcomeDetail = this.getElementText("./PREMIS:eventOutcomeInformation/PREMIS:eventOutcomeDetail/PREMIS:eventOutcomeDetailNote", false);		//	OPTIONAL

		this.linkingObjectIdType = this.getElementText("./PREMIS:linkingObjectIdentifier/PREMIS:linkingObjectIdentifierType", true);
		this.linkingObjectIdValue = this.getElementText("./PREMIS:linkingObjectIdentifier/PREMIS:linkingObjectIdentifierValue", true);
	}


	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read. Create the list of Digiprov Events out of the given parent AMDSectionDigiprov
	 */
	static public List<Event> parse(DigiprovWithPremis parent, Element premisElement)
	{
		List<Event> fileList = new ArrayList<Event>();

		for (java.lang.Object o: premisElement.selectNodes("./PREMIS:event"))		fileList.add(new Event(parent, (Element)o));

		return fileList;
	}


	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getIdType()
	{
		return this.idType;
	}

	public String getType()
	{
		if (this.type == null)			return "";
		return this.type.getValue();
	}

	public String getDateTime()
	{
		return this.dateTime;
	}

	public String getDetail()
	{
		return this.detail;
	}

	public String getOutcome()
	{
		if (this.outcome == null)		return "";
		return this.outcome.getValue();
	}

	public String getOutcomeDetail()
	{
		return this.outcomeDetail;
	}

	public String getLinkingObjectIdType()
	{
		return this.linkingObjectIdType;
	}

	public String getLinkingObjectIdValue()
	{
		return this.linkingObjectIdValue;
	}

	public Object getLinkedObject()
	{
		return ((DigiprovWithPremis)this.parent).getObject(this.linkingObjectIdValue);
	}


	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return
			new StringBuilder("[PREMISEvent:id=")
				.append(this.id)
				.append("/Object->")
				.append(this.linkingObjectIdValue)
				.append("(")
				.append(this.dateTime)
				.append("/")
				.append(this.getType())
				.append("/")
				.append(this.detail)
				.append("/")
				.append(this.getOutcome())
				.append("/")
				.append(this.outcomeDetail)
				.append("]")
				.toString();
	}


	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
