/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;
/**
 *
 */



import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.text.DecimalFormat;
import java.util.List;
import java.util.Vector;

import ch.docuteam.tools.file.exception.FileDeletionNoParentPermissionException;
import ch.docuteam.tools.file.exception.FileDeletionNoPermissionException;
import ch.docuteam.tools.file.exception.FileIsNotReadableException;
import ch.docuteam.tools.file.exception.FileIsNotWritableException;
import ch.docuteam.tools.file.exception.FileUtilException;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.file.exception.FolderIsNotEmptyException;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;

/**
 * This abstract class offers convenient file system operations.
 *
 * @author denis
 *
 */
public abstract class FileUtil
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static public final Charset				AsciiCharset = Charset.forName("ASCII");
	static public final String				SafeFileNameAllowedChars = "_.-";
	static public final Character			SafeFileNameReplacementChar = '_';

	static public final String				FileExtensionSeparator = ".";

	static private final int				CopyBufferSize		= 4096;
	static private final String				DefaultTempFolder	= "./temp";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private String					TempFolder = DefaultTempFolder;

	static private List<FileUtilException>	FileUtilExceptions;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	--------		Temp Folder			-------------------------------------------------------

	static public String getTempFolder()
	{
		return TempFolder;
	}


	static public void setTempFolder(String newTempFolderName)
	{
		if (newTempFolderName == null || newTempFolderName.trim().isEmpty())			return;

		File newTempFolder = new File(newTempFolderName);
		if (!newTempFolder.exists() && !newTempFolder.mkdirs())		throw new IllegalArgumentException("Could not create new temp-folder: " + newTempFolderName);
		TempFolder = newTempFolderName;

		Logger.getLogger().info("Temp-folder set to: " + newTempFolderName);
	}

	//	--------		Copy				-------------------------------------------------------
	//	--------		(recursive)			-------------------------------------------------------

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Replacing means that the destination folder will contain EXACTLY the same files like the source, no matter if the
	 * destination folder already existed and contained files or folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderOverwriting(String fileName, String toFolderName) throws IOException, FileUtilExceptionListException
	{
		return copyToFolderOverwriting(fileName, toFolderName, false);		//	As a default (for now), DON'T preserve the access rights in the copy!
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Replacing means that the destination folder will contain EXACTLY the same files like the source, no matter if the
	 * destination folder already existed and contained files or folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderOverwriting(File sourceFile, File toFolder) throws IOException, FileUtilExceptionListException
	{
		return copyToFolderOverwriting(sourceFile, toFolder, false);		//	As a default (for now), DON'T preserve the access rights in the copy!
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderMerging(String fileName, String toFolderName) throws IOException, FileUtilExceptionListException
	{
		return copyToFolderMerging(fileName, toFolderName, false);			//	As a default (for now), DON'T preserve the access rights in the copy!
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderMerging(File sourceFile, File toFolder) throws IOException, FileUtilExceptionListException
	{
		return copyToFolderMerging(sourceFile, toFolder, false);			//	As a default (for now), DON'T preserve the access rights in the copy!
	}


	/**
	 * This method copies single files and whole directory trees.
	 * Non-existing destination folders will be created. Any existing destination files or folders will be deleted.
	 * @param sourceFile file or folder to copy
	 * @param destFile file to copy to
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToOverwriting(File sourceFile, File destFile) throws IOException, FileUtilExceptionListException
	{
		return copyToOverwriting(sourceFile, destFile, false);				//	As a default (for now), DON'T preserve the access rights in the copy!
	}


	/**
	 * This method copies single files and whole directory trees.
	 * Non-existing destination folders will be created. Any existing destination files or folders will be deleted.
	 * @param sourceFileName Name of file or folder to copy
	 * @param destFileName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToOverwriting(String sourceFileName, String destFileName) throws IOException, FileUtilExceptionListException
	{
		return copyToOverwriting(sourceFileName, destFileName, false);		//	As a default (for now), DON'T preserve the access rights in the copy!
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToMerging(File sourceFile, File toFolder) throws IOException, FileUtilExceptionListException
	{
		return copyToMerging(sourceFile, toFolder, false);					//	As a default (for now), DON'T preserve the access rights in the copy!
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToMerging(String fileName, String toFolderName) throws IOException, FileUtilExceptionListException
	{
		return copyToMerging(fileName, toFolderName, false);				//	As a default (for now), DON'T preserve the access rights in the copy!
	}

	//	--------		Preserving access rights	------------------------------------

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Replacing means that the destination folder will contain EXACTLY the same files like the source, no matter if the
	 * destination folder already existed and contained files or folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderOverwriting(String fileName, String toFolderName, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return basicCopyToFolder(fileName, toFolderName, true, preserving);
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Replacing means that the destination folder will contain EXACTLY the same files like the source, no matter if the
	 * destination folder already existed and contained files or folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderOverwriting(File sourceFile, File toFolder, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return copyToFolderOverwriting(sourceFile.getPath(), toFolder.getPath(), preserving);
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderMerging(String fileName, String toFolderName, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return basicCopyToFolder(fileName, toFolderName, false, preserving);
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToFolderMerging(File sourceFile, File toFolder, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return copyToFolderMerging(sourceFile.getPath(), toFolder.getPath(), preserving);
	}


	/**
	 * This method copies single files and whole directory trees.
	 * Non-existing destination folders will be created. Any existing destination files or folders will be deleted.
	 * @param sourceFile file or folder to copy
	 * @param destFile file to copy to
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToOverwriting(File sourceFile, File destFile, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return basicCopyTo(sourceFile, destFile, true, preserving);
	}


	/**
	 * This method copies single files and whole directory trees.
	 * Non-existing destination folders will be created. Any existing destination files or folders will be deleted.
	 * @param sourceFileName Name of file or folder to copy
	 * @param destFileName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToOverwriting(String sourceFileName, String destFileName, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return copyToOverwriting(new File(sourceFileName), new File(destFileName), preserving);
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToMerging(File sourceFile, File toFolder, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return basicCopyTo(sourceFile, toFolder, false, preserving);
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException
	 *
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 * @throws FileUtilExceptionListException
	 */
	static public File copyToMerging(String fileName, String toFolderName, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return copyToMerging(new File(fileName), new File(toFolderName), preserving);
	}

	//	--------		Delete				-------------------------------------------------------
	//	--------		(recursive)			-------------------------------------------------------

	/**
	 * This method deletes single files and whole directory trees.
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param fileName Name of File or Directory to delete
	 * @throws FileUtilExceptionListException
	 */
	static public File delete(String fileName) throws FileUtilExceptionListException
	{
		return delete(new File(fileName));
	}


	/**
	 * Delete recursively single files and whole directory trees.
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param file File or Directory to delete
	 * @throws FileUtilExceptionListException
	 */
	static public File delete(File file) throws FileUtilExceptionListException
	{
		return basicDelete(file);
	}


	/**
	 * This method marks single files and whole directory trees to be deleted when the Java VM terminates.
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param fileName Name of File or Directory to delete
	 * @throws FileUtilExceptionListException
	 */
	static public void deleteOnExit(String fileName) throws IOException, FileUtilExceptionListException
	{
		deleteOnExit(new File(fileName));
	}


	/**
	 * This method marks single files and whole directory trees to be deleted when the Java VM terminates.
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param file File or Directory to delete on exit
	 * @throws FileUtilExceptionListException
	 */
	static public void deleteOnExit(File file) throws IOException, FileUtilExceptionListException
	{
		basicDeleteOnExit(file);
	}


	/**
	 * Set recursively single files and whole directory trees to be writable.
	 * If the file or folder doesn't exist, ignore it silently.
	 * If making the file or folder writable doesn't work, just continue.
	 * @param file File or Directory to make writable
	 */
	static public void setWritable(String fileName)
	{
		setWritable(new File(fileName));
	}


	/**
	 * Set recursively single files and whole directory trees to be writable.
	 * If the file or folder doesn't exist, ignore it silently.
	 * If making the file or folder writable doesn't work, just continue.
	 * @param file File or Directory to make writable
	 */
	static public void setWritable(File file)
	{
		basicSetWritableRecursively(file);
	}

	//	--------		Rename				-------------------------------------------------------

	/**
	 * This method moves or renames files and folders. Non-existing destination folders will be created.
	 * @param oldName Name of File or Directory to move or rename
	 * @param newName Name of File or Directory to move or rename to
	 * @throws IOException
	 */
	static public File renameTo(String oldName, String newName) throws IOException
	{
		return renameTo(new File(oldName), new File(newName));
	}


	/**
	 * This method moves or renames files and folders. Non-existing destination folders will be created.
	 * @param oldFile File or Directory to move or rename
	 * @param newFile File or Directory to move or rename to
	 * @throws IOException
	 */
	static public File renameTo(File oldFile, File newFile) throws IOException
	{
		Logger.getLogger().info("Renaming file: '" + oldFile.getPath() + "' to: '" + newFile.getPath() + "'");

		if (!oldFile.exists())				throw new IOException("Source File '" + oldFile.getPath() + "' doesn't exist.");

		newFile.getParentFile().mkdirs();

		//	Rename the file, retry 20 times, wait 1/2 sec between each try. Give up after 20 tries:
		boolean success = false;
		for (int i = 20; i > 0; i--)
		{
			if (oldFile.renameTo(newFile))
			{
				Logger.getLogger().debug("Renamed file: '" + oldFile.getPath() + "' to: '" + newFile.getPath() + "'");

				success = true;
				break;
			}

			Logger.getLogger().debug("Retrying: " + i);

			//	Wait 1/2 sec and then retry:
			try { Thread.sleep(500); } catch (InterruptedException x){}
		}

		if (!success)		throw new IOException("Could not rename file '" + oldFile.getPath() + "' to: '" + newFile.getPath() + "'");

		return newFile;
	}

	//	--------		Move				-------------------------------------------------------

	/**
	 * This method moves files and folders to another folder. Non-existing destination folders will be created.
	 * @param fileName Name of File or Directory to move or rename
	 * @param toFolderName Name of Directory to move to
	 * @throws IOException
	 */
	static public File moveToFolder(String fileName, String toFolderName) throws IOException
	{
		return moveToFolder(new File(fileName), new File(toFolderName));
	}


	/**
	 * This method moves files and folders to another folder. Non-existing destination folders will be created.
	 * @param sourceFile File or Directory to move
	 * @param toFolder Directory to move to
	 * @throws IOException
	 */
	static public File moveToFolder(File sourceFile, File toFolder) throws IOException
	{
		return renameTo(sourceFile, new File(toFolder.getPath() + "/" + sourceFile.getName()));
	}


	/**
	 * This method moves or renames files and folders. Non-existing destination folders will be created.
	 * @param oldName Name of File or Directory to move or rename
	 * @param newName Name of File or Directory to move or rename to
	 * @throws IOException
	 */
	static public File moveTo(String oldName, String newName) throws IOException
	{
		return moveTo(new File(oldName), new File(newName));
	}


	/**
	 * This method moves or renames files and folders. Non-existing destination folders will be created.
	 * @param oldFile File or Directory to move or rename
	 * @param newFile File or Directory to move or rename to
	 * @throws IOException
	 */
	static public File moveTo(File oldFile, File newFile) throws IOException
	{
		return renameTo(oldFile, newFile);
	}

	//	--------		Create Folder		-------------------------------------------------------

	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be emptied.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static public File createFolderOverwriting(String folderName, String inFolderName) throws IOException, FileUtilExceptionListException
	{
		return createFolderOverwriting(folderName, new File(inFolderName));
	}


	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be emptied.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static public File createFolderOverwriting(String folderName, File inFolder) throws IOException, FileUtilExceptionListException
	{
		return createFolderOverwriting(inFolder.getPath() + "/" + folderName);
	}


	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be emptied.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static public File createFolderOverwriting(String newFolderName) throws IOException, FileUtilExceptionListException
	{
		return createFolderOverwriting(new File(newFolderName));
	}


	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be emptied.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static public File createFolderOverwriting(File newFolder) throws IOException, FileUtilExceptionListException
	{
		Logger.getLogger().info("Creating folder (overwriting): '" + newFolder.getPath() + "'");

		if (newFolder.exists())		delete(newFolder);

		if (!newFolder.mkdirs())	throw new IOException("Could not create folder '" + newFolder.getPath() + "'");

		return newFolder;
	}


	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	static public File createFolderMerging(String folderName, String inFolderName) throws IOException
	{
		return createFolderMerging(folderName, new File(inFolderName));
	}


	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	static public File createFolderMerging(String folderName, File inFolder) throws IOException
	{
		return createFolderMerging(inFolder.getPath() + "/" + folderName);
	}


	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	static public File createFolderMerging(String newFolderName) throws IOException
	{
		return createFolderMerging(new File(newFolderName));
	}


	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	static public File createFolderMerging(File newFolder) throws IOException
	{
		Logger.getLogger().info("Creating folder (merging): '" + newFolder.getPath() + "'");

		if (!newFolder.exists())
		{
			if (!newFolder.mkdirs())		throw new IOException("Could not create folder '" + newFolder.getPath() + "'");
		}

		return newFolder;
	}


	//	--------		File Content		-------------------------------------------------------

	static public String getFileContentAsString(File file)
	{
		return getFileContentAsString(file.getPath());
	}

	static public String getFileContentAsString(String filePath)
	{
		StringBuilder text = new StringBuilder();
		BufferedReader reader = null;
		try
		{
			reader = new BufferedReader(new FileReader(filePath));
			String line;
			do
			{
				line = reader.readLine();
				if (line == null)		break;		//	EOF reached

				text.append(line).append("\n");
			}
			while (line != null);
		}
		catch (FileNotFoundException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		finally
		{
			try
			{
				if (reader != null)		reader.close();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
		}

		return text.toString();
	}


	/**
	 * Create or overwrite the file named "filePath" and fill it with the string "content". Content may be null.
	 * CAVEAT: For some reason, java always appends a "/n" to the end of the file created!
	 * @param filePath
	 * @param content
	 * @return the created file.
	 * @throws IOException
	 */
	static public File createFileWithContent(String filePath, String content) throws IOException
	{
		return createFileWithContent(new File(filePath), content);
	}


	/**
	 * Create or overwrite the file and fill it with the string "content". Content may be null.
	 * CAVEAT: For some reason, java always appends a "/n" to the end of the file created!
	 * @param filePath
	 * @param content
	 * @return the created file.
	 * @throws IOException
	 */
	static public File createFileWithContent(File file, String content) throws IOException
	{
		//	Create parent folders if necessary:
		file.getParentFile().mkdirs();

		OutputStream out = null;
		try
		{
			out = new BufferedOutputStream(new FileOutputStream(file));
			if (content != null)	out.write(content.getBytes());
		}
		finally
		{
			if (out != null)		out.close();
		}

		return file;
	}



	/**
	 * Count the lines in the file named "filePath".
	 * @param fileName
	 * @return
	 */
	static public Integer countLines(String filePath)
	{
		return countLines(new File(filePath));
	}


	/**
	 * Count the lines in the file.
	 * @param fileName
	 * @return
	 */
	static public Integer countLines(File file)
	{
		FileReader reader = null;
		LineNumberReader lineNumberReader = null;

		int lineCounter = 0;
		try
		{
			reader = new FileReader(file);
			lineNumberReader = new LineNumberReader(reader);
			while(lineNumberReader.readLine() != null)		++lineCounter;
		}
		catch (FileNotFoundException e)
		{
			e.printStackTrace();
			return null;
		}
		catch (IOException e)
		{
			e.printStackTrace();
			return null;
		}
		finally
		{
			if (lineNumberReader != null)		try { lineNumberReader.close(); } catch (IOException e) {};
			if (reader != null)					try { reader.close(); } catch (IOException e) {};
		}

		//	Add one because the last line has no line end:
		return lineCounter + 1;
	}

	//	--------		Inquiring			-------------------------------------------------------

	static public boolean areOnSameVolume(String fileName1, String fileName2)
	{
		return areOnSameVolume(new File(fileName1), new File(fileName2));
	}


	static public boolean areOnSameVolume(File file1, File file2)
	{
		String path1 = file1.getAbsolutePath();
		String path2 = file2.getAbsolutePath();

		if ((path1.charAt(1) == ':') && (path2.charAt(1) == ':'))
		{
			//	Windows platform: compare drive letter:
			return (path1.charAt(0) == path2.charAt(0));
		}

		//	Other platforms: compare /Volume/xyz/:
		int i1 = StringUtil.indexOf(path1, "/", 3);
		int i2 = StringUtil.indexOf(path2, "/", 3);

		String vol1 = path1.substring(0, (i1 != -1)? i1: path1.length());
		String vol2 = path2.substring(0, (i2 != -1)? i2: path2.length());

		return vol1.equals(vol2);
	}

	public static long parseStringAsFileSize(String size) 
	{
		if (!size.matches("\\d+(KB|MB|GB)"))
		{
			throw new IllegalArgumentException("Input \"" + size + "\" does not meet requirements.");
		}
		long length = Long.parseLong(size.substring(0, size.length() - 2));
		switch (size.substring(size.length() - 2)) {
		case "GB":
			return length * (long) Math.pow(1024L, 3L);
		case "MB":
			return length * (long) Math.pow(1024L, 2L);
		case "KB":
			return length * 1024;
		}
		return -1;
	}

	public static String getHumanReadableFileSize(File file) {
		if (!file.exists())
			return null;
		return getHumanReadableFileSize(file.length());
	}

	public static String getHumanReadableFileSize(long size) {
		if (size <= 0)
			return "0";
		
		final String[] units = new String[] { "B", "KB", "MB", "GB", "TB", "PB" };
		int digitGroups = (int) (Math.log10(size) / Math.log10(1024));
		return new DecimalFormat("#,##0.#").format(size
				/ Math.pow(1024, digitGroups))
				+ " " + units[digitGroups];
	}

	//	--------		File Names			-------------------------------------------------------

	/**
	 * Return the file name with all potentially problematic charactes replaced either by their unproblematic variant or by "_".
	 * @param f
	 * @return
	 */
	static public String asSafeFileName(String s)
	{
		String safeFileName = new String(s);

		//	Convert known special characters:
		safeFileName = safeFileName.replace("ä", "ae");
		safeFileName = safeFileName.replace("Ä", "Ae");
		safeFileName = safeFileName.replace("ö", "oe");
		safeFileName = safeFileName.replace("Ö", "Oe");
		safeFileName = safeFileName.replace("ü", "ue");
		safeFileName = safeFileName.replace("Ü", "Ue");
		safeFileName = safeFileName.replace("ß", "ss");
		safeFileName = safeFileName.replace("é", "e");
		safeFileName = safeFileName.replace("É", "E");
		safeFileName = safeFileName.replace("è", "e");
		safeFileName = safeFileName.replace("È", "E");
		safeFileName = safeFileName.replace("ê", "e");
		safeFileName = safeFileName.replace("Ê", "E");
		safeFileName = safeFileName.replace("à", "a");
		safeFileName = safeFileName.replace("À", "A");
		safeFileName = safeFileName.replace("â", "a");
		safeFileName = safeFileName.replace("Â", "A");
		safeFileName = safeFileName.replace("ç", "c");
		safeFileName = safeFileName.replace("Ç", "C");
		safeFileName = safeFileName.replace("ñ", "n");
		safeFileName = safeFileName.replace("Ñ", "N");

		//	Turn all remaining non-ascii characters to a '?':
		safeFileName = new String(safeFileName.getBytes(AsciiCharset));

		//	Turn all characters except letters, digits, and those defined in SafeFileNameAllowedChars, to the character defined in SafeFileNameReplacementChar:
		StringBuffer result = new StringBuffer();
		for (int i = 0; i < safeFileName.length(); i++)
		{
			char c = safeFileName.charAt(i);
			if (Character.isLetterOrDigit(c) || SafeFileNameAllowedChars.indexOf(c) != -1)		result.append(c);
			else	result.append(SafeFileNameReplacementChar);
		}

		return result.toString();
	}


	/**
	 * Return the file name with all potentially problematic charactes replaced either by their unproblematic variant or by "_".
	 * @param f
	 * @return
	 */
	static public String asSafeFileName(File f)
	{
		return asSafeFileName(f.getName());
	}



	/**
	 * Return the file's canonical file name or, if this would throw an exception, it's absolute file name:
	 * @param file
	 * @return
	 */
	static public String asCanonicalFileName(String fileName)
	{
		return asCanonicalFileName(new File(fileName));
	}


	/**
	 * Return the file's canonical file name or, if this would throw an exception, it's absolute file name:
	 * @param file
	 * @return
	 */
	static public String asCanonicalFileName(File file)
	{
		try
		{
			return file.getCanonicalPath();
		}
		catch (IOException ex)
		{
			return file.getAbsolutePath();
		}
	}


	/**
	 * This method returns true if a file or folder with this filename could successfully be created in the current file system, and false otherwise.
	 * @param s File name to be tested.
	 * @return
	 */
	static public boolean isFileNameAllowed(String s)
	{
		//	Don't allow these file names generally:
		if ("/".equals(s) || "\\".equals(s) || ".".equals(s) || "..".equals(s))		return false;

		File f = new File(OperatingSystem.javaTempDir() + s);
		if (f.exists())					try { delete(f); } catch (Exception x) {};

		try
		{
			//	Try to create a file with this name:
			if (!f.createNewFile())		return false;
		}
		catch (IOException e)
		{
			return false;
		}
		finally
		{
			//	Delete the file directly after is has been created:
			f.delete();
		}

		try
		{
			//	Try to create a folder with this name:
			if (!f.mkdir())				return false;
		}
		catch (Exception e)
		{
			return false;
		}
		finally
		{
			//	Delete the folder directly after is has been created:
			f.delete();
		}

		return true;
	}


	/**
	 * Interpret the string as a fileName and append the suffix to the end of the filename, but before the extension.
	 * If the extensionCharacter doesn't exist, append the suffix to the end of the whole string.
	 * @param filePath
	 * @return
	 */
	static public String appendSuffixToFileName(String filePath, String suffix)
	{
		String fileName = asFileName(filePath);

		if (fileName.contains(FileExtensionSeparator))
		{
			int fileExtensionSeparatorPos = filePath.lastIndexOf(FileExtensionSeparator);
			return filePath.substring(0, fileExtensionSeparatorPos) + suffix + filePath.substring(fileExtensionSeparatorPos);
		}

		return filePath + suffix;
	}


	/**
	 * Interpret the string as a fileName and append the suffix to the end of the filename, but before the extension.
	 * If the extensionCharacter doesn't exist, append the suffix to the end of the whole string.
	 * @param filePath
	 * @return
	 */
	static public String appendSuffixToFileName(File file, String suffix)
	{
		return appendSuffixToFileName(file.getPath(), suffix);
	}


	/**
	 * Interpret the string as a filePath and return the filePath without extension. If the extensionCharacter doesn't exist, return the complete string.
	 * @param filePath
	 * @return
	 */
	static public String asFilePathWithoutExtension(String filePath)
	{
		String fileName = asFileName(filePath);

		if (fileName.contains(FileExtensionSeparator))		return filePath.substring(0, filePath.lastIndexOf(FileExtensionSeparator));

		return filePath;
	}


	/**
	 * Interpret the string as a filePath and return the filePath without extension. If the extensionCharacter doesn't exist, return the complete string.
	 * @param filePath
	 * @return
	 */
	static public String asFilePathWithoutExtension(File file)
	{
		return asFilePathWithoutExtension(file.getPath());
	}


	/**
	 * Interpret the string as a fileName and return the fileName without path and without extension.
	 * If the extensionCharacter doesn't exist, return the file name without path.
	 * @param filePath
	 * @return
	 */
	static public String asFileNameWithoutExtension(String filePath)
	{
		String fileName = asFileName(filePath);

		if (fileName.contains(FileExtensionSeparator))		return fileName.substring(0, fileName.lastIndexOf(FileExtensionSeparator));

		return fileName;
	}


	/**
	 * Interpret the string as a fileName and return the fileName without path and without extension.
	 * If the extensionCharacter doesn't exist, return the file name without path.
	 * @param filePath
	 * @return
	 */
	static public String asFileNameWithoutExtension(File file)
	{
		return asFileNameWithoutExtension(file.getName());
	}


	/**
	 * Interpret the string as a filePath and return the file name extension. If the extensionCharacter doesn't exist, return the empty string.
	 * @param filePath
	 * @return
	 */
	static public String asFileNameExtension(String filePath)
	{
		String fileName = asFileName(filePath);

		if (fileName.contains(FileExtensionSeparator))		return filePath.substring(filePath.lastIndexOf(FileExtensionSeparator) + 1);

		return "";
	}


	/**
	 * Interpret the string as a filePath and return the file name extension. If the extensionCharacter doesn't exist, return the empty string.
	 * @param filePath
	 * @return
	 */
	static public String asFileNameExtension(File file)
	{
		return asFileNameExtension(file.getName());
	}


	static public String asFileName(String filePath)
	{
		return asFileName(new File(filePath));
	}


	static public String asFileName(File file)
	{
		return file.getName();
	}


	static public String asParentPath(String filePath)
	{
		return asParentPath(new File(filePath));
	}


	static public String asParentPath(File file)
	{
		return file.getParent();
	}

	//	========	Static Private			=======================================================

	//	--------		Exceptions			-------------------------------------------------------

	/**
	 * Return the list of FileUtilExceptions and clear the list afterwards.
	 *
	 * IMPORTANT NOTE: I find it peculiar that emptying the FileUtilExceptions List happens AFTER the list was returned!
	 */
	static private List<FileUtilException> consumeFileUtilExceptions()
	{
		try
		{
			return FileUtilExceptions;
		}
		finally
		{
			clearFileUtilExceptions();
		}
	}

	static private void clearFileUtilExceptions()
	{
		FileUtilExceptions = new Vector<FileUtilException>(10);
	}

	//	--------		Basic Ops			-------------------------------------------------------

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @param overwriting If true, the destination will contain EXACTLY the same files like the source. If false, the
	 * destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static private File basicCopyToFolder(String fileName, String toFolderName, boolean overwriting, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		return basicCopyTo(new File(fileName), new File(toFolderName + "/" + new File(fileName).getName()), overwriting, preserving);
	}


	static private File basicCopyTo(File sourceFile, File destFile, boolean overwriting, boolean preserving) throws IOException, FileUtilExceptionListException
	{
		Logger.getLogger().info("Copying file or folder: '" + sourceFile.getPath() + "' to file or folder: '" + destFile.getPath() + "'");

		if (!sourceFile.exists())		throw new FileNotFoundException(sourceFile.getAbsolutePath());

		//	Prepare (= empty) the Exception List:
		clearFileUtilExceptions();

		if (overwriting)
			//	IMPORTANT: The method "basicDeleteRecursively()" does NOT clear the FileUtilExceptionList afterwards,
			//	so the exceptions being thrown during this operation remain in the FileUtilExceptionList!
			basicDeleteRecursively(destFile);

		basicCopyToRecursively(sourceFile, destFile, preserving);

		//	If any errors occurred, throw exception:
		if (!FileUtilExceptions.isEmpty())		throw new FileUtilExceptionListException(consumeFileUtilExceptions());

		return destFile;
	}


	/**
	 * Delete recursively single files and whole directory trees.
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param file File or Directory to delete
	 * @throws FileUtilExceptionListException
	 */
	static private File basicDelete(File file) throws FileUtilExceptionListException
	{
		//	Prepare (= empty) the Exception List:
		clearFileUtilExceptions();

		File returnFile = basicDeleteRecursively(file);

		//	If any errors occurred, throw exception:
		if (!FileUtilExceptions.isEmpty())		throw new FileUtilExceptionListException(consumeFileUtilExceptions());

		return returnFile;
	}


	/**
	 * This method marks single files and whole directory trees to be deleted when the Java VM terminates.
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param fileName Name of File or Directory to delete
	 * @throws FileUtilExceptionListException
	 */
	static private void basicDeleteOnExit(File file) throws IOException, FileUtilExceptionListException
	{
		//	Prepare (= empty) the Exception List:
		clearFileUtilExceptions();

		basicDeleteOnExitRecursively(file);

		//	If any errors occurred, throw exception:
		if (!FileUtilExceptions.isEmpty())		throw new FileUtilExceptionListException(consumeFileUtilExceptions());
	}

	//	--------		Recursive			-------------------------------------------------------

	/**
	 * Copy recursively a file or folder to the destination file or folder.
	 */
	static private void basicCopyToRecursively(File sourceFile, File destFile, boolean doPreserveAccessRights) throws IOException
	{
		if (!sourceFile.exists())		throw new FileNotFoundException(sourceFile.getAbsolutePath());

		//	If no read access on the source file or directory is granted to the current user, don't copy but collect exception:
		if (!sourceFile.canRead())
		{
			FileUtilExceptions.add(new FileIsNotReadableException(sourceFile));
			return;
		}

		//	Do not overwrite unreadable file or folder:
		if (destFile.exists() && !destFile.canRead())
		{
			FileUtilExceptions.add(new FileIsNotReadableException(destFile));
			return;
		}

		//	Do not overwrite unwritable file or folder:
		if (destFile.exists() && !destFile.canWrite())
		{
			FileUtilExceptions.add(new FileIsNotWritableException(destFile));
			return;
		}

		if (sourceFile.isDirectory())
		{
			Logger.getLogger().debug("Creating folder: '" + destFile + "'");

			destFile.mkdirs();

			for (File f: sourceFile.listFiles())		basicCopyToRecursively(f, new File(destFile + "/" + f.getName()), doPreserveAccessRights);		//	Recursion!
		}
		else
		{
			destFile.getParentFile().mkdirs();

			basicCopyFilePhysically(sourceFile, destFile);
		}

		//	Apply the access rights of the source file or folder to the destination file or folder:
		if (doPreserveAccessRights)
		{
			//	destFile.setReadable(sourceFile.canRead(), false);		This would be ridiculous because if I can't read a file, I couldn't have made a copy of it!
			destFile.setWritable(sourceFile.canWrite(), true);			//	true means: set grants only for the owner, not for all
			destFile.setExecutable(sourceFile.canExecute(), true);		//	true means: set grants only for the owner, not for all
		}
	}


	/**
	 * Delete recursively single files and whole directory trees.
	 * If the file or folder to be deleted doesn't exist, ignore it silently.
	 * If deletion doesn't work, retry 20 times, with a break of 1/2 sec between each try.
	 * If exceptions occur during deletion, collect them and continue.
	 * @param file File or Directory to delete
	 */
	static private File basicDeleteRecursively(File file)
	{
		if (!file.exists())		return null;

		File parent = file.getParentFile();

		//	If no WRITE access on the PARENT of the file or folder is granted to the current user, don't delete and collect exception:
		if (!parent.canWrite())
		{
			FileUtilExceptions.add(new FileDeletionNoParentPermissionException(file));
			return null;
		}

		if (file.isDirectory())
		{
			//	Grumble if no READ access on the folder is granted to the current user:
			if (!file.canRead() || file.list() == null)
			{
				FileUtilExceptions.add(new FileIsNotReadableException(file));
				return null;
			}

			Logger.getLogger().info("Deleting folder: '" + file.getPath() + "'");

			for (File f: file.listFiles())	basicDeleteRecursively(f);		//	Recursion!

			//	Grumble if folder is still not empty:
			if (file.list().length != 0)
			{
				FileUtilExceptions.add(new FolderIsNotEmptyException(file));
				return null;
			}
		}
		else
		{
			//	Change default behaviour and do not delete restricted files from writable folders:
			if (!file.canRead() || !file.canWrite())
			{
				FileUtilExceptions.add(new FileDeletionNoPermissionException(file));
				return null;
			}
			Logger.getLogger().info("Deleting file: '" + file.getPath() + "'");
		}

		//	Delete the file or folder, retry 20 times, wait 1/2 sec between each try. Give up after 20 tries:
		boolean success = false;
		for (int i = 20; i > 0; i--)
		{
			if (file.delete())
			{
				Logger.getLogger().debug("Deleted: '" + file.getPath() + "'");

				success = true;
				break;
			}

			Logger.getLogger().debug("Retrying: " + i);

			//	Wait 1/2 sec and then retry:
			try { Thread.sleep(500); } catch (InterruptedException x){}
		}

		//	Could still not delete file:
		if (!success)
		{
			FileUtilExceptions.add(new FileDeletionNoPermissionException(file));
			return null;
		}

		return file;
	}


	/**
	 * Mark recursively single files and whole directory trees to be deleted when the Java VM terminates.
	 * If the file or folder to be deleted doesn't exist, ignore it silently.
	 * @param file File or Directory to delete on exit
	 */
	static private void basicDeleteOnExitRecursively(File file) throws IOException
	{
		if (!file.exists())			return;

		File parent = file.getParentFile();
		if (!parent.canWrite() || !parent.canRead())
		{
			FileUtilExceptions.add(new FileDeletionNoParentPermissionException(file));
			return;
		}

		if (!file.canWrite() || !file.canRead())
		{
			FileUtilExceptions.add(new FileDeletionNoPermissionException(file));
			return;
		}

		file.deleteOnExit();

		if (file.isDirectory())
		{
			//	Note the order!!! Mark FIRST the folder for deletion, THEN the included files and folders!!! It doesn't work the other way around.

			Logger.getLogger().info("Deleting on exit folder: '" + file.getPath() + "'");

			for (File f: file.listFiles())		basicDeleteOnExitRecursively(f);		//	Recursion!
		}
		else
		{
			Logger.getLogger().info("Deleting on exit file  : '" + file.getPath() + "'");
		}

		return;
	}


	/**
	 * Set recursively single files and whole directory trees to be writable.
	 * If the file or folder doesn't exist, ignore it silently.
	 * If making the file or folder writable doesn't work, just continue.
	 * @param file File or Directory to delete
	 */
	static private void basicSetWritableRecursively(File file)
	{
		if (!file.exists())			return;

		file.setWritable(true);

		if (file.isDirectory())
		{
			Logger.getLogger().info("Making writable folder: '" + file.getPath() + "'");

			if (file.listFiles() == null)		return;								//	In case the folder is not readable

			for (File f: file.listFiles())		basicSetWritableRecursively(f);		//	Recursion!
		}
		else
		{
			Logger.getLogger().info("Making writable file  : '" + file.getPath() + "'");
		}

		return;
	}

	//	--------		Copy physically		-------------------------------------------------------

	/**
	 * Physically copy the source file to the destination file.
	 * If the destination file doesn't exist yet, it will be created. If it exists, it will be overwritten.
	 */
	static private void basicCopyFilePhysically(File sourceFile, File destFile) throws IOException
	{
		Logger.getLogger().debug("Physically copying file: '" + sourceFile.getPath() + "' to file: '" + destFile.getPath() + "'");

		InputStream in = null;
		OutputStream out = null;

		try
		{
			in = new BufferedInputStream(new FileInputStream(sourceFile));
			out = new BufferedOutputStream(new FileOutputStream(destFile));

			byte[] buffer = new byte[CopyBufferSize];
			int bytesRead;

			while ((bytesRead = in.read(buffer)) > 0)		out.write(buffer, 0, bytesRead);
		}
		finally
		{
			if (in != null)		in.close();
			if (out != null)	out.close();
		}
	}

}
