/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.*;

import javax.swing.JFrame;
import javax.swing.JTextField;
import javax.swing.text.*;

import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.string.StringUtil;

/**
 * @author denis
 *
 */
public class LimitedIntegerTextField extends JTextField
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	private static final Color	BadValueBackgroundColor = Color.RED;

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String				originalText;

	private Integer				minValue = null;
	private Integer				maxValue = null;

	private final Color			defaultBackgroundColor = this.getBackground();

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	static public void main(String... args)
	{
		JFrame f = new JFrame();
		f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		f.add(new LimitedIntegerTextField(), BorderLayout.NORTH);
		f.add(new LimitedIntegerTextField(10, 100), BorderLayout.SOUTH);
		f.setSize(100, 80);
		f.setLocationRelativeTo(null);
		f.setVisible(true);
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	public LimitedIntegerTextField()
	{
		this.setInputConstraints();
		this.setToolTipText("0 <= Num");
	}

	public LimitedIntegerTextField(Integer maxValue)
	{
		this.maxValue = maxValue;
		this.setColumns(this.maxValue.toString().length());

		this.setInputConstraints();
		this.setToolTipText("0 <= Num <= " + maxValue);
	}

	public LimitedIntegerTextField(Integer minValue, Integer maxValue)
	{
		if (minValue > maxValue)
		{
			this.minValue = maxValue;
			this.maxValue = minValue;
		}
		else
		{
			this.minValue = minValue;
			this.maxValue = maxValue;
		}

		this.setColumns(this.maxValue.toString().length());

		this.setInputConstraints();
		this.setToolTipText(this.minValue + " <= Num <= " + this.maxValue);
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public Integer getValue()
	{
		if (this.getText() == null || this.getText().isEmpty())		return null;

		return new Integer(this.getText());
	}

	public void setValue(Byte value)
	{
		this.setText(value == null? "": "" + value);
	}

	public void setValue(Short value)
	{
		this.setText(value == null? "": "" + value);
	}

	public void setValue(Integer value)
	{
		this.setText(value == null? "": "" + value);
	}

	public void setValue(Long value)
	{
		this.setText(value == null? "": "" + value);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------

	private void setInputConstraints()
	{
		//	Evaluate content after loosing focus. Select all content when gaining focus:
		this.addFocusListener(new FocusAdapter()
				{ @Override public void focusLost(FocusEvent e)   { if (e.isTemporary()) return; LimitedIntegerTextField.this.evaluate(); }
				  @Override public void focusGained(FocusEvent e) { if (e.isTemporary()) return; LimitedIntegerTextField.this.rememberOriginalText(); LimitedIntegerTextField.this.selectAll(); }});

		//	Evaluate content on action:
		this.addActionListener(new ActionListener()
				{ @Override public void actionPerformed(ActionEvent e) { LimitedIntegerTextField.this.evaluate(); }});

		this.setDocument(new PlainDocument()
		{
			@Override
			public void insertString(int offset, String str, AttributeSet attr) throws BadLocationException
			{
				if (str == null)		return;

				//	Don't accept non-numeric input:
				if (!StringUtil.isNumeric(str))
				{
					OperatingSystem.beep();
					return;
				}

				//	Don't accept too long input:
				if (LimitedIntegerTextField.this.getColumns() != 0
				&& (this.getLength() + str.length()) > LimitedIntegerTextField.this.getColumns())
				{
					OperatingSystem.beep();
					return;
				}

				super.insertString(offset, str, attr);
			}
		});
	}

	private void rememberOriginalText()
	{
		this.originalText = this.getText();
	}

	private void evaluate()
	{
		Integer newValue = LimitedIntegerTextField.this.getValue();

		if (newValue == null)
		{
			//	If new value is null or text field is empty:
			LimitedIntegerTextField.this.setBackground(this.defaultBackgroundColor);
		}
		else if (  (LimitedIntegerTextField.this.minValue != null && newValue < LimitedIntegerTextField.this.minValue)
				|| (LimitedIntegerTextField.this.maxValue != null && newValue > LimitedIntegerTextField.this.maxValue))
		{
			//	If new value is out of bounds:
			OperatingSystem.beep();
			LimitedIntegerTextField.this.setText(this.originalText);
			LimitedIntegerTextField.this.setBackground(BadValueBackgroundColor);
		}
		else
		{
			//	If new value is within bounds:
			LimitedIntegerTextField.this.setValue(newValue);							//	This turns "01" to "1"
			LimitedIntegerTextField.this.setBackground(this.defaultBackgroundColor);
		}
	}

	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
