/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.string;
/**
 * 
 */


import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ch.docuteam.tools.out.Tracer;

/**
 * Here, sets of <a href="./StringSetElement.html">StringSetElement</a>s are held and administered.
 * The users of <a href="./StringSetElement.html">StringSetElement</a>s usually don't need to make any direct calls to this class 
 * since all necessary functionality is offered by the class <a href="./StringSetElement.html">StringSetElement</a> and it's subclasses.
 * @author denis
 *
 */
public abstract class StringSet
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	/**
	 * This variable "All" contains all instances of the subclasses of StringSetElement.
	 * It is a double Map (or matrix):
	 * The outer Map has the StringSetElement's class Name as key and a Map with the StringSetElements as the value.
	 * The inner Maps have the StringSetElement's key as key and the StringSetElement itself as value.
	 * See <a href="./StringSetElement.html" title="class StringSetElement"><code>StringSetElement</code></a> for details.
	 */
	static private Map<String, Map<String, StringSetElement>>		All = new HashMap<String, Map<String, StringSetElement>>();

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	/**
	 * Return the StringSetElement for the given class and with the given key, or null if the elementClass is not a StringSetElement subclass or the key could not be found.
	 */
	static public StringSetElement get(Class<? extends StringSetElement> elementClass, String value)
	{
		try
		{
			return All.get(elementClass.getName()).get(StringSetElement.asKey(value));
		}
		catch(java.lang.NullPointerException ex)
		{
			return null;
		}
	}
	

	/**
	 * Reteurn all StringSetElements for the given class.
	 * @param elementClass The StringSetElement subclass
	 * @return The list of all StringSetElements for the given class, or null if the elementClass is not a StringSetElement subclass.
	 */
	static public List<StringSetElement> getAll(Class<? extends StringSetElement> elementClass)
	{
		try
		{
			List<StringSetElement> all = new ArrayList<StringSetElement>(All.get(elementClass.getName()).values());
			Collections.sort(all);
			return all;
		}
		catch(java.lang.NullPointerException ex)
		{
			return null;
		}
	}


	//	--------		Debugging			-------------------------------------------------------

	/**
	 * Write all StringSetElements to System.out.
	 */
	static public void traceAll(Class<? extends StringSetElement> elementClass)
	{
		for (StringSetElement e: getAll(elementClass))		Tracer.trace(e);
	}
	
	//	========	Static Private			=======================================================

	/**
	 * Add the StringSetElement to "All".
	 */
	protected static void add(StringSetElement newElement)
	{
		String elementClassName = newElement.getClass().getName();

		// Does All contain key "className"? If no: create a new map with the className as the key:
		if (!All.containsKey(elementClassName))		All.put(elementClassName, new HashMap<String, StringSetElement>());

		// Put the element into the (now surely existing) map:
		(All.get(elementClassName)).put(newElement.getValueAsKey(), newElement);
	}
	
}
