/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.filesec;

import java.util.*;

import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.tools.exception.Exception;


/**
 * This class, used by the class <a href="../Document.html">Document</a>, represents the METS FileSection.
 * It contains a collection of <a href="./File.html">File</a>s.
 * <p>
 * This class inserts Exceptions into the <a href="../../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when the number of file groups found in the METS file is not exactly 1</li>
 * </ul>
 * @author denis
 */
public class FileSection extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	/**
	 * Map containing this FileSection's files. Key is the File's id, value is the File itself.
	 */
	protected Map<String, File>	files = new HashMap<String, File>();

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	protected FileSection() {}

	//	========	Constructors Private	=======================================================

	/**
	 * 	This constructor is used only when a METS-File is being read
	 */
	@SuppressWarnings("unchecked")
	private FileSection(Document document)
	{
		this.document = document;
		this.element = (Element)this.document.selectSingleNode("./METS:mets/METS:fileSec");

		//	Check if a fileSection exists at all - if not, create one and place it at the right position (after the amdSection):
		if (this.element == null)
		{
			this.element = DocumentHelper.createElement("METS:fileSec");
			this.document.getRootElement().content().add(2, this.element);
			this.element.addElement("METS:fileGrp");
		}

		//	Consistency check: no multiple fileGroups are allowed:
		if (this.element.selectNodes("./METS:fileGrp").size() >= 2)
			Exception.remember("Bad number of file groups in the METS file's File Section. Expected: 1, found: " + this.element.selectNodes("./METS:fileGrp").size());

		for (File file: File.parse(this))		this.files.put(file.getId(), file);
	}

	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read.
	 */
	static public FileSection parse(Document document)
	{
		return new FileSection(document);
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public List<File> getFiles()
	{
		return new ArrayList<File>(this.files.values());
	}

	public File getFile(String id)
	{
		return this.files.get(id);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * Delete the file from my file list.
	 */
	public void delete(File file)
	{
		this.files.remove(file.getId());
	}


	/**
	 * Add the file to my file list.
	 */
	public void add(File file)
	{
		this.files.put(file.getId(), file);
		this.document.setIsModified();
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder("\n[FileSection:(files:" + this.files.size() + ")");
		for (File file: this.files.values())		buf.append("\n\t" + file.toString());
		buf.append("\n]");

		return buf.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
