/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.structmap;

import java.io.*;
import java.util.*;

import org.dom4j.Element;
import org.jdesktop.swingx.treetable.MutableTreeTableNode;

import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.amdsec.DigiprovWithPremis;
import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.file.FileFilter;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.exception.FileIsNotReadableException;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;



/**
 * This class, used by the class <a href="./StructureMap.html">StructureMap</a>, represents a folder within the METS StructMap.
 * <p>
 * This class inserts Exceptions into the <a href="../../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Inserting a file, when the file is not readable</li>
 * <li>Inserting a file, when the Submission Agreement doesn't allow the file</li>
 * <li>Inserting a file, when the file could not be copied</li>
 * <li>Inserting a folder, when the folder is not readable</li>
 * </ul>
 *
 * @author denis
 */
public class NodeFolder extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Final Static Public		=======================================================

	//	========	Final Static Private	=======================================================

	static final private String			Type = "folder";
	static final private String			RootType = "rootfolder";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	protected NodeFolder() {}

	//	========	Constructors Private	=======================================================

	/**
	 * This constructor is used only when a METS-File is being read, and for the root node only.
	 * Create recursively the tree structure.
	 */
	protected NodeFolder(StructureMap parent, Element element)
	{
		super(parent.getDocument(), element);
		this.initialize();
	}


	/**
	 * 	This constructor is used only when a METS-File is being read, and for non-root nodes only.
	 * Create recursively the tree structure.
	 */
	protected NodeFolder(NodeFolder parent, Element element)
	{
		super(parent.getDocument(), element);
		parent.add(this);
		this.initialize();
	}


	/**
	 * 	This constructor is used when a new ROOT folder is created programmatically.
	 * @param label The new folder label
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 */
	protected NodeFolder(StructureMap parent, String label) throws IOException, FileUtilExceptionListException
	{
		super(parent, label);

		//	Create corresponding folder in the file system:
		FileUtil.createFolderOverwriting(this.getFile());
		this.initializeFileAccessRights();

		this.allowsChildren = true;
		this.type = RootType;

		this.element = parent.getElement()
			.addElement("METS:div")
				.addAttribute("LABEL", this.label)
				.addAttribute("TYPE", this.type)
				.addAttribute("ADMID", this.admId);

		//	Create new AMDSectionDigiprov:
		new DigiprovWithPremis(this);

		//	Insert associated DMDSectionWithEAD:
		this.createDMDSectionWithEADIfNecessary();

		//	Only now, when the DMDSectionWithEAD is present for sure, I can initialize my dynamic metadata elements:
		this.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();
	}


	/**
	 * 	This constructor is used when a new NON-root folder is created programmatically.
	 *
	 * NEW: I can assume that the folder already exists in the file system, so I can initialize me with the access rights.
	 *
	 * @param parent The parent
	 * @param label The new folder label
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 */
	private NodeFolder(NodeFolder parent, String label) throws IOException, FileUtilExceptionListException
	{
		super(parent, label);

		//	Create corresponding folder in the file system:
		FileUtil.createFolderOverwriting(this.getFile());
		this.initializeFileAccessRights();

		this.allowsChildren = true;
		this.type = Type;

		this.element = ((NodeFolder)this.parent).getElement()
			.addElement("METS:div")
				.addAttribute("LABEL", this.label)
				.addAttribute("TYPE", this.type)
				.addAttribute("ADMID", this.admId);

		//	Create new AMDSectionDigiprov:
		new DigiprovWithPremis(this);

		//	Insert associated DMDSectionWithEAD:
		this.createDMDSectionWithEADIfNecessary();

		//	After the DMDSectionWithEAD was created, set my level to my parent's default sublevel:
		try
		{
			this.setLevel(parent.getLevel().getDefaultSubLevel());
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
		};

		//	Only now, when the DMDSectionWithEAD is present for sure, I can initialize my dynamic metadata elements:
		this.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();
	}

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	@Override
	public String getChecksum()
	{
		return "";
	}

	@Override
	public String getChecksumType()
	{
		return "";
	}

	@Override
	public String getMimeType()
	{
		return "";
	}

	@Override
	public String getFormatKey()
	{
		return "";
	}

	@Override
	public String getFormatName()
	{
		return "";
	}

	/**
	 * @return the total number of children + sub-children
	 */
	@Override
	public int getDescendantCount()
	{
		Integer cumulatedCount = this.getChildCount();
		for (MutableTreeTableNode child: this.children)		cumulatedCount = cumulatedCount + ((NodeAbstract)child).getDescendantCount();

		return cumulatedCount;
	}


	/**
	 * @return The depth of my subtree: it's 1 + the maximal treeDepth of my children. The treeDepth of a leaf is 0.
	 */
	@Override
	public int getTreeDepth()
	{
		if (this.children.size() == 0)		return 0;

		int maxDepth = 1;
		for (MutableTreeTableNode child: this.children)		maxDepth = Math.max(maxDepth, ((NodeAbstract)child).getTreeDepth() + 1);

		return maxDepth;
	}


	public List<MutableTreeTableNode> getChildren()
	{
		return this.children;
	}

	//	----------		Inquiring				---------------------------------------------------

	@Override
	public boolean isFile()
	{
		return false;
	}

	@Override
	public boolean isFolder()
	{
		return true;
	}


	public boolean doesLabelHaveNumericPrefix()
	{
		return Character.isDigit(this.label.charAt(0));
	}


	public boolean isPredecessorOf(NodeAbstract child)
	{
		NodeFolder parent = (NodeFolder)child.getParent();
		while (parent != null)
		{
			if (this == parent)			return true;
			parent = (NodeFolder)parent.getParent();
		}

		return false;
	}


	public boolean hasDescendantNotReadableByCurrentUser()
	{
		for (MutableTreeTableNode c: this.children)
		{
			NodeAbstract child = (NodeAbstract)c;
			if (!child.fileExists || !child.canRead)							return true;
			if (child.isFolder()
				&& ((NodeFolder)child).hasDescendantNotReadableByCurrentUser())	return true;
		}

		return false;
	}


	public boolean hasDescendantNotWritableByCurrentUser()
	{
		for (MutableTreeTableNode c: this.children)
		{
			NodeAbstract child = (NodeAbstract)c;
			if (!child.fileExists || !child.canWrite)							return true;
			if (child.isFolder()
				&& ((NodeFolder)child).hasDescendantNotWritableByCurrentUser())	return true;
		}

		return false;
	}


	public boolean hasDescendantNotReadableOrWritableByCurrentUser()
	{
		for (MutableTreeTableNode c: this.children)
		{
			NodeAbstract child = (NodeAbstract)c;
			if (!child.fileExists || !child.canWrite || !child.canRead)						return true;
			if (child.isFolder()
				&& ((NodeFolder)child).hasDescendantNotReadableOrWritableByCurrentUser())	return true;
		}

		return false;
	}


	@Override
	public boolean checkFixity()
	{
		//	Go through all children (don't stop on 1st "false") in order to collect all exceptions:
		boolean isOK = true;
		for (MutableTreeTableNode t: this.children)		if (((NodeAbstract)t).checkFixity() == false)		isOK = false;

		return isOK;
	}


	@Override
	public boolean checkAgainstSubmissionAgreement()
	{
		return this.checkAgainstSubmissionAgreement(((Document)this.document).getSubmissionAgreement(), ((Document)this.document).getDSSId());
	}

	@Override
	public List<NodeFile> filesNotAllowedBySubmissionAgreement()
	{
		return this.filesNotAllowedBySubmissionAgreement(((Document)this.document).getSubmissionAgreement(), ((Document)this.document).getDSSId());
	}


	@Override
	public boolean checkAgainstSubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		//	Go through all children (don't stop on 1st "false") in order to collect all exceptions:
		boolean isOK = true;
		for (MutableTreeTableNode t: this.children)		if (((NodeAbstract)t).checkAgainstSubmissionAgreement(sa, dssId) == false)		isOK = false;

		return isOK;
	}

	@Override
	public List<NodeFile> filesNotAllowedBySubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		//	Go through all children and collect NodeFiles not complying to the SA:
		List<NodeFile> result = new Vector<NodeFile>();
		for (MutableTreeTableNode t: this.children)		result.addAll(((NodeAbstract)t).filesNotAllowedBySubmissionAgreement(sa, dssId));

		return result;
	}


	/**
	 * Folders are actually always allowed by any SA, but returning true means: 'OK'
	 */
	@Override
	public boolean isAllowedBySA()
	{
		return true;
	}


	/**
	 * Folders actually don't have a mime type, but returning true means: 'OK'
	 */
	@Override
	public boolean hasMimeType()
	{
		return true;
	}


	/**
	 * Folders actually don't have a format key, but returning true means: 'OK'
	 */
	@Override
	public boolean hasFormatKey()
	{
		return true;
	}

	//	----------		Searching 			-------------------------------------------------------

	/**
	 * Return this node, if it has this admId, or pass on the search to the children.
	 */
	@Override
	public NodeAbstract searchId(String admId)
	{
		if (this.admId.equals(admId))		return this;

		for (MutableTreeTableNode child: this.children)
		{
			NodeAbstract found = ((NodeAbstract)child).searchId(admId);
			if (found != null)		return found;
		}

		return null;
	}


	/**
	 * Return the (possibly empty) list of nodes with this label among me and my children.
	 */
	@Override
	public List<NodeAbstract> searchLabel(String label)
	{
		List<NodeAbstract> found = new Vector<NodeAbstract>(10);

		if (this.label.equals(label))		found.add(this);
		for (MutableTreeTableNode child: this.children)		found.addAll(((NodeAbstract)child).searchLabel(label));

		return found;
	}


	/**
	 * Pass on the search to the children.
	 */
	@Override
	public NodeAbstract searchFileId(String fileId)
	{
		for (MutableTreeTableNode child: this.children)
		{
			NodeAbstract found = ((NodeAbstract)child).searchFileId(fileId);
			if (found != null)		return found;
		}

		return null;
	}

//	----------		Business Ops			---------------------------------------------------

	/**
	 * Change the name of this folder, then update all my descendants in the FileSection and the corresponding folder in the file system.
	 * @throws FileAlreadyExistsException
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 * @throws Exception In case that a folder or file with this name already exists in the parent folder
	 */
	@Override
	public void rename(String label) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException
	{
		this.checkIfFileOperationNotAllowed(label, "Rename");

		String oldRelativePath = this.getPathString();
		String oldAbsolutePath = this.getAbsolutePathString();
		super.rename(label);
		String newRelativePath = this.getPathString();
		String newAbsolutePath = this.getAbsolutePathString();

		//	Update recursively for all descendant files their corresponding File in the FileSection:
		for (NodeFile file: this.getDescendantFiles())
		{
			file.createNewEvent("Path Modification", "Manually", "Success", "");
			((Document)this.document).getFileSection().getFile(file.getFileId()).replaceInFilePath(oldRelativePath, newRelativePath);
		}

		//	Change folder name in file system:
		FileUtil.renameTo(oldAbsolutePath, newAbsolutePath);
	}

	/**
	 * Move this folder to another folder, then update all my descendants in the FileSection and the corresponding folder in the file system.
	 * @throws FileAlreadyExistsException
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 */
	@Override
	public void moveTo(NodeFolder toFolder) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException
	{
		this.checkIfFileOperationNotAllowed(toFolder.label, "MoveTo");

		String oldRelativePath = this.getPathString();
		String oldAbsolutePath = this.getAbsolutePathString();
		super.moveTo(toFolder);
		String newRelativePath = this.getPathString();
		String newAbsolutePath = this.getAbsolutePathString();

		//	Update recursively for all descendant files their corresponding FileSectionFile in the FileSection:
		for (NodeFile file: this.getDescendantFiles())
		{
			file.createNewEvent("Path Modification", "Manually", "Success", "");
			((Document)this.document).getFileSection().getFile(file.getFileId()).replaceInFilePath(oldRelativePath, newRelativePath);
		}

		//	Change folder name in file system:
		FileUtil.renameTo(oldAbsolutePath, newAbsolutePath);
	}


	/**
	 * First delete all my descendants, then delete this folder itself, and then finally delete the corresponding folder in the file system.
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	@Override
	public void delete() throws IOException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(this.label, "Delete");

		//	Delete recursively all descendants, then delete me:
		for (NodeAbstract node: this.getDescendantsDepthFirst())
		{
			node.delete();
		}

		String oldAbsolutePath = this.getAbsolutePathString();
		super.delete();

		//	Delete corresponding folder in the file system:
		FileUtil.delete(oldAbsolutePath);
	}


	/**
	 * Insert file or folder into me. In the case of a folder, exclude hidden elements. Preserve access rights.
	 * @param fileOrFolderPath
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws FileAlreadyExistsException
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public NodeAbstract insertFileOrFolder(String fileOrFolderPath, boolean renameIfNecessary) throws FileNotFoundException, IOException, FileAlreadyExistsException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(fileOrFolderPath, renameIfNecessary? "InsertFileOrFolderRenameIfNecessary": "InsertFileOrFolder");

		return
			new File(fileOrFolderPath).isFile()
				? this.insertFile(fileOrFolderPath, renameIfNecessary)
				: this.insertFolder(fileOrFolderPath, renameIfNecessary);
	}


	/**
	 * Insert file or folder into me. In the case of a folder, exclude hidden elements. Preserve access rights.
	 * @param fileOrFolderPath
	 * @return
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @throws FileAlreadyExistsException
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public NodeAbstract insertFileOrFolder(String fileOrFolderPath) throws FileNotFoundException, IOException, FileAlreadyExistsException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		return this.insertFileOrFolder(fileOrFolderPath, false);
	}


	/**
	 * Create a new folder in the file system, then create the corresponding folder node as a new child of mine.
	 * @throws FileAlreadyExistsException
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 * @throws FolderNameIsEmptyException
	 * @throws FileUtilExceptionListException
	 * @throws Exception
	 */
	public NodeFolder createNewFolder(String newFolderName) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, FolderNameIsEmptyException, FileUtilExceptionListException
	{
		return this.createNewFolder(newFolderName, false);
	}


	/**
	 * Create a new folder in the file system, then create the corresponding folder node as a new child of mine.
	 * @throws FileAlreadyExistsException
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 * @throws FolderNameIsEmptyException
	 * @throws FileUtilExceptionListException
	 * @throws Exception
	 */
	public NodeFolder createNewFolder(String newFolderName, boolean renameIfNecessary) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, FolderNameIsEmptyException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(newFolderName, "InsertNewFolder");
		if (newFolderName == null || newFolderName.isEmpty())		throw new FolderNameIsEmptyException();

		//	In "don't rename" mode: does file already exist as a child of mine? If yes: exception!
		if (!renameIfNecessary && this.doesAlreadyContainChildWithThisName(newFolderName))		throw new FileAlreadyExistsException(newFolderName, this.label);

		String usedFolderName = newFolderName;
		if (renameIfNecessary)
		{
			int i = 1;
			//	Find a fileName which is not already used:
			while (this.doesAlreadyContainChildWithThisName(usedFolderName))
			{
				usedFolderName = FileUtil.appendSuffixToFileName(newFolderName, "-" + StringUtil.padUpLeftWith("" + i++, '0', 3));
				Logger.getLogger().debug("Trying other folder name: " + usedFolderName);
			}
		}

		NodeFolder newNodeFolder = new NodeFolder(this, usedFolderName);

		if (renameIfNecessary && !usedFolderName.equals(newFolderName))
		{
			//	ToDo: Add a new PREMIS Event as the FIRST event (doesn't work correctly yet):
			newNodeFolder.getMyDigiprov().addNewFirstEvent("Renaming", "Renamed from '" + newFolderName + "' to '" + usedFolderName + "'", "Success", "");
		}

		return newNodeFolder;
	}

	//	--------		Interface			-------------------------------------------------------

	/* (non-Javadoc)
	 * @see JXTreeTable.FileSystem.FileSystemAbstract#getSize()
	 */
	@Override
	public Long getSize()
	{
		Long cumulatedSize = 0L;
		for (MutableTreeTableNode child: this.children)	cumulatedSize = cumulatedSize + ((NodeAbstract)child).getSize();

		return cumulatedSize;
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	public boolean doesAlreadyContainChildWithThisName(String newFileOrFolderName)
	{
		//	I use equalsIgnoreCase here because file names are assumed equal when they have the same name NOT considering the case:
		for (MutableTreeTableNode t: this.children)		if (((NodeAbstract)t).label.equalsIgnoreCase(newFileOrFolderName))		return true;
		return false;
	}


	/**
	 * We have to sort the children NOW (when INSERTING a child) because other objects access the children using the FIELD instead of the METHOD! >:-/
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public void insert(MutableTreeTableNode child, int index)
	{
		super.insert(child, index);
		Collections.sort((List<NodeAbstract>)(List)this.children);		//	Double casting: Workaround to make the sun javac compiler (used by ant) compile this >8-/

		this.document.setIsModified();
	}


	/**
	 *
	 * @return A list of all descendants (= children + grandchildren + grandgrandChildren etc.)
	 */
	public List<NodeAbstract> getDescendants()
	{
		List<NodeAbstract> descendants = new Vector<NodeAbstract>();

		for (MutableTreeTableNode child: this.children)
		{
			NodeAbstract aChild = ((NodeAbstract)child);
			descendants.add(aChild);
			if (aChild.isFolder())		descendants.addAll(((NodeFolder)aChild).getDescendants());
		}

		return descendants;
	}


	/**
	 *
	 * @return A list of all descendants (= children + grandchildren + grandgrandChildren etc.)
	 */
	public List<NodeAbstract> getDescendantsDepthFirst()
	{
		List<NodeAbstract> descendants = new Vector<NodeAbstract>();

		for (MutableTreeTableNode child: this.children)
		{
			NodeAbstract aChild = ((NodeAbstract)child);
			if (aChild.isFolder())		descendants.addAll(((NodeFolder)aChild).getDescendantsDepthFirst());
			descendants.add(aChild);
		}

		return descendants;
	}


	/**
	 * @return A list containing me and all my descendants (for a leaf: only me)
	 */
	@Override
	public List<NodeAbstract> getWithDescendants()
	{
		List<NodeAbstract> all = new ArrayList<NodeAbstract>();
		all.add(this);
		all.addAll(this.getDescendants());
		return all;
	}


	/**
	 * @return A list containing me and all my descendants
	 */
	@Override
	public List<NodeAbstract> getWithDescendantsDepthFirst()
	{
		List<NodeAbstract> all = new ArrayList<NodeAbstract>();
		all.addAll(this.getDescendantsDepthFirst());
		all.add(this);
		return all;
	}


	/**
	 *
	 * @return A list of all descendants (= children + grandchildren + grandgrandChildren etc.) that are files (not folders)
	 */
	public List<NodeFile> getDescendantFiles()
	{
		List<NodeFile> descendantFiles = new Vector<NodeFile>();

		for (MutableTreeTableNode child: this.children)
		{
			NodeAbstract aChild = ((NodeAbstract)child);
			if (aChild.isFile())		descendantFiles.add((NodeFile)aChild);
			else						descendantFiles.addAll(((NodeFolder)aChild).getDescendantFiles());
		}

		return descendantFiles;
	}


	/**
	 *
	 * @return A list of all descendants (= children + grandchildren + grandgrandChildren etc.) that are folders (not files)
	 */
	public List<NodeFolder> getDescendantFolders()
	{
		List<NodeFolder> descendantFolders = new Vector<NodeFolder>();

		for (MutableTreeTableNode child: this.children)
		{
			NodeAbstract node = ((NodeAbstract)child);
			if (node.isFolder())
			{
				NodeFolder folder = (NodeFolder)node;
				descendantFolders.add(folder);
				descendantFolders.addAll(folder.getDescendantFolders());
			}
		}

		return descendantFolders;
	}


	//	--------		Utilities			-------------------------------------------------------

	/**
	 * When sorting Nodes, all Files come behind Folders. Then sort folders alphabetically.
	 */
	@Override
	protected String compareString()
	{
		return "1" + this.getLabel();
	}


	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return new StringBuilder("[")
		.append((this.isRoot()? "Root": ""))
		.append("StructureMapFolder:")
		.append(this.label)
		.append("[")
		.append(this.getLevel())
		.append("](admId=")
		.append(this.admId)
		.append("/dmdIdOAI=")
		.append(this.dmdIdOAI_DC)
		.append("/dmdIdEAD=")
		.append(this.dmdIdEAD)
		.append(")(children:")
		.append(this.children.size())
		.append(")")
		.append(this.fileExists? "D": "d")
		.append(this.canRead? "R": "r")
		.append(this.canWrite? "W": "w")
		.append(this.canExecute? "X": "x")
		.append("^")
		.append(this.hasPredecessorNotReadableByCurrentUser()? "r": "R")
		.append(this.hasPredecessorNotWritableByCurrentUser()? "w": "W")
		.append("v")
		.append(this.hasDescendantNotReadableByCurrentUser()? "r": "R")
		.append(this.hasDescendantNotWritableByCurrentUser()? "w": "W")
		.append("(")
		.append(this.getSubmitStatus())
		.append(")]")
		.toString();
	}


	/**
	 * @return A string describing this folder, but indented to show the tree structure. Then recurse into my children with incremented indent.
	 */
	@Override
	public String treeString(Integer indent)
	{
		StringBuilder buf = new StringBuilder(200);
		buf.append("\n");
		for (int i = 0; i < indent; i++)	buf.append("\t");
		buf.append(this.toString());
		for (MutableTreeTableNode child: this.children)		buf.append(((NodeAbstract)child).treeString(indent + 1));

		return buf.toString();
	}


	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initialization, used only when reading a METS file.
	 */
	protected void initialize()
	{
		this.initializeFileAccessRights();
		this.allowsChildren = true;

		//	Continue parsing children elements, but only file and folder types (e.g. skip metadata types):
		for (Object o: this.element.selectNodes("./METS:div[@TYPE = 'file' or @TYPE = 'folder']"))		parse(this, (Element)o);
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	protected NodeFile insertFile(String filePath, boolean renameIfNecessary) throws FileNotFoundException, IOException, FileAlreadyExistsException
	{
		Logger.getLogger().debug(this.label + ": Inserting file: '" + filePath + "'");

		File file = new File(filePath);

		//	Does file exist? If not: exception!
		if (!file.exists())			throw new FileNotFoundException("File to be inserted does not exist: " + filePath);

		//	Is it a folder? If yes: exception!
		if (file.isDirectory())		throw new FileNotFoundException("File to be inserted is a folder: " + filePath);

		//	Is it hidden? If yes: ignore!
		if (file.isHidden())		return null;

		//	Is it readable? If no: remember and skip!
		if (!file.canRead())
		{
			ch.docuteam.tools.exception.Exception.remember(new FileIsNotReadableException(file));
			return null;
		}

		//	In "don't rename" mode: does file already exist as a child of mine? If yes: exception!
		if (!renameIfNecessary && this.doesAlreadyContainChildWithThisName(file.getName()))		throw new FileAlreadyExistsException(filePath, this.label);

		//	Does SA allow this file? If not: remember exception and continue!
		SubmissionAgreement sa = ((Document)this.document).getSubmissionAgreement();
		if ((sa != null) && !sa.allowsFile(((Document)this.document).getDSSId(), filePath))
		{
			ch.docuteam.tools.exception.Exception.remember("Submission Agreement '" + ((Document)this.document).getSAId() + "' Data Submission Session '" + ((Document)this.document).getDSSId() + "' doesn't allow file '" + filePath + "'");
			return null;
		}

		String usedFileName = file.getName();
		if (renameIfNecessary)
		{
			int i = 1;
			//	Find a fileName which is not already used:
			while (this.doesAlreadyContainChildWithThisName(usedFileName))
			{
				usedFileName = FileUtil.appendSuffixToFileName(file.getName(), "-" + StringUtil.padUpLeftWith("" + i++, '0', 3));
				Logger.getLogger().debug("Trying other file name: " + usedFileName);
			}
		}

		//	Somebody (a GUI?) may be interested in what I'm doing:
		this.document.distributeMessage(usedFileName);

		//	FileSystem: copy the file, retaining it's access rights. If Exceptions occur, remember them:
		try
		{
			FileUtil.copyToOverwriting(filePath, this.getAbsolutePathString() + "/" + usedFileName, true);
		}
		catch (FileUtilExceptionListException ex)
		{
			ch.docuteam.tools.exception.Exception.remember(ex);
		}

		//	FileSection:
		//	Create new FileSection file:
		ch.docuteam.darc.mets.filesec.File newFileSectionFile = new ch.docuteam.darc.mets.filesec.File(this.document, this.getPathString() + "/" + usedFileName);

		//	StructMap and AMDSection:
		//	Create a new NodeFile with its FILEID taken from newFileSectionFile.
		//	(This will add the new nodeFile as a new child of mine, and create a new Digiprov section with a new corresponding PREMIS object and PREMIS event)
		NodeFile newNodeFile = new NodeFile(this, newFileSectionFile);

		//	If the mimeType or format ID of the new file is undef, remember it:
		if (newNodeFile.getMimeType().isEmpty())		Exception.remember("No MimeType found for file: '" + newNodeFile.getPathString() + "'");
		if (newNodeFile.getFormatKey().isEmpty())		Exception.remember("No FormatID found for file: '" + newNodeFile.getPathString() + "'");

		if (renameIfNecessary && !usedFileName.equals(file.getName()))
		{
			newNodeFile.getMyDigiprov().addNewFirstEvent("Renaming", "Old name: '" + file.getName() + "', new name: '" + usedFileName + "'", "Success", "");
		}

		Logger.getLogger().debug(this.label + ": Inserted file: '" + usedFileName + "'");

		return newNodeFile;
	}


	/**
	 * Insert recursively the folder and all its elements (except hidden) into me.
	 * @param folderPath
	 * @return
	 * @throws FileAlreadyExistsException
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 * @throws FolderNameIsEmptyException
	 */
	@SuppressWarnings("null")		//	folderPath is not empty for sure.
	protected NodeAbstract insertFolder(String folderPath, boolean renameIfNecessary) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		Logger.getLogger().debug(this.label + ": Inserting folder: '" + folderPath + "'");

		File folder = new File(folderPath);

		//	Does folder exist? If no: exception!
		if (!folder.exists())			throw new FileNotFoundException("Folder to be inserted does not exist: " + folderPath);

		//	Is it a file? If yes: exception!
		if (folder.isFile())			throw new FileNotFoundException("Folder to be inserted is a file: " + folderPath);

		//	Is it hidden? If yes: ignore!
		if (folder.isHidden())			return null;

		//	Is it readable? If no: remember and skip!
		if (!folder.canRead())
		{
			ch.docuteam.tools.exception.Exception.remember(new FileIsNotReadableException(folder));
			return null;
		}

		String newFolderName = folder.getName();
		//	Special case (Thanx to Andi for finding this!):
		//	If the folder path is only a drive specification (like "A:\"), use the drive letter as the folder name:
		if (newFolderName.isEmpty())	newFolderName = folderPath.substring(0, 1);

		String usedFolderName = newFolderName;
		if (renameIfNecessary)
		{
			int i = 1;
			//	Find a fileName which is not already used:
			while (this.doesAlreadyContainChildWithThisName(usedFolderName))
			{
				usedFolderName = FileUtil.appendSuffixToFileName(folder.getName(), "-" + StringUtil.padUpLeftWith("" + i++, '0', 3));
				Logger.getLogger().debug("Trying other folder name: " + usedFolderName);
			}
		}

		NodeFolder newNodeFolder = null;
		try
		{
			newNodeFolder = this.createNewFolder(usedFolderName);
		}
		catch (FolderNameIsEmptyException ex) {}	//	Ignore this exception because the folder name is not empty for sure.

		if (renameIfNecessary && !usedFolderName.equals(folder.getName()))
		{
			newNodeFolder.getMyDigiprov().addNewFirstEvent("Renaming", "Old name: '" + folder.getName() + "', new name: '" + usedFolderName + "'", "Success", "");
		}

		Logger.getLogger().debug(this.label + ": Inserted folder: '" + folderPath + "'");

		//	Recurse into the folder, but skip hidden files and folders:
		File[] listFiles = folder.listFiles((FilenameFilter)FileFilter.VisibleAll);
		//	The list of files is null when the folder is not readable:
		if (listFiles != null)		for (File f: listFiles)		newNodeFolder.insertFileOrFolder(f.getPath());		//	Recursion!

		//	Copy access rights, but AFTER copying the files and folders within:
		newNodeFolder.getFile().setExecutable(folder.canExecute());
		newNodeFolder.getFile().setWritable(folder.canWrite());
		newNodeFolder.getFile().setReadable(folder.canRead());			//	This of course can't ever happen.
		newNodeFolder.initializeFileAccessRights();						//	The node must know the folder's access restrictions

		return newNodeFolder;
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
