/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.common;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.tree.DefaultElement;

import ch.docuteam.darc.util.XMLUtil;
import ch.docuteam.tools.exception.Exception;


/**
 * Abstract superclass of all elements that are contained in a <a href="../mets/Document.html">Document</a>.
 * All these elements have these four properties in common:
 * <ul>
 * <li>document: The METS document</li>
 * <li>parent: The node containing me</li>
 * <li>element: My dom4j xpath element</li>
 * <li>id: (optional) An identification string</li>
 * </ul>
 * This class inserts Exceptions into the <a href="../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when a mandatory element is missing</li>
 * </ul>
 * @author denis
 *
 */
public abstract class NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected DocumentAbstract			document;
	protected NodeAbstract				parent;
	protected Element					element;
	protected String					id;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public DocumentAbstract getDocument()
	{
		return this.document;
	}

	public NodeAbstract getParent()
	{
		return this.parent;
	}

	public Element getElement()
	{
		return this.element;
	}

	public String getId()
	{
		return this.id;
	}


	public String getElementText(String xPath, Boolean isMandatory)
	{
		Node subNode = this.element.selectSingleNode(xPath);
		if (isMandatory && (subNode == null))
			Exception.remember("Mandatory element missing in File: '" + this.document.getName() + "':\n\t'" + xPath + "'\nin location: '" + this.element.getPath() + "'\nID: " + this.id);

		return ((subNode == null)? "": subNode.getText());
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	//	Set the id. Protected because only subclasses may set the id.
	protected void setId(String id)
	{
		this.id = id;
		this.element.addAttribute("ID", this.id);
	}

	//	--------		Accessing Dynamic	-------------------------------------------------------

	/**
	 * Get the texts of a dynamic element. If it is repeatable there are as many nodes as repeated elements.
	 * Each element may have or have not several lines. If it has one line, the text can be extracted from this element,
	 * if there are several lines, the element has several child elements, each containing text or EAD:lb.
	 * In this case we are only interested in text elements by building one line of text of them, separated by a \n.
	 * @param xPath
	 * @return
	 */
	protected List<String> getDynamicElementTexts(String xPath)
	{
		@SuppressWarnings("unchecked")
		List<Node> nodes = this.element.selectNodes(xPath);
		if (nodes.isEmpty())		return null;

		List<String> results = new ArrayList<String>(nodes.size());
		for (Node node: nodes)
		{
			//	Check if node contains <EAD:lb> elements
			if (node.selectNodes("EAD:lb").size() > 0)
			{
				//	select child text nodes from node, if any
				@SuppressWarnings("unchecked")
				List<Node> children = node.selectNodes("node()");
				StringBuilder builder = new StringBuilder();
				for (int i = 0; i < children.size(); i++)
				{
					//	Build a string by appending each node in the list, converting <EAD:lb> into regular line breaks
					Node child = children.get(i);
					if (child instanceof DefaultElement && ((DefaultElement)child).getQualifiedName().equals("EAD:lb"))
						builder = builder.append("\n");
					else
						builder = builder.append(child.getText().trim());
				}
				results.add(builder.toString());
			}
			else
				results.add(node.getText().trim());
		}
		
		return results;
	}

	protected void setDynamicElementTexts(List<String> values, String xPath)
	{
		XMLUtil.set(this.element, xPath, values);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
