/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.ead;

import java.util.*;

import org.dom4j.*;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.mdconfig.LevelOfDescription;
import ch.docuteam.darc.mets.dmdsec.DMDSectionWithEAD;
import ch.docuteam.darc.mets.structmap.NodeAbstract.SubmitStatus;


/**
 * This abstract class is the superclass of <a href="./C.html">C</a> and <a href="./ArchDesc.html">ArchDesc</a>.
 * It contains most of the dynamic metadata objects. Exempt are those that are stored in the <a href="./DID.html">DID</a> element.
 * In addition, it contains one <a href="./DID.html">DID</a> and a List of <a href="./C.html">C</a> elements; so it is recursive.
 *
 * @author denis
 *
 */
public abstract class ComponentAbstract extends EADNodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static protected final String	Level							= "otherlevel";
	static protected final String	Role							= "simple";

	static protected final String	Xmlns_xlink						= "http://www.w3.org/1999/xlink";

	static protected final String	XPathPID						= "EAD:dao[@xlink:role = '" + Role + "']";

	static protected final String	XPathAuthor						= "EAD:controlaccess/EAD:persname[@role = 'author']";
	static protected final String	XPathComment					= "EAD:note/EAD:p";

	static protected final String	XPathAccessNr							= "EAD:acqinfo/EAD:p";
	static protected final String	XPathScopeContent						= "EAD:scopecontent/EAD:p";
	static protected final String	XPathAccessPolicy						= "EAD:accessrestrict[@type = 'accessPolicy']/EAD:p";
	static protected final String	XPathAccessRestriction					= "EAD:accessrestrict[@type = 'restrictions']/EAD:p";
	static protected final String	XPathAccessRestrictionExplanation		= "EAD:accessrestrict[@type = 'accessRestrictionsExplanation']/EAD:p";
	static protected final String	XPathAccessRestrictionPeriod			= "EAD:accessrestrict[@type = 'accessRestrictionsPeriod']/EAD:p";
	static protected final String	XPathAccessRestrictionPeriodBaseYear	= "EAD:accessrestrict[@type = 'accessRestrictionsPeriodBaseYear']/EAD:p";
	static protected final String	XPathAccessRestrictionClassification	= "EAD:accessrestrict[@type = 'accessRestrictionsClassification']/EAD:p";
	static protected final String	XPathAccessRestrictionPrivacy			= "EAD:accessrestrict[@type = 'accessRestrictionsPrivacy']/EAD:p";
	static protected final String	XPathAccessRestrictionStatus			= "EAD:accessrestrict[@type = 'accessRestrictionsStatus']/EAD:p";
	static protected final String	XPathAccessRestrictionStatusExplanation	= "EAD:accessrestrict[@type = 'accessRestrictionsStatusExplanation']/EAD:p";
	static protected final String	XPathUsagePermission					= "EAD:accessrestrict[@type = 'usagePermission']/EAD:p";
	static protected final String	XPathUsagePermissionExpiringDate		= "EAD:accessrestrict[@type = 'usagePermissionExpiringDate']/EAD:p";
	
	static protected final String	XPathRetentionPolicy			= "EAD:accessrestrict[@type = 'retentionPolicy']/EAD:p";
	static protected final String	XPathRetentionPeriodBaseYear	= "EAD:accessrestrict[@type = 'baseYear']/EAD:p";
	static protected final String	XPathArchivalHistory			= "EAD:custodhist/EAD:p";
	static protected final String	XPathSourceType					= "EAD:custodhist/EAD:note/EAD:p";
	static protected final String	XPathModeOfAcquisition			= "EAD:custodhist/EAD:acqinfo/EAD:p";
	static protected final String	XPathObjectType					= "EAD:controlaccess/EAD:genreform";
	static protected final String	XPathConditionsOfReproductions	= "EAD:userestrict/EAD:p";
	static protected final String	XPathCharacteristics			= "EAD:phystech/EAD:p";
	static protected final String	XPathFindingAids				= "EAD:otherfindaid/EAD:p";
	static protected final String	XPathLocationOfOriginals		= "EAD:originalsloc/EAD:p";
	static protected final String	XPathReproductions				= "EAD:altformavail/EAD:p";
	static protected final String	XPathRelatedMaterial			= "EAD:relatedmaterial/EAD:p";
	static protected final String	XPathInstitution				= "EAD:controlaccess/EAD:corpname";
	static protected final String	XPathAppraisalAndDestruction	= "EAD:appraisal/EAD:p";
	static protected final String	XPathArrangement				= "EAD:arrangement/EAD:p";
	static protected final String	XPathBibliography				= "EAD:bibliography/EAD:p";
	static protected final String	XPathDescriptionLevel			= "EAD:processinfo[@type = 'level']/EAD:p";
	static protected final String	XPathDescriptionLevelNotes		= "EAD:processinfo[@type = 'levelNotes']/EAD:p";
	static protected final String	XPathDescriptionRules			= "EAD:processinfo[@type = 'rules']/EAD:p";
	static protected final String	XPathRevisions					= "EAD:processinfo[@type = 'revisions']/EAD:p";
	static protected final String	XPathDigitization				= "EAD:processinfo[@type = 'digitization']/EAD:p";

	static protected final String	XPathBiographicalHistory		= "EAD:bioghist/EAD:p";
	static protected final String	XPathDeathOfAuthor				= "EAD:bioghist/EAD:note/EAD:p/EAD:date[@type = 'deathAuthor']";
	static protected final String	XPathAccruals					= "EAD:accruals/EAD:p";
	static protected final String	XPathCharacteristicsNote		= "EAD:phystech/EAD:note/EAD:p";
	static protected final String	XPathSubject					= "EAD:controlaccess/EAD:subject";

	static protected final String	XPathResponsible				= "EAD:controlaccess/EAD:persname[@role = 'responsible']";
	static protected final String	XPathInvolved					= "EAD:controlaccess/EAD:persname[@role = 'involved']";
	static protected final String	XPathStaff						= "EAD:controlaccess/EAD:persname[@role = 'staff']";
	static protected final String	XPathProjectName				= "EAD:odd[@type = 'projectName']/EAD:p";
	static protected final String	XPathProjectTitle				= "EAD:odd[@type = 'projectTitle']/EAD:p";
	static protected final String	XPathProject					= "EAD:odd[@type = 'project']/EAD:p";
	static protected final String	XPathProjectAbbreviation		= "EAD:odd[@type = 'projectAbbreviation']/EAD:p";
	static protected final String	XPathLocation					= "EAD:controlaccess/EAD:name[@role = 'location']";
	static protected final String	XPathJournal					= "EAD:controlaccess/EAD:name[@role = 'journal']";
	static protected final String	XPathUniversity					= "EAD:controlaccess/EAD:name[@role = 'university']";
	static protected final String	XPathInstitute					= "EAD:controlaccess/EAD:name[@role = 'institute']";
	static protected final String	XPathCompartment				= "EAD:controlaccess/EAD:name[@role = 'compartment']";
	static protected final String	XPathMethod						= "EAD:controlaccess/EAD:name[@role = 'method']";
	static protected final String	XPathKeyword					= "EAD:controlaccess/EAD:name[@role = 'keyword']";
	static protected final String	XPathFundingSource				= "EAD:controlaccess/EAD:name[@role = 'fundingSource']";
	static protected final String	XPathStatus						= "EAD:controlaccess/EAD:name[@role = 'status']";
	static protected final String	XPathPublisher					= "EAD:controlaccess/EAD:name[@role = 'publisher']";
	static protected final String	XPathDOIJournal					= "EAD:odd[@type = 'doiJournal']/EAD:p";
	static protected final String	XPathRetentionPeriod			= "EAD:accessrestrict[@type = 'retentionPeriod']/EAD:p";
	static protected final String	XPathUsage						= "EAD:odd[@type = 'usage']/EAD:p";

	static protected final String	XPathSubmitStatus				= "EAD:controlaccess/EAD:name[@role = 'submitStatus']";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected int					depth = 0;

	protected LevelOfDescription	otherLevel = LevelOfDescription.getUndefined();

	protected DID					did;
	protected List<C>				cs = new ArrayList<C>();

	/**
	 * pid is an optional value, containing the PIDs of this C or ArchDesc element.
	 */
	protected List<String>			pid;

	/**
	 * submitStatus is an optional value, containing a SubmitStatus. Some submitStatuses disallow editing.
	 */
	protected SubmitStatus			submitStatus;

	//	--------		Dynamic Metadata	-------------------------------------------------------

	protected List<String>			author;
	protected List<String>			comment;

	protected List<String>			accessNr;
	protected List<String>			scopeContent;
	protected List<String>			accessPolicy;
	protected List<String>			accessRestriction;
	protected List<String>			accessRestrictionExplanation;
	protected List<String>			accessRestrictionPeriod;
	protected List<String>			accessRestrictionPeriodBaseYear;
	protected List<String>			accessRestrictionClassification;
	protected List<String>			accessRestrictionPrivacy;
	protected List<String>			accessRestrictionStatus;
	protected List<String>			accessRestrictionStatusExplanation;
	protected List<String>			usagePermission;
	protected List<String>			usagePermissionExpiringDate;
	protected List<String>			retentionPolicy;
	protected List<String>			retentionPeriodBaseYear;
	protected List<String>			archivalHistory;
	protected List<String>			sourceType;
	protected List<String>			modeOfAcquisition;
	protected List<String>			objectType;
	protected List<String>			conditionsOfReproductions;
	protected List<String>			characteristics;
	protected List<String>			findingAids;
	protected List<String>			locationOfOriginals;
	protected List<String>			reproductions;
	protected List<String>			relatedMaterial;
	protected List<String>			institution;
	protected List<String>			appraisalAndDestruction;
	protected List<String>			arrangement;
	protected List<String>			bibliography;
	protected List<String>			descriptionLevel;
	protected List<String>			descriptionLevelNotes;
	protected List<String>			descriptionRules;
	protected List<String>			revisions;
	protected List<String>			digitization;

	protected List<String>			biographicalHistory;
	protected List<String>			deathOfAuthor;
	protected List<String>			accruals;
	protected List<String>			characteristicsNote;
	protected List<String>			subject;

	protected List<String>			responsible;
	protected List<String>			involved;
	protected List<String>			staff;
	protected List<String>			projectName;
	protected List<String>			projectTitle;
	protected List<String>			project;
	protected List<String>			projectAbbreviation;
	protected List<String>			location;
	protected List<String>			journal;
	protected List<String>			university;
	protected List<String>			institute;
	protected List<String>			compartment;
	protected List<String>			method;
	protected List<String>			keyword;
	protected List<String>			fundingSource;
	protected List<String>			status;
	protected List<String>			publisher;
	protected List<String>			doiJournal;
	protected List<String>			retentionPeriod;
	protected List<String>			usage;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	/**
	 * This constructor is used when an EAD-File or METS-file is being read.
	 */
	protected ComponentAbstract(NodeAbstract parent, Element element, int depth)
	{
		this(parent.getDocument(), element, depth);
		this.parent = parent;
	}


	/**
	 * This constructor is used when an EAD-File or METS-file is being read.
	 * NOTE: If I am an ArchDesc (without a parent node - my parent is a Document), this constructor is called directly,
	 * 		NOT via the constructor ComponentAbstract(NodeAbstract parent, Element element, int depth)!
	 */
	protected ComponentAbstract(DocumentAbstract document, Element element, int depth)
	{
		this.document = document;
		this.element = element;

		this.depth = depth;

		this.otherLevel = this.document.getLevels().get(element.attributeValue("otherlevel"));

		List<String> pids = new Vector<String>(3);
		for (Object e: element.selectNodes(XPathPID))		pids.add(((Element)e).attributeValue("href"));
		if (!pids.isEmpty())		this.pid = pids;

		this.submitStatus = SubmitStatus.named(this.getElementText(XPathSubmitStatus, false));

		this.author = this.getDynamicElementTexts(XPathAuthor);
		this.comment = this.getDynamicElementTexts(XPathComment);

		this.accessNr = this.getDynamicElementTexts(XPathAccessNr);
		this.scopeContent = this.getDynamicElementTexts(XPathScopeContent);
		this.accessPolicy = this.getDynamicElementTexts(XPathAccessPolicy);
		this.accessRestriction = this.getDynamicElementTexts(XPathAccessRestriction);
		this.accessRestrictionExplanation = this.getDynamicElementTexts(XPathAccessRestrictionExplanation);
		this.accessRestrictionPeriod = this.getDynamicElementTexts(XPathAccessRestrictionPeriod);
		this.accessRestrictionPeriodBaseYear = this.getDynamicElementTexts(XPathAccessRestrictionPeriodBaseYear);
		this.accessRestrictionClassification = this.getDynamicElementTexts(XPathAccessRestrictionClassification);
		this.accessRestrictionPrivacy = this.getDynamicElementTexts(XPathAccessRestrictionPrivacy);
		this.accessRestrictionStatus = this.getDynamicElementTexts(XPathAccessRestrictionStatus);
		this.accessRestrictionStatusExplanation = this.getDynamicElementTexts(XPathAccessRestrictionStatusExplanation);
		this.usagePermission = this.getDynamicElementTexts(XPathUsagePermission);
		this.usagePermissionExpiringDate = this.getDynamicElementTexts(XPathUsagePermissionExpiringDate);
		this.retentionPolicy = this.getDynamicElementTexts(XPathRetentionPolicy);
		this.retentionPeriodBaseYear = this.getDynamicElementTexts(XPathRetentionPeriodBaseYear);
		this.archivalHistory = this.getDynamicElementTexts(XPathArchivalHistory);
		this.sourceType = this.getDynamicElementTexts(XPathSourceType);
		this.modeOfAcquisition = this.getDynamicElementTexts(XPathModeOfAcquisition);
		this.objectType = this.getDynamicElementTexts(XPathObjectType);
		this.conditionsOfReproductions = this.getDynamicElementTexts(XPathConditionsOfReproductions);
		this.characteristics = this.getDynamicElementTexts(XPathCharacteristics);
		this.findingAids = this.getDynamicElementTexts(XPathFindingAids);
		this.locationOfOriginals = this.getDynamicElementTexts(XPathLocationOfOriginals);
		this.reproductions = this.getDynamicElementTexts(XPathReproductions);
		this.relatedMaterial = this.getDynamicElementTexts(XPathRelatedMaterial);
		this.institution = this.getDynamicElementTexts(XPathInstitution);
		this.appraisalAndDestruction = this.getDynamicElementTexts(XPathAppraisalAndDestruction);
		this.arrangement = this.getDynamicElementTexts(XPathArrangement);
		this.bibliography = this.getDynamicElementTexts(XPathBibliography);
		this.descriptionLevel = this.getDynamicElementTexts(XPathDescriptionLevel);
		this.descriptionLevelNotes = this.getDynamicElementTexts(XPathDescriptionLevelNotes);
		this.descriptionRules = this.getDynamicElementTexts(XPathDescriptionRules);
		this.revisions = this.getDynamicElementTexts(XPathRevisions);
		this.digitization = this.getDynamicElementTexts(XPathDigitization);

		this.biographicalHistory = this.getDynamicElementTexts(XPathBiographicalHistory);
		this.deathOfAuthor = this.getDynamicElementTexts(XPathDeathOfAuthor);
		this.accruals = this.getDynamicElementTexts(XPathAccruals);
		this.characteristicsNote = this.getDynamicElementTexts(XPathCharacteristicsNote);
		this.subject = this.getDynamicElementTexts(XPathSubject);

		this.responsible = this.getDynamicElementTexts(XPathResponsible);
		this.involved = this.getDynamicElementTexts(XPathInvolved);
		this.staff = this.getDynamicElementTexts(XPathStaff);
		this.projectName = this.getDynamicElementTexts(XPathProjectName);
		this.projectTitle = this.getDynamicElementTexts(XPathProjectTitle);
		this.project = this.getDynamicElementTexts(XPathProject);
		this.projectAbbreviation = this.getDynamicElementTexts(XPathProjectAbbreviation);
		this.location = this.getDynamicElementTexts(XPathLocation);
		this.journal = this.getDynamicElementTexts(XPathJournal);
		this.university = this.getDynamicElementTexts(XPathUniversity);
		this.institute = this.getDynamicElementTexts(XPathInstitute);
		this.compartment = this.getDynamicElementTexts(XPathCompartment);
		this.method = this.getDynamicElementTexts(XPathMethod);
		this.keyword = this.getDynamicElementTexts(XPathKeyword);
		this.fundingSource = this.getDynamicElementTexts(XPathFundingSource);
		this.status = this.getDynamicElementTexts(XPathStatus);
		this.publisher = this.getDynamicElementTexts(XPathPublisher);
		this.doiJournal = this.getDynamicElementTexts(XPathDOIJournal);
		this.retentionPeriod = this.getDynamicElementTexts(XPathRetentionPeriod);
		this.usage = this.getDynamicElementTexts(XPathUsage);

		this.cs = C.parse(this, element, depth + 1);
		this.did = DID.parse(this, this.element);
	}


	/**
	 * This constructor is used when a new ComponentAbstract is created programmatically within an EAD Document:
	 */
	protected ComponentAbstract(ComponentAbstract parent, Element element, String title)
	{
		this.document = parent.getDocument();
		this.element = element;
		this.element.addAttribute("level", Level);		//	'level' is a mandatory attribute so I have to set it

		this.depth = parent.depth + 1;
		this.did = new DID(this, title);

		this.document.setIsModified();
	}


	/**
	 * This constructor is used when a new ComponentAbstract is created programmatically within a METS Document:
	 */
	protected ComponentAbstract(DMDSectionWithEAD parent, Element element, String title)
	{
		this.document = parent.getDocument();
		this.element = element;
		this.element
			.addAttribute("level", Level)				//	'level' is a mandatory attribute so I have to set it
			.addAttribute("otherlevel", this.otherLevel.getName());

		this.depth = 0;
		this.did = new DID(this, title);

		this.document.setIsModified();
	}

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * This is done on EAD export.
	 */
	protected void copyEADPropertiesFrom(ComponentAbstract source)
	{
		//	Copy all EAD properties from the source component to me:
		this.setOtherLevel(source.getOtherLevel());
		this.setPID(source.getPID());

		this.setAuthor(source.getAuthor());
		this.setComment(source.getComment());

		this.setAccessNr(source.getAccessNr());
		this.setScopeContent(source.getScopeContent());
		this.setAccessPolicy(source.getAccessPolicy());
		this.setAccessRestriction(source.getAccessRestriction());
		this.setAccessRestrictionExplanation(source.getAccessRestrictionExplanation());
		this.setAccessRestrictionPeriod(source.getAccessRestrictionPeriod());
		this.setAccessRestrictionPeriodBaseYear(source.getAccessRestrictionPeriodBaseYear());
		this.setAccessRestrictionClassification(source.getAccessRestrictionClassification());
		this.setAccessRestrictionPrivacy(source.getAccessRestrictionPrivacy());
		this.setAccessRestrictionStatus(source.getAccessRestrictionStatus());
		this.setAccessRestrictionStatusExplanation(source.getAccessRestrictionStatusExplanation());
		this.setUsagePermission(source.getUsagePermission());
		this.setUsagePermissionExpiringDate(source.getUsagePermissionExpiringDate());
		this.setRetentionPolicy(source.getRetentionPolicy());
		this.setRetentionPeriodBaseYear(source.getRetentionPeriodBaseYear());
		this.setArchivalHistory(source.getArchivalHistory());
		this.setSourceType(source.getSourceType());
		this.setModeOfAcquisition(source.getModeOfAcquisition());
		this.setObjectType(source.getObjectType());
		this.setConditionsOfReproductions(source.getConditionsOfReproductions());
		this.setCharacteristics(source.getCharacteristics());
		this.setFindingAids(source.getFindingAids());
		this.setLocationOfOriginals(source.getLocationOfOriginals());
		this.setReproductions(source.getReproductions());
		this.setRelatedMaterial(source.getRelatedMaterial());
		this.setInstitution(source.getInstitution());
		this.setAppraisalAndDestruction(source.getAppraisalAndDestruction());
		this.setArrangement(source.getArrangement());
		this.setBibliography(source.getBibliography());
		this.setDescriptionLevel(source.getDescriptionLevel());
		this.setDescriptionLevelNotes(source.getDescriptionLevelNotes());
		this.setDescriptionRules(source.getDescriptionRules());
		this.setRevisions(source.getRevisions());
		this.setDigitization(source.getDigitization());

		this.setResponsible(source.getResponsible());
		this.setInvolved(source.getInvolved());
		this.setStaff(source.getStaff());
		this.setProjectName(source.getProjectName());
		this.setProjectTitle(source.getProjectTitle());
		this.setProject(source.getProject());
		this.setProjectAbbreviation(source.getProjectAbbreviation());
		this.setLocation(source.getLocation());
		this.setJournal(source.getJournal());
		this.setUniversity(source.getUniversity());
		this.setInstitute(source.getInstitute());
		this.setCompartment(source.getCompartment());
		this.setMethod(source.getMethod());
		this.setKeyword(source.getKeyword());
		this.setFundingSource(source.getFundingSource());
		this.setStatus(source.getStatus());
		this.setPublisher(source.getPublisher());
		this.setDoiJournal(source.getDoiJournal());
		this.setRetentionPeriod(source.getRetentionPeriod());
		this.setUsage(source.getUsage());
		
		this.setBiographicalHistory(source.getBiographicalHistory());
		this.setDeathOfAuthor(source.getDeathOfAuthor());
		this.setAccruals(source.getAccruals());
		this.setCharacteristicsNote(source.getCharacteristicsNote());
		this.setSubject(source.getSubject());

		//	Data stored in my did element:
		this.setUnitTitle(source.getUnitTitle());
		this.setUnitTitleAdditional(source.getUnitTitleAdditional());
		this.setOrigination(source.getOrigination());
		this.setCreationPeriod(source.getCreationPeriod());
		this.setCreationPeriodNotes(source.getCreationPeriodNotes());
		this.setFromYear(source.getFromYear());
		this.setToYear(source.getToYear());
		this.setRelationPeriod(source.getRelationPeriod());
		this.setLanguage(source.getLanguage());
		this.setLanguageNotes(source.getLanguageNotes());
		this.setRefCode(source.getRefCode());
		this.setRefCodeOld(source.getRefCodeOld());
		this.setRefCodeAdmin(source.getRefCodeAdmin());
		this.setAccessionNumber(source.getAccessionNumber());
		this.setExtent(source.getExtent());
		this.setMaterial(source.getMaterial());

		this.setYear(source.getYear());
		this.setAbstract(source.getAbstract());
	}

	//	--------		Accessing			-------------------------------------------------------

	public DID getDid()
	{
		return this.did;
	}

	public List<C> getCs()
	{
		return this.cs;
	}

	public C getC(int i)
	{
		return this.cs.get(i);
	}


	public String getOtherLevelName()
	{
		return this.otherLevel.getName();
	}

	public void setOtherLevelName(String newLevel)
	{
		this.setOtherLevel(this.document.getLevels().get(newLevel));
	}

	public LevelOfDescription getOtherLevel()
	{
		return this.otherLevel;
	}

	public void setOtherLevel(LevelOfDescription level)
	{
		this.otherLevel = level;
		this.element
//			.addAttribute("level", Level)		//	This attribute is already set because it is mandatory
			.addAttribute("otherlevel", this.otherLevel.getName());

		this.document.setIsModified();
	}


	public String getUnitTitle()
	{
		return this.did.getUnitTitle();
	}

	public void setUnitTitle(String unitTitle)
	{
		this.did.setUnitTitle(unitTitle);
	}


	/**
	 * Return the submit status. If it is null, return SubmitStatus.SubmitUndefined.
	 * @return
	 */
	public SubmitStatus getSubmitStatus()
	{
		if (this.submitStatus == null)	return SubmitStatus.SubmitUndefined;

		return this.submitStatus;
	}

	public void setSubmitStatus(SubmitStatus submitStatus)
	{
		if (submitStatus == null)		submitStatus = SubmitStatus.SubmitUndefined;

		this.submitStatus = submitStatus;

		Node submitStatusNode = this.element.selectSingleNode(XPathSubmitStatus);
		if (submitStatusNode == null)
		{
			Element controlAccessNode = (Element)this.element.selectSingleNode("EAD:controlaccess");
			if (controlAccessNode == null)		controlAccessNode = this.element.addElement("EAD:controlaccess");

			submitStatusNode = controlAccessNode.addElement("EAD:name").addAttribute("role", "submitStatus");
		}
		submitStatusNode.setText(this.submitStatus.name());

		this.document.setIsModified();
	}

	//	--------		Accessing Dynamic	-------------------------------------------------------

	/**
	 * Dynamic Metadata
	 */
	public List<String> getPID()
	{
		return this.pid;
	}

	/**
	 * Dynamic Metadata: special way to set the values - don't use the standard way with this.setDynamicElementTexts(value, XPath);
	 */
	public void setPID(List<String> values)
	{
		this.pid = values;

		//	First delete all elements (if any)...:
		for (Object node: this.element.selectNodes(XPathPID))		((Node)node).detach();

		//	... then set them (if not null):
		if (values != null)
		{
			for (String pid: values)
			{
				this.element
					.addElement("EAD:dao")
						.addAttribute(new QName("role", new Namespace("xlink", Xmlns_xlink)), Role)
						.addAttribute(new QName("href", new Namespace("xlink", Xmlns_xlink)), pid);
			}
		}

		this.document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getUnitTitleAdditional()
	{
		return this.did.getUnitTitleAdditional();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setUnitTitleAdditional(List<String> s)
	{
		this.did.setUnitTitleAdditional(s);
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getAuthor()
	{
		return this.author;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAuthor(List<String> s)
	{
		this.author = s;

		this.setDynamicElementTexts(this.author, XPathAuthor);
		if (this.author == null)		this.deleteParentElementIfEmpty(XPathAuthor);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getComment()
	{
		return this.comment;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setComment(List<String> values)
	{
		this.comment = values;

		this.setDynamicElementTexts(this.comment, XPathComment);
		if (this.comment == null)		this.deleteParentElementIfEmpty(XPathComment);

		this.document.setIsModified();
	}



	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessNr()
	{
		return this.accessNr;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessNr(List<String> values)
	{
		this.accessNr = values;

		this.setDynamicElementTexts(this.accessNr, XPathAccessNr);
		if (this.accessNr == null)		this.deleteParentElementIfEmpty(XPathAccessNr);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getOrigination()
	{
		return this.did.getOrigination();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setOrigination(List<String> s)
	{
		this.did.setOrigination(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getCreationPeriod()
	{
		return this.did.getCreationPeriod();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setCreationPeriod(List<String> s)
	{
		this.did.setCreationPeriod(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getCreationPeriodNotes()
	{
		return this.did.getCreationPeriodNotes();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setCreationPeriodNotes(List<String> s)
	{
		this.did.setCreationPeriodNotes(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getFromYear()
	{
		return this.did.getFromYear();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setFromYear(List<String> s)
	{
		this.did.setFromYear(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getToYear()
	{
		return this.did.getToYear();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setToYear(List<String> s)
	{
		this.did.setToYear(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRelationPeriod()
	{
		return this.did.getRelationPeriod();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRelationPeriod(List<String> s)
	{
		this.did.setRelationPeriod(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getScopeContent()
	{
		return this.scopeContent;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setScopeContent(List<String> values)
	{
		this.scopeContent = values;

		this.setDynamicElementTexts(this.scopeContent, XPathScopeContent);
		if (this.scopeContent == null)		this.deleteParentElementIfEmpty(XPathScopeContent);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getLanguage()
	{
		return this.did.getLanguage();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setLanguage(List<String> s)
	{
		this.did.setLanguage(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getLanguageNotes()
	{
		return this.did.getLanguageNotes();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setLanguageNotes(List<String> s)
	{
		this.did.setLanguageNotes(s);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessPolicy()
	{
		return this.accessPolicy;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessPolicy(List<String> values)
	{
		this.accessPolicy = values;

		this.setDynamicElementTexts(this.accessPolicy, XPathAccessPolicy);
		if (this.accessPolicy == null)		this.deleteParentElementIfEmpty(XPathAccessPolicy);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestriction()
	{
		return this.accessRestriction;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestriction(List<String> values)
	{
		this.accessRestriction = values;

		this.setDynamicElementTexts(this.accessRestriction, XPathAccessRestriction);
		if (this.accessRestriction == null)		this.deleteParentElementIfEmpty(XPathAccessRestriction);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestrictionExplanation()
	{
		return this.accessRestrictionExplanation;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestrictionExplanation(List<String> values)
	{
		this.accessRestrictionExplanation = values;

		this.setDynamicElementTexts(this.accessRestrictionExplanation, XPathAccessRestrictionExplanation);
		if (this.accessRestrictionExplanation == null)		this.deleteParentElementIfEmpty(XPathAccessRestrictionExplanation);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestrictionPeriod()
	{
		return this.accessRestrictionPeriod;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestrictionPeriod(List<String> values)
	{
		this.accessRestrictionPeriod = values;

		this.setDynamicElementTexts(this.accessRestrictionPeriod, XPathAccessRestrictionPeriod);
		if (this.accessRestrictionPeriod == null)		this.deleteParentElementIfEmpty(XPathAccessRestrictionPeriod);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestrictionPeriodBaseYear()
	{
		return this.accessRestrictionPeriodBaseYear;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestrictionPeriodBaseYear(List<String> values)
	{
		this.accessRestrictionPeriodBaseYear = values;

		this.setDynamicElementTexts(this.accessRestrictionPeriodBaseYear, XPathAccessRestrictionPeriodBaseYear);
		if (this.accessRestrictionPeriodBaseYear == null)		this.deleteParentElementIfEmpty(XPathAccessRestrictionPeriodBaseYear);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestrictionClassification()
	{
		return this.accessRestrictionClassification;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestrictionClassification(List<String> values)
	{
		this.accessRestrictionClassification = values;

		this.setDynamicElementTexts(this.accessRestrictionClassification, XPathAccessRestrictionClassification);
		if (this.accessRestrictionClassification == null)		this.deleteParentElementIfEmpty(XPathAccessRestrictionClassification);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestrictionPrivacy()
	{
		return this.accessRestrictionPrivacy;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestrictionPrivacy(List<String> values)
	{
		this.accessRestrictionPrivacy = values;

		this.setDynamicElementTexts(this.accessRestrictionPrivacy, XPathAccessRestrictionPrivacy);
		if (this.accessRestrictionPrivacy == null)		this.deleteParentElementIfEmpty(XPathAccessRestrictionPrivacy);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestrictionStatus()
	{
		return this.accessRestrictionStatus;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestrictionStatus(List<String> values)
	{
		this.accessRestrictionStatus = values;

		this.setDynamicElementTexts(this.accessRestrictionStatus, XPathAccessRestrictionStatus);
		if (this.accessRestrictionStatus == null)		this.deleteParentElementIfEmpty(XPathAccessRestrictionStatus);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessRestrictionStatusExplanation()
	{
		return this.accessRestrictionStatusExplanation;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessRestrictionStatusExplanation(List<String> values)
	{
		this.accessRestrictionStatusExplanation = values;

		this.setDynamicElementTexts(this.accessRestrictionStatusExplanation, XPathAccessRestrictionStatusExplanation);
		if (this.accessRestrictionStatusExplanation == null)		this.deleteParentElementIfEmpty(XPathAccessRestrictionStatusExplanation);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getUsagePermission()
	{
		return this.usagePermission;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setUsagePermission(List<String> values)
	{
		this.usagePermission = values;

		this.setDynamicElementTexts(this.usagePermission, XPathUsagePermission);
		if (this.usagePermission == null)		this.deleteParentElementIfEmpty(XPathUsagePermission);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getUsagePermissionExpiringDate()
	{
		return this.usagePermissionExpiringDate;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setUsagePermissionExpiringDate(List<String> values)
	{
		this.usagePermissionExpiringDate = values;

		this.setDynamicElementTexts(this.usagePermissionExpiringDate, XPathUsagePermissionExpiringDate);
		if (this.usagePermissionExpiringDate == null)		this.deleteParentElementIfEmpty(XPathUsagePermissionExpiringDate);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRetentionPolicy()
	{
		return this.retentionPolicy;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRetentionPolicy(List<String> retentionPolicy)
	{
		this.retentionPolicy = retentionPolicy;

		this.setDynamicElementTexts(this.retentionPolicy, XPathRetentionPolicy);
		if (this.retentionPolicy == null)		this.deleteParentElementIfEmpty(XPathRetentionPolicy);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRetentionPeriodBaseYear()
	{
		return this.retentionPeriodBaseYear;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRetentionPeriodBaseYear(List<String> retentionPeriodBaseYear)
	{
		this.retentionPeriodBaseYear = retentionPeriodBaseYear;

		this.setDynamicElementTexts(this.retentionPeriodBaseYear, XPathRetentionPeriodBaseYear);
		if (this.retentionPeriodBaseYear == null)		this.deleteParentElementIfEmpty(XPathRetentionPeriodBaseYear);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getArchivalHistory()
	{
		return this.archivalHistory;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setArchivalHistory(List<String> values)
	{
		this.archivalHistory = values;

		this.setDynamicElementTexts(this.archivalHistory, XPathArchivalHistory);
		if (this.archivalHistory == null)		this.deleteParentElementIfEmpty(XPathArchivalHistory);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getSourceType()
	{
		return this.sourceType;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setSourceType(List<String> values)
	{
		this.sourceType = values;

		this.setDynamicElementTexts(this.sourceType, XPathSourceType);
		if (this.sourceType == null)		this.deleteParentElementIfEmpty(XPathSourceType);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getModeOfAcquisition()
	{
		return this.modeOfAcquisition;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setModeOfAcquisition(List<String> values)
	{
		this.modeOfAcquisition = values;

		this.setDynamicElementTexts(this.modeOfAcquisition, XPathModeOfAcquisition);
		if (this.modeOfAcquisition == null)		this.deleteParentElementIfEmpty(XPathModeOfAcquisition);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getObjectType()
	{
		return this.objectType;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setObjectType(List<String> objectType)
	{
		this.objectType = objectType;

		this.setDynamicElementTexts(this.objectType, XPathObjectType);
		if (this.objectType == null)		this.deleteParentElementIfEmpty(XPathObjectType);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCode()
	{
		return this.did.getRefCode();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCode(List<String> refCode)
	{
		this.did.setRefCode(refCode);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCodeOld()
	{
		return this.did.getRefCodeOld();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCodeOld(List<String> refCodeOld)
	{
		this.did.setRefCodeOld(refCodeOld);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCodeAdmin()
	{
		return this.did.getRefCodeAdmin();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCodeAdmin(List<String> refCodeAdmin)
	{
		this.did.setRefCodeAdmin(refCodeAdmin);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessionNumber()
	{
		return this.did.getAccessionNumber();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessionNumber(List<String> accessionNumber)
	{
		this.did.setAccessionNumber(accessionNumber);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getExtent()
	{
		return this.did.getExtent();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setExtent(List<String> extent)
	{
		this.did.setExtent(extent);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getMaterial()
	{
		return this.did.getMaterial();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setMaterial(List<String> material)
	{
		this.did.setMaterial(material);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getConditionsOfReproductions()
	{
		return this.conditionsOfReproductions;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setConditionsOfReproductions(List<String> values)
	{
		this.conditionsOfReproductions = values;

		this.setDynamicElementTexts(this.conditionsOfReproductions, XPathConditionsOfReproductions);
		if (this.conditionsOfReproductions == null)		this.deleteParentElementIfEmpty(XPathConditionsOfReproductions);

		this.document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getCharacteristics()
	{
		return this.characteristics;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setCharacteristics(List<String> values)
	{
		this.characteristics = values;

		this.setDynamicElementTexts(this.characteristics, XPathCharacteristics);
		if (this.characteristics == null)		this.deleteParentElementIfEmpty(XPathCharacteristics);

		this.document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getFindingAids()
	{
		return this.findingAids;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setFindingAids(List<String> values)
	{
		this.findingAids = values;

		this.setDynamicElementTexts(this.findingAids, XPathFindingAids);
		if (this.findingAids == null)		this.deleteParentElementIfEmpty(XPathFindingAids);

		this.document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getLocationOfOriginals()
	{
		return this.locationOfOriginals;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setLocationOfOriginals(List<String> values)
	{
		this.locationOfOriginals = values;

		this.setDynamicElementTexts(this.locationOfOriginals, XPathLocationOfOriginals);
		if (this.locationOfOriginals == null)		this.deleteParentElementIfEmpty(XPathLocationOfOriginals);

		this.document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getReproductions()
	{
		return this.reproductions;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setReproductions(List<String> values)
	{
		this.reproductions = values;

		this.setDynamicElementTexts(this.reproductions, XPathReproductions);
		if (this.reproductions == null)		this.deleteParentElementIfEmpty(XPathReproductions);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRelatedMaterial()
	{
		return this.relatedMaterial;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRelatedMaterial(List<String> values)
	{
		this.relatedMaterial = values;

		this.setDynamicElementTexts(this.relatedMaterial, XPathRelatedMaterial);
		if (this.relatedMaterial == null)		this.deleteParentElementIfEmpty(XPathRelatedMaterial);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getInstitution()
	{
		return this.institution;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setInstitution(List<String> institution)
	{
		this.institution = institution;

		this.setDynamicElementTexts(this.institution, XPathInstitution);
		if (this.institution == null)		this.deleteParentElementIfEmpty(XPathInstitution);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAppraisalAndDestruction()
	{
		return this.appraisalAndDestruction;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAppraisalAndDestruction(List<String> values)
	{
		this.appraisalAndDestruction = values;

		this.setDynamicElementTexts(this.appraisalAndDestruction, XPathAppraisalAndDestruction);
		if (this.appraisalAndDestruction == null)		this.deleteParentElementIfEmpty(XPathAppraisalAndDestruction);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getArrangement()
	{
		return this.arrangement;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setArrangement(List<String> values)
	{
		this.arrangement = values;

		this.setDynamicElementTexts(this.arrangement, XPathArrangement);
		if (this.arrangement == null)		this.deleteParentElementIfEmpty(XPathArrangement);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getBibliography()
	{
		return this.bibliography;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setBibliography(List<String> values)
	{
		this.bibliography = values;

		this.setDynamicElementTexts(this.bibliography, XPathBibliography);
		if (this.bibliography == null)		this.deleteParentElementIfEmpty(XPathBibliography);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getDescriptionLevel()
	{
		return this.descriptionLevel;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setDescriptionLevel(List<String> descriptionLevel)
	{
		this.descriptionLevel = descriptionLevel;

		this.setDynamicElementTexts(this.descriptionLevel, XPathDescriptionLevel);
		if (this.descriptionLevel == null)		this.deleteParentElementIfEmpty(XPathDescriptionLevel);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getDescriptionLevelNotes()
	{
		return this.descriptionLevelNotes;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setDescriptionLevelNotes(List<String> values)
	{
		this.descriptionLevelNotes = values;

		this.setDynamicElementTexts(this.descriptionLevelNotes, XPathDescriptionLevelNotes);
		if (this.descriptionLevelNotes == null)		this.deleteParentElementIfEmpty(XPathDescriptionLevelNotes);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getDescriptionRules()
	{
		return this.descriptionRules;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setDescriptionRules(List<String> values)
	{
		this.descriptionRules = values;

		this.setDynamicElementTexts(this.descriptionRules, XPathDescriptionRules);
		if (this.descriptionRules == null)		this.deleteParentElementIfEmpty(XPathDescriptionRules);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRevisions()
	{
		return this.revisions;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRevisions(List<String> revisions)
	{
		this.revisions = revisions;

		this.setDynamicElementTexts(this.revisions, XPathRevisions);
		if (this.revisions == null)		this.deleteParentElementIfEmpty(XPathRevisions);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getDigitization()
	{
		return this.digitization;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setDigitization(List<String> value)
	{
		this.digitization = value;

		this.setDynamicElementTexts(this.digitization, XPathDigitization);
		if (this.digitization == null)		this.deleteParentElementIfEmpty(XPathDigitization);

		this.document.setIsModified();
	}

	//	--------			ETH Forschungsdaten:

	/**
	 * Dynamic Metadata
	 */
	public List<String> getResponsible()
	{
		return this.responsible;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setResponsible(List<String> responsible)
	{
		this.responsible = responsible;

		this.setDynamicElementTexts(this.responsible, XPathResponsible);
		if (this.responsible == null)		this.deleteParentElementIfEmpty(XPathResponsible);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getInvolved()
	{
		return this.involved;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setInvolved(List<String> involved)
	{
		this.involved = involved;

		this.setDynamicElementTexts(this.involved, XPathInvolved);
		if (this.involved == null)		this.deleteParentElementIfEmpty(XPathInvolved);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getStaff()
	{
		return this.staff;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setStaff(List<String> staff)
	{
		this.staff = staff;

		this.setDynamicElementTexts(this.staff, XPathStaff);
		if (this.staff == null)		this.deleteParentElementIfEmpty(XPathStaff);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getProjectName()
	{
		return this.projectName;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setProjectName(List<String> projectName)
	{
		this.projectName = projectName;

		this.setDynamicElementTexts(this.projectName, XPathProjectName);
		if (this.projectName == null)		this.deleteParentElementIfEmpty(XPathProjectName);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getProjectTitle()
	{
		return this.projectTitle;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setProjectTitle(List<String> projectTitle)
	{
		this.projectTitle = projectTitle;

		this.setDynamicElementTexts(this.projectTitle, XPathProjectTitle);
		if (this.projectTitle == null)		this.deleteParentElementIfEmpty(XPathProjectTitle);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getProject()
	{
		return this.project;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setProject(List<String> project)
	{
		this.project = project;

		this.setDynamicElementTexts(this.project, XPathProject);
		if (this.project == null)		this.deleteParentElementIfEmpty(XPathProject);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getProjectAbbreviation()
	{
		return this.projectAbbreviation;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setProjectAbbreviation(List<String> projectAbbreviation)
	{
		this.projectAbbreviation = projectAbbreviation;

		this.setDynamicElementTexts(this.projectAbbreviation, XPathProjectAbbreviation);
		if (this.projectAbbreviation == null)		this.deleteParentElementIfEmpty(XPathProjectAbbreviation);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getLocation()
	{
		return this.location;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setLocation(List<String> location)
	{
		this.location = location;

		this.setDynamicElementTexts(this.location, XPathLocation);
		if (this.location == null)		this.deleteParentElementIfEmpty(XPathLocation);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getJournal()
	{
		return this.journal;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setJournal(List<String> journal)
	{
		this.journal = journal;

		this.setDynamicElementTexts(this.journal, XPathJournal);
		if (this.journal == null)		this.deleteParentElementIfEmpty(XPathJournal);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getUniversity()
	{
		return this.university;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setUniversity(List<String> university)
	{
		this.university = university;

		this.setDynamicElementTexts(this.university, XPathUniversity);
		if (this.university == null)		this.deleteParentElementIfEmpty(XPathUniversity);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getInstitute()
	{
		return this.institute;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setInstitute(List<String> institute)
	{
		this.institute = institute;

		this.setDynamicElementTexts(this.institute, XPathInstitute);
		if (this.institute == null)		this.deleteParentElementIfEmpty(XPathInstitute);

		this.document.setIsModified();
	}

	public List<String> getCompartment()
	{
		return this.compartment;
	}

	public void setCompartment(List<String> compartment)
	{
		this.compartment = compartment;

		this.setDynamicElementTexts(this.compartment, XPathCompartment);
		if (this.compartment == null)		this.deleteParentElementIfEmpty(XPathCompartment);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getMethod()
	{
		return this.method;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setMethod(List<String> method)
	{
		this.method = method;

		this.setDynamicElementTexts(this.method, XPathMethod);
		if (this.method == null)		this.deleteParentElementIfEmpty(XPathMethod);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getYear()
	{
		return this.did.getYear();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setYear(List<String> year)
	{
		this.did.setYear(year);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getKeyword()
	{
		return this.keyword;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setKeyword(List<String> keyword)
	{
		this.keyword = keyword;

		this.setDynamicElementTexts(this.keyword, XPathKeyword);
		if (this.keyword == null)		this.deleteParentElementIfEmpty(XPathKeyword);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getFundingSource()
	{
		return this.fundingSource;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setFundingSource(List<String> fundingSource)
	{
		this.fundingSource = fundingSource;

		this.setDynamicElementTexts(this.fundingSource, XPathFundingSource);
		if (this.fundingSource == null)		this.deleteParentElementIfEmpty(XPathFundingSource);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getStatus()
	{
		return this.status;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setStatus(List<String> status)
	{
		this.status = status;

		this.setDynamicElementTexts(this.status, XPathStatus);
		if (this.status == null)		this.deleteParentElementIfEmpty(XPathStatus);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getPublisher()
	{
		return this.publisher;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setPublisher(List<String> publisher)
	{
		this.publisher = publisher;

		this.setDynamicElementTexts(this.publisher, XPathPublisher);
		if (this.publisher == null)		this.deleteParentElementIfEmpty(XPathPublisher);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAbstract()
	{
		return this.did.getAbstract();
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAbstract(List<String> abstract_)
	{
		this.did.setAbstract(abstract_);
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getDoiJournal()
	{
		return this.doiJournal;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setDoiJournal(List<String> doiJournal)
	{
		this.doiJournal = doiJournal;

		this.setDynamicElementTexts(this.doiJournal, XPathDOIJournal);
		if (this.doiJournal == null)		this.deleteParentElementIfEmpty(XPathDOIJournal);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRetentionPeriod()
	{
		return this.retentionPeriod;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRetentionPeriod(List<String> retentionPeriod)
	{
		this.retentionPeriod = retentionPeriod;

		this.setDynamicElementTexts(this.retentionPeriod, XPathRetentionPeriod);
		if (this.retentionPeriod == null)		this.deleteParentElementIfEmpty(XPathRetentionPeriod);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getUsage()
	{
		return this.usage;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setUsage(List<String> usage)
	{
		this.usage = usage;

		this.setDynamicElementTexts(this.usage, XPathUsage);
		if (this.usage == null)		this.deleteParentElementIfEmpty(XPathUsage);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getBiographicalHistory()
	{
		return this.biographicalHistory;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setBiographicalHistory(List<String> values)
	{
		this.biographicalHistory = values;

		this.setDynamicElementTexts(this.biographicalHistory, XPathBiographicalHistory);
		if (this.biographicalHistory == null)		this.deleteParentElementIfEmpty(XPathBiographicalHistory);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getDeathOfAuthor()
	{
		return this.deathOfAuthor;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setDeathOfAuthor(List<String> values)
	{
		this.deathOfAuthor = values;

		this.setDynamicElementTexts(this.deathOfAuthor, XPathDeathOfAuthor);
		if (this.deathOfAuthor == null)		this.deleteParentElementIfEmpty(XPathDeathOfAuthor);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccruals()
	{
		return this.accruals;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccruals(List<String> values)
	{
		this.accruals = values;

		this.setDynamicElementTexts(this.accruals, XPathAccruals);
		if (this.accruals == null)		this.deleteParentElementIfEmpty(XPathAccruals);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getCharacteristicsNote()
	{
		return this.characteristicsNote;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setCharacteristicsNote(List<String> values)
	{
		this.characteristicsNote = values;

		this.setDynamicElementTexts(this.characteristicsNote, XPathCharacteristicsNote);
		if (this.characteristicsNote == null)		this.deleteParentElementIfEmpty(XPathCharacteristicsNote);

		this.document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getSubject()
	{
		return this.subject;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setSubject(List<String> values)
	{
		this.subject = values;

		this.setDynamicElementTexts(this.subject, XPathSubject);
		if (this.subject == null)		this.deleteParentElementIfEmpty(XPathSubject);

		this.document.setIsModified();
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	public ComponentAbstract searchUnitTitle(String title)
	{
		if (this.did.getUnitTitle().equals(title))	return this;

		for (C c: this.cs)
		{
			ComponentAbstract found = c.searchUnitTitle(title);		//	Recursion!
			if (found != null)		return found;
		}

		return null;
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------

	/**
	 * This is done on EAD export.
	 */
	public abstract void fillEADElementsFromNodeRecursively(ch.docuteam.darc.mets.structmap.NodeAbstract node);

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	protected String toString(String name)
	{
		StringBuilder stringBuilder = new StringBuilder();

		//	Indent:
		for (int i = 0; i < this.depth; i++)		stringBuilder.append("\t");

		stringBuilder.append("[" + name + "(")
			.append(this.depth)
			.append("):'")
			.append(this.otherLevel)
			.append("':")
			.append(this.did);

		for (C c: this.cs)		stringBuilder.append("\n").append(c);

		return stringBuilder.append("]")
			.toString();
	}


	protected String toStringShort(String name)
	{
		return new StringBuilder()
			.append("[" + name + "(")
			.append(this.depth)
			.append("):'")
			.append(this.otherLevel)
			.append("':")
			.append(this.did)
			.append("]")
			.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
