/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Observer;
import java.util.Vector;

import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.exceptions.CantCreateTemplateWithRootFileException;
import ch.docuteam.darc.exceptions.DocumentIsReadOnlyException;
import ch.docuteam.darc.exceptions.FileOperationNotAllowedException;
import ch.docuteam.darc.exceptions.FileOrFolderIsInUseException;
import ch.docuteam.darc.exceptions.FolderIsNotEmptyException;
import ch.docuteam.darc.exceptions.FolderNameIsEmptyException;
import ch.docuteam.darc.exceptions.MetadataElementCantDeleteException;
import ch.docuteam.darc.exceptions.OriginalSIPIsMissingException;
import ch.docuteam.darc.exceptions.ZIPDoesNotContainMETSFileException;
import ch.docuteam.darc.ingest.AIPCreatorProxy;
import ch.docuteam.darc.mdconfig.MetadataElementInstance;
import ch.docuteam.darc.mets.amdsec.AMDSection;
import ch.docuteam.darc.mets.dmdsec.DMDSectionAbstract;
import ch.docuteam.darc.mets.filesec.FileSection;
import ch.docuteam.darc.mets.metshdr.Header;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.mets.structmap.NodeAbstract.SubmitStatus;
import ch.docuteam.darc.mets.structmap.NodeFile;
import ch.docuteam.darc.mets.structmap.NodeFolder;
import ch.docuteam.darc.mets.structmap.StructureMap;
import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.exception.ExceptionCollector;
import ch.docuteam.tools.exception.ExceptionCollectorException;
import ch.docuteam.tools.file.FileFilter;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.Zipper;
import ch.docuteam.tools.file.exception.FileUtilException;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.out.Tracer;
import ch.docuteam.tools.string.DateFormatter;


/**
 * An instance of this class is the java-equivalent of a METS-file.
 * It contains
 * <ul>
 * <li>a <a href="./metshdr/Header.html">Header</a></li>
 * <li>an <a href="./amdsec/AMDSection.html">AMDSection</a></li>
 * <li>a <a href="./filesec/FileSection.html">FileSection</a></li>
 * <li>and a <a href="./structmap/StructureMap.html">StructureMap</a></li>
 * </ul>
 * <p>
 * This class inserts Exceptions into the <a href="../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>When some files don't exist in the file system but should exist according to the METS file</li>
 * </ul>
 * @author denis
 */
@SuppressWarnings("serial")
public class Document extends DocumentAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Final Static Public		=======================================================

	static final public String			DefaultMETSFileName = "mets.xml";
	static final public String			LockFileSuffix = ".lock";

	//	========	Final Static Private	=======================================================

	static final protected String		WorkingFolderPrefix = "METSDocument_read_";
	static final protected String		WorkingFolderPrefixReadZip = "METSDocument_readZIP_";
	static final protected String		WorkingFolderPrefixCreateZip = "METSDocument_createZIP_";

	static final protected String		DefaultMETS_SIPTemplateFile = "resources/templates/mets_sip.xml";

	static final protected Integer		DefaultKeepBackupsCount = 10;

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static protected String				METS_SIPTemplateFile = DefaultMETS_SIPTemplateFile;

	/**
	 * If the backup folder is null, backups are created in the folder the SIP is in; otherwise in the specified backup folder (which can be either absolute or relative).
	 */
	static protected String				BackupFolder = null;

	/**
	 * How many backups to keep.
	 * 		0:		keep none
	 * 		3:		keep 3
	 * 		< 0:	is illegal and throws an exception
	 */
	static protected Integer			KeepBackupsCount = DefaultKeepBackupsCount;

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	/**
	 * The mode tells in how far the SIP can be modified.
	 */
	protected Mode						mode = Mode.Undefined;

	/**
	 * The folder where the original SIP is located. It can be both relative or absolute. It is never null.
	 */
	protected String					originalSIPFolder;

	/**
	 * True when the current SIP is a working copy, false when working directly on a SIP.
	 * In the latter case, all file operations are disabled.
	 * NOTE: When the original SIP is a ZIP archive, a working copy always exists, therefore isWorkingCopy is always true.
	 * Nevertheless file operations may be disabled when the SIP was opened in ReadOnly or NoFileOps mode.
	 */
	protected boolean					isWorkingCopy = true;

	/**
	 * The name of the user who locked this SIP. If null, the current user locked this SIP.
	 */
	protected String					lockedBy;


	//	Root element properties:
	protected String					namespaceMETS;
	protected String					namespacePREMIS;
	protected String					profile;
	protected String					label;
	protected String					type;
	protected String					saId;
	protected String					dssId;
	protected String					objId;


	/**
	 * The METS Header of this document.
	 */
	protected Header					header;

	/**
	 * Map containing the optional DMDSections. Key is the DMDSection's id, value is the DMDSection itself.
	 */
	protected Map<String, DMDSectionAbstract>
										dmdSections = new HashMap<String, DMDSectionAbstract>();

	/**
	 * The AMD Section of this document.
	 */
	protected AMDSection				amdSection;

	/**
	 * The File Section of this document.
	 */
	protected FileSection				fileSection;

	/**
	 * The Structure Map of this document.
	 */
	protected StructureMap				structureMap;

	/**
	 * Who is currently using this Document?
	 */
	protected String					currentOperatorName;

	/**
	 * The submission agreement, initialized lazily only when needed.
	 */
	protected SubmissionAgreement		submissionAgreement;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * For testing.
	 *
	 * @param args not needed
	 */
	public static void main(String... args)
	{
		String desktop = OperatingSystem.userHome() + "Desktop";
		String workspace = desktop + "/Workspace/Packer";
		String sips = workspace + "/SIPs";

		FileUtil.setTempFolder(workspace + "/Temp");
		setBackupFolder(workspace + "/Backups");

		Document doc = null;

		try
		{
			//	----------	Reading a SIP:
//			doc = Document.openReadOnly(sips + "/ch-001194-4_459", "Document.main");
//			doc = Document.openReadWriteFilesLocked(sips + "/ch-001194-4_459", "Document.main");
//			doc = Document.openReadWrite(sips + "/ch-001194-4_459", "Document.main");
//			doc = Document.openReadOnly(sips + "/ch-001194-4_459.zip", "Document.main");
//			doc = Document.openReadWriteFilesLocked(sips + "/ch-001194-4_459.zip", "Document.main");
//			doc = Document.openReadWrite(sips + "/ch-001194-4_459.zip", "Document.main");
//			doc = Document.openReadOnly(sips + "/Locked", "Document.main");
//			doc = Document.openReadWriteFilesLocked(sips + "/Locked", "Document.main");
//			doc = Document.openReadWrite(sips + "/Locked", "Document.main");

			//	----------	Inserting Files and Folders:
//			doc = Document.openReadWrite(sips + "/MetsWithMetadata", "Document.main");
//			NodeFolder folder = (NodeFolder)doc.getStructureMap().searchId("_20120606105132179");
//			folder.insertFileOrFolder(desktop + "/TestNoMimeTypeAndFormatID.denis");

			//	----------	Creating new SIPs:
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new", "files/Handbuch_Arena.pdf", "sa_all-formats-01", "dss-01", "Document.main");	//	Create a new Document with a file as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new.zip", "files/Handbuch_Arena.pdf", "sa_all-formats-01", "dss-01", "Document.main");	//	Create a new ZIP Document with a file as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new", "files/Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new Document with a folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new.zip", "files/Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new ZIP Document with a folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/SIPs/Empty Folder", OperatingSystem.userHome() + "/Desktop/Empty Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new Document with an empty folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/SIPs/Empty Folder.zip", OperatingSystem.userHome() + "/Desktop/Empty Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new ZIP Document with an empty folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/SIPs/Test", OperatingSystem.userHome() + "/Desktop/Test/TopFolder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new ZIP Document with an empty folder as the root element
			doc = Document.createNewWithRootFolderName(OperatingSystem.userHome() + "/Desktop/SIPs/Test", "TopFolder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new ZIP Document with an empty folder as the root element

			//	----------	Reading the created SIPs:
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/created/new/mets.xml", "Document.main");						//	Backcheck the previously created Document
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/created/new.zip", "Document.main");							//	Backcheck the previously created Document

			//	----------	Rename:
//			doc = Document.openReadWrite(sips + "/RepeatedFileNames", "Document.main");
//			NodeFolder node = (NodeFolder)doc.getStructureMap().getRoot();
//			NodeFolder node = (NodeFolder)doc.getStructureMap().searchId("_20140512110705492");
//			node.rename("imageRenamed");
//			doc.saveWithBackup();

			//	----------	Dynamic metadata:
//			doc = Document.openReadWriteFilesLocked(sips + "/MetsWithMetadata", "Document.main");
//			NodeAbstract node = doc.getStructureMap().getRoot();
//			doc = Document.openReadWrite("files/NestedFolders/mets.xml", "Document.main");									//	METS with Levels
//			NodeAbstract node = doc.getStructureMap().searchId("_20120606105132179");
//			doc = Document.openReadWrite("files/EmptyMets/mets.xml", "Document.main");										//	Empty SIP
//			NodeAbstract node = doc.getStructureMap().getRoot();

			//		Check special Exceptions:
//			doc = Document.openReadOnly(sips + "/SIP_20130307_Test1_Lem", "Document.main");
//			NodeAbstract paketNode = doc.getStructureMap().searchId("_20140411101229193");					//	Node "Ablieferung"
//
//			Tracer.trace(paketNode.getLevel());
//			Tracer.trace(paketNode.getAllDynamicMetadataValuesForName("refCode"));							//	OK and filled
//			Tracer.trace(paketNode.getAllDynamicMetadataValuesForName("material"));							//	OK and empty
//			try
//			{
//				Tracer.trace(paketNode.getAllDynamicMetadataValuesForName("bibliography"));					//	Not for this level
//			}
//			catch (MetadataElementIsNotAllowedException ex)
//			{
//				ex.printStackTrace();
//			}
//			try
//			{
//				Tracer.trace(paketNode.getAllDynamicMetadataValuesForName("bad metadata name"));			//	Not defined at all
//			}
//			catch (MetadataElementIsNotDefinedException ex)
//			{
//				ex.printStackTrace();
//			}
//
//			Tracer.trace(paketNode.getAllDynamicMetadataValuesForName_NoCheck("refCode"));					//	OK and filled
//			Tracer.trace(paketNode.getAllDynamicMetadataValuesForName_NoCheck("material"));					//	OK and empty
//			Tracer.trace(paketNode.getAllDynamicMetadataValuesForName_NoCheck("bibliography"));				//	Not for this level
//			try
//			{
//				Tracer.trace(paketNode.getAllDynamicMetadataValuesForName_NoCheck("bad metadata name"));	//	Not defined at all
//			}
//			catch (MetadataElementIsNotDefinedException ex)
//			{
//				ex.printStackTrace();
//			}

//			node.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();

//			node.setDynamicMetadataValueForName("unitTitleAdditional", "Additional Title");
//			node.setDynamicMetadataValueForName("accessNr", "Acqckg-Nr");
////			node.setDynamicMetadataValueForName("creationPeriod", "1963-02-09");								//	Default value is specified
//			node.setDynamicMetadataValueForName("creationPeriodNotes", "Starting on the 11th of June 2012");
//			node.setDynamicMetadataValueForName("origination", "Docuteam");
//			node.setDynamicMetadataValueForName("origination", "Docuteam, Dättwil");
//			node.setDynamicMetadataValueForName("origination", "Docuteam, Dättwil, Langacker 16");
////			node.setDynamicMetadataValueForName("origination", null);
////			node.setDynamicMetadataValueForName("title", "Great New Title");
//			node.addDynamicMetadataElementInstanceWithName("language");
//			node.setDynamicMetadataValueForName("language", 0, "DE");
//			node.addDynamicMetadataElementInstanceWithName("language");
//			node.setDynamicMetadataValueForName("language", 1, "EN");
//			node.addDynamicMetadataElementInstanceWithName("language");
//			node.setDynamicMetadataValueForName("language", 2, "FR");
//			node.setDynamicMetadataValueForName("language", 0, "XX");
//			node.setDynamicMetadataValueForName("languageNotes", "Language may be any");
////			node.setDynamicMetadataValueForName("languageNotes", null);
//			node.setDynamicMetadataValueForName("comment", "I think this is really not bad");
//			node.setDynamicMetadataValueForName("extent", "3 mal 3 mal 3 Meter");
//			node.setDynamicMetadataValueForName("material", "Pappe");
//			node.setDynamicMetadataValueForName("refCode", "12345-67890");
//			node.setDynamicMetadataValueForName("refCodeOld", "12345-67890-X");
//			node.setDynamicMetadataValueForName("refCodeAdmin", "12345-67890-A");
//			node.setDynamicMetadataValueForName("scopeContent", "This contains lots of wierd stuff");
//			node.setDynamicMetadataValueForName("accessRestriction", "Access Restriction");
//			node.setDynamicMetadataValueForName("archivalHistory", "Archival History");
//			node.setDynamicMetadataValueForName("conditionsOfReproductions", "Conditions of Reproductions");
//			node.setDynamicMetadataValueForName("reproductions", "Reproductions");
//			node.setDynamicMetadataValueForName("appraisalAndDestruction", "A and D");
//			node.setDynamicMetadataValueForName("arrangement", "Arrangement");
//			node.setDynamicMetadataValueForName("bibliography", "Bib");
//			node.setDynamicMetadataValueForName("descriptionLevelNotes", "description level notes");
//			node.setDynamicMetadataValueForName("descriptionRules", "description rules");
//			node.setDynamicMetadataValueForName("author", "Lem");
//			node.setDynamicMetadataValueForName("institution", "Docuteam, Dättwil");							//	Default value is specified
////			node.setDynamicMetadataValueForName("objectType", "Korrespondenzstück");							//	Default value is specified
//			node.setDynamicMetadataValueForName("revisions", "Revisionen");
////			node.setDynamicMetadataValueForName("retentionPolicy", "Retention policy");							//	Default value is specified
////			node.setDynamicMetadataValueForName("retentionPeriodBaseYear", "2000");								//	Default value is specified
//			node.setDynamicMetadataValueForName("responsible", "Responsible: ich");
//			node.setDynamicMetadataValueForName("involved", "Involved: Egon");
//			node.setDynamicMetadataValueForName("staff", "Staff: Luise");
//			node.setDynamicMetadataValueForName("projectName", "ProjectName");
//			node.setDynamicMetadataValueForName("projectTitle", "ProjectTitle");
//			node.setDynamicMetadataValueForName("project", "Project");
//			node.setDynamicMetadataValueForName("projectAbbreviation", "ProjectAbbr.");
//			node.setDynamicMetadataValueForName("location", "Location");
//			node.setDynamicMetadataValueForName("journal", "Journal");
//			node.setDynamicMetadataValueForName("university", "University");
//			node.setDynamicMetadataValueForName("institute", "Institute");
//			node.setDynamicMetadataValueForName("compartment", "Compartment");
//			node.setDynamicMetadataValueForName("method", "Method");
//			node.setDynamicMetadataValueForName("keyword", "Keyword");
//			node.setDynamicMetadataValueForName("fundingSource", "FundingSource");
//			node.setDynamicMetadataValueForName("status", "Status");
//			node.setDynamicMetadataValueForName("publisher", "Publisher");
//			node.setDynamicMetadataValueForName("doiJournal", "DOI Journal");
//			node.setDynamicMetadataValueForName("retentionPeriod", "Retention Period");
//			node.setDynamicMetadataValueForName("usage", "Use for this and that");
//			node.setDynamicMetadataValueForName("year", "Year 1999");
//			node.setDynamicMetadataValueForName("abstract", "Abstract");

//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstancesWhichAreMandatoryOrSet())		Tracer.trace(mde);
//			node.setLevel(LevelOfDescription.get("Archiv"));		//	Change the level -> some metadata disappear, some are re-initialized

//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstances())							Tracer.trace(mde);
//			Tracer.trace();
//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstancesToBeDisplayed())				Tracer.trace(mde);
//			Tracer.trace();
//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstancesWhichAreMandatoryButNotSet())	Tracer.trace(mde);

			//	Setter post-action (here: a simple trace):
//			doc = Document.openReadWriteFilesLocked(sips + "/MetsWithMetadata", "Document.main");
//			NodeAbstract node = doc.getStructureMap().getRoot();

//			Tracer.trace(node.getDynamicMetadataValueForName("fromYear"));
//			node.setDynamicMetadataValueForName("fromYear", "1234");
//			Tracer.trace(node.getDynamicMetadataValueForName("fromYear"));
//			node.setDynamicMetadataValueForName("fromYear", "2345");
//			Tracer.trace(node.getDynamicMetadataValueForName("fromYear"));
//			node.setDynamicMetadataValueForName("fromYear", null);
//			Tracer.trace(node.getDynamicMetadataValueForName("fromYear"));

			//	----------	Templates:
//			doc = Document.createNewFromTemplate("files/Templates/Template1", OperatingSystem.userHome() + "/Desktop/Created/SIP1", "Document.main");
//			doc = Document.createNewFromTemplate("files/Templates/Template1", OperatingSystem.userHome() + "/Desktop/Created/SIP2.zip", "Document.main");

//			doc = Document.openReadWrite("files/ch-001194-4_459 Nested", "Document.main");
//			doc = Document.openReadWriteFilesLocked("files/MetsWithMetadata", "Document.main");
//			doc = Document.openReadWriteFilesLocked(OperatingSystem.userHome() + "/Desktop/SIPs/MetsWithMetadata", "Document.main");
//			doc.saveAsTemplate(OperatingSystem.userHome() + "/Desktop/Templates/MetsWithMetadata");

			//	----------	Locking:
//			doc = Document.openReadWrite("files/Locked/mets.xml", "Document.main");
//			doc = Document.openReadWrite("files/ch-001194-4_459 Nested/mets.xml", "Document.main");
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/SIP/mets.xml", "Document.main");
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/SIP.zip", "Document.main");
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/SIP.zip", "Document.main");

			//	----------	Creating EAD File:
//			doc = Document.openReadWriteFilesLocked("files/NestedFolders/mets.xml", "Document.main");
//			doc.createEADFile(OperatingSystem.userHome() + "/Desktop/EAD/NestedFolders.xml");
//			doc = Document.openReadOnly("files/MetsWithMetadata/mets.xml", "Document.main");
//			doc.createEADFile(OperatingSystem.userHome() + "/Desktop/EADs/MetsWithMetadata.xml");

			//	----------	Non-standard levels:
//			Document doc1 = Document.openReadOnly("files/MetsWithMetadata", "Document.main");
//			Document doc2 = Document.openReadOnly("files/MetsWithMetadataNonstandardLevels", "Document.main");
//			Tracer.trace(doc1.getLevels());
//			Tracer.trace(doc2.getLevels());

			//	----------	Submitting:
//			doc = Document.openReadWrite(sips + "/SubmitTest_NOK", "Document.main");
//			doc = Document.openReadWrite(sips + "/SubmitTest_OK (submitted)", "Document.main");
//			doc = Document.openReadWrite(sips + "/SubmitTest_OK (submitted).zip", "Document.main");
//			doc = Document.openReadWrite(sips + "/X-ray Tomography all (submitted)", "Document.main");
//			doc = Document.openReadWrite(sips + "/X-ray Tomography all (submitted).zip", "Document.main");
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setUnitTitle("Test1");
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setSubmitStatus(SubmitStatus.SubmitRequested);
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setUnitTitle("Test2");
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setSubmitRequested_recursive();
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());
//			Tracer.trace(doc.submit(OperatingSystem.userHome() + "/Desktop/Workspace/IngestDropFolder"));
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());
//			doc.getStructureMap().getRoot().setSubmitStatusRecursivelyAllOrNone(SubmitStatus.SubmitUndefined);
//			doc.getStructureMap().getRoot().setSubmitStatusRecursivelyAllOrNone(SubmitStatus.SubmitRequestPending);

//			AIPCreatorProxy.initializeImpl(AIPCreatorEmpty.class);
//			Tracer.trace(doc.checkSubmission(workspace + "/_Feeder/dropbox"));
//			Tracer.trace(doc.submit(workspace + "/_Feeder/dropbox"));
//			Tracer.trace(Document.checkIngestFeedback(sips, workspace + "/_Feeder/dropbox", workspace + "/_Feeder/feedback"));

			//	----------	Saving:
//			setBackupFolder(OperatingSystem.userHome() + "/Desktop/Workspace/Backups");
//			setKeepBackupsCount(3);
//			doc.saveWithBackup();
//			doc.saveWithoutBackup();
//			doc.saveTo(desktop + "/Test");
//			doc = Document.openReadOnly(OperatingSystem.userHome() + "/Desktop/Created/Test", "Document.main");		//	Reread saved doc

			//	----------	Saving and reverting to last backup:
//			doc = Document.openReadOnly(sips + "/ch-001194-4_459", "Document.main");		//	Must throw a "DocumentIsReadOnlyException"!
//			doc = Document.openReadWrite(sips + "/ch-001194-4_459", "Document.main");
//			doc = Document.openReadWriteFilesLocked(sips + "/ch-001194-4_459", "Document.main");
//			doc = Document.openReadOnly(sips + "/ch-001194-4_459.zip", "Document.main");	//	Must throw a "DocumentIsReadOnlyException"!
//			doc = Document.openReadWrite(sips + "/ch-001194-4_459.zip", "Document.main");
//			doc = Document.openReadWriteFilesLocked(sips + "/ch-001194-4_459.zip", "Document.main");
//			doc = Document.openReadWrite(sips + "/SIP.zip", "Document.main");				//	Must throw a "DocumentIsReadOnlyException"!
//			Tracer.trace(doc);

//			doc.saveWithBackup();
//			doc.revertToLastBackup();

//			AIPCreatorProxy.initializeImpl(AIPCreatorEmpty.class);
//			Tracer.trace(doc.submit("/Volumes/DoesNotExist"));			//	THIS destination will throw an exception for sure.

//			FileUtil.delete(sips + "/ch-001194-4_459_COPY.zip");
//			FileUtil.delete(sips + "/ch-001194-4_459_COPY");
//			copy(sips + "/ch-001194-4_459", sips + "/ch-001194-4_459_COPY");
//			copy(sips + "/ch-001194-4_459.zip", sips + "/ch-001194-4_459_COPY.zip");
//			copy(sips + "/ch-001194-4_459", sips + "/ch-001194-4_459_COPY.zip");
//			copy(sips + "/ch-001194-4_459.zip", sips + "/ch-001194-4_459_COPY");
		}
		catch (Throwable x)
		{
			x.printStackTrace();
		}
		finally
		{
//			Tracer.trace(doc);				//	ToDo

			if (!ExceptionCollector.isEmpty())		Tracer.trace(ExceptionCollector.toStringAll());

			try
			{
				if (doc != null)
				{
					doc.unlockIfNecessary();
					doc.cleanupWorkingCopy();
				}
			}
			catch (java.lang.Exception e)
			{
				e.printStackTrace();
			}
		}
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	----------		Initializing			---------------------------------------------------

	static public void setBackupFolder(String backupFolder)
	{
		BackupFolder = backupFolder;
	}

	/**
	 * Set the keepBackupsCount value. If trying to set to null, ignore it and leave as is.
	 * @param keepBackupsCount Must not be negative.
	 */
	static public void setKeepBackupsCount(Integer keepBackupsCount)
	{
		if (keepBackupsCount == null)		return;
		if (keepBackupsCount < 0)			throw new IllegalArgumentException("KeepBackupsCount must not be negative: " + keepBackupsCount);

		KeepBackupsCount = keepBackupsCount;
	}

	//	--------		Reading				-------------------------------------------------------

	/**
	 * Create a new, empty SIP from a template.
	 * If the zipOrMETSFilePath ends with ".zip" or ".ZIP", create a ZIP file, otherwise a folder.
	 * @param templatePath The path to the template file.
	 * @param zipOrMETSFilePath The name of the destination SIP
	 * @param operatorName
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document createNewFromTemplate(String templatePath, String zipOrMETSFilePath, String operatorName) throws java.lang.Exception
	{
		return createNewFromTemplate(templatePath, zipOrMETSFilePath, operatorName, null);
	}


	/**
	 * Create a new, empty SIP from a template.
	 * If the zipOrMETSFilePath ends with ".zip" or ".ZIP", create a ZIP file, otherwise a folder.
	 * @param templatePath The path to the template file.
	 * @param zipOrMETSFilePath The name of the destination SIP
	 * @param operatorName
	 * @param observer
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document createNewFromTemplate(String templateFolderPath, String zipOrMETSFilePath, String operatorName, Observer observer) throws java.lang.Exception
	{
		Document document;

		if (zipOrMETSFilePath.toLowerCase().endsWith(".zip"))
		{
			//	It is a ZIP file:
			String sipFolderPath = FileUtil.asFilePathWithoutExtension(zipOrMETSFilePath);

			//	Copy the template to the destination path and zip it:
			FileUtil.copyToOverwriting(templateFolderPath, sipFolderPath);
			Zipper.zip(sipFolderPath, zipOrMETSFilePath, true);
			FileUtil.delete(sipFolderPath);

			document = openReadWrite(zipOrMETSFilePath, operatorName, observer);
		}
		else
		{
			//	Copy the template to the destination path:
			FileUtil.copyToOverwriting(templateFolderPath, zipOrMETSFilePath);

			//	Open the copy:
			document = openReadWrite(zipOrMETSFilePath + "/" + DefaultMETSFileName, operatorName, observer);
		}

		//	Initialize dynamic metadata elements in all nodes from the template:
		for (NodeAbstract node: document.getStructureMap().getRoot().getWithDescendants())	node.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();

		document.saveWithoutBackup();
		return document;
	}


	static public Document createNewWithRoot(String newZIPOrFolderFilePath, String sourceFilePath, String saId, String dssId, String operatorName) throws java.lang.Exception
	{
		return createNewWithRoot(newZIPOrFolderFilePath, sourceFilePath, saId, dssId, operatorName, null);
	}


	static public Document createNewWithRoot(String newZIPOrFolderFilePath, String sourceFilePath, String saId, String dssId, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (!new File(sourceFilePath).exists())		throw new FileNotFoundException(sourceFilePath);

		Document document = createEmpty(newZIPOrFolderFilePath, saId, dssId, operatorName, observer);
		try
		{
			//	Insert file or folder recursively as the new root in structMap:
			document.structureMap.createRootNode(sourceFilePath);
		}
		catch (FileOperationNotAllowedException ex)
		{
			//	this exception will not be thrown because the METS was not opened as 'read-only'.
		}

		//	Save SIP to original folder:
		document.saveWithoutBackup();

		return document;
	}


	static public Document createNewWithRootFolderName(String newZIPOrFolderFilePath, String rootFolderName, String saId, String dssId, String operatorName) throws java.lang.Exception
	{
		return createNewWithRootFolderName(newZIPOrFolderFilePath, rootFolderName, saId, dssId, operatorName, null);
	}


	static public Document createNewWithRootFolderName(String newZIPOrFolderFilePath, String rootFolderName, String saId, String dssId, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (rootFolderName == null || rootFolderName.isEmpty())	throw new FolderNameIsEmptyException();

		Document document = createEmpty(newZIPOrFolderFilePath, saId, dssId, operatorName, observer);
		try
		{
			//	Insert new empty folder as the new root in structMap:
			document.structureMap.createNewRootFolder(rootFolderName);
		}
		catch (FileOperationNotAllowedException ex)
		{
			//	Ignore the FileOperationNotAllowedException because this newly created ZIP will be saved anyway.
			//	Apart from that - this exception will not be thrown anyway because the METS was not opened as 'read-only'.
		}

		//	Save SIP to original folder:
		document.saveWithoutBackup();

		return document;
	}


	/**
	 * Create a document on the SIP denoted by sipPath. Open this SIP in the "readWriteFilesLocked" mode and set a lock on this SIP.
	 * If this SIP is already locked by somebody else, open it in "readOnly" mode.
	 * The sipPath can be the path to a ZIP file, a folder, or a mets.xml file.
	 * @param sipPath
	 * @param operatorName
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document openReadWriteFilesLocked(String sipPath, String operatorName) throws java.lang.Exception
	{
		return openReadWriteFilesLocked(sipPath, operatorName, null);
	}


	/**
	 * Create a document on the SIP denoted by sipPath. Open this SIP in the "readWriteFilesLocked" mode and set a lock on this SIP.
	 * If this SIP is already locked by somebody else, open it in "readOnly" mode.
	 * The sipPath can be the path to a ZIP file, a folder, or a mets.xml file.
	 * @param sipPath
	 * @param operatorName
	 * @param observer
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document openReadWriteFilesLocked(String sipPath, String operatorName, Observer observer) throws java.lang.Exception
	{
		return open(sipPath, Mode.ReadWriteNoFileOps, operatorName, observer);
	}


	/**
	 * Create a document on the SIP denoted by sipPath. Open this SIP in the "readOnly" mode.
	 * The sipPath can be the path to a ZIP file, a folder, or a mets.xml file.
	 * @param sipPath
	 * @param operatorName
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document openReadOnly(String sipPath, String operatorName) throws java.lang.Exception
	{
		return openReadOnly(sipPath, operatorName, null);
	}


	/**
	 * Create a document on the SIP denoted by sipPath. Open this SIP in the "readOnly" mode.
	 * The sipPath can be the path to a ZIP file, a folder, or a mets.xml file.
	 * @param sipPath
	 * @param operatorName
	 * @param observer
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document openReadOnly(String sipPath, String operatorName, Observer observer) throws java.lang.Exception
	{
		return open(sipPath, Mode.ReadOnly, operatorName, observer);
	}


	/**
	 * Create a document on the SIP denoted by sipPath. Open this SIP in the "readWrite" mode and set a lock on this SIP.
	 * If this SIP is already locked by somebody else, open it in "readOnly" mode.
	 * The sipPath can be the path to a ZIP file, a folder, or a mets.xml file.
	 * @param sipPath
	 * @param operatorName
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document openReadWrite(String sipPath, String operatorName) throws java.lang.Exception
	{
		return openReadWrite(sipPath, operatorName, null);
	}


	/**
	 * Create a document on the SIP denoted by sipPath. Open this SIP in the "readWrite" mode and set a lock on this SIP.
	 * If this SIP is already locked by somebody else, open it in "readOnly" mode.
	 * The sipPath can be the path to a ZIP file, a folder, or a mets.xml file.
	 * @param sipPath
	 * @param operatorName
	 * @param observer
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document openReadWrite(String sipPath, String operatorName, Observer observer) throws java.lang.Exception
	{
		return open(sipPath, Mode.ReadWrite, operatorName, observer);
	}

	//	--------		Accessing			-------------------------------------------------------

	static public void setMETS_SIPTemplateFile(String newMETS_SIPTemplateFile)
	{
		METS_SIPTemplateFile = newMETS_SIPTemplateFile;
	}

	static public String getBackupFolder()
	{
		return BackupFolder;
	}

	static public Integer getKeepBackupsCount()
	{
		return KeepBackupsCount;
	}

	//	--------		Inquiring			-------------------------------------------------------

	static public boolean isZIPFile(File sip)
	{
		return isZIPFile(sip.getName());
	}

	static public boolean isZIPFile(String sipName)
	{
		return sipName.toLowerCase().endsWith(".zip");
	}


	static public boolean isValidSIPFolder(String folderPath)
	{
		return isValidSIPFolder(new File(folderPath));
	}

	static public boolean isValidSIPFolder(File folder)
	{
		//	Must be an existing and readable folder...
		if (folder == null || !folder.exists() || !folder.canRead())			return false;
		if (folder.isFile())													return false;

		//	... and must contain a mets.xml file:
		File[] metsFilesWithin = folder.listFiles();
		if (metsFilesWithin == null)											return false;
		for (File file: metsFilesWithin)
			if (file.isFile()
				&& file.getName().toLowerCase().equals(DefaultMETSFileName))	return true;

		return false;
	}


	/**
	 * Return true if the METS-file and the corresponding SIP appears to be OK; false otherwise.
	 * As a default, regard missing files als errors.
	 *
	 * @param metsFileName contains the file name of the METS file to parse
	 */
	static public boolean check(String metsFileName)
	{
		return check(metsFileName, true);
	}


	/**
	 * Return true if the METS-file and the corresponding SIP appears to be OK; false otherwise.
	 *
	 * @param metsFileName
	 * @param regardMissingFilesAsErrors If true, regard missing files as errors and fill the ExceptionCollector with corresponding exceptions; otherwise don't.
	 * @return
	 */
	static public boolean check(String metsFileName, boolean regardMissingFilesAsErrors)
	{
		Document document = null;

		ExceptionCollector.clear();
		try
		{
			document = openReadOnly(metsFileName, null);
		}
		catch (java.lang.Exception e)
		{
			e.printStackTrace();
			return false;
		}

		if (regardMissingFilesAsErrors && document.isAtLeastOneFileNotReadable)
			for (NodeAbstract n: document.getStructureMap().getRoot().getWithDescendants())
			{
				File f = n.getFile();
				if (!f.exists())		Exception.remember(new FileNotFoundException(f.getPath()));
			}

		return ExceptionCollector.isEmpty();
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Locking				-------------------------------------------------------

	static public boolean isLocked(File sip)
	{
		return isLocked(sip.getPath());
	}

	/**
	 * Return true if this SIP is locked by somebody else, false otherwise.
	 */
	static public boolean isLocked(String sipName)
	{
		return lockedByWhom(sipName) != null;
	}


	/**
	 * Return true if this SIP is locked by somebody else, false otherwise.
	 */
	static public boolean isLockedBySomebodyElse(File sip)
	{
		return isLockedBySomebodyElse(sip.getPath());
	}

	/**
	 * Return true if this SIP is locked by somebody else, false otherwise.
	 */
	static public boolean isLockedBySomebodyElse(String sipName)
	{
		String lockedBy = lockedByWhom(sipName);

		if (lockedBy == null)		return false;
		return !OperatingSystem.userName().equals(lockedBy);
	}


	/**
	 * Return the login name of the user who locked this SIP, or null if no lock exists.
	 *
	 * @param sipName
	 * @return
	 */
	static public String lockedByWhom(File sip)
	{
		return lockedByWhom(sip.getPath());
	}

	/**
	 * Return the login name of the user who locked this SIP, or null if no lock exists.
	 *
	 * @param sipName
	 * @return
	 */
	static public String lockedByWhom(String sipName)
	{
		File lockFile = new File(sipName + LockFileSuffix);

		//	If no lockfile exists, the SIP is not locked for sure:
		if (!lockFile.exists())		return null;

		//	The lockfile exists. Who locked it?
		return FileUtil.getFileContentAsString(lockFile).trim();		//	trim() is here to cut off any possible newlines.
	}


	/**
	 * Overwrite any existing lock for the given SIP with a new "Submitted-lock".
	 */
	static protected void lockAsSubmitted(String sipFolder)
	{
		try
		{
			FileUtil.createFileWithContent(sipFolder + LockFileSuffix, SubmitStatus.getSubmittedLockId());
		}
		catch (IOException ex)
		{
			ex.printStackTrace();
		}
	}

	//	--------		Submitting			-------------------------------------------------------

	/**
	 * Test if the folder ingestFeedbackFolderPath contains a folder that exists in the workspaceFolderPath,
	 * and that this folder contains only and exactly the file "mets.xml".
	 * If yes, copy the file "mets.xml" and overwrite it in the SIP in the workspace, then delete it from the ingestFeedackFolder
	 * and the zipped SIP from the ingestSubmitFolder.
	 * This works for zipped SIPs as well.
	 * @param workspaceFolderPath
	 * @param ingestFeedbackFolderPath
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 */
	static public List<String> checkIngestFeedback(String workspaceFolderPath, String ingestSubmitFolderPath, String ingestFeedbackFolderPath) throws IOException, FileUtilExceptionListException
	{
		List<String> ingestFeedbackFoundForSIPs = new Vector<String>();

		Logger.getLogger().debug("Checking ingest feedback in folder: '" + ingestFeedbackFolderPath + "' for workspace folder: '" + workspaceFolderPath + "'");

		if (ingestFeedbackFolderPath == null)					return ingestFeedbackFoundForSIPs;
		File ingestFeedbackFolder = new File(ingestFeedbackFolderPath);
		if (!ingestFeedbackFolder.exists())						return ingestFeedbackFoundForSIPs;
		if (!ingestFeedbackFolder.isDirectory())				return ingestFeedbackFoundForSIPs;
		if (ingestFeedbackFolder.list().length == 0)			return ingestFeedbackFoundForSIPs;

		if (workspaceFolderPath == null)						return ingestFeedbackFoundForSIPs;
		File workspaceFolder = new File(workspaceFolderPath);
		if (!workspaceFolder.exists())							return ingestFeedbackFoundForSIPs;
		if (!workspaceFolder.isDirectory())						return ingestFeedbackFoundForSIPs;
		if (workspaceFolder.list().length == 0)					return ingestFeedbackFoundForSIPs;

		for (File f: ingestFeedbackFolder.listFiles((FilenameFilter)FileFilter.SIPs))
		{
			String feedbackSIPName = f.getName();

			Logger.getLogger().debug("Found ingest feedback for SIP: '" + feedbackSIPName + "'");

			//	This feedback folder must contain only the mets.xml file, nothing else:
			String[] fileList = f.list(FileFilter.VisibleFiles);
			if (fileList.length != 1)							continue;
			if (!fileList[0].equalsIgnoreCase("mets.xml"))		continue;

			if (feedbackSIPName.endsWith("_zip"))
			{
				//	SIP is zipped:

				//	Replace the trailing "_zip" by ".zip":
				String sipName = feedbackSIPName.substring(0, feedbackSIPName.length() - 4) + ".zip";
				String sipPath = workspaceFolder + "/" + sipName;

				if (!new File(sipPath).exists())
				{
					Logger.getLogger().debug("No SIP in Workspace: '" + sipName + "'");
					continue;
				}
				if (!new File(sipPath).isFile())				continue;

				//	If this document is not locked as SubmitRequestPending, ignore it:
				String lockedBy = Document.lockedByWhom(sipPath);
				if (!	(Document.isLocked(sipPath)
						&& Document.lockedByWhom(sipPath).contains(SubmitStatus.SubmitRequestPending.toString())))
				{
					Logger.getLogger().debug("No SubmitRequestPending lock: '" + lockedBy + "' for SIP: '" + feedbackSIPName + "'");
																continue;
				}

				Logger.getLogger().debug("Replacing METS-file from ingest feedback for SIP: '" + feedbackSIPName + ".zip'");

				String feedbackSIPPath = ingestFeedbackFolderPath + "/" + feedbackSIPName;

				//	Unzip:
				String workingPath = FileUtil.getTempFolder() + "/" + WorkingFolderPrefixReadZip + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "/" + feedbackSIPName;
				Zipper.unzip(sipPath, workingPath);
				//	Replace mets-file:
				FileUtil.copyToFolderOverwriting(feedbackSIPPath + "/mets.xml", workingPath);

				//	Create security copy of the original:
				String securitySuffix = "_COPY_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + ".zip";
				FileUtil.renameTo(sipPath, sipPath + securitySuffix);

				//	Zip:
				Zipper.zip(workingPath, sipPath, true);

				//	Lock:
				lockAsSubmitted(sipPath);

				//	Cleanup feedback folder:
				FileUtil.delete(sipPath + securitySuffix);
				FileUtil.delete(FileUtil.asParentPath(workingPath));
				FileUtil.delete(feedbackSIPPath);

				//	Cleanup ingest folder:
				String ingestSIPPath = ingestSubmitFolderPath + "/" + feedbackSIPName + ".zip";
				FileUtil.delete(ingestSIPPath);

				ingestFeedbackFoundForSIPs.add(feedbackSIPName);
			}
			else
			{
				//	SIP is folder:

				String sipPath = workspaceFolder + "/" + feedbackSIPName;

				if (!new File(sipPath).exists())
				{
					Logger.getLogger().debug("No SIP in Workspace: '" + feedbackSIPName + "'");
					continue;
				}
				if (!new File(sipPath).isDirectory())			continue;

				//	If this document is not locked as SubmitRequestPending, ignore it:
				String lockedBy = Document.lockedByWhom(sipPath);
				if (!	(Document.isLocked(sipPath)
						&& lockedBy.contains(SubmitStatus.SubmitRequestPending.toString())))
				{
					Logger.getLogger().debug("No SubmitRequestPending lock: '" + lockedBy + "' for SIP: '" + feedbackSIPName + "'");
																continue;
				}

				Logger.getLogger().debug("Replacing METS-file from ingest feedback for SIP: '" + feedbackSIPName + "'");

				String feedbackSIPPath = ingestFeedbackFolderPath + "/" + feedbackSIPName;

				//	Replace mets-file:
				FileUtil.copyToFolderOverwriting(feedbackSIPPath + "/mets.xml", sipPath);
				//	Lock:
				lockAsSubmitted(sipPath);

				//	Cleanup feedback folder:
				FileUtil.delete(feedbackSIPPath);

				//	Cleanup ingest folder:
				String ingestSIPPath = ingestSubmitFolderPath + "/" + feedbackSIPName + ".zip";
				FileUtil.delete(ingestSIPPath);

				ingestFeedbackFoundForSIPs.add(feedbackSIPName);
			}
		}

		return ingestFeedbackFoundForSIPs;
	}


	//	--------		Copying				-------------------------------------------------------

	/**
	 * Copy a SIP. Distinguish between zipped and non-zipped SIPs in the source as well as in the target SIP by means of their file name extension.
	 * @param sourceSIPName
	 * @param destinationSIPName
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	static public void copy(String sourceSIPPath, String targetSIPPath) throws IOException, FileUtilExceptionListException
	{
		if (isZIPFile(sourceSIPPath))
		{
			if (isZIPFile(targetSIPPath))
			{
				//	zip to zip:
				FileUtil.copyToOverwriting(sourceSIPPath, targetSIPPath);
			}
			else
			{
				//	zip to folder:
				Zipper.unzip(sourceSIPPath, targetSIPPath);
			}
		}
		else
		{
			if (isZIPFile(targetSIPPath))
			{
				//	folder to zip:
				Zipper.zip(sourceSIPPath, targetSIPPath, true);
			}
			else
			{
				//	folder to folder:
				FileUtil.copyToOverwriting(sourceSIPPath, targetSIPPath);
			}
		}
	}

	//	========	Static Private			=======================================================

	//	--------		Initialization		-------------------------------------------------------

	static private Document createEmpty(String newZIPOrFolderFilePath, String saId, String dssId, String operatorName, Observer observer) throws java.lang.Exception
	{
		File newZIPOrFolderFile = new File(newZIPOrFolderFilePath);
		if (newZIPOrFolderFile.exists())
		{
			if (newZIPOrFolderFile.isFile())					FileUtil.delete(newZIPOrFolderFile);
			else if (newZIPOrFolderFile.isDirectory()
				&& newZIPOrFolderFile.list().length != 0)		throw new FolderIsNotEmptyException(newZIPOrFolderFilePath);
		}

			if (FileUtil.asFileNameExtension(newZIPOrFolderFilePath).toLowerCase().equals("zip"))
			{
				//	Create a ZIP file:
	
				String destinationFolderPath = newZIPOrFolderFile.getParent();
				String metsFilePath = destinationFolderPath + File.separator + DefaultMETSFileName;
	
				initializeEmptyMetsFile(metsFilePath, saId, dssId);
				Zipper.zip(metsFilePath, newZIPOrFolderFilePath);
				FileUtil.delete(metsFilePath);
			}
			else
			{
				//	Create a folder:
				String metsFilePath = newZIPOrFolderFilePath + File.separator + DefaultMETSFileName;
				newZIPOrFolderFile.mkdirs();
				initializeEmptyMetsFile(metsFilePath, saId, dssId);
			}

		//	Read the newly created empty SIP:
		Document document = openReadWrite(newZIPOrFolderFilePath, operatorName, observer);

		return document;
	}
	
	static private void initializeEmptyMetsFile(String metsFilePath, String saId, String dssId) throws DocumentException, IOException, FileUtilExceptionListException
	{
		FileUtil.copyToOverwriting(METS_SIPTemplateFile, metsFilePath);

		SAXReader reader = new SAXReader();
		org.dom4j.Document document = reader.read(metsFilePath);
		document.getRootElement().addAttribute("TYPE", saId + "_" + dssId);
		
		writeMETSFile(document, metsFilePath);
	}

	//	--------		Reading				-------------------------------------------------------

	/**
	 * Create a document on the SIP denoted by sipPath in the mode "mode".
	 * The sipPath can be the path to a ZIP file, a folder, or a mets.xml file.
	 * @param sipPath
	 * @param operatorName
	 * @return
	 * @throws java.lang.Exception
	 */
	static private Document open(String sipPath, Mode mode, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (!new File(sipPath).exists())		throw new FileNotFoundException(sipPath);

		String		originalSIPFolder = null;	//	If originalSIPFolder is null, we are working on the original SIP without a working copy
		String		metsFilePath = null;

		if (sipPath.endsWith(".zip") || sipPath.endsWith(".ZIP"))
		{
			//	It's a zip file - unzip into working folder:

			Logger.getLogger().info("Opening zipped SIP in mode: '" + mode + "': '" + sipPath + "'");

			String sipName = (new File(sipPath).getName());
			sipName = sipName.substring(0, sipName.length() - 4);
			String workingFolder = FileUtil.getTempFolder() + File.separator + WorkingFolderPrefixReadZip + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + File.separator + sipName;

			Zipper.unzip(sipPath, workingFolder);

			//	Break if the ZIP doesn't contain the mets file in the top level:
			File metsFile = new File(workingFolder + File.separator + DefaultMETSFileName);
			if (!metsFile.exists())
			{
				try
				{
					FileUtil.delete(new File(workingFolder).getParent());
				}
				catch (FileUtilExceptionListException ex)
				{
					ex.printStackTrace();
				}

				throw new ZIPDoesNotContainMETSFileException(sipPath);
			}

			originalSIPFolder = sipPath;
			metsFilePath = FileUtil.asCanonicalFileName(metsFile);
		}
		else
		{
			//	It's a file or a folder:

			if (new File(sipPath).isFile())
			{
				originalSIPFolder = new File(sipPath).getParent();
			}
			else
			{
				originalSIPFolder = sipPath;
				sipPath = sipPath + File.separator + DefaultMETSFileName;
			}

			if (   isLockedBySomebodyElse(originalSIPFolder)
				|| mode == Mode.ReadOnly
				|| mode == Mode.ReadWriteNoFileOps)
			{
				//	Don't create working copy:
				//	In case of Mode.ReadWriteNoFileOps: The lock will be set later, in document.initialize(...).

				originalSIPFolder = null;
				metsFilePath = FileUtil.asCanonicalFileName(sipPath);

				Logger.getLogger().info("Opening SIP directly in mode: '" + mode + "': '" + sipPath + "'");
			}
			else
			{
				//	Create working copy:
				//	The lock will be set later, in document.initialize(...).

				String workingFolder = FileUtil.getTempFolder() + File.separator + WorkingFolderPrefix + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);

				try
				{
					FileUtil.copyToFolderOverwriting(originalSIPFolder, workingFolder, true);	//	true = Preserve all access rights
				}
				catch (FileUtilExceptionListException ex)
				{
					//	If Exceptions occurred during creating the copy, ignore them and continue!
				}

				metsFilePath = FileUtil.asCanonicalFileName(workingFolder + File.separator + FileUtil.asFileName(originalSIPFolder) + File.separator + FileUtil.asFileName(sipPath));

				Logger.getLogger().info("Opening copy of SIP in mode: '" + mode + "': '" + originalSIPFolder + "'");
			}
		}

		Logger.getLogger().debug("METS-file: '" + metsFilePath + "'");

		File metsFile = new File(metsFilePath);
		SAXReader reader = new SAXReader();
		reader.setDocumentFactory(new DocumentFactory());
		Document document = (Document)reader.read(metsFile);

		try
		{
			document.initialize(metsFilePath, originalSIPFolder, mode, operatorName, observer);
			return document;
		}
		catch (java.lang.Exception ex)
		{
			//	Cleanup if parsing was not successful:
			document.unlockIfNecessary();
			document.cleanupWorkingCopy();

			//	Re-throw the exception so it doesn't get lost:
			throw ex;
		}
	}

	//	========	Instance Public			=======================================================

	//	----------		Initializing			---------------------------------------------------
	//	----------		Accessing				---------------------------------------------------

	public String getNamespaceMETS()
	{
		return this.namespaceMETS;
	}

	public String getNamespacePREMIS()
	{
		return this.namespacePREMIS;
	}

	public Mode getMode()
	{
		return this.mode;
	}

	public String getProfile()
	{
		return this.profile;
	}

	public String getLabel()
	{
		return this.label;
	}

	public String getType()
	{
		return this.type;
	}

	public void setType(String type)
	{
		this.type = type;
		this.getRootElement().addAttribute("TYPE", type);
		this.setIsModified();
	}

	public String getSAId()
	{
		return this.saId;
	}

	public String getDSSId()
	{
		return this.dssId;
	}

	public String getObjectId()
	{
		return this.objId;
	}

	public void setObjectId(String objId)
	{
		this.objId = objId;
		this.getRootElement().addAttribute("OBJID", objId);
		this.setIsModified();
	}

	@Deprecated
	/**
	 * @deprecated Use getSIPFolder() instead
	 */
	public String getSipFolder()
	{
		return this.getSIPFolder();
	}

	public String getSIPFolder()
	{
		return new File(this.filePath).getParent();
	}

	@Deprecated
	/**
	 * @deprecated  Use getOriginalSIPFolder() instead
	 */
	public String getOriginalSipFolder()
	{
		return this.getOriginalSIPFolder();
	}

	public String getOriginalSIPFolder()
	{
		return this.originalSIPFolder;
	}

	@Deprecated
	/**
	 * @deprecated  Use getSIPName() instead
	 */
	public String getOriginalSipName()
	{
		return this.getSIPName();
	}

	public String getSIPName()
	{
		return FileUtil.asFileName(this.originalSIPFolder);
	}


	public Header getHeader()
	{
		return this.header;
	}

	public AMDSection getAMDSection()
	{
		return this.amdSection;
	}

	public Collection<DMDSectionAbstract> getDMDSections()
	{
		return this.dmdSections.values();
	}

	public DMDSectionAbstract getDMDSection(String id)
	{
		return this.dmdSections.get(id);
	}

	public void addDMDSection(DMDSectionAbstract dmdSection)
	{
		this.dmdSections.put(dmdSection.getId(), dmdSection);
		this.setIsModified();
	}

	public FileSection getFileSection()
	{
		return this.fileSection;
	}

	public StructureMap getStructureMap()
	{
		return this.structureMap;
	}


	public String getCurrentOperatorName()
	{
		return this.currentOperatorName;
	}

	public void setCurrentOperatorName(String opName)
	{
		this.currentOperatorName = opName;
		this.setIsModified();
	}


	public List<NodeFile> setSubmissionAgreement( String saId, String dssId) throws FileNotFoundException, DocumentException, ExceptionCollectorException
	{
		SubmissionAgreement submissionAgreement = SubmissionAgreement.read(saId + ".xml");
		if (submissionAgreement == null)		return null;

		this.saId = submissionAgreement.getSaId();
		this.dssId = dssId;
		this.setType(this.saId + "_" + dssId);
		this.submissionAgreement = submissionAgreement;

		return this.filesNotAllowedBySubmissionAgreement();
	}


	public SubmissionAgreement getSubmissionAgreement()
	{
		if (this.saId == null || this.saId.isEmpty())		return null;

		try
		{
			//	Cache the submissionAgreement once it is requested:
			if (this.submissionAgreement == null)		this.submissionAgreement = SubmissionAgreement.read(this.saId + ".xml");
			return this.submissionAgreement;
		}
		catch (java.lang.Exception e)
		{
			e.printStackTrace();
			return null;
		}
	}


	public String getLockedBy()
	{
		return this.lockedBy;
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean isZIPFile()
	{
		return isZIPFile(this.originalSIPFolder);
	}

	@Override
	public boolean isReadOnly()
	{
		return this.mode == Mode.ReadOnly;
	}

	public boolean isLocked()
	{
		return this.mode == Mode.Locked;
	}

	public boolean isReadWrite()
	{
		return this.mode == Mode.ReadWrite;
	}

	public boolean isReadWriteNoFileOps()
	{
		return this.mode == Mode.ReadWriteNoFileOps;
	}

	public boolean canWrite()
	{
		return this.mode == Mode.ReadWrite || this.mode == Mode.ReadWriteNoFileOps;
	}

	public boolean areFileOperationsAllowed()
	{
		return this.mode == Mode.ReadWrite;
	}

	public boolean hasNodesWithDynamicMetadataElementInstancesWhichAreMandatoryButNotSet()
	{
		return this.structureMap.hasNodesWithDynamicMetadataElementInstancesWhichAreMandatoryButNotSet();
	}


	public boolean checkFixity()
	{
		return this.structureMap.checkFixity();
	}


	public boolean checkAgainstSubmissionAgreement()
	{
		return this.structureMap.checkAgainstSubmissionAgreement();
	}

	public List<NodeFile> filesNotAllowedBySubmissionAgreement()
	{
		return this.structureMap.filesNotAllowedBySubmissionAgreement();
	}


	public boolean checkAgainstSubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		return this.structureMap.checkAgainstSubmissionAgreement(sa, dssId);
	}

	public List<NodeFile> filesNotAllowedBySubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		return this.structureMap.filesNotAllowedBySubmissionAgreement(sa, dssId);
	}

	//	--------		Interface			-------------------------------------------------------

	/**
	 * Return the type of this document: METS, EAD, ...
	 */
	@Override
	public Type getDocumentType()
	{
		return Type.METS;
	}

	//	--------		Business Ops		-------------------------------------------------------

	public void createEADFile(String eadFilePath) throws DocumentException, ExceptionCollectorException, IOException
	{
		ch.docuteam.darc.ead.Document eadDocument = ch.docuteam.darc.ead.Document.createEmpty();
		eadDocument.setId(this.getType());

		//	Since I know that the archDesc already exists (initialized with default values), I have only to initialize it explicitly:
		eadDocument.getArchDesc().initialize(this.structureMap.getRoot().getUnitTitle());
		eadDocument.getArchDesc().fillEADElementsFromNodeRecursively(this.structureMap.getRoot());

		eadDocument.saveAs(eadFilePath);
	}

	//	--------		Submitting			-------------------------------------------------------

	/**
	 * Return the SIP fileName to be used when submitting this SIP. Since the submitted SIP will be zipped, it must have the ".zip" suffix.
	 * In addition, if the *original* SIP is zipped, the file name contains "_zip" for distinction; this is needed for the ingest feedback.
	 * @return
	 */
	public String getSIPNameForSubmit()
	{
		return FileUtil.asFileNameWithoutExtension(this.getSIPName()) + (this.isZIPFile()? "_zip" : "") + ".zip";
	}


	/**
	 * Check if a submit can be issued.
	 *
	 * @return a list of error messages. If this list is empty, a submit can be issued.
	 */
	public List<String> checkSubmission(String submitFolderPath)
	{
		String submitSIPName = this.getSIPNameForSubmit();

		//	Check if this document already exists in the submit folder. If yes, reject:
		if (new File(submitFolderPath + File.separator + submitSIPName).exists())
		{
			List<String> rejectMessages = new Vector<String>();
			rejectMessages.add("MessageSubmitSIPExistsAlreadyInSubmitFolder '" + submitSIPName + "'");
			return rejectMessages;
		}

		try
		{
			return AIPCreatorProxy.checkSubmission(this);
		}
		catch (java.lang.Exception ex)
		{
			return Arrays.asList(new String[]{ "MessageSubmitExceptionOccurred " + ex.getMessage() });
		}
	}


	/**
	 * First check if submit can be issued.
	 * If yes, set all submit status entries that are "SubmitRequested" to "SubmitRequestPending", save and lock this document,
	 * then trigger the feeder-workflow "Ingest" (Here: by zipping the SIP into the submit folder).
	 * @return a list of error messages. If this list is empty, the submit was successful.
	 * @throws java.lang.Exception
	 */
	public List<String> submit(String submitFolderPath) throws java.lang.Exception
	{
		//	Check document before submitting. If there are messages, don't submit but return the messages:
		List<String> rejectMessages = this.checkSubmission(submitFolderPath);
		if (!rejectMessages.isEmpty())		return rejectMessages;

		String sipName = this.getSIPName();

		//	Set all nodes to "SubmitRequestPending":
		Logger.getLogger().debug("Setting all submit status entries to 'SubmitRequestPending' in SIP: '" + sipName + "'");
		this.structureMap.getRoot().setSubmitStatusRecursivelyAllOrNone(SubmitStatus.SubmitRequestPending);

		//	Create a backup:
		this.saveWithBackup();

		//	Remember the lock mode and file paths - in case I have to revert to the just created backup:
		Mode rememberedMode = this.mode;
		String rememberedFilePath = this.filePath;
		String rememberedOriginalSIPFolder = this.originalSIPFolder;
		boolean rememberedIsWorkingCopy = this.isWorkingCopy;

		//	Lock me as "SubmitRequestPending":
		Logger.getLogger().debug("Creating 'SubmitRequestPending' lock for SIP: '" + sipName + "'");
		this.lockAsSubmitRequestPending();

		try
		{
			//	Trigger feeder-workflow "Ingest" by placing a ZIP of this SIP into the ingest folder:
			Logger.getLogger().debug("Zipping to ingest-folder: '" + submitFolderPath + "' SIP: '" + sipName + "'");

			if (this.isZIPFile())
				FileUtil.copyToOverwriting(this.originalSIPFolder, submitFolderPath + File.separator + this.getSIPNameForSubmit());
			else
				Zipper.zip(this.originalSIPFolder, submitFolderPath + File.separator + this.getSIPNameForSubmit(), true);
		}
		catch(java.lang.Exception ex)
		{
			Logger.getLogger().error("Copying to submit folder failed, reverting to original SIP: '" + sipName + "'");

			//	Prepare for reverting to last backup: remove the SubmitRequestPending lock, reset the original lock and the file paths:
			this.unlock_dontCheck();
			this.mode = rememberedMode;
			this.filePath = rememberedFilePath;
			this.originalSIPFolder = rememberedOriginalSIPFolder;
			this.isWorkingCopy = rememberedIsWorkingCopy;
			this.revertToLastBackup();

			throw new java.lang.Exception("Copying to submit folder failed, reverting to original SIP: '" + sipName + "'", ex);
		}

		//	At this point the list "rejectMessages" is empty for sure:
		return rejectMessages;
	}

	//	--------		Persistence			-------------------------------------------------------

	/**
	 * Save the METS-file and create a backup of the original.
	 * If this document has at least one file which is not readable, save without backup!
	 * If this document has at least one file which is not writable, save with backup but ignore FileUtilExceptionListException!
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 * @throws OriginalSIPIsMissingException The original SIP could not be found so no backup could be made. Nevertheless the saving was successful.
	 */
	public void saveWithBackup() throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException, OriginalSIPIsMissingException
	{
		if (this.isAtLeastOneFileNotReadable)
			//	If some files or folders are not readable, I can't make a backup!
			try
			{
				this.save(false);
			}
			catch (FileUtilExceptionListException ex)
			{
				//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
			}
		else if (this.isAtLeastOneFileNotWritable)
			try
			{
				this.save(true);
			}
			catch (FileUtilExceptionListException ex)
			{
				//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
			}
		else
			this.save(true);
	}


	/**
	 * Save the METS-file overwriting the original.
	 * If this document has at least one file which is not readable, ignore FileUtilExceptionListException!
	 * If this document has at least one file which is not writable, ignore FileUtilExceptionListException!
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 */
	public void saveWithoutBackup() throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException
	{
		try
		{
			if (this.isAtLeastOneFileNotReadable)
				//	If some files or folders are not readable, I can't make a backup!
				try
				{
					this.save(false);
				}
				catch (FileUtilExceptionListException ex)
				{
					//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
				}
			else if (this.isAtLeastOneFileNotWritable)
				try
				{
					this.save(false);
				}
				catch (FileUtilExceptionListException ex)
				{
					//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
				}
			else
				this.save(false);
		}
		catch (OriginalSIPIsMissingException ex)
		{
			//	This exception gets only thrown when saving WITH backup, so it will never happen here
		}
	}


	/**
	 * Save the SIP and rename the METS-file.
	 * This operation is allowed even when this document was opened in read-only mode.
	 * @param newMETSFilePath
	 * @throws DocumentIsReadOnlyException
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 */
	public void saveAs(String newMETSFilePath) throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException
	{
		if (newMETSFilePath.toLowerCase().endsWith(".xml"))
		{
			//	The mets.xml file gets a new name:

			String oldMETSFileName = FileUtil.asFileName(this.filePath);
			String newSIPFolderPath = FileUtil.asParentPath(newMETSFilePath);
			String newMETSFileName = FileUtil.asFileName(newMETSFilePath);

			//	Save to folder:
			this.saveTo(newSIPFolderPath);

			//	Rename the mets.xml file (if isWorkingCopy, then this is the working copy, otherwise this is the original):
			File newFilePath = FileUtil.renameTo(this.filePath, this.getSIPFolder() + "/" + newMETSFileName);
			this.filePath = newFilePath.getPath();

			if (this.isWorkingCopy)
			{
				//	If working on a copy, rename the ORIGINAL copy of the mets.xml file too:
				FileUtil.renameTo(newSIPFolderPath + "/" + oldMETSFileName, newMETSFilePath);
			}
		}
		else
		{
			this.saveTo(newMETSFilePath);
		}
	}


	/**
	 * Save the SIP into the given directory.
	 * This operation is allowed even when this document was opened in read-only mode.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 */
	public void saveTo(String newSipFolder) throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException
	{
		boolean wasWorkingCopy = this.isWorkingCopy;

		try
		{
			//	Set isWorkingCopy to false, so nothing gets written to the original SIP:
			this.isWorkingCopy = false;

			//	Save METS-File before copying to new destination (working copy only), but only if it was modified:
			if (this.isModified)	this.saveWithoutBackup();

			//	Copy to new destination:
			if (newSipFolder.toLowerCase().endsWith(".zip"))
			{
				//	Create a zip file:
				List<String> files = new ArrayList<String>();
				for (File file: new File(this.getSIPFolder()).listFiles())		files.add(file.getPath());

				Zipper.zip(files.toArray(new String[]{}), newSipFolder);
			}
			else
			{
				//	Copy the SIP, don't preserve the access rights but set all to read/write:
				FileUtil.copyToOverwriting(this.getSIPFolder(), newSipFolder, false);
			}
		}
		finally
		{
			this.unlockIfNecessary();

			if (wasWorkingCopy)
			{
				//	This was a working copy, so set the original folder to the one just created;
				//	the working folder (= file path) remains as it was:
				this.originalSIPFolder = newSipFolder;
				this.isWorkingCopy = true;
			}
			else
			{
				//	This was NOT a working copy, so set my working folder (= file path) to the path just created
				//	and the original folder to its parent folder.
				File newFile = new File(FileUtil.asCanonicalFileName(newSipFolder + "/" + new File(this.filePath).getName()));
				this.filePath = newFile.getPath();
				this.originalSIPFolder = newFile.getParent();
			}

			this.mode = Mode.ReadWrite;
			this.lockIfPossibleAndRequired();
		}
	}


	public void saveAsTemplate(String templatePath) throws java.lang.Exception
	{
		//	Can't create a template when the root node is a file:
		if (this.structureMap.getRoot().isFile())		throw new CantCreateTemplateWithRootFileException();

		//	First save the document if necessary:
		if (this.isModified)		this.saveWithoutBackup();

		//	Then create a copy of this document in the template folder with the given template name (templatePath):
		if (this.isWorkingCopy)
		{
			if (this.isZIPFile())
			{
				Zipper.unzip(this.originalSIPFolder, templatePath);
			}
			else
			{
				FileUtil.copyToOverwriting(this.originalSIPFolder, templatePath, false);
			}
		}
		else
		{
			FileUtil.copyToOverwriting(new File(this.filePath).getParent(), templatePath, false);
		}

		//	Read the just created template:
		Document templateDocument = openReadWrite(templatePath + "/" + DefaultMETSFileName, "Document.saveAsTemplate");

		//	Then strip away all files:
		for (NodeFile f: ((NodeFolder)templateDocument.getStructureMap().getRoot()).getDescendantFiles())
		{
			try
			{
				f.delete();
			}
			catch (FileOperationNotAllowedException ex){}		//	This can't happen because I opened the document in read/write mode.
		}

		//	Then strip away all dynamic metadata:
		for (NodeAbstract n: ((NodeFolder)templateDocument.getStructureMap().getRoot()).getWithDescendants())
		{
			for (MetadataElementInstance mdei: n.getDynamicMetadataElementInstances())
			{
				//	Don't remove element if marked with "keepInTemplate='true'":
				if (mdei.keepInTemplate())		continue;

				if (mdei.isRepeatable())
				{
					try
					{
						//	No matter what index THIS metadata instance has, delete always the 0th index.
						//	This way all repeating instances of this metadata will be deleted:
						n.deleteDynamicMetadataElementInstanceWithName(mdei.getName(), 0);
					}
					catch (MetadataElementCantDeleteException ex)
					{
						//	If I can't delete this element, I clear it:
						n.setDynamicMetadataValueForName(mdei.getName(), 0, null);
					}
				}
				else
				{
					try
					{
						//	Delete this element:
						n.deleteDynamicMetadataElementInstanceWithName(mdei.getName(), mdei.getIndex());
					}
					catch (MetadataElementCantDeleteException ex)
					{
						//	If I can't delete this element, I clear it:
						n.setDynamicMetadataValueForName_force(mdei.getName(), mdei.getIndex(), null);
					}
				}
			}
		}

		//	Finally save the new template:
		templateDocument.saveWithoutBackup();

		//	... and cleanup:
		templateDocument.unlockIfNecessary();
		templateDocument.cleanupWorkingCopy();
	}


	/**
	 * Remove the working copy directory. If no working copy was created, ignore it.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	public void cleanupWorkingCopy() throws IOException, FileUtilExceptionListException
	{
		//	The following applies only if the document was opened using "createWorkingCopyAndRead()",
		//	OR if the source was a zip-file:
		if (this.isWorkingCopy)
		{
			//	Set cleanup folder recursively to writable (if possible) so cleanup can be done:
			FileUtil.setWritable(new File(this.filePath).getParentFile().getParentFile());

			//	Cleanup working folder:
			FileUtil.deleteOnExit(new File(this.filePath).getParentFile().getParentFile());
		}
	}



	/**
	 *	Revert document to last backup.
	 * @throws java.lang.Exception
	 */
	public void revertToLastBackup() throws java.lang.Exception
	{
		//	If this SIP was opened in ReadOnly mode or is locked by somebody else, reverting to the last backup is not possible:
		if (!this.canWrite())				throw new DocumentIsReadOnlyException(this);

		//	Does a backup exist? If not, return:
		String lastBackupFilePath = this.getLastBackupFilePath();
		if (lastBackupFilePath == null)		throw new FileNotFoundException("No backup file found: '" + this.originalSIPFolder + "'");


		//	Before reverting to the last backup, make a security copy of the current document:
		String securitySuffix = "_COPY_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);

		//	Create security copy of the original:
		FileUtil.renameTo(this.originalSIPFolder, this.originalSIPFolder + securitySuffix);

		//	Move last backup to original:
		FileUtil.moveTo(lastBackupFilePath, this.originalSIPFolder);

		//	Finally delete the security copy:
		FileUtil.delete(this.originalSIPFolder + securitySuffix);

		//	Remember the mode (ReadWrite or ReadWriteNoFileOps) - because the unlocking sets the mode back to Mode.Undefined.
		//	Then unlock the document.
		//	Delete my working copy (if any), I will work on the newDoc's working copy (if any):
		Mode originalMode = this.mode;
		this.unlock_dontCheck();
		this.cleanupWorkingCopy();

		//	Create a new document from the reverted backup:
		Document newDoc = null;
		String lastMetsBackupFilePath = this.getLastMetsBackupFilePath();
		if (lastMetsBackupFilePath == null)
		{
			newDoc = open(this.originalSIPFolder, originalMode, this.currentOperatorName, null);
		}
		else
		{
			newDoc = open(lastMetsBackupFilePath, Mode.ReadOnly, this.currentOperatorName, null);
		}
		
		//	Make me point to the reverted backup:
		Element rootElement = newDoc.getRootElement();
		rootElement.detach();
		this.setRootElement(rootElement);
		this.initialize(newDoc.filePath, (this.isWorkingCopy? this.originalSIPFolder: null), originalMode, this.currentOperatorName, null);
	}

	//	--------		Locking				-------------------------------------------------------

	/**
	 * If it was me who locked this archive, unlock it.
	 * This method has to be public because it may be called on shutdown or when exceptions occur.
	 */
	public void unlockIfNecessary()
	{
		if (this.lockedBy == null)		this.unlock();
	}

	//	--------		Searching			-------------------------------------------------------

	/**
	 * Search in the metadata of all nodes for the searchString
	 * @param searchString
	 * @return the list of hits
	 */
	public List<NodeAbstract>searchFor(String searchString)
	{
		List<NodeAbstract> searchHits = new Vector<NodeAbstract>();
		for (NodeAbstract node: this.structureMap.getRoot().getWithDescendants())	if (node.searchFor(searchString))	searchHits.add(node);

		return searchHits;
	}


	/**
	 * Search in the metadata of all nodes for the searchString.
	 * Ignore the case. Search for each word separately. Consider quotes.
	 * Perform an AND search, that means: ALL search words must be present.
	 * @param searchString
	 * @return the list of hits
	 */
	public List<NodeAbstract>searchForAllQuoted(String searchString)
	{
		List<NodeAbstract> searchHits = new Vector<NodeAbstract>();
		for (NodeAbstract node: this.structureMap.getRoot().getWithDescendants())	if (node.searchForAllQuoted(searchString))	searchHits.add(node);

		return searchHits;
	}


	/**
	 * Search in the metadata of all nodes for the searchString.
	 * Ignore the case. Search for each word separately. Consider quotes.
	 * Perform an OR search, that means: AT LEAST ONE of the search words must be present.
	 * @param searchString
	 * @return the list of hits
	 */
	public List<NodeAbstract>searchForAnyQuoted(String searchString)
	{
		List<NodeAbstract> searchHits = new Vector<NodeAbstract>();
		for (NodeAbstract node: this.structureMap.getRoot().getWithDescendants())	if (node.searchForAnyQuoted(searchString))	searchHits.add(node);

		return searchHits;
	}

	//	--------		Support				-------------------------------------------------------

	public void deleteDMDSection(DMDSectionAbstract dmdSection)
	{
		this.dmdSections.remove(dmdSection.getId());
	}

	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder("\n====\tMETS-Document on '")
			.append(this.filePath)
			.append("'\t====\n====\tOriginal SIP:    '")
			.append(this.originalSIPFolder)
			.append("'\t====\n\t")
			.append(this.label)
			.append("/")
			.append(this.type)
			.append("/")
			.append(this.objId)
			.append(this.isModified? " (modified)": " (not modified)")
			.append(this.isAtLeastOneFileNotReadable? " (not all readable)": " (all readable)")
			.append(this.isAtLeastOneFileNotWritable? " (not all writable)": " (all writable)")
			.append(" (")
			.append(this.mode)
			.append(")")
			.append(this.isWorkingCopy? " (workingCopy)": " (direct)")
			.append(this.lockedBy == null? "": " (locked by: '" + this.lockedBy + "')")
			.append(this.header);

		for (DMDSectionAbstract dmdSection: this.dmdSections.values())		buf.append(dmdSection.toString());

		buf.append(this.amdSection)
			.append(this.fileSection)
			.append(this.structureMap)
			.append("\n");

		return buf.toString();
	}

	//	----------		Temporary				---------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initialize my properties filePath, originalSIPFolder, isWorkingCopy, mode, and currentOperatorName.
	 * originalSIPFolder might be null in the case when the filePath does NOT point to a working copy but to the original SIP.
	 * Lock the file if necessary. Add the observer to my observer list.
	 * Finally parse the document and create the properties header, amdSection, fileSection, structMap, and optionally the dmdSections.
	 */
	protected void initialize(String metsFilePath, String originalSIPFolder, Mode mode, String operatorName, Observer observer)
	{
		this.mode = mode;
		this.filePath = FileUtil.asCanonicalFileName(metsFilePath);
		this.currentOperatorName = operatorName;
		this.addObserver(observer);

		//	originalSipFolder is either a folder path, or a ZIP file path, or null.
		//	If originalSipFolder is null, I am working on the original SIP (no working copy),
		//	so set my originalSipFolder to the parent of the metsFilePath, and set the property isWorkingCopy to false:
		if (originalSIPFolder == null)
		{
			this.originalSIPFolder = FileUtil.asParentPath(metsFilePath);
			this.isWorkingCopy = false;
		}
		else
		{
			this.originalSIPFolder = originalSIPFolder;
			this.isWorkingCopy = true;
		}

		//	Locking:
		this.lockIfPossibleAndRequired();

		//	Parsing:
		Element root = this.getRootElement();
		this.namespaceMETS = root.getNamespaceForPrefix("METS").getStringValue();
		Namespace premis = root.getNamespaceForPrefix("PREMIS");
		this.namespacePREMIS = (premis != null)? premis.getStringValue(): "info:lc/xmlns/premis-v2";
		this.profile = root.attributeValue("PROFILE");
		this.label = root.attributeValue("LABEL");
		this.type = root.attributeValue("TYPE");
		if ((this.type == null) || this.type.isEmpty())
		{
			Exception.remember("METS file does not contain a submission agreement definition.");
		}
		else
		{
			try
			{
				this.saId = this.type.substring(0, this.type.lastIndexOf("_"));
				this.dssId = this.type.substring(this.type.lastIndexOf("_") + 1, this.type.length());
			}
			catch (StringIndexOutOfBoundsException e)
			{
				Exception.remember("METS file does not contain a valid submission agreement definition: " + this.type);
			}
		}
		this.objId = root.attributeValue("OBJID");

		//	Optional elements:
		this.distributeMessage("DMD Section...");
		this.dmdSections = DMDSectionAbstract.parse(this);

		//	Mandatory elements:
		this.distributeMessage("AMD Section...");
		this.amdSection = AMDSection.parse(this);
		this.distributeMessage("Header...");			//	Since the Header now needs the AMD Section, parse the header after the AMD Section
		this.header = Header.parse(this);
		this.distributeMessage("File Section...");
		this.fileSection = FileSection.parse(this);
		this.distributeMessage("Struct Map...");
		this.structureMap = StructureMap.parse(this);
		this.distributeMessage("OK");

		this.isModified = false;
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------

	/**
	 * Write back to original SIP-folder or ZIP-file.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException This Document can't be saved because it it read-only.
	 * @throws FileOrFolderIsInUseException A file or folder inside this SIP is being used by another application.
	 * @throws OriginalSIPIsMissingException The original SIP could not be found so no backup could be made. Nevertheless the saving was successful.
	 */
	protected void save(boolean withBackup) throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException, OriginalSIPIsMissingException
	{
		//	Can't save if this document is read-only:
		if (!this.canWrite())		throw new DocumentIsReadOnlyException(this);

		//	If the original SIP is missing, remember this and throw an exception after saving.
		boolean originalSIPIsMissing = false;
		this.header.setLastModificationDateToNow();
		this.header.setSwNameAndType();

		if (this.isWorkingCopy)
		{
			//	Write the mets-file in working copy:
			this.writeMETSFile();

			if (this.isZIPFile())
			{
				//	Original SIP is a ZIP file:

				if (withBackup)
				{
					if (!new File(this.originalSIPFolder).exists())
					{
						originalSIPIsMissing = true;
					}
					else
					{
						//	Backup original ZIP-file:
						if (this.isReadWriteNoFileOps())
						{
							String filename = null;
							if (BackupFolder == null)
							{
								filename = FileUtil.asFilePathWithoutExtension(this.originalSIPFolder) + "_ORIGINAL-METS_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "." + FileUtil.asFileNameExtension(this.originalSIPFolder);
							}
							else
							{
								filename = BackupFolder + "/" + FileUtil.asFileNameWithoutExtension(this.originalSIPFolder) + "_ORIGINAL-METS_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "." + FileUtil.asFileNameExtension(this.originalSIPFolder);
							}
							Zipper.zip(this.filePath, filename);
						}
						else
						{
							if (BackupFolder != null)
							{
								FileUtil.renameTo(this.originalSIPFolder, BackupFolder + "/" + FileUtil.asFileNameWithoutExtension(this.originalSIPFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "." + FileUtil.asFileNameExtension(this.originalSIPFolder));
							}
							else
							{
								FileUtil.renameTo(this.originalSIPFolder, FileUtil.asFilePathWithoutExtension(this.originalSIPFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "." + FileUtil.asFileNameExtension(this.originalSIPFolder));
							}
						}

						this.purgeBackups();
					}
				}

				//	Zip working folder back to original ZIP-file:
				List<String> files = new ArrayList<String>();
				for (File file: new File(this.getSIPFolder()).listFiles())		files.add(file.getPath());

				Zipper.zip(files.toArray(new String[]{}), this.originalSIPFolder);
			}
			else
			{
				//	Original SIP is a folder:

				if (withBackup)
				{
					if (!new File(this.originalSIPFolder).exists())
					{
						originalSIPIsMissing = true;
					}
					else
					{
						// 	this case never happens, as the SIP has not been extracted to a working folder
						// 	if (this.isReadWriteNoFileOps()) ...
						// 	else ...
						
						//	Backup original folder, DO preserve the access rights:
						if (BackupFolder != null)
						{
							FileUtil.copyToOverwriting(this.originalSIPFolder, BackupFolder + "/" + FileUtil.asFileName(this.originalSIPFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
						}
						else
						{
							FileUtil.copyToOverwriting(this.originalSIPFolder, this.originalSIPFolder + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
						}

						this.purgeBackups();
					}
				}

				//	Redesign: instead of just copying the working copy back over the original SIP, now copy the working copy to a temporary SIP,
				//	then delete the original SIP and finally rename the temporary SIP to the original SIP name.
				//	So I can be sure there is a saved copy in place.
				String securitySuffix = "_COPY_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);

				//	If there are access-restricted files, I might not be able to delete them - so I overwrite whatever I can:
				if (this.isAtLeastOneFileNotReadable || this.isAtLeastOneFileNotWritable)
				{
					//	There are access-restricted files in this SIP: report but ignore exceptions:

					try
					{
						//	Create security copy:
						FileUtil.copyToOverwriting(this.getSIPFolder(), this.originalSIPFolder + securitySuffix, true);
					}
					catch (FileUtilExceptionListException ex)
					{
						//	If I have at least one access-restricted file, report but ignore this exception:
						ex.printStackTrace();
					}

					try
					{
						//	Copy security copy to original:
						FileUtil.copyToOverwriting(this.originalSIPFolder + securitySuffix, this.originalSIPFolder, true);
					}
					catch (FileUtilExceptionListException ex)
					{
						//	If I have at least one access-restricted file, report but ignore this exception:
						ex.printStackTrace();
					}

					//	Delete security copy:
					FileUtil.setWritable(this.originalSIPFolder + securitySuffix);
					FileUtil.delete(this.originalSIPFolder + securitySuffix);
				}
				else
				{
					//	There are supposed to be NO access-restricted files in this SIP, so don't catch exceptions:

					//	Create security copy:
					FileUtil.copyToOverwriting(this.getSIPFolder(), this.originalSIPFolder + securitySuffix, true);

					try
					{
						//	Delete original:
						FileUtil.delete(this.originalSIPFolder);
					}
					catch (FileUtilExceptionListException ex)
					{
						List<FileUtilException> exFiles = new Vector<FileUtilException>();
						for (FileUtilException x: ex.getAllExceptions())		exFiles.add(x);

						//	A file must be in use by another application or the Windows explorer:
						throw new FileOrFolderIsInUseException(exFiles, this.originalSIPFolder, this.originalSIPFolder + securitySuffix);
					}

					//	Rename security copy back to original:
					FileUtil.renameTo(this.originalSIPFolder + securitySuffix, this.originalSIPFolder);
				}
			}
		}
		else
		{
			//	No working copy:

			if (withBackup)
			{
				if (!new File(this.originalSIPFolder).exists())
				{
					originalSIPIsMissing = true;
				}
				else
				{
					if (this.isReadWriteNoFileOps())
					{
						String filename = null;
						if (BackupFolder == null)
						{
							filename = FileUtil.asFilePathWithoutExtension(this.originalSIPFolder) + "_ORIGINAL-METS_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);
						}
						else
						{
							filename = BackupFolder + "/" + FileUtil.asFileNameWithoutExtension(this.originalSIPFolder) + "_ORIGINAL-METS_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);
						}
						File backup = new File(filename);
						if (!backup.exists())
						{
							backup.mkdirs();
						}
						FileUtil.copyToFolderOverwriting(this.filePath, filename);
					}
					else
					{
						//	Backup original folder, DO preserve the access rights:
						if (BackupFolder != null)
						{
							FileUtil.copyToOverwriting(this.originalSIPFolder, BackupFolder + "/" + FileUtil.asFileName(this.originalSIPFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
						}
						else
						{
							FileUtil.copyToOverwriting(this.originalSIPFolder, this.originalSIPFolder + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
						}
					}
					
					this.purgeBackups();
				}
			}

			//	In the case that no working copy was created, write back only the METS-File:
			this.writeMETSFile();
		}

		this.isModified = false;

		if (originalSIPIsMissing)		throw new OriginalSIPIsMissingException(this.originalSIPFolder);
	}


	private void writeMETSFile() throws IOException
	{
		writeMETSFile(this, this.filePath);
	}


	private static void writeMETSFile(org.dom4j.Document document, String metsFilePath) throws IOException
	{
		Writer oswriter = new java.io.OutputStreamWriter(new FileOutputStream(metsFilePath), "utf-8");
		OutputFormat outformat = new OutputFormat();
		outformat.setEncoding("UTF-8");
		outformat.setIndent(false);
		outformat.setNewlines(false);
		XMLWriter writer = new XMLWriter(oswriter, outformat);
		writer.write(document);
		writer.flush();
		writer.close();
	}

	private void purgeBackups()
	{
		//	If KeepBackupsCount is null, keep all backups:
		if (KeepBackupsCount == null)		return;

		//	Define the correct filename filter:
		final String backupPattern =
			this.isZIPFile()
				? FileUtil.asFileNameWithoutExtension(this.originalSIPFolder) + "_ORIGINAL_"
				: FileUtil.asFileName(this.originalSIPFolder) + "_ORIGINAL_";
		FilenameFilter myBackups = new FilenameFilter()
		{	@Override public boolean accept(File dir, String name) { return name.startsWith(backupPattern); }};

		//	Apply the filename filter to the correct backup folder and sort the result according to the file name:
		File backupFolder = BackupFolder != null? new File(BackupFolder): new File(this.originalSIPFolder).getParentFile();
		List<String> backups = Arrays.asList(backupFolder.list(myBackups));
		Collections.sort(backups);

		//	Purge all superflouos SIPs:
		for (int i = 0; i < backups.size() - KeepBackupsCount; i++)
		{
			Logger.getLogger().info("Purging '" + backupFolder + "/" + backups.get(i) + "'");
			try
			{
				String backup = backups.get(i);

				FileUtil.delete(backupFolder + "/" + backup);

				// Delete precedence backup files which only content mets.xml files
				if (backups.size() > i + 1)
				{
					final String backupMetsPattern = backups.get(i + 1).replace("_ORIGINAL_", "_ORIGINAL-METS_");
					FilenameFilter myMetsBackups = new FilenameFilter()
					{ @Override public boolean accept(File dir, String name) { return name.contains("_ORIGINAL-METS_") && name.compareTo(backupMetsPattern) < 0; }
					};
					List<String> metsBackups = Arrays.asList(backupFolder.list(myMetsBackups));
					for (String metsBackup : metsBackups)
					{
						Logger.getLogger().info("Purging '" + backupFolder + "/" + metsBackup + "'");
						try
						{
							FileUtil.delete(backupFolder + "/" + metsBackup);
						}
						catch (FileUtilExceptionListException ex)
						{
							ex.printStackTrace();
						}
					}
				}
			}
			catch (FileUtilExceptionListException ex)
			{
				ex.printStackTrace();
			}
		}
	}

	/**
	 * @return the last backup file path, if a last backup exists; otherwise return null.
	 */
	private String getLastBackupFilePath()
	{
		//	Define the correct filename filter:
		final String backupPattern =
			this.isZIPFile()
				? FileUtil.asFileNameWithoutExtension(this.originalSIPFolder) + "_ORIGINAL_"
				: FileUtil.asFileName(this.originalSIPFolder) + "_ORIGINAL_";
		FilenameFilter myBackupsFilter =
			new FilenameFilter()
			{	@Override public boolean accept(File dir, String name) { return name.startsWith(backupPattern); }};

		//	Apply the filename filter to the correct backup folder and sort the result according to the file name:
		File backupFolder = BackupFolder != null? new File(BackupFolder): new File(this.originalSIPFolder).getParentFile();

		String[] backupsArray = backupFolder.list(myBackupsFilter);
		if (backupsArray == null)	return null;

		List<String> backups = Arrays.asList(backupFolder.list(myBackupsFilter));
		Collections.sort(backups);
		if (backups.isEmpty())		return null;

		return backupFolder + File.separator + backups.get(backups.size() - 1);
	}

	private String getLastMetsBackupFilePath()
	{
		//	Define the correct filename filter:
		final String backupPattern =
			this.isZIPFile()
				? FileUtil.asFileNameWithoutExtension(this.originalSIPFolder) + "_ORIGINAL-METS_"
				: FileUtil.asFileName(this.originalSIPFolder) + "_ORIGINAL-METS_";
		FilenameFilter myBackupsFilter =
			new FilenameFilter()
			{	@Override public boolean accept(File dir, String name) { return name.startsWith(backupPattern); }};

		//	Apply the filename filter to the correct backup folder and sort the result according to the file name:
		File backupFolder = BackupFolder != null? new File(BackupFolder): new File(this.originalSIPFolder).getParentFile();

		String[] backupsArray = backupFolder.list(myBackupsFilter);
		if (backupsArray == null)	return null;

		List<String> backups = Arrays.asList(backupFolder.list(myBackupsFilter));
		Collections.sort(backups);
		if (backups.size() < 2)		return null;

		return backupFolder + File.separator + backups.get(backups.size() - 2);
	}

	//	--------		Locking				-------------------------------------------------------

	/**
	 * If this document is not locked by somebody else, lock it:
	 */
	protected void lockIfPossibleAndRequired()
	{
		String lockedBy = this.lockedByWhom();
		if (lockedBy == null)
		{
			//	Not yet locked:
			if (this.canWrite())		this.lock_dontCheck();
			//	(The mode is already Mode.ReadWrite or Mode.ReadWriteNoFileOps)
		}
		else if (OperatingSystem.userName().equals(lockedBy))
		{
			//	Is already locked by me:
			this.lockedBy = null;
		}
		else
		{
			//	Is locked by somebody else:
			this.lockedBy = lockedBy;
			this.mode = Mode.Locked;
		}
	}


	/**
	 * Return the login name of the user who locked this SIP, or null if no lock exists.
	 *
	 * @param sipName
	 * @return
	 */
	protected String lockedByWhom()
	{
		return lockedByWhom(this.originalSIPFolder);
	}


	/**
	 * Lock this SIP by creating a lock file.
	 * IMPORTANT NOTE: This method does NOT check if locking is allowed, it just overwrites any existing lock!
	 */
	protected void lock_dontCheck()
	{
		try
		{
			FileUtil.createFileWithContent(this.originalSIPFolder + LockFileSuffix, OperatingSystem.userName());
		}
		catch (IOException ex)
		{
			ex.printStackTrace();
		}
	}


	/**
	 * Unlock this SIP by deleting the lock file (only if allowed).
	 * return true if unlock was succesful, false otherwise.
	 */
	protected boolean unlock()
	{
		//	Can't unlock if I am not allowed to write.
		if (!this.canWrite())		return false;

		return this.unlock_dontCheck();
	}


	/**
	 * Unlock this SIP by deleting the lock file.
	 * IMPORTANT NOTE: This method does NOT check if unlocking is allowed, it just deletes any existing lock!
	 * return true if unlock was succesful, false otherwise.
	 */
	protected boolean unlock_dontCheck()
	{
		try
		{
			FileUtil.delete(this.originalSIPFolder + LockFileSuffix);
			this.mode = Mode.Undefined;
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
			return false;
		}

		return true;
	}


	/**
	 * Remove the existing lock, create a new one ("SubmitRequestPending-lock"),
	 * then (for not-zipped SIPs) remove the working copy and switch to the original SIP as read-only.
	 */
	protected void lockAsSubmitRequestPending()
	{
		//	Remove the existing lock:
		this.unlock();

		try
		{
			//	Create a new lock:
			FileUtil.createFileWithContent(this.originalSIPFolder + LockFileSuffix, SubmitStatus.getSubmitRequestPendingLockId());
			this.mode = Mode.Locked;
			this.lockedBy = SubmitStatus.getSubmitRequestPendingLockId();

			//	For ZIP-files, the working copy will be removed anyway when closing this SIP:
			if (!this.isZIPFile())
			{
				//	Remove the working copy:
				this.cleanupWorkingCopy();

				// 	Switch to the original SIP (so it becomes read-only):
				this.filePath = FileUtil.asCanonicalFileName(this.getOriginalSIPFolder() + File.separator + DefaultMETSFileName);
				this.isWorkingCopy = false;
			}
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
		}
	}

	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	/**
	 * Document mode:
	 *
	 * Undefined:	Temporary. The document is just being read; the mode will soon change.
	 * ReadWrite:	The user can change whatever he wants. This is the case when the SIP was opened with a working copy.
	 * NoFileOps:	The user can only change things that don't affect the file system. This is the case when the SIP was opened without a working copy.
	 * ReadOnly:	The user can not make any changes to the SIP. This is the case when the SIP was opened in ReadOnly mode.
	 * Locked:		The user can not make any changes to the SIP. This is the case when the SIP was opened by somebody else in ReadWrite or NoFileOps mode.
	 */
	public enum Mode
	{
		Undefined, ReadWrite, ReadWriteNoFileOps, ReadOnly, Locked
	}


	/**
	 * This class is needed within the class <a href="../Document.html">Document</a> to make the SAXReader generate an instance of the Document class.
	 *
	 * @author denis
	 */
	static private class DocumentFactory extends org.dom4j.DocumentFactory
	{
		// Factory methods
		@Override
		public Document createDocument()
		{
			ch.docuteam.darc.mets.Document document = new ch.docuteam.darc.mets.Document();
			document.setDocumentFactory(this);

			return document;
		}
	}

}
