/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.amdsec;

import java.util.*;

import org.dom4j.Element;

import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.id.UniqueID;


/**
 * This class, used by <a href="../Document.html">Document</a>, represents the METS Administrative Metadata Section.
 * It contains a collection of <a href="./DigiprovWithPremis.html">DigiprovWithPremis</a> Elements.
 * <p>
 * This class inserts Exceptions into the <a href="../../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when the number of AMDSections found in the METS file is not exactly 1</li>
 * </ul>
 * @author denis
 */
public class AMDSection extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	/**
	 * The Map of Digiprovs. Key is the digiprov's id, value is the digiprov itself.
	 */
	protected Map<String, DigiprovAbstract>	digiprovs = new HashMap<String, DigiprovAbstract>();

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	protected AMDSection() {}

	//	========	Constructors Private	=======================================================

	/**
	 * Create the AMDSection out of the given document.
	 */
	private AMDSection(Document document)
	{
		this.document = document;

		//	Consistency check: no multiple AMDSections are allowed:
		if (this.document.selectNodes("./METS:mets/METS:amdSec").size() >= 2)
			Exception.remember("Bad number of AMD sections in the METS file. Expected: 1, found: " + this.document.selectNodes("./METS:mets/METS:amdSec").size());

		this.element = (Element)this.document.selectSingleNode("./METS:mets/METS:amdSec");

		//	Check if an amdSection exists at all - if not, create one:
		if (this.element == null)
			this.element = this.document.getRootElement()
								.addElement("METS:amdSec")
									.addAttribute("ID", UniqueID.getXML());

		this.id = this.element.attributeValue("ID");

		this.addAllDigiprovs(DigiprovWithPremis.parse(this));
	}

	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read.
	 */
	static public AMDSection parse(Document document)
	{
		return new AMDSection(document);
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * Return a collection with all digiprovs (unspecified order)
	 */
	public Collection<? extends DigiprovAbstract> getDigiprovs()
	{
		return this.digiprovs.values();
	}

	/**
	 * Return the digiprov with the given id. IF it doesn't exist, return null.
	 * @param id The id of the digiprov to get
	 * @return the digiprov with the id, or null
	 */
	public DigiprovWithPremis getDigiprov(String id)
	{
		return (DigiprovWithPremis)this.digiprovs.get(id);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------

	public void addAllDigiprovs(List<? extends DigiprovAbstract> digiprovs)
	{
		for (DigiprovAbstract digiprov: digiprovs)		this.addNewDigiprov(digiprov);
	}


	/**
	 * Add a new digiprov to the digiprov map.
	 */
	public void addNewDigiprov(DigiprovAbstract digiprov)
	{
		this.digiprovs.put(digiprov.getId(), digiprov);
	}


	/**
	 * Delete this digiprov from the digiprov map. This is called only when creating a template out of this METS-File.
	 */
	public void deleteDigiprov(DigiprovAbstract digiprov)
	{
		this.digiprovs.remove(digiprov.getId());
	}

	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder("\n[AMDSection:" + this.id + "(digiprovs:" + this.digiprovs.size() + ")");
		for (DigiprovAbstract digiprov: this.digiprovs.values())		buf.append("\n\t" + digiprov.toString());
		buf.append("\n]");

		return buf.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
