/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.filesec;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.*;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.structmap.NodeFolder;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.file.MetadataProviderDROID;
import ch.docuteam.tools.id.UniqueID;


/**
 * This class, used by the class <a href="./FileSection.html">FileSection</a>, represents the METS File entities within the file section.
 * <p>
 * This class inserts Exceptions into the <a href="../../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when the referenced file is not present</li>
 * <li>Replacing a file, when the Checksum Calculator throws an exception</li>
 * </ul>
 * @author denis
 */
public class File extends ch.docuteam.darc.common.NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static final protected String		HrefPrefix = "file:///";
	static final protected String		LocType = "URL";
	static final protected String		Xmlns_xlink = "http://www.w3.org/1999/xlink";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected String		mimeType;
	protected String		locType = LocType;
	protected String		href;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	protected File() {}


	/**
	 * 	This constructor is used when a new file is created programmatically
	 * @param parent The parent
	 * @param label The new folder label
	 */
	public File(DocumentAbstract document, String newRelativeFilePath)
	{
		this.document = document;
		this.parent = ((Document)this.document).getFileSection();

		//	To set the element, the parent already has to be defined:
		this.element = ((Element)this.parent.getElement().selectSingleNode("./METS:fileGrp"))
				.addElement("METS:file");
		this.element
				.addElement("METS:FLocat")
				.addAttribute("LOCTYPE", this.locType);

		//	To set the ID, the element already has to be defined:
		this.setId(UniqueID.getXML());

		//	To set me as one of my parent's children, the ID already has to be defined:
		((FileSection)this.parent).add(this);

		//		set href:
		this.setHrefAddPrefix(newRelativeFilePath);

		//		set mime type:
		try
		{
			this.setMimeType(MetadataProviderDROID.getMimeType(this.getAbsoluteFilePath()));
		}
		catch (java.lang.Exception ex)
		{
			//	Another exception ("No format ID found for file ...") will be remembered in NodeFolder.insertFile(),
			//	so there's no need to remember it here:
//			Exception.remember(ex);

			//	If DROID doesn't return exactly one MimeType, don't leave it null but set it to "":
			this.setMimeType("");
		}

		this.document.setIsModified();
	}

	//	========	Constructors Private	=======================================================

	/**
	 * 	This constructor is used only when a METS-File is being read
	 */
	private File(FileSection parent, Element fileElement)
	{
		this.document = parent.getDocument();
		this.parent = parent;
		this.element = fileElement;

		this.id = fileElement.attributeValue("ID");
		this.mimeType = fileElement.attributeValue("MIMETYPE");

		Element fileLocationSubNode = (Element)fileElement.selectSingleNode("./METS:FLocat");

		this.locType = fileLocationSubNode.attributeValue("LOCTYPE");
		this.href = fileLocationSubNode.attributeValue("href");
	}

	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read. Create the list of Files out of the given parent FileSection.
	 */
	static protected List<File> parse(FileSection parent)
	{
		List<File> fileList = new ArrayList<File>();

		for (Object o: parent.getElement().selectNodes("./METS:fileGrp/METS:file"))		fileList.add(new File(parent, (Element)o));

		return fileList;
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getMimeType()
	{
		return this.mimeType;
	}

	public String getLocType()
	{
		return this.locType;
	}

	public String getHref()
	{
		return this.href;
	}

	public String getAbsoluteFilePath()
	{
		return ((Document)this.document).getSIPFolder() + "/" + this.href.substring(HrefPrefix.length());
	}

	/**
	 *
	 * @return The java.io.File instance referring to the absolute path of this node
	 */
	public java.io.File getFile()
	{
		return new java.io.File(this.getAbsoluteFilePath());
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	public void delete()
	{
		((FileSection)this.parent).delete(this);
		this.element.detach();
		this.document.setIsModified();
	}


	public void rename(String newName)
	{
		this.setHrefAddPrefix(newName);
	}


	/**
	 * Moving a file means to change its href. So set the href to the new folder path + my file name.
	 */
	public void moveTo(NodeFolder toFolder)
	{
		this.setHrefAddPrefix(toFolder.getPathString() + this.href.substring(this.href.lastIndexOf("/")));
	}


	/**
	 * Replace in href the string "old" by the string "replace". Replace only the 1st occurrence of "old" and only with trailing "file:///".
	 * This is used when a superfolder changed its name or location.
	 * @param old The string to be replaced
	 * @param replace The string to replace the old one
	 */
	public void replaceInFilePath(String old, String replace)
	{
		this.setHref(this.href.replaceFirst(HrefPrefix + old, HrefPrefix + replace));
	}


	public void replaceFile(String fileName)
	{
		//		change href:
		this.setHref(this.href.substring(0, this.href.lastIndexOf("/") + 1) + fileName);

		//		change mime type:
		try
		{
			this.setMimeType(MetadataProviderDROID.getMimeType(this.getAbsoluteFilePath()));
		}
		catch (java.lang.Exception e)
		{
			Exception.remember(e);
		}
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return new StringBuilder("[FileSectionFile:id=")
			.append(this.id + "(")
			.append(this.mimeType + ")href=")
			.append(this.href + "]")
			.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	protected void setMimeType(String mimeType)
	{
		this.mimeType = mimeType;
		this.element.addAttribute("MIMETYPE", this.mimeType);

		this.document.setIsModified();
	}

	protected void setHref(String href)
	{
		this.href = href;
		((Element)this.element.selectSingleNode("./METS:FLocat"))
			.addAttribute(new QName("href", new Namespace("xlink", Xmlns_xlink)), this.href);

		this.document.setIsModified();
	}

	/**
	 * Set the href but add the prefix "file:///" before it.
	 * @param href
	 */
	public void setHrefAddPrefix(String href)
	{
		this.setHref(HrefPrefix + href);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
