/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.structmap;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.util.*;

import javax.swing.tree.TreePath;

import org.dom4j.Element;
import org.jdesktop.swingx.treetable.AbstractMutableTreeTableNode;

import bsh.EvalError;
import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.dc.OAI_DC;
import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.mdconfig.*;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.amdsec.DigiprovWithPremis;
import ch.docuteam.darc.mets.dmdsec.*;
import ch.docuteam.darc.premis.Event;
import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.file.FileFilter;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.id.UniqueID;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;


/**
 * This class is the abstract superclass of <a href="./NodeFile.html">NodeFile</a> and <a href="./NodeFolder.html">NodeFolder</a>.
 * It contains many common and abstract methods for these two subclasses.
 *
 * @author denis
 *
 */
public abstract class NodeAbstract extends AbstractMutableTreeTableNode implements Comparable<NodeAbstract>
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected DocumentAbstract		document;
	protected Element				element;

	protected String				label;
	protected String				type;
	protected String				admId;
	protected String				dmdIdOAI_DC;
	protected String				dmdIdEAD;

	//	Does the referenced file really exist?
	protected boolean				fileExists;
	//	Can the current user read, write, or execute the referenced file or folder?
	protected boolean				canRead;
	protected boolean				canWrite;
	protected boolean				canExecute;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	protected NodeAbstract() {}

	//	========	Constructors Private	=======================================================

	/**
	 * 	This constructor is used only when a METS-File is being read.
	 */
	protected NodeAbstract(DocumentAbstract document, Element element)
	{
		super();
		this.document = document;
		this.element = element;

		this.label = element.attributeValue("LABEL");
		this.type = element.attributeValue("TYPE");
		this.admId = element.attributeValue("ADMID");

		//	Optional elements:
		Element e = (Element)element.selectSingleNode("./METS:div[@TYPE='metadata' and @LABEL='OAI_DC']");
		if (e != null)		this.dmdIdOAI_DC = e.attributeValue("DMDID");

		e = (Element)element.selectSingleNode("./METS:div[@TYPE='metadata' and @LABEL='EAD']");
		if (e != null)		this.dmdIdEAD = e.attributeValue("DMDID");

		//	Insert associated DMDSectionWithEAD if none is present:
		this.createDMDSectionWithEADIfNecessary();
	}


	/**
	 * 	This constructor is used when a new NON-root node is created programmatically.
	 * @param parent
	 * @param label
	 */
	protected NodeAbstract(NodeFolder parent, String label)
	{
		super();
		this.document = parent.getDocument();
		this.parent = parent;

		//	At this point, this.element() is yet undefined, so I can't set the element's properties!
		this.label = label;
		this.admId = UniqueID.getXML();

		parent.add(this);

		this.document.setIsModified();
	}


	/**
	 * 	This constructor is used when a new ROOT node is created programmatically.
	 * @param parent
	 * @param label
	 */
	protected NodeAbstract(StructureMap parent, String label)
	{
		//	The parent of the root node has to remain null!
		super();
		this.document = parent.getDocument();

		//	At this point, this.element() is yet undefined, so I can't set the element's properties!
		this.label = label;
		this.admId = UniqueID.getXML();

		this.document.setIsModified();
	}


	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read, and for the root node only.
	 * Create recursively the tree structure.
	 * NOTE: This can not be implemented within a constructor because here I decide, which class actually to instantiate.
	 * @param document
	 * @param element
	 */
	static protected NodeAbstract parse(StructureMap parent, Element element)
	{
		String type = element.attributeValue("TYPE");

		//	Is the element a rootfile or a rootfolder?
		if (type == null)						throw new NullPointerException("Undefined root node type in the METS file's Structure Map");

		else if (type.equals("rootfolder"))		return new NodeFolder(parent, element);
		else if (type.equals("rootfile"))		return new NodeFile(parent, element);
		else 									throw new IllegalArgumentException("Bad root node type in the METS file's Structure Map: '" + type + "'");
	}


	/**
	 * This method is used only when a METS-File is being read, and for NON-root nodes only.
	 * Create recursively the tree structure.
	 * NOTE: This can not be implemented within a constructor because here I decide, which class actually to instantiate.
	 * @param parent The parent of this object
	 * @param element The element of this object
	 */
	static protected NodeAbstract parse(NodeFolder parent, Element element)
	{
		String type = element.attributeValue("TYPE");

		//	Is the element a file or a folder?
		if (type == null)						throw new NullPointerException("Undefined node type in the METS file's Structure Map");

		else if (type.equals("folder"))			return new NodeFolder(parent, element);
		else if (type.equals("file"))			return new NodeFile(parent, element);
		else									throw new IllegalArgumentException("Bad node type in the METS file's Structure Map: '" + type + "'");
	}



	/**
	 * Create a new root node out of the sourceFilePath. Create recursively the tree structure.
	 * NOTE: This can not be implemented within a constructor because here I decide, which class actually to instantiate.
	 * @param document
	 * @param element
	 * @throws FileAlreadyExistsException
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	static protected NodeAbstract createRootNode(StructureMap structMap, File file) throws FileNotFoundException, IOException, FileAlreadyExistsException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		Document document = (Document)structMap.getDocument();
		if (!document.areFileOperationsAllowed())		throw new FileOperationNotAllowedException(document, file.getPath(), "CreateRootNode");

		String sipRelativeFilePath = file.getName();
		String newRootFilePath = document.getSIPFolder() + "/" + sipRelativeFilePath;

		if (file.isFile())
		{
			//	Does file exist? If not: exception!
			if (!file.exists())
			{
				throw new FileNotFoundException("File to be inserted does not exist: '" + file.getPath() + "'");
			}

			//	Is it hidden? If yes: ignore!
			if (file.isHidden())
			{
				return null;
			}

			//	Does SA allow this file? If not: remember exception and continue!
			SubmissionAgreement sa = document.getSubmissionAgreement();
			if ((sa != null) && !sa.allowsFile(document.getDSSId(), file.getPath()))
			{
				ch.docuteam.tools.exception.Exception.remember("Submission Agreement '" + document.getSAId() + "' Data Submission Session '" + document.getDSSId() + "' doesn't allow file '" + file.getPath() + "'");
				return null;
			}

			//	Copy the file to here:
			FileUtil.copyToOverwriting(file.getPath(), newRootFilePath);

			ch.docuteam.darc.mets.filesec.File fileSecFile = new ch.docuteam.darc.mets.filesec.File(structMap.getDocument(), sipRelativeFilePath);

			NodeFile nodeFile = new NodeFile(structMap, fileSecFile);

			//	If the mimeType or format ID of the new file is undef, remember it:
			if (nodeFile.getMimeType().isEmpty())		Exception.remember("No MimeType found for file: '" + nodeFile.getPathString() + "'");
			if (nodeFile.getFormatKey().isEmpty())		Exception.remember("No FormatID found for file: '" + nodeFile.getPathString() + "'");

			return nodeFile;
		}

		//	Read recursively the whole directory structure:

		NodeFolder rootFolder = new NodeFolder(structMap, sipRelativeFilePath);
		for (File f: file.listFiles((FilenameFilter)FileFilter.VisibleAll))
		{
			rootFolder.insertFileOrFolder(f.getPath());
		}

		return rootFolder;
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * Cast the variable "this.document" from "DocumentAbstract" to "mets.Document" because I (a node) am in a mets.Document for sure.
	 */
	public Document getDocument()
	{
		return (Document)this.document;
	}


	public Element getElement()
	{
		return this.element;
	}


	public String getLabel()
	{
		return this.label;
	}

	public void setLabel(String label)
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		this.label = label;
		this.element.addAttribute("LABEL", this.label);

		//	Change unitTitle in myDMDSectionWithEAD.c.did:
		//	NO: Once set when created, my label and my unitTitle are no longer connected with each other.
		//	    Hence don't update the unitTitle when changing the label!
//		this.setUnitTitle(this.label);

		this.document.setIsModified();
	}

	public String getType()
	{
		return this.type;
	}

	public void setType(String type)
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		this.type = type;
		this.element.addAttribute("TYPE", this.type);

		this.document.setIsModified();
	}

	public String getAdmId()
	{
		return this.admId;
	}

	public void setAdmId(String admId)
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		this.admId = admId;
		this.element.addAttribute("ADMID", this.admId);

		this.document.setIsModified();
	}

	public String getDmdIdOAI_DC()
	{
		return this.dmdIdOAI_DC;
	}

	public String getDmdIdEAD()
	{
		return this.dmdIdEAD;
	}


	/**
	 * Return the submit status. If it is null, return SubmitStatus.SubmitUndefined.
	 * @return
	 */
	public SubmitStatus getSubmitStatus()
	{
		if (this.getMyDMDSectionWithEAD() == null)		return SubmitStatus.SubmitUndefined;

		return this.getMyDMDSectionWithEAD().getC().getSubmitStatus();
	}

	/**
	 * Set the submit status, but only if my current submitStatus allows this.
	 * @param submitStatus
	 * @throws CantSetSubmitStatusNotAllowedException This exception gets thrown when the current submitStatus doesn't allow the new one.
	 */
	public void setSubmitStatus(SubmitStatus newSubmitStatus) throws CantSetSubmitStatusNotAllowedException
	{
		//	Check if the new submit status could be set. If not, throw CantSetSubmitStatusNotAllowedException:
		this.setSubmitStatus_check(newSubmitStatus);

		//	Now set the new submit status. I know this works because I just checked it:
		this.setSubmitStatus_force(newSubmitStatus);
	}

	/**
	 * Check if the new submitStatus could be set, but doesn't actually set it.
	 * @param newSubmitStatus
	 * @throws CantSetSubmitStatusNotAllowedException This exception gets thrown when the current submitStatus doesn't allow the new one.
	 */
	public void setSubmitStatus_check(SubmitStatus newSubmitStatus) throws CantSetSubmitStatusNotAllowedException
	{
		SubmitStatus currentSubmitStatus = this.getSubmitStatus();

		if (currentSubmitStatus == null)		currentSubmitStatus = SubmitStatus.SubmitUndefined;
		if (!currentSubmitStatus.isNextSubmitStatusAllowed(newSubmitStatus))
		{
			Logger.getLogger().debug("Can't set submit status from: '" + currentSubmitStatus + "' to: '" + newSubmitStatus + "' in node: " + this.getPathString());
			throw new CantSetSubmitStatusNotAllowedException(this, currentSubmitStatus, newSubmitStatus);
		}
	}

	/**
	 * Set the submit status, don't check if this is allowed.
	 * @param submitStatus
	 */
	public void setSubmitStatus_force(SubmitStatus newSubmitStatus)
	{
		this.getMyDMDSectionWithEAD().getC().setSubmitStatus(newSubmitStatus);
	}


	/**
	 * Set the submitStatus to newSubmitStatus for me and all my subnodes, but only if their current submitStatus allows this for all nodes.
	 * If at least one node doesn't allow setting newSubmitStatus, don't set any of them.
	 * @param newSubmitStatus
	 * @throws CantSetSubmitStatusRecursiveException gets thrown when at least one of my subnodes can't set newSubmitStatus
	 */
	public void setSubmitStatusRecursivelyAllOrNone(SubmitStatus newSubmitStatus) throws CantSetSubmitStatusRecursiveException
	{
		List<NodeAbstract> meAndAllMyDescendants = this.getWithDescendants();
		List<String> rejectMessages = new Vector<String>();

		//	First check if the submit status can be set for me and all my subnodes:
		for (NodeAbstract n: meAndAllMyDescendants)
			try
			{
				n.setSubmitStatus_check(newSubmitStatus);
			}
			catch (CantSetSubmitStatusNotAllowedException ex)
			{
				rejectMessages.add("MessageSubmitNodeCurrentStatusDoesntAllowNextStatus '" + ex.getNode().getPathString() + "'");
			}

		if (!rejectMessages.isEmpty())			throw new CantSetSubmitStatusRecursiveException(this, newSubmitStatus, rejectMessages);

		//	If ok, set the submit status for me and all my subnodes:
		for (NodeAbstract n: meAndAllMyDescendants)
			try { n.setSubmitStatus(newSubmitStatus); }
			catch (CantSetSubmitStatusNotAllowedException ex){}	//	This ex can't ever occur here because it was already handled further up.
	}

	//	----------		Accessing calculated	---------------------------------------------------

	/**
	 * Return my DigiprovWithPremis. If none exists, create one and return this.
	 * @return
	 */
	public DigiprovWithPremis getMyDigiprov()
	{
		if (this.admId == null)
		{
			this.setAdmId(UniqueID.getXML());
			new DigiprovWithPremis(this);
		}

		return this.getDocument().getAMDSection().getDigiprov(this.admId);
	}


	public DMDSectionWithEAD getMyDMDSectionWithEAD()
	{
		if (this.dmdIdEAD == null)		return null;

		return (DMDSectionWithEAD)this.getDocument().getDMDSection(this.dmdIdEAD);
	}


	public DMDSectionWithOAI_DC getMyDMDSectionWithOAI_DC()
	{
		if (this.dmdIdOAI_DC == null)	return null;

		return (DMDSectionWithOAI_DC)this.getDocument().getDMDSection(this.dmdIdOAI_DC);
	}


	public LevelOfDescription getLevel()
	{
		if (this.dmdIdEAD == null)		return null;

		return this.getMyDMDSectionWithEAD().getC().getOtherLevel();
	}


	public void setLevel(LevelOfDescription newLevel) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		if (this.dmdIdEAD == null)		return;

		//	Cleanup Metadata values: clear those values the old level did have and the new level does not:
		Collection<LevelMetadataElement> oldMDElements = this.getDynamicMetadataElementsOrdered();

		this.getMyDMDSectionWithEAD().getC().setOtherLevel(newLevel);

		for (LevelMetadataElement lmde: oldMDElements)
		{
			if (!newLevel.getDynamicMetadataElements().containsKey(lmde.getId()))
			{
				Logger.getLogger().debug("Clearing: " + lmde.getId() + " in Node: " + this);
				lmde.clearValueInNode(this);
			}
		}

		this.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();
	}


	public String getUnitTitle()
	{
		if (this.dmdIdEAD == null)		return null;

		return this.getMyDMDSectionWithEAD().getC().getDid().getUnitTitle();
	}


	public void setUnitTitle(String unitTitle)
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		if (this.dmdIdEAD == null)		return;

		this.getMyDMDSectionWithEAD().getC().getDid().setUnitTitle(unitTitle);
	}


	/**
	 * @return The list of AMDSectionDigiprovEvents for this node
	 */
	public List<Event> getMyEvents()
	{
		return new ArrayList<Event>(this.getMyDigiprov().getEvents());
	}


	/**
	 * @return The index-th event for this node
	 */
	public Event getMyEvent(Integer index)
	{
		return this.getMyEvents().get(index);
	}

	/**
	 * @return The root node
	 */
	public NodeAbstract getRoot()
	{
		return this.getDocument().getStructureMap().getRoot();
	}


	/**
	 * @return The depth of this node within the tree. Root has the depth 0, for any other node is depth = parent.depth + 1.
	 */
	public int getDepth()
	{
		if (this.isRoot())		return 0;

		return ((NodeAbstract)this.parent).getDepth() + 1;
	}


	/**
	 *
	 * @return The tree path of this node = the list of all nodes from root to this.
	 */
	public TreePath getTreePath()
	{
		Vector<NodeAbstract> path = new Vector<NodeAbstract>(10);
		NodeAbstract parent = this;
		do
		{
			path.add(0, parent);
			parent = (NodeAbstract)parent.parent;
		}
		while (parent != null);

		return new TreePath(path.toArray(new NodeAbstract[]{}));
	}


	/**
	 *
	 * @return The relative path string (relative to the SIP root folder), constructed from all parents of this node
	 */
	public String getPathString()
	{
		StringBuilder path = new StringBuilder(200);
		NodeAbstract parent = this;
		do
		{
			path.insert(0, "/" + parent.getLabel());
			parent = (NodeAbstract)parent.parent;
		}
		while (parent != null);

		//	Cut off leading "/":
		return path.toString().substring(1);
	}


	/**
	 *
	 * @return The absolute file path string
	 */
	public String getAbsolutePathString()
	{
		return this.getDocument().getSIPFolder() + "/" + this.getPathString();
	}


	/**
	 *
	 * @return The java.io.File instance referring to the absolute path of this node
	 */
	public File getFile()
	{
		return new File(this.getAbsolutePathString());
	}


	/**
	 *
	 * @return The ORIGINAL file path string.
	 * In the case when there is no original SIP folder (the SIP was opened directly or the original SIP is a ZIP file), return the working file.
	 */
	public String getOriginalPathString()
	{
		String originalSIPFolder = this.getDocument().getOriginalSIPFolder();

		if (originalSIPFolder == null
		||  originalSIPFolder.toLowerCase().endsWith(".zip"))		return this.getAbsolutePathString();

		return originalSIPFolder + "/" + this.getPathString();
	}


	/**
	 *
	 * @return The java.io.File instance referring to the ORIGINAL path of this node.
	 * If the original path name is undefined (= no working copy), return the path to this file.
	 */
	public File getOriginalFile()
	{
		return new File(this.getOriginalPathString());
	}


	/**
	 *
	 * @return The size of this node (in percent) relative to the size of the root node
	 */
	public Long getRelativeSize()
	{
		if (this.getRoot().getSize() == 0)		return 0L;		//	(This happens when the root node has no children)

		return (100 * (this.getSize()) / this.getRoot().getSize());
	}


	public List<String> getPID()
	{
		return this.getMyDMDSectionWithEAD().getC().getPID();
	}


	//	Dynamic Metadata:	getting and setting values

	/**
	 * Get the dynamic metadata value for the non-repeatable item with the specified name.
	 * @param name
	 * @param index
	 * @return
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementIsNotAllowedException
	 * @throws MetadataElementIsNotDefinedException
	 */
	public String getDynamicMetadataValueForName(String name) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException
	{
		return this.getDynamicMetadataValueForName(name, 0);
	}

	/**
	 * Get the dynamic metadata value for the repeatable item with the specified name at the indicated index position. If the index is out of bounds, raise an exception.
	 * @param name
	 * @param index
	 * @return
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementIsNotAllowedException
	 * @throws MetadataElementIsNotDefinedException
	 */
	public String getDynamicMetadataValueForName(String name, int index) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException
	{
		return this.getLevel().getDynamicMetadataElement(name).getValueFromNode(index, this);
	}

	/**
	 * Get all dynamic metadata values for the repeatable item with the specified name. May return an empty List.
	 * @param name
	 * @param index
	 * @return
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementIsNotAllowedException
	 * @throws MetadataElementIsNotDefinedException
	 */
	public List<String> getAllDynamicMetadataValuesForName(String name) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException
	{
		return this.getLevel().getDynamicMetadataElement(name).getAllValuesFromNode(this);
	}


	/**
	 * This method is to retrieve dynamic metadata values, irrespective of whether this metadata is configured (= allowed) for this node's level, or not.
	 * USE WITH CARE!
	 * @param name
	 * @return
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementIsNotDefinedException
	 */
	public List<String> getAllDynamicMetadataValuesForName_NoCheck(String name) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementIsNotDefinedException
	{
		MetadataElement mde = MetadataElement.get(name);
		if (mde == null)		throw new MetadataElementIsNotDefinedException(name);

		return mde.getValueFromNode(this);
	}


	/**
	 * Set the dynamic metadata value for the non-repeatable item with the specified name.
	 * @param name
	 * @param index
	 * @param value
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementValidatorException
	 * @throws MetadataElementIsNotAllowedException
	 * @throws MetadataElementIsNotDefinedException
	 * @throws LevelMetadataElementIsReadOnly
	 * @throws MetadataElementSetterPostActionException
	 * @throws MetadataElementAllowedValuesException
	 */
	public void setDynamicMetadataValueForName(String name, String value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		this.setDynamicMetadataValueForName(name, 0, value);
	}

	/**
	 * Set the dynamic metadata value for the repeatable item with the specified name at the indicated index position. If the index is out of bounds, raise an exception.
	 * @param name
	 * @param index
	 * @param value
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementValidatorException
	 * @throws MetadataElementIsNotAllowedException
	 * @throws MetadataElementIsNotDefinedException
	 * @throws LevelMetadataElementIsReadOnly
	 * @throws MetadataElementSetterPostActionException
	 * @throws MetadataElementAllowedValuesException
	 */
	public void setDynamicMetadataValueForName(String name, int index, String value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		this.getLevel().getDynamicMetadataElement(name).setValueInNode(index, value, this);
	}


	/**
	 * Set the dynamic metadata value for the non-repeatable item with the specified name. Don't care if this metadata element is read-only for this level.
	 * @param name
	 * @param index
	 * @param value
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementValidatorException
	 * @throws MetadataElementIsNotAllowedException
	 * @throws MetadataElementIsNotDefinedException
	 * @throws LevelMetadataElementIsReadOnly
	 * @throws MetadataElementSetterPostActionException
	 * @throws MetadataElementAllowedValuesException
	 */
	public void setDynamicMetadataValueForName_force(String name, String value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		this.setDynamicMetadataValueForName_force(name, 0, value);
	}

	/**
	 * Set the dynamic metadata value for the repeatable item with the specified name at the indicated index position. If the index is out of bounds, raise an exception.
	 * Don't care if this metadata element is read-only for this level.
	 * @param name
	 * @param index
	 * @param value
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementValidatorException
	 * @throws MetadataElementIsNotAllowedException
	 * @throws MetadataElementIsNotDefinedException
	 * @throws LevelMetadataElementIsReadOnly
	 * @throws MetadataElementSetterPostActionException
	 * @throws MetadataElementAllowedValuesException
	 */
	public void setDynamicMetadataValueForName_force(String name, int index, String value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		this.getLevel().getDynamicMetadataElement(name).setValueInNode_NoCheck(index, value, this);
	}


	//	Dynamic Metadata:	MetadataElementInstances

	public void addDynamicMetadataElementInstanceWithName(String name) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementCantAddException, EvalError, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException
	{
		this.getLevel().getDynamicMetadataElement(name).addMetadataElementInstanceToNode(this);
	}

	public void deleteDynamicMetadataElementInstanceWithName(String name, int index) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementCantDeleteException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException
	{
		this.getLevel().getDynamicMetadataElement(name).deleteMetadataElementInstanceFromNode(this, index);
	}


	/**
	 * Initialize dynamic metadata elements which are mandatory or alwaysDisplayed with their default value (if defined).
	 * If the elements are repeatable, first insert one element if necessary.
	 * @return
	 */
	public List<MetadataElementInstance> initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed()
	{
		List<MetadataElementInstance> mandatoryOrAlwaysDisplayedInstances = new Vector<MetadataElementInstance>();

		for (LevelMetadataElement lmde: this.getDynamicMetadataElementsOrdered())
		{
			if (lmde.isMandatory() || lmde.isAlwaysDisplayed())
			{
				try
				{
					List<MetadataElementInstance> mdeis = lmde.getMetadataElementInstancesOfNode(this);

					//	If this mandatory or alwaysDisplayed element is repeatable but no instances exist yet, add one instance:
					if (lmde.isRepeatable() && mdeis.isEmpty())		lmde.addMetadataElementInstanceToNode(this);

					for (MetadataElementInstance mdei: mdeis)
					{
						try
						{
							//	Initialize with the default value (if it is defined) only if the existing value is null or empty:
							mdei.initializeValueIfNecessary();
						}
						catch (java.lang.Exception ex)
						{
							ex.printStackTrace();
						}

						mandatoryOrAlwaysDisplayedInstances.add(mdei);
					}
				}
				catch (java.lang.Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}

		return mandatoryOrAlwaysDisplayedInstances;
	}

	public List<MetadataElementInstance> getDynamicMetadataElementInstances()
	{
		List<MetadataElementInstance> allInstances = new Vector<MetadataElementInstance>();

		for (LevelMetadataElement lmde: this.getDynamicMetadataElementsOrdered())
		{
			try
			{
				allInstances.addAll(lmde.getMetadataElementInstancesOfNode(this));
			}
			catch (java.lang.Exception ex)
			{
				ex.printStackTrace();
			}
		}

		return allInstances;
	}

	/**
	 * This method return MDInstances to be displayed: they are either mandatory, or must always be displayed, or they are set.
	 * 		NOTE: Repeatable MDInstances to be displayed are initialized with an empty String if necessary.
	 * @return
	 */
	public List<MetadataElementInstance> getDynamicMetadataElementInstancesToBeDisplayed()
	{
		List<MetadataElementInstance> instancesToBeDisplayed = new Vector<MetadataElementInstance>();

		boolean wasDocumentModified = this.document.isModified();

		for (LevelMetadataElement lmde: this.getDynamicMetadataElementsOrdered())
		{
			try
			{
				//	If this mandatory or alwaysDisplayed element is repeatable but no instances exist yet, add one instance so it gets displayed:
				if ((lmde.isMandatory() || lmde.isAlwaysDisplayed()) && lmde.isRepeatable()
					&& lmde.getMetadataElementInstancesOfNode(this).isEmpty())
						lmde.addMetadataElementInstanceToNode(this);

				for (MetadataElementInstance mdei: lmde.getMetadataElementInstancesOfNode(this))
					if (mdei.isToBeDisplayed())		instancesToBeDisplayed.add(mdei);
			}
			catch (java.lang.Exception ex)
			{
				ex.printStackTrace();
			}
		}

		//	Restore the document's "modified" status if necessary:
		if (this.document.isModified() && !wasDocumentModified)		this.document.setIsNotModified();

		return instancesToBeDisplayed;
	}

	/**
	 * This method is to find out which mandatory fields are not set
	 * @return
	 */
	public List<MetadataElementInstance> getDynamicMetadataElementInstancesWhichAreMandatoryButNotSet()
	{
		List<MetadataElementInstance> mandatoryButNotSetInstances = new Vector<MetadataElementInstance>();

		for (LevelMetadataElement lmde: this.getDynamicMetadataElementsOrdered())
		{
			try
			{
				for (MetadataElementInstance mdei: lmde.getMetadataElementInstancesOfNode(this))
					if (mdei.isMandatoryButNotSet())	mandatoryButNotSetInstances.add(mdei);
			}
			catch (java.lang.Exception ex)
			{
				ex.printStackTrace();
			}
		}

		return mandatoryButNotSetInstances;
	}

	/**
	 * This method is to find out if there are mandatory fields which are not set
	 * @return
	 */
	public boolean hasDynamicMetadataElementInstancesWhichAreMandatoryButNotSet()
	{
		for (LevelMetadataElement lmde: this.getDynamicMetadataElementsOrdered())
		{
			try
			{
				for (MetadataElementInstance mdei: lmde.getMetadataElementInstancesOfNode(this))
					if (mdei.isMandatoryButNotSet())	return true;
			}
			catch (java.lang.Exception ex)
			{
				ex.printStackTrace();
			}
		}

		return false;
	}

	public List<LevelMetadataElement> getDynamicMetadataElementsWhichCanBeAdded()
	{
		List<LevelMetadataElement> elementsWhichCanBeAdded = new Vector<LevelMetadataElement>();

		for (LevelMetadataElement lmde: this.getDynamicMetadataElementsOrdered())
		{
			try
			{
				if (lmde.canAddOneInstanceToNode(this))		elementsWhichCanBeAdded.add(lmde);
			}
			catch (java.lang.Exception ex)
			{
				ex.printStackTrace();
			}
		}

		return elementsWhichCanBeAdded;
	}

	//	----------		Accessing abstract		---------------------------------------------------

	/**
	 * @return the size
	 * The size (in bytes) is defined for files, and calculated for folders.
	 */
	abstract public Long getSize();
	abstract public int getDescendantCount();
	abstract public List<NodeAbstract> getWithDescendants();
	abstract public List<NodeAbstract> getWithDescendantsDepthFirst();
	abstract public int getTreeDepth();
	abstract public String getChecksum();
	abstract public String getChecksumType();
	abstract public String getMimeType();
	abstract public String getFormatKey();
	abstract public String getFormatName();

	//	----------		Inquiring 				---------------------------------------------------

	public boolean fileExists()
	{
		return this.fileExists;
	}

	public boolean canRead()
	{
		return this.canRead;
	}

	public boolean canWrite()
	{
		return this.canWrite;
	}

	public boolean canExecute()
	{
		return this.canExecute;
	}

	public Boolean isRoot()
	{
		return (this.parent == null);
	}


	public void checkIfFileOperationNotAllowed(String fileName, String fileOp) throws FileOperationNotAllowedException
	{
		if (this.document.getDocumentType() == DocumentAbstract.Type.METS
				&& !this.getDocument().areFileOperationsAllowed())
					throw new FileOperationNotAllowedException(this.document, fileName, fileOp);
	}


	public boolean doesParentAllowSubLevel(LevelOfDescription level)
	{
		//	If level is undefined, return true:
		if (level == null)		return true;

		return this.doesParentAllowSubLevel(level.getName());
	}

	public boolean doesParentAllowSubLevel(String levelName)
	{
		//	Allow all sublevels for the root node:
		if (this.isRoot())		return true;

		return ((NodeFolder)this.parent).getLevel().allowsSubLevel(levelName);
	}

	public boolean doesParentAllowMyLevel()
	{
		return this.doesParentAllowSubLevel(this.getLevel());
	}


	public boolean hasPredecessorNotReadableByCurrentUser()
	{
		NodeFolder parent = (NodeFolder)this.parent;
		if (parent == null)								return false;
		if (!parent.fileExists || !parent.canRead)		return true;

		return parent.hasPredecessorNotReadableByCurrentUser();
	}


	public boolean hasPredecessorNotWritableByCurrentUser()
	{
		NodeFolder parent = (NodeFolder)this.parent;
		if (parent == null)								return false;
		if (!parent.fileExists || !parent.canWrite)		return true;

		return parent.hasPredecessorNotWritableByCurrentUser();
	}


	public boolean hasPredecessorNotReadableOrWritableByCurrentUser()
	{
		NodeFolder parent = (NodeFolder)this.parent;
		if (parent == null)													return false;
		if (!parent.fileExists || !parent.canRead || !parent.canWrite)		return true;

		return parent.hasPredecessorNotReadableOrWritableByCurrentUser();
	}


	public boolean doesSubmitStatusAllowEditing()
	{
		return this.getSubmitStatus().isEditingAllowed();
	}

	//	----------		Inquiring abstract		---------------------------------------------------

	abstract public boolean isFile();
	abstract public boolean isFolder();

	abstract public boolean checkFixity();

	abstract public boolean checkAgainstSubmissionAgreement();
	abstract public List<NodeFile> filesNotAllowedBySubmissionAgreement();

	abstract public boolean checkAgainstSubmissionAgreement(SubmissionAgreement sa, String dssId);
	abstract public List<NodeFile> filesNotAllowedBySubmissionAgreement(SubmissionAgreement sa, String dssId);

	abstract public boolean isAllowedBySA();

	abstract public boolean hasMimeType();
	abstract public boolean hasFormatKey();

	//	----------		Searching abstract		---------------------------------------------------

	/**
	 * Return the node with this admId, if it exists in the tree, or null.
	 */
	abstract public NodeAbstract searchId(String admId);

	/**
	 * Return the file node with this fileId, if it exists in the tree, or null.
	 */
	abstract public NodeAbstract searchFileId(String labelfileId);

	/**
	 * Return the (possibly empty) list of nodes with this label.
	 */
	abstract public List<NodeAbstract> searchLabel(String label);

	//	--------		Searching			-------------------------------------------------------

	/**
	 * Search in the metadata for the searchString. Ignore the case.
	 * @param searchString
	 * @return true if the file name, label, level, mime type, format key, format name, or at least one dynamic metadata element contains the search string.
	 */
	public boolean searchFor(String searchString)
	{
		String searchStringLower = searchString.trim().toLowerCase();
		if (searchStringLower.isEmpty())											return false;

		if (	this.getFile().getName().toLowerCase().contains(searchStringLower)
			||	this.getUnitTitle().toLowerCase().contains(searchStringLower)
			||	this.getLevel().getName().toLowerCase().contains(searchStringLower)
			||	(this.getMimeType() != null && this.getMimeType().toLowerCase().contains(searchStringLower))
			||	(this.getFormatKey() != null && this.getFormatKey().toLowerCase().contains(searchStringLower))
			||	(this.getFormatName() != null && this.getFormatName().toLowerCase().contains(searchStringLower)))
																					return true;

		for (MetadataElementInstance mdei: this.getDynamicMetadataElementInstances())
			if (	mdei.getValue() != null
				&&	mdei.getValue().toLowerCase().contains(searchStringLower))		return true;

		return false;
	}


	/**
	 * Search in the metadata for the searchString. Ignore the case. Search for each word separately. Consider quotes.
	 * Perform an AND search, that means: ALL search words must be present, each in any of the searched areas.
	 * @param quotedSearchString
	 * @return true if the file name, label, level, mime type, format key, format name, or at least one dynamic metadata element contains the search string.
	 */
	public boolean searchForAllQuoted(String quotedSearchString)
	{
		//	Return false if and only if one of the quoted search strings is not found in any of the searched areas.

		String searchStringLower = quotedSearchString.trim().toLowerCase();
		if (searchStringLower.isEmpty())													return false;

		for (String searchString: StringUtil.splitQuoted(searchStringLower))
		{
			//	Search the dynamic metadata:
			boolean aMetadataElementContainsSearchString = false;
			for (MetadataElementInstance mdei: this.getDynamicMetadataElementInstances())
				if ((mdei.getValue() != null && mdei.getValue().toLowerCase().contains(searchString)))
				{
					aMetadataElementContainsSearchString = true;
					break;
				}

			//	Search the other areas:
			if (!(		aMetadataElementContainsSearchString
					||	this.getFile().getName().toLowerCase().contains(searchString)
					||	this.getUnitTitle().toLowerCase().contains(searchString)
					||	this.getLevel().getName().toLowerCase().contains(searchString)
					||	(this.getMimeType() != null && this.getMimeType().contains(searchString))
					||	(this.getFormatKey() != null && this.getFormatKey().contains(searchString))
					||	(this.getFormatName() != null && this.getFormatName().contains(searchString))
				))																			return false;	//	Not found: return false
		}

		return true;
	}



	/**
	 * Search in the metadata for the searchString. Ignore the case. Search for each word separately. Consider quotes.
	 * Perform an OR search, that means: AT LEAST ONE of the search words must be present.
	 * @param searchString
	 * @return true if the file name, label, level, mime type, format key, format name, or at least one dynamic metadata element contains the search string.
	 */
	public boolean searchForAnyQuoted(String searchString)
	{
		String searchStringLower = searchString.trim().toLowerCase();
		if (searchStringLower.isEmpty())													return false;

		List<String> searchStrings = StringUtil.splitQuoted(searchStringLower);

		if (	StringUtil.containsAny(this.getFile().getName().toLowerCase(), searchStrings)
			||	StringUtil.containsAny(this.getUnitTitle().toLowerCase(), searchStrings)
			||	StringUtil.containsAny(this.getLevel().getName().toLowerCase(), searchStrings)
			||	(this.getMimeType() != null && StringUtil.containsAny(this.getMimeType().toLowerCase(), searchStrings))
			||	(this.getFormatKey() != null && StringUtil.containsAny(this.getFormatKey().toLowerCase(), searchStrings))
			||	(this.getFormatName() != null && StringUtil.containsAny(this.getFormatName().toLowerCase(), searchStrings)))
																							return true;

		for (MetadataElementInstance mdei: this.getDynamicMetadataElementInstances())
			if (	mdei.getValue() != null
				&&	StringUtil.containsAny(mdei.getValue().toLowerCase(), searchStrings))	return true;

		return false;
	}

	//	--------		Business Ops Abstract	---------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * @throws IOException The IOException is actually thrown in some subclasses, but not in this implementation.
	 * @throws FileOperationNotAllowedException
	 */
	public void rename(String newLabel) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException
	{
		this.checkIfFileOperationNotAllowed(newLabel, "Rename");

		//	Check if I can be renamed:
		NodeFolder parent = (NodeFolder)this.parent;

		if ((parent != null) && parent.doesAlreadyContainChildWithThisName(newLabel))
		{
			throw new FileAlreadyExistsException(newLabel, parent.label);
		}

		String oldLabel = this.getLabel();
		this.setLabel(newLabel);

		//	Create new Event in my DigiprovMD section:
		this.createNewEvent("Renaming", "Renamed from '" + oldLabel + "' to '" + newLabel + "'", "Success", "");
	}


	/**
	 * @throws IOException The IOException is actually thrown in some subclasses, but not in this implementation.
	 * @throws FileOperationNotAllowedException
	 */
	public void moveTo(NodeFolder toFolder) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException
	{
		this.checkIfFileOperationNotAllowed(toFolder.label, "Move");

		//	Check if I can be moved:
		if (toFolder.doesAlreadyContainChildWithThisName(this.label))
		{
			throw new FileAlreadyExistsException(this.label, toFolder.getLabel());
		}

		File file = new File(this.getPathString());
		String oldPath = file.getParent() == null ? "/" : file.getParent();
		
		this.removeFromParent();

		this.parent = toFolder;
		toFolder.add(this);
		toFolder.getElement().add(this.element);

		//	Create new Event in my DigiprovMD section:
		this.createNewEvent("Path Modification", "Moved '" + this.getLabel() + "' from '" + oldPath + "'", "Success", "");
	}


	/**
	 * @throws IOException The IOException is actually thrown in some subclasses, but not in this implementation.
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public void delete() throws IOException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(this.label, "Delete");

		String path = this.getPathString();
		
		this.removeFromParent();

		//	Create new Event in my DigiprovMD section:
		this.getRoot().createNewEvent("Deletion", "Deleted '" + path + "'", "Success", "");

		//	Delete my DMDSections:
		if (this.getMyDMDSectionWithEAD() != null)			this.getMyDMDSectionWithEAD().delete();
		if (this.getMyDMDSectionWithOAI_DC() != null)		this.getMyDMDSectionWithOAI_DC().delete();
		if (this.getMyDigiprov() != null)					this.getMyDigiprov().delete();
	}


	public void addOAI_DCFromDocument(org.dom4j.Document oaiDcDocument)
	{
		OAI_DC oaiDc = OAI_DC.create(this.getDocument(), oaiDcDocument);
		if (oaiDc == null)			return;

		this.setDmdIdOAI_DC(((DMDSectionAbstract)oaiDc.getParent()).getId());
	}

	//	----------		Interface				---------------------------------------------------

	@Override
	public int compareTo(NodeAbstract node)
	{
		return this.compareString().compareTo(node.compareString());
	}


	/** (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.TreeTableNode#getColumnCount()
	 */
	@Override
	@Deprecated
	public int getColumnCount()
	{
		throw new RuntimeException("This method should never be used! It violates the MVC paradigm! It doesn't belong here!");
	}

	/** (non-Javadoc)
	 * @see org.jdesktop.swingx.treetable.TreeTableNode#getValueAt(int)
	 */
	@Override
	@Deprecated
	public Object getValueAt(int column)
	{
		throw new RuntimeException("This method should never be used! It violates the MVC paradigm! It doesn't belong here!");
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeFromParent()
	{
		//	In the case that I am the root node, my parent is null, and super.removeFromParent() won't work:
		if (this.parent != null)		super.removeFromParent();
		this.element.detach();

		this.document.setIsModified();
	}


	/**
	 * Create an associated DMDSectionWithEAD, if none is already present.
	 */
	public void createDMDSectionWithEADIfNecessary()
	{
		if (this.getMyDMDSectionWithEAD() != null)		return;

		DMDSectionWithEAD newDMDSection = new DMDSectionWithEAD(this);
		this.setDmdIdEAD(newDMDSection.getId());
	}

	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	abstract public String toString();
	abstract public String treeString(Integer indent);
	abstract protected String compareString();

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * At this point, the parent is already defined, so I can retrieve the file path:
	 */
	protected void initializeFileAccessRights()
	{
		File file = this.getFile();

		this.fileExists = file.exists();
		this.canRead = file.canRead();
		this.canWrite = file.canWrite();
		this.canExecute = file.canExecute();

		if (!this.fileExists || !this.canRead)		this.document.setIsAtLeastOneFileNotReadable();
		if (!this.canWrite)							this.document.setIsAtLeastOneFileNotWritable();
	}

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * Set the dmdIdOAI_DC value. If NO appropriate METS:div section exists, create it.
	 */
	protected void setDmdIdOAI_DC(String dmdIdOAI_DC)
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		this.dmdIdOAI_DC = dmdIdOAI_DC;

		//	Does a METS:div element exist with TYPE="metadata" and LABEL="OAI_DC"? If no, create it:
		Element metsDivOAI_DCLinkToDMD = (Element)this.element.selectSingleNode("./METS:div[@TYPE='metadata' and @LABEL='OAI_DC']");
		if (metsDivOAI_DCLinkToDMD == null)
			metsDivOAI_DCLinkToDMD = this.element.addElement("METS:div").addAttribute("LABEL", "OAI_DC").addAttribute("TYPE", "metadata");

		metsDivOAI_DCLinkToDMD.addAttribute("DMDID", this.dmdIdOAI_DC);

		this.document.setIsModified();
	}

	/**
	 * Set the dmdIdOAI_DC value. If NO appropriate METS:div section exists, create it.
	 */
	protected void setDmdIdEAD(String dmdIdEAD)
	{
		if (!this.doesSubmitStatusAllowEditing())		return;

		this.dmdIdEAD = dmdIdEAD;

		//	Does a METS:div element exist with TYPE="metadata" and LABEL="EAD"? If no, create it:
		Element metsDivEADLinkToDMD = (Element)this.element.selectSingleNode("./METS:div[@TYPE='metadata' and @LABEL='EAD']");
		if (metsDivEADLinkToDMD == null)
			metsDivEADLinkToDMD = this.element.addElement("METS:div").addAttribute("LABEL", "EAD").addAttribute("TYPE", "metadata");

		metsDivEADLinkToDMD.addAttribute("DMDID", this.dmdIdEAD);

		this.document.setIsModified();
	}

	//	----------		Accessing calculated	---------------------------------------------------

	protected List<LevelMetadataElement> getDynamicMetadataElementsOrdered()
	{
		return this.getLevel().getDynamicMetadataElementsOrdered();
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	protected void createNewEvent(String type, String detail, String outcome, String outcomeDetail)
	{
		this.getMyDigiprov().addNewEvent(type, detail, outcome, outcomeDetail);
	}

	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------
	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	public enum SubmitStatus
	{
		SubmitUndefined, SubmitRequested, SubmitRequestPending, Submitted, SubmitFailed;


		/**
		 * Return the SubmitStatus with this name or, if key is null or empty, "Undefined".
		 * Throw IllegalArgumentException if key is not an existing SubmitStatus name.
		 * @param key
		 * @return
		 */
		static public SubmitStatus named(String key)
		{
			if (key == null || key.isEmpty())		return SubmitUndefined;

			return valueOf(key);
		}


		/**
		 * Return the String that will be the content of the lock file when the Document is "SubmitRequestPending".
		 * @return
		 */
		static public String getSubmitRequestPendingLockId()
		{
			return OperatingSystem.userName() + " (" + SubmitRequestPending + ")";
		}

		/**
		 * Return the String that will be the content of the lock file when the Document is "Submitted".
		 * @return
		 */
		static public String getSubmittedLockId()
		{
			return OperatingSystem.userName() + " (" + Submitted + ")";
		}


		public boolean isEditingAllowed()
		{
			return this == SubmitUndefined
				|| this == SubmitFailed;
		}


		public boolean isNextSubmitStatusAllowed(SubmitStatus nextSubmitStatus)
		{
			switch (this)
			{
				case SubmitUndefined:
				{
					return nextSubmitStatus == SubmitUndefined
						|| nextSubmitStatus == SubmitRequested
						|| nextSubmitStatus == SubmitRequestPending;	//	NOTE: This is to enable request + submit in one step
				}
				case SubmitRequested:
				{
					return nextSubmitStatus == null
						|| nextSubmitStatus == SubmitUndefined
						|| nextSubmitStatus == SubmitRequested
						|| nextSubmitStatus == SubmitRequestPending;
				}
				case SubmitRequestPending:
				{
					return nextSubmitStatus == Submitted
						|| nextSubmitStatus == SubmitFailed;
				}
				case Submitted:
				{
					return false;
				}
				case SubmitFailed:
				{
					return nextSubmitStatus == null
						|| nextSubmitStatus == SubmitUndefined
						|| nextSubmitStatus == SubmitRequested;
				}
			}

			return false;
		}
	}

}
