/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.structmap;

import java.io.*;
import java.util.*;

import org.dom4j.Element;

import ch.docuteam.darc.exceptions.FileAlreadyExistsException;
import ch.docuteam.darc.exceptions.FileOperationNotAllowedException;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.amdsec.DigiprovWithPremis;
import ch.docuteam.darc.premis.Event;
import ch.docuteam.darc.premis.Object;
import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.tools.exception.Exception;
import ch.docuteam.tools.file.FileChecksumCalculator;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.id.UniqueID;
import ch.docuteam.tools.out.Logger;

/**
 * This class, used by the class <a href="./StructureMap.html">StructureMap</a>, represents a file within the StructMap.
 * <p>
 * This class inserts Exceptions into the <a href="../../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Replacing a file, when the replacement file is not present</li>
 * <li>Replacing a file, when an existing file would be overwritten</li>
 * <li>Replacing a file, when a file operation fails</li>
 * <li>Fixity check, when the checksum verification fails</li>
 * <li>Fixity check, when the checksum verification throws an exception</li>
 * <li>In the constructor, when the corresponding file is not present</li>
 * </ul>
 *
 * @author denis
 */
public class NodeFile extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Final Static Public		=======================================================

	//	========	Final Static Private	=======================================================

	static final protected String	Type		= "file";
	static final protected String	RootType	= "rootfile";

	static final protected String	DivLabel	= "Content";
	static final protected String	DivType		= "content";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	/**
	 * The file id
	 */
	protected String				fileId;

	/**
	 * The file size (retrieved from the file system)
	 */
	protected Long					size		= 0L;

	protected boolean				isAllowedBySA = true;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================


	/**
	 * This constructor is used only when a METS-File is being read, and for the root node only.
	 */
	protected NodeFile(StructureMap parent, Element element)
	{
		super(parent.getDocument(), element);
		this.initialize();
	}


	/**
	 * This constructor is used only when a METS-File is being read, and for non-root nodes.
	 *
	 * @param parent
	 * @param element
	 */
	protected NodeFile(NodeFolder parent, Element element)
	{
		super(parent.getDocument(), element);
		parent.add(this);
		this.initialize();
	}


	/**
	 * This constructor is used when a new file is created programmatically, and when a new Digiprov has to be created for that file.
	 *
	 * @param parent
	 *            The parent
	 * @param label
	 *            The new file name
	 */
	protected NodeFile(NodeFolder parent, ch.docuteam.darc.mets.filesec.File file)
	{
		this(parent, file, null);
		this.initializeFileAccessRights();
	}


	/**
	 * This constructor is used when a new NON-root file is created programmatically.
	 * If reuseExistingAdmId is undefined, create a new AMDSectionDigiprov with my admId, including a new PREMIS Object and a new PREMIS Event.
	 *
	 * @param parent The parent
	 * @param label The new file name
	 */
	protected NodeFile(NodeFolder parent, ch.docuteam.darc.mets.filesec.File file, String reuseExistingAdmId)
	{
		super(parent, file.getFile().getName());
		if (reuseExistingAdmId != null)		this.admId = reuseExistingAdmId; // If defined, override the admId set in the super constructor!
		this.allowsChildren = false;
		this.type = Type;
		this.fileId = file.getId();
		this.size = file.getFile().length();

		this.element = ((NodeFolder)this.parent).getElement()
			.addElement("METS:div")
				.addAttribute("LABEL", this.label)
				.addAttribute("TYPE", this.type)
				.addAttribute("ADMID", this.admId);

		Element intermediaryDiv = this.element
			.addElement("METS:div")
				.addAttribute("LABEL", DivLabel)
				.addAttribute("TYPE", DivType);
		intermediaryDiv
			.addElement("METS:fptr")
				.addAttribute("FILEID",this.fileId);

		// if reuseExistingAdmId is undefined, create a new AMDSectionDigiprov with my admId, including a new PREMIS Object and a new PREMIS Event:
		if (reuseExistingAdmId == null)		new DigiprovWithPremis(this);

		//	Insert associated DMDSectionWithEAD:
		this.createDMDSectionWithEADIfNecessary();

		//	After the DMDSectionWithEAD was created, set my level to my parent's default sublevel:
		try
		{
			this.setLevel(parent.getLevel().getDefaultSubLevel());
		}
		catch (java.lang.Exception e)
		{
			e.printStackTrace();
		}

		//	Only now, when the DMDSectionWithEAD is present for sure, I can initialize my dynamic metadata elements:
		this.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();
	}


	/**
	 * This constructor is used when a new ROOT file is created programmatically.
	 *
	 * @param parent
	 *            The parent
	 * @param label
	 *            The new file name
	 */
	protected NodeFile(StructureMap parent, ch.docuteam.darc.mets.filesec.File file)
	{
		super(parent, file.getFile().getName());
		this.allowsChildren = false;
		this.type = RootType;
		this.fileId = file.getId();
		this.size = file.getFile().length();

		this.element = parent.getElement()
			.addElement("METS:div")
				.addAttribute("LABEL", this.label)
				.addAttribute("TYPE", this.type)
				.addAttribute("ADMID", this.admId);

		Element intermediaryDiv = this.element
			.addElement("METS:div")
				.addAttribute("LABEL", DivLabel)
				.addAttribute("TYPE", DivType);
		intermediaryDiv
			.addElement("METS:fptr")
				.addAttribute("FILEID", this.fileId);

		new DigiprovWithPremis(this);

		//	Insert associated DMDSectionWithEAD:
		this.createDMDSectionWithEADIfNecessary();

		//	Only now, when the DMDSectionWithEAD is present for sure, I can initialize my dynamic metadata elements:
		this.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getFileId()
	{
		return this.fileId;
	}

	@Override
	public String getChecksum()
	{
		return this.getMyLastDigiprovObject().getMessageDigest();
	}

	@Override
	public String getChecksumType()
	{
		return this.getMyLastDigiprovObject().getMessageDigestAlgorithm();
	}

	@Override
	public String getMimeType()
	{
		return this.getMyFileSectionFile().getMimeType();
	}

	@Override
	public String getFormatKey()
	{
		return this.getMyLastDigiprovObject().getFormatRegistryKey();
	}

	@Override
	public String getFormatName()
	{
		return this.getMyLastDigiprovObject().getFormatName();
	}

	@Override
	public Long getSize()
	{
		return this.size;
	}

	/**
	 * @return The total number of children & sub-children (for files always 0)
	 */
	@Override
	public int getDescendantCount()
	{
		return 0;
	}


	/**
	 * @return A list containing me and all my descendants (for a leaf: only me)
	 */
	@Override
	public List<NodeAbstract> getWithDescendants()
	{
		List<NodeAbstract> all = new ArrayList<NodeAbstract>();
		all.add(this);
		return all;
	}


	/**
	 * @return A list containing me and all my descendants (for a leaf: only me)
	 */
	@Override
	public List<NodeAbstract> getWithDescendantsDepthFirst()
	{
		return this.getWithDescendants();
	}


	/**
	 * @return The depth of my subtree is 0 for a leaf.
	 */
	@Override
	public int getTreeDepth()
	{
		return 0;
	}


	public void setIsNotAllowedBySA()
	{
		this.isAllowedBySA = false;
	}

	//	----------		Inquiring				---------------------------------------------------

	@Override
	public boolean isFile()
	{
		return true;
	}

	@Override
	public boolean isFolder()
	{
		return false;
	}


	@Override
	public boolean checkFixity()
	{
		String filePath = this.getAbsolutePathString();
		Logger.getLogger().info("Fixity Check: " + filePath + "... ");

		String docType = ((Document)this.document).getType();

		try
		{
			if (FileChecksumCalculator.getMD5(filePath).equalsIgnoreCase(this.getChecksum()))
			{
				this.createNewEvent("Fixity Check", "Based on " + docType, "Success", "");
			}
			else
			{
				this.createNewEvent("Fixity Check", "Based on " + docType, "Failure", "Calculated checksum differs from stored checksum");

				Logger.getLogger().warn("Checksum verification failed on file: " + filePath);
				Exception.remember("Checksum verification failed on file: " + filePath);
				return false;
			}
		}
		catch (FileNotFoundException e)
		{
			this.createNewEvent("Fixity Check", "Based on " + docType, "Failure", e.getMessage());

			Logger.getLogger().error(e);
			Exception.remember(e);
			return false;
		}
		catch (IOException e)
		{
			this.createNewEvent("Fixity Check", "Based on " + docType, "Failure", e.getMessage());

			Logger.getLogger().error(e);
			Exception.remember(e);
			return false;
		}

		Logger.getLogger().info("Checksum OK for file: " + filePath);
		return true;
	}


	@Override
	public boolean checkAgainstSubmissionAgreement()
	{
		return this.checkAgainstSubmissionAgreement(((Document)this.document).getSubmissionAgreement(), ((Document)this.document).getDSSId());
	}

	@Override
	public List<NodeFile> filesNotAllowedBySubmissionAgreement()
	{
		return this.filesNotAllowedBySubmissionAgreement(((Document)this.document).getSubmissionAgreement(), ((Document)this.document).getDSSId());
	}


	@Override
	public boolean checkAgainstSubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		return sa.allowsFileType(dssId, this.getAbsolutePathString(), this.getFormatKey(), this.getMimeType());
	}

	@Override
	public List<NodeFile> filesNotAllowedBySubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		List<NodeFile> result = new Vector<NodeFile>();
		if (!this.checkAgainstSubmissionAgreement(sa, dssId))		result.add(this);

		return result;
	}

	@Override
	public boolean isAllowedBySA()
	{
		return this.isAllowedBySA;
	}


	@Override
	public boolean hasMimeType()
	{
		String mimeType = this.getMimeType();
		return mimeType == null || mimeType.isEmpty();
	}


	@Override
	public boolean hasFormatKey()
	{
		String formatKey = this.getFormatKey();
		return formatKey == null || formatKey.isEmpty();
	}

	//	----------		Searching 			-------------------------------------------------------

	/**
	 * Return this node, if it has this amdId, or null if not.
	 */
	@Override
	public NodeAbstract searchId(String admId)
	{
		return ((this.admId.equals(admId)) ? this : null);
	}

	/**
	 * Return this node, if it is the one searched for, or null if not.
	 */
	@Override
	public List<NodeAbstract> searchLabel(String label)
	{
		List<NodeAbstract> found = new ArrayList<NodeAbstract>(1);
		if (this.label.equals(label))		found.add(this);

		return found;
	}

	/**
	 * Return this node, if it has this fileId, or null if not.
	 */
	@Override
	public NodeAbstract searchFileId(String fileId)
	{
		return ((this.fileId.equals(fileId)) ? this : null);
	}

	//	----------		Business Ops			---------------------------------------------------

	/**
	 * Change the name of this file, then update the FileSection and the
	 * corresponding file in the file system.
	 * @throws FileAlreadyExistsException
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 *
	 * @throws java.lang.Exception
	 *             In case that a folder or file with this name already exists
	 *             in the parent folder
	 */
	@Override
	public void rename(String label) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException
	{
		this.checkIfFileOperationNotAllowed(label, "Rename");

		String oldAbsolutePath = this.getAbsolutePathString();
		super.rename(label);
		String newAbsolutePath = this.getAbsolutePathString();

		// Change corresponding FileSectionFile:
		this.getMyFileSectionFile().rename(this.getPathString());

		// Change corresponding file in file system:
		FileUtil.renameTo(oldAbsolutePath, newAbsolutePath);
	}


	/**
	 * Move this file to another folder, then update the FileSection and the
	 * corresponding file in the file system.
	 * @throws FileAlreadyExistsException
	 *
	 * @throws java.lang.IOException
	 * @throws FileOperationNotAllowedException
	 * @throws java.lang.Exception
	 */
	@Override
	public void moveTo(NodeFolder toFolder) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException
	{
		this.checkIfFileOperationNotAllowed(toFolder.label, "Move");

		String oldAbsolutePath = this.getAbsolutePathString();
		super.moveTo(toFolder);
		String newAbsolutePath = this.getAbsolutePathString();

		// Update corresponding File in the FileSection:
		this.getMyFileSectionFile().moveTo(toFolder);

		// Move corresponding file in the file system:
		FileUtil.renameTo(oldAbsolutePath, newAbsolutePath);
	}


	/**
	 * Delete this file, then update the FileSection and then finally delete the
	 * corresponding file in the file system.
	 *
	 * @throws IOException
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	@Override
	public void delete() throws IOException, FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(this.label, "Delete");

		String oldAbsolutePath = this.getAbsolutePathString();
		super.delete();

		// Delete corresponding FileSectionFile from the FileSection:
		this.getMyFileSectionFile().delete();

		// Delete corresponding file in the file system:
		FileUtil.delete(oldAbsolutePath);
	}


	public void migrateToFileKeepOriginal(String filePath) throws FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.migrateToFileKeepOriginal(filePath, null);
	}


	/**
	 * Replace in this node the current file by the new one, but keep the current one.
	 * This works even when the current and new file have equal names.
	 * @param newFilePath
	 * @param migrationToolName
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public void migrateToFileKeepOriginal(String filePath, String migrationToolName) throws FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(filePath, "MigrateToFileKeepOriginal");

		File newFile = new File(filePath);

		// Does file exist? If no: exception!
		if (!newFile.exists())
		{
			Exception.remember("Migration file does not exist: " + filePath);
			return;
		}

		if (this.isRoot())
		{
			try
			{
				//	Create a new root folder and place me into it:
				((Document)this.document).getStructureMap().moveRootFileToNewRootFolder(this);
			}
			catch(java.lang.Exception x)
			{
				Exception.remember(x);
				return;
			}
		}

		String newFileName = newFile.getName();
		try
		{
			File oldFile = this.getFile();

			if (new File(oldFile.getParent() + "/" + newFileName).exists())
			{
				//	In the case that the current and the new file have equals names, add to the new file name a unique identifier:
				int lastDot = newFileName.lastIndexOf(".");
				newFileName = newFileName.substring(0, lastDot) + UniqueID.getXML() + newFileName.substring(lastDot);
			}

			//	... Add the new file to the parent folder:
			FileUtil.copyToOverwriting(newFile.getPath(), oldFile.getParent() + "/" + newFileName);
		}
		catch (IOException x)
		{
			Exception.remember(x);
			return;
		}

		String newRelativeFilePath = ((NodeFolder)this.getParent()).getPathString() + "/" + newFileName;

		// FileSection:
		// Create new file section file:
		ch.docuteam.darc.mets.filesec.File newFileSectionFile = new ch.docuteam.darc.mets.filesec.File(this.document, newRelativeFilePath);

		// StructMap:
		// Create a new NodeFile with its FILEID taken from newFileSectionFile.
		// This will add the new file as a new child to my parent and create a new Digiprov section with a new corresponding PREMIS object and PREMIS event:
		NodeFile newStructMapFile = new NodeFile((NodeFolder)this.getParent(), newFileSectionFile);

		// AMDSection:
		// This new event links to the ORIGINAL Object:
		String eventType =
			(migrationToolName == null)
				? "File '" + this.label + "' migrated to file '" + newFileName + "'"
				: "Application:" + migrationToolName + " - Source:" + this.label + " - Target:" + newFileName;
		Event newMigrationEvent = this.getMyDigiprov().addNewEvent("Migration", eventType, "Success", "");

		Object newLastDigiprovObject = newStructMapFile.getMyLastDigiprovObject();
		//	Set the NEW object's relationships to the new migration event and the original object:
		newLastDigiprovObject.addRelationships(newMigrationEvent);
		//	Set the NEW object's original name to the original object's label:
		newLastDigiprovObject.setOriginalName(this.label);

		Object originalLastDigiprovObject = this.getMyLastDigiprovObject();
		//	Set the ORIGINAL object's relationships to the new migration event:
		originalLastDigiprovObject.addRelationships(newMigrationEvent);
		//	Set the ORIGINAL object's object relationship to the NEW object:
		originalLastDigiprovObject.setRelatedObject(newLastDigiprovObject, "source", "source for");
	}


	/**
	 * Replace in this node the current file by the new one.
	 * This works even when the current and new file have equal names.
	 * @param newFilePath
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public void migrateToFile(String filePath) throws FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.migrateToFile(filePath, null);
	}


	/**
	 * Replace in this node the current file by the new one.
	 * This works even when the current and new file have equal names.
	 * @param newFilePath
	 * @param migrationToolName
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public void migrateToFile(String newFilePath, String migrationToolName) throws FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(newFilePath, "MigrateToFile");

		File newFile = new File(newFilePath);

		// Does file exist? If no: exception!
		if (!newFile.exists())
		{
			Exception.remember("Migration file does not exist: " + newFilePath);
			return;
		}

		String oldLabel = this.label;
		this.replaceByFile(newFile);

		// AMDSection:
		// These new events link to the ORIGINAL Object:
		String eventType =
			(migrationToolName == null)
				? "File '" + oldLabel + "' migrated to file '" + this.label + "'"
				: "Application:" + migrationToolName + " - Source:" + oldLabel + " - Target:" + this.label;
		Event newMigrationEvent = this.getMyDigiprov().addNewEvent("Migration", eventType, "Success", "");

		// This new object has a relationship to the ORIGINAL Object and the previously created migration event:
		Object newMigrationObject = this.getMyDigiprov().addNewObjectWithRelationship(this, newMigrationEvent);
		// Set the original name to the OLD label:
		newMigrationObject.setOriginalName(oldLabel);
	}


	/**
	 * Replace in this node the current file by the new one.
	 * This works even when the current and new file have equal names.
	 * The difference between this method and migrateToFile() are only different events.
	 * @param newFilePath
	 * @throws FileOperationNotAllowedException
	 * @throws FileUtilExceptionListException
	 */
	public void replaceByFile(String newFilePath) throws FileOperationNotAllowedException, FileUtilExceptionListException
	{
		this.checkIfFileOperationNotAllowed(newFilePath, "ReplaceByFile");

		File newFile = new File(newFilePath);

		// Does file exist? If no: exception!
		if (!newFile.exists())
		{
			Exception.remember("Replacement file does not exist: " + newFilePath);
			return;
		}

		String oldLabel = this.label;
		this.replaceByFile(newFile);

		// AMDSection:
		// These new events link to the ORIGINAL Object:
		Event newMigrationEvent = this.getMyDigiprov().addNewEvent("Replacement", "Replaced file '" + oldLabel + "' by file '" + this.label + "'", "Success", "");

		// This new object has a relationship to the ORIGINAL Object and the previously created migration event:
		Object newMigrationObject = this.getMyDigiprov().addNewObjectWithRelationship(this, newMigrationEvent);
		// Set the original name to the OLD label:
		newMigrationObject.setOriginalName(oldLabel);
	}

	//	--------		Interface			-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return new StringBuilder("[StructureMapFile:")
				.append(this.label)
				.append("[")
				.append(this.getLevel())
				.append("](admId=")
				.append(this.admId)
				.append("/dmdIdOAI=")
				.append(this.dmdIdOAI_DC)
				.append("/dmdIdEAD=")
				.append(this.dmdIdEAD)
				.append("/fileId=")
				.append(this.fileId)
				.append(")(")
				.append(this.size)
				.append(")")
				.append(this.fileExists? "F": "f")
				.append(this.canRead? "R": "r")
				.append(this.canWrite? "W": "w")
				.append(this.canExecute? "X": "x")
				.append("^")
				.append(this.hasPredecessorNotReadableByCurrentUser()? "r": "R")
				.append(this.hasPredecessorNotWritableByCurrentUser()? "w": "W")
				.append("v--")
				.append("(")
				.append(this.getSubmitStatus())
				.append(")]")
			.toString();
	}

	/**
	 * @return A string describing this file, but indented to show the tree
	 *         structure
	 */
	@Override
	public String treeString(Integer indent)
	{
		StringBuilder buf = new StringBuilder(50);
		buf.append("\n");
		for (int i = 0; i < indent; i++)		buf.append("\t");
		buf.append(this.toString());

		return buf.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initialization, common for all constructors
	 */
	protected void initialize()
	{
		this.initializeFileAccessRights();
		this.allowsChildren = false;

		this.fileId = ((Element)this.element
				.selectSingleNode("./METS:div/METS:fptr"))
				.attributeValue("FILEID");

		this.size = this.getFile().length();
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * Replace in this node the current file by the new one.
	 * This works even when the current and new file have equal names.
	 * Don't create any PREMIS events or objects.
	 * @param newFile
	 * @throws FileUtilExceptionListException
	 */
	private void replaceByFile(File newFile) throws FileUtilExceptionListException
	{
		Logger.getLogger().debug("Replacing file in node: '" + this + "' by file: '" + newFile + "'");

		String newFileName = newFile.getName();
		try
		{
			File oldFile = this.getFile();

			//	Delete my old file, it will be replaced by the new one:
			FileUtil.delete(oldFile);

			//	Now it can happen that another file with the same name as the new file already exists in the destination folder:
			if (new File(oldFile.getParent() + "/" + newFileName).exists())
			{
				//	In this case, add a unique identifier to the new file name:
				int lastDot = newFileName.lastIndexOf(".");
				newFileName = newFileName.substring(0, lastDot) + UniqueID.getXML() + newFileName.substring(lastDot);
			}

			//	... Add the new file to the parent folder:
			FileUtil.copyToOverwriting(newFile.getPath(), oldFile.getParent() + "/" + newFileName);
		}
		catch (IOException x)
		{
			Exception.remember(x);
			return;
		}

		this.setLabel(newFileName);
		this.size = newFile.length();

		// FileSection:
		this.getMyFileSectionFile().replaceFile(this.label);
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	private ch.docuteam.darc.mets.filesec.File getMyFileSectionFile()
	{
		return ((Document)this.document).getFileSection().getFile(this.fileId);
	}

	/**
	 *
	 * @return The last AMDSectionDigiprovObject for this File
	 */
	private Object getMyLastDigiprovObject()
	{
		return this.getMyDigiprov().getLastObject();
	}

	//	--------		Utilities			-------------------------------------------------------

	/**
	 * When sorting Nodes, all Files come behind Folders. Then sort files alphabetically.
	 */
	@Override
	protected String compareString()
	{
		return "2" + this.getLabel();
	}

	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
