/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.packer.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FilenameFilter;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.SwingWorker;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileView;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.io.FileUtils;

import ch.docuteam.converter.FileConverter;
import ch.docuteam.darc.mdconfig.LevelOfDescription;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.Document.Mode;
import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.darc.sa.SubmissionAgreement.Overview;
import ch.docuteam.packer.gui.FilePreviewer.FilePreviewPanel;
import ch.docuteam.tools.exception.ExceptionCollector;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.PropertyFile;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.gui.GUIUtil;
import ch.docuteam.tools.gui.GridBagPanel;
import ch.docuteam.tools.gui.MultiLineLabel;
import ch.docuteam.tools.gui.ScrollableMessageDialog;
import ch.docuteam.tools.gui.SmallPeskyMessageWindow;
import ch.docuteam.tools.gui.SystemOutView;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.DateFormatter;
import ch.docuteam.tools.translations.I18N;

/**
 * @author denis
 *
 */
public class LauncherView extends JFrame
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static protected final String			DefaultPropertyFileName = "./config/docuteamPacker.properties";
	static protected final String			DefaultLevelsConfigFileName = "./config/levels.xml";

	static protected final Dimension		DefaultScreenSizeWithoutWorkspaceManager = new Dimension(800, 120);
	static protected final Dimension		DefaultScreenSizeWithWorkspaceManager = new Dimension(900, 300);
	static protected final Point			ScreenPosition = new Point(20, 20);
	static protected final boolean			DoInitiallyOpenWorkspaceManager = true;

	static protected final String			PropertyFilePathOSSuffix = OperatingSystem.isWindows()? ".Win": (OperatingSystem.isMacOSX()? ".OSX": ".Linux");

	//	========	Static Public			=======================================================

	//	These fields should be overwritten in subclasses, e.g. in a static initializer:

	static public String					VersionProduct = ch.docuteam.packer.admin.Version.Product;
	static public String					VersionNumber = ch.docuteam.packer.admin.Version.Number;
	static public String					VersionLastChange = ch.docuteam.packer.admin.Version.LastChange;

	static public String					Operator = "docuteam packer";

	//	The following static fields contain default values - they might be overwritten when reading the property file.
	//	In addition, some of these fields can be overwritten using command line parameters:

	static public String					LevelsConfigFileName = DefaultLevelsConfigFileName;

	static public String					SIPDirectory = OperatingSystem.userHome();
	static public String					DataDirectory = SIPDirectory;
	static public String					TemplateDirectory = SIPDirectory;
	static public String					IngestFeedbackDirectory = null;

	static public Boolean					UseSystemLAF = true;
	static public Boolean					IsDevelopingMode = false;
	static public Boolean					NewSIPDefaultsToZipped = true;

	//	========	Static Private			=======================================================

	static public String					LastUsedOpenOrSaveDirectory = SIPDirectory;

	//	Visual:

	static protected LauncherView			LauncherView;

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	//	BO:

	protected File							selectedSIP;
	protected List<SIPView>					openedSIPViews = new Vector<SIPView>();
	protected List<String>					sipsWithSavingInProgress = new Vector<String>();

	//	Actions:

	//	General actions:
	protected Action						aboutAction;
	protected Action						openExceptionWindowAction;
	protected Action						openDocuteamHomepageAction;
	protected Action						helpAction;
	protected Action						quitAction;

	//	SA actions:
	protected Action						updateSAsFromServerAction;

	//	SIP actions:
	protected Action						createNewSIPAction;
	protected Action						createNewSIPFromTemplateAction;
	protected Action						openSIPAction;
	protected Action						openSIPReadWriteNoFileOpsAction;
	protected Action						openSIPReadOnlyAction;

	//	Workspace actions:
	protected Action						showWorkspaceManagerAction;
	protected Action						hideWorkspaceManagerAction;

	protected Action						rereadWorkspaceFolderAction;
	protected Action						selectWorkspaceFolderAction;
	protected Action						searchWorkspaceAction;

	protected Action						checkIngestFeedbackAction;

	protected Action						openSIPInWorkspaceAction;
	protected Action						openSIPInWorkspaceReadWriteNoFileOpsAction;
	protected Action						openSIPInWorkspaceReadOnlyAction;
	protected Action						renameSIPInWorkspaceAction;
	protected Action						copySIPInWorkspaceAction;
	protected Action						deleteSIPInWorkspaceAction;

	//	Visual:

	protected JPopupMenu					popupMenu;
	protected JMenuBar						menuBar;
	protected JMenu							programMenu;
	protected JMenu							workspaceMenu;
	protected JMenu							sipMenu;
	protected JMenu							saMenu;
	protected JMenu							windowMenu;

	protected JPanel						northPanel;
	protected JPanel						workspaceManagerPanel;

	protected JButton						showWorkspaceManagerButton;
	protected JButton						hideWorkspaceManagerButton;

	protected JTable						sipTable;

	protected JTextField					footerTextField;

	//	The following fields contain default values - they will be overwritten when the user resizes the view.

	protected Dimension						ScreenSizeWithoutWorkspaceManager = DefaultScreenSizeWithoutWorkspaceManager;
	protected Dimension						ScreenSizeWithWorkspaceManager = DefaultScreenSizeWithWorkspaceManager;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * @param args
	 */
	public static void main(String[] args)
	{
		//	Create shudown hook:
		Runtime.getRuntime().addShutdownHook(new Thread(){ @Override public void run() { shutdown(); }});

		//	Initialize:
		CommandLine commandLine = getCommandLine(args);
		initialize(commandLine);

		//	Create and open the launcher:
		LauncherView = new LauncherView();
		LauncherView.setVisible(true);

		//	Check ingest feedback on startup:
		LauncherView.checkIngestFeedback(false);

		//	Open a SIP on startup (if specified in the commandLine):
		LauncherView.openSIP(commandLine);
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors protected	=======================================================

	protected LauncherView()
	{
		super(SIPDirectory);
		this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);		//	Because the quit event is handled using a windowListener below
		this.addWindowListener(
				new WindowAdapter() { @Override public void windowClosing(WindowEvent e) { LauncherView.this.requestQuit(); }});
		this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));

		//	Tables:

		//	Right-align numbers in "size" and "last modified" column:
		DefaultTableCellRenderer rightAlignmentRenderer = new DefaultTableCellRenderer();
		rightAlignmentRenderer.setHorizontalAlignment(DefaultTableCellRenderer.RIGHT);

		this.sipTable = new JTable(new SIPTableModel());
		this.sipTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		this.sipTable.getSelectionModel().addListSelectionListener(
				new ListSelectionListener() { @Override public void valueChanged(ListSelectionEvent e){ if (!e.getValueIsAdjusting()) LauncherView.this.listSelectionChanged(e); }});
		this.sipTable.addMouseListener(
				new MouseAdapter() { @Override public void mousePressed(MouseEvent e){ LauncherView.this.listSelectionWasClicked(e); }});
		this.sipTable.getColumnModel().getColumn(1).setMinWidth(50);
		this.sipTable.getColumnModel().getColumn(1).setPreferredWidth(150);
		this.sipTable.getColumnModel().getColumn(1).setMaxWidth(500);
		this.sipTable.getColumnModel().getColumn(2).setMinWidth(80);
		this.sipTable.getColumnModel().getColumn(2).setMaxWidth(80);
		this.sipTable.getColumnModel().getColumn(2).setCellRenderer(rightAlignmentRenderer);
		this.sipTable.getColumnModel().getColumn(3).setMinWidth(150);
		this.sipTable.getColumnModel().getColumn(3).setMaxWidth(150);
		this.sipTable.getColumnModel().getColumn(3).setCellRenderer(new DateTableCellRenderer());
		this.sipTable.getColumnModel().getColumn(4).setMinWidth(40);
		this.sipTable.getColumnModel().getColumn(4).setMaxWidth(40);

		//	Make sipTable sortable:
		this.sipTable.setAutoCreateRowSorter(true);

		//	Text Fields:

		this.footerTextField = new JTextField();
		this.footerTextField.setEditable(false);

		//	Actions:

		//	General actions:
		this.aboutAction = new AbstractAction(I18N.translate("ActionAbout"), new ImageIcon("./resources/images/Info.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openAboutWindow(); }};
		this.aboutAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipAbout"));
		this.aboutAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_I, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));

		this.openExceptionWindowAction = new AbstractAction(I18N.translate("ActionOpenExceptionWindow"), new ImageIcon("./resources/images/Exception.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openExceptionWindow(); }};
		this.openExceptionWindowAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenExceptionWindow"));

		this.openDocuteamHomepageAction = new AbstractAction(I18N.translate("ActionOpenDocuteamHomepage"), new ImageIcon("./resources/images/Home.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openDocuteamHomepage(); }};
		this.openDocuteamHomepageAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenDocuteamHomepage"));

		this.helpAction = new AbstractAction(I18N.translate("ActionHelp"), new ImageIcon("./resources/images/Help.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openHelpPage(); }};
		this.helpAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipHelp"));
		this.helpAction.putValue(Action.ACCELERATOR_KEY, getKeyStroke(KeyEvent.VK_H, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask(), KeyEvent.VK_F1, 0));

		this.quitAction = new AbstractAction(I18N.translate("ActionQuit"), new ImageIcon("./resources/images/Quit.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.requestQuit(); }};
		this.quitAction.putValue(Action.ACCELERATOR_KEY, getKeyStroke(KeyEvent.VK_Q, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask(), KeyEvent.VK_F4, InputEvent.ALT_DOWN_MASK));
		this.quitAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipQuit"));

		//	Use OSXAdapter to handle cmd-q and the OS X "About" menu entry:
		try
		{
			ch.docuteam.tools.os.OSXAdapter.setQuitHandler(this, this.getClass().getMethod("requestQuit"));
			ch.docuteam.tools.os.OSXAdapter.setAboutHandler(this, this.getClass().getMethod("openAboutWindow"));
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}

		//	SA actions:
		this.updateSAsFromServerAction = new AbstractAction(I18N.translate("ActionLoadSAsFromServer"), new ImageIcon("./resources/images/Download.png"))
		{	@Override public void actionPerformed(ActionEvent e){ updateSAsFromServer(); }};
		this.updateSAsFromServerAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipLoadSAsFromServer"));

		//	Workspace actions:
		this.showWorkspaceManagerAction = new AbstractAction(I18N.translate("ActionShowWorkspaceManager"), new ImageIcon("./resources/images/WorkspaceManagerShow.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.showWorkspaceManager(true); }};
		this.showWorkspaceManagerAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipShowWorkspaceManager"));

		this.hideWorkspaceManagerAction = new AbstractAction(I18N.translate("ActionHideWorkspaceManager"), new ImageIcon("./resources/images/WorkspaceManagerHide.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.showWorkspaceManager(false); }};
		this.hideWorkspaceManagerAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipHideWorkspaceManager"));

		this.selectWorkspaceFolderAction = new AbstractAction(I18N.translate("ActionSelectWorkspaceFolder"), new ImageIcon("./resources/images/OpenFolder.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.selectWorkspaceFolder(); }};
		this.selectWorkspaceFolderAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSelectWorkspaceFolder"));

		this.searchWorkspaceAction = new AbstractAction(I18N.translate("ActionSearchWorkspace"), new ImageIcon("./resources/images/Search.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openSearchView(); }};
		this.searchWorkspaceAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_F, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
		this.searchWorkspaceAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearchWorkspace"));

		this.rereadWorkspaceFolderAction = new AbstractAction(I18N.translate("ActionRereadWorkspaceFolder"), new ImageIcon("./resources/images/Redisplay.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.rereadWorkspaceFolder(); }};
		this.rereadWorkspaceFolderAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipRereadWorkspaceFolder"));
		this.rereadWorkspaceFolderAction.putValue(Action.ACCELERATOR_KEY, getKeyStroke(KeyEvent.VK_R, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask(), KeyEvent.VK_F5, 0));

		this.createNewSIPAction = new AbstractAction(I18N.translate("ActionCreateNew"), new ImageIcon("./resources/images/New.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.createNewSIP(); }};
		this.createNewSIPAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_N, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
		this.createNewSIPAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipCreateNew"));

		this.createNewSIPFromTemplateAction = new AbstractAction(I18N.translate("ActionCreateNewFromTemplate"), new ImageIcon("./resources/images/NewFromTemplate.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.createNewSIPFromTemplate(); }};
		this.createNewSIPFromTemplateAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_N, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() | InputEvent.ALT_DOWN_MASK));
		this.createNewSIPFromTemplateAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipCreateNewFromTemplate"));

		this.openSIPAction = new AbstractAction(I18N.translate("ActionOpen"), new ImageIcon("./resources/images/Open.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openSIP(Mode.ReadWrite); }};
		this.openSIPAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_O, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
		this.openSIPAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpen"));

		this.openSIPReadWriteNoFileOpsAction = new AbstractAction(I18N.translate("ActionOpenReadWriteNoFileOps"), new ImageIcon("./resources/images/OpenReadWriteNoFileOps.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openSIP(Mode.ReadWriteNoFileOps); }};
		this.openSIPReadWriteNoFileOpsAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_O, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() | InputEvent.ALT_DOWN_MASK));
		this.openSIPReadWriteNoFileOpsAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenReadWriteNoFileOps"));

		this.openSIPReadOnlyAction = new AbstractAction(I18N.translate("ActionOpenReadOnly"), new ImageIcon("./resources/images/OpenReadOnly.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openSIP(Mode.ReadOnly); }};
		this.openSIPReadOnlyAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_O, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() | InputEvent.SHIFT_DOWN_MASK));
		this.openSIPReadOnlyAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenReadOnly"));

		this.openSIPInWorkspaceAction = new AbstractAction(I18N.translate("ActionOpenInWorkspace"), new ImageIcon("./resources/images/OpenInWorkspace.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openSelectedSIPInWorkspace(Mode.ReadWrite); }};
		this.openSIPInWorkspaceAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0));
		this.openSIPInWorkspaceAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenInWorkspace"));

		this.openSIPInWorkspaceReadWriteNoFileOpsAction = new AbstractAction(I18N.translate("ActionOpenInWorkspaceReadWriteNoFileOps"), new ImageIcon("./resources/images/OpenInWorkspaceReadWriteNoFileOps.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openSelectedSIPInWorkspace(Mode.ReadWriteNoFileOps); }};
		this.openSIPInWorkspaceReadWriteNoFileOpsAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.ALT_DOWN_MASK));
		this.openSIPInWorkspaceReadWriteNoFileOpsAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenInWorkspaceReadWriteNoFileOps"));

		this.openSIPInWorkspaceReadOnlyAction = new AbstractAction(I18N.translate("ActionOpenInWorkspaceReadOnly"), new ImageIcon("./resources/images/OpenInWorkspaceReadOnly.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.openSelectedSIPInWorkspace(Mode.ReadOnly); }};
		this.openSIPInWorkspaceReadOnlyAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.SHIFT_DOWN_MASK));
		this.openSIPInWorkspaceReadOnlyAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipOpenInWorkspaceReadOnly"));

		this.renameSIPInWorkspaceAction = new AbstractAction(I18N.translate("ActionRenameInWorkspace"), new ImageIcon("./resources/images/Rename.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.renameSIP(); }};
		this.renameSIPInWorkspaceAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipRenameSIP"));
		this.renameSIPInWorkspaceAction.putValue(Action.ACCELERATOR_KEY, getKeyStroke(KeyEvent.VK_R, InputEvent.ALT_DOWN_MASK, KeyEvent.VK_F2, InputEvent.ALT_DOWN_MASK, KeyEvent.VK_F2, InputEvent.CTRL_DOWN_MASK));

		this.copySIPInWorkspaceAction = new AbstractAction(I18N.translate("ActionCopyInWorkspace"), new ImageIcon("./resources/images/Copy.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.copySIP(); }};
		this.copySIPInWorkspaceAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipCopySIP"));
		this.copySIPInWorkspaceAction.putValue(Action.ACCELERATOR_KEY, getKeyStroke(KeyEvent.VK_C, Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() | InputEvent.ALT_DOWN_MASK));

		this.deleteSIPInWorkspaceAction = new AbstractAction(I18N.translate("ActionDeleteInWorkspace"), new ImageIcon("./resources/images/Delete.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.deleteSIP(); }};
		this.deleteSIPInWorkspaceAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipDeleteSIP"));
		this.deleteSIPInWorkspaceAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0));

		this.checkIngestFeedbackAction = new AbstractAction(I18N.translate("ActionCheckIngestFeedback"), new ImageIcon("./resources/images/SubmitCheckIngestFeedback.png"))
		{	@Override public void actionPerformed(ActionEvent e){ LauncherView.this.checkIngestFeedback(true); }};
		this.checkIngestFeedbackAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipCheckIngestFeedback"));
		this.checkIngestFeedbackAction.setEnabled(!(IngestFeedbackDirectory == null || IngestFeedbackDirectory.isEmpty()));

		//  Table:
		// Use Enter to show underlying sip
		String open = "open.sip.readwrite";
		KeyStroke enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
		this.sipTable.getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(enter, open);
		this.sipTable.getActionMap().put(open, this.openSIPInWorkspaceAction);

		open = "open.sip.readwrite.no.file.ops.action";
		enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.ALT_DOWN_MASK);
		this.sipTable.getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(enter, open);
		this.sipTable.getActionMap().put(open, this.openSIPInWorkspaceReadWriteNoFileOpsAction);

		open = "open.sip.readonly";
		enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.SHIFT_DOWN_MASK);
		this.sipTable.getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(enter, open);
		this.sipTable.getActionMap().put(open, this.openSIPInWorkspaceReadOnlyAction);


		
		//	Menus:

		this.popupMenu = new JPopupMenu();
		this.popupMenu.add(new JMenuItem(this.openSIPInWorkspaceAction));
		this.popupMenu.add(new JMenuItem(this.openSIPInWorkspaceReadWriteNoFileOpsAction));
		this.popupMenu.add(new JMenuItem(this.openSIPInWorkspaceReadOnlyAction));
		this.popupMenu.addSeparator();
		this.popupMenu.add(new JMenuItem(this.renameSIPInWorkspaceAction));
		this.popupMenu.add(new JMenuItem(this.copySIPInWorkspaceAction));
		this.popupMenu.add(new JMenuItem(this.deleteSIPInWorkspaceAction));

		this.programMenu = new JMenu(VersionProduct);
		this.programMenu.setIcon(new ImageIcon("./resources/images/DocuteamPackerSmall.png"));
		this.programMenu.add(new JMenuItem(this.aboutAction));
		this.programMenu.add(new JMenuItem(this.helpAction));
		this.programMenu.add(new JMenuItem(this.openDocuteamHomepageAction));
		this.programMenu.addSeparator();
		this.programMenu.add(new JMenuItem(this.quitAction));

		this.workspaceMenu = new JMenu(I18N.translate("MenuWorkspace"));
		this.workspaceMenu.setIcon(new ImageIcon("./resources/images/Workspace.png"));
		this.workspaceMenu.add(new JMenuItem(this.selectWorkspaceFolderAction));
		this.workspaceMenu.add(new JMenuItem(this.rereadWorkspaceFolderAction));
		this.workspaceMenu.addSeparator();
		this.workspaceMenu.add(new JMenuItem(this.showWorkspaceManagerAction));
		this.workspaceMenu.add(new JMenuItem(this.hideWorkspaceManagerAction));
		this.workspaceMenu.addSeparator();
		this.workspaceMenu.add(new JMenuItem(this.searchWorkspaceAction));
		this.workspaceMenu.addSeparator();
		this.workspaceMenu.add(new JMenuItem(this.checkIngestFeedbackAction));

		this.sipMenu = new JMenu(I18N.translate("MenuFile"));
		this.sipMenu.setIcon(new ImageIcon("./resources/images/MenuFile.png"));
		this.sipMenu.add(new JMenuItem(this.createNewSIPAction));
		this.sipMenu.add(new JMenuItem(this.createNewSIPFromTemplateAction));
		this.sipMenu.addSeparator();
		this.sipMenu.add(new JMenuItem(this.openSIPAction));
		this.sipMenu.add(new JMenuItem(this.openSIPReadWriteNoFileOpsAction));
		this.sipMenu.add(new JMenuItem(this.openSIPReadOnlyAction));
		this.sipMenu.addSeparator();
		this.sipMenu.add(new JMenuItem(this.openSIPInWorkspaceAction));
		this.sipMenu.add(new JMenuItem(this.openSIPInWorkspaceReadWriteNoFileOpsAction));
		this.sipMenu.add(new JMenuItem(this.openSIPInWorkspaceReadOnlyAction));
		this.sipMenu.addSeparator();
		this.sipMenu.add(new JMenuItem(this.renameSIPInWorkspaceAction));
		this.sipMenu.add(new JMenuItem(this.copySIPInWorkspaceAction));
		this.sipMenu.add(new JMenuItem(this.deleteSIPInWorkspaceAction));

		this.saMenu = new JMenu(I18N.translate("MenuSA"));
		this.saMenu.setIcon(new ImageIcon("./resources/images/MenuSA.png"));
		this.saMenu.add(new JMenuItem(this.updateSAsFromServerAction));

		this.windowMenu = new JMenu(I18N.translate("MenuWindow"));
		this.windowMenu.setIcon(new ImageIcon("./resources/images/MenuWindow.png"));
		this.windowMenu.add(new JMenuItem(this.openExceptionWindowAction));

		this.menuBar = new JMenuBar();
		this.menuBar.add(this.programMenu);
		this.menuBar.add(this.workspaceMenu);
		this.menuBar.add(this.sipMenu);
		this.menuBar.add(this.saMenu);
		this.menuBar.add(this.windowMenu);
		this.menuBar.setVisible(true);
		this.setJMenuBar(this.menuBar);

		//	Buttons:

		JButton createNewSIPButton = new JButton(this.createNewSIPAction);
		createNewSIPButton.setHideActionText(true);

		JButton createNewSIPFromTemplateButton = new JButton(this.createNewSIPFromTemplateAction);
		createNewSIPFromTemplateButton.setHideActionText(true);

		JButton openSIPButton = new JButton(this.openSIPAction);
		openSIPButton.setHideActionText(true);

		JButton openSIPReadWriteNoFileOpsButton = new JButton(this.openSIPReadWriteNoFileOpsAction);
		openSIPReadWriteNoFileOpsButton.setHideActionText(true);

		JButton openSIPReadOnlyButton = new JButton(this.openSIPReadOnlyAction);
		openSIPReadOnlyButton.setHideActionText(true);

		JButton selectWorkspaceFolderButton = new JButton(this.selectWorkspaceFolderAction);
		selectWorkspaceFolderButton.setHideActionText(true);

		this.showWorkspaceManagerButton = new JButton(this.showWorkspaceManagerAction);
		this.showWorkspaceManagerButton.setHideActionText(true);

		this.hideWorkspaceManagerButton = new JButton(this.hideWorkspaceManagerAction);
		this.hideWorkspaceManagerButton.setHideActionText(true);

		JButton openSIPInWorkspaceButton = new JButton(this.openSIPInWorkspaceAction);
		openSIPInWorkspaceButton.setHideActionText(true);

		JButton openSIPInWorkspaceReadWriteNoFileOpsButton = new JButton(this.openSIPInWorkspaceReadWriteNoFileOpsAction);
		openSIPInWorkspaceReadWriteNoFileOpsButton.setHideActionText(true);

		JButton openSIPInWorkspaceReadOnlyButton = new JButton(this.openSIPInWorkspaceReadOnlyAction);
		openSIPInWorkspaceReadOnlyButton.setHideActionText(true);

		JButton renameSIPButton = new JButton(this.renameSIPInWorkspaceAction);
		renameSIPButton.setHideActionText(true);

		JButton copySIPButton = new JButton(this.copySIPInWorkspaceAction);
		copySIPButton.setHideActionText(true);

		JButton deleteSIPButton = new JButton(this.deleteSIPInWorkspaceAction);
		deleteSIPButton.setHideActionText(true);

		JButton searchWorkspaceButton = new JButton(this.searchWorkspaceAction);
		searchWorkspaceButton.setHideActionText(true);

		JButton logoButton = new JButton(new ImageIcon("./resources/images/Logo_docuteam_packer.png"));
		logoButton.setEnabled(true);
		logoButton.setHideActionText(true);
		logoButton.setContentAreaFilled(false);
		logoButton.setBorderPainted(false);
		logoButton.setToolTipText(I18N.translate("ToolTipOpenDocuteamHomepage"));
		logoButton.addActionListener(new ActionListener()
		{	@Override public void actionPerformed(ActionEvent e) { LauncherView.this.openDocuteamHomepage(); }});

		Box buttonBox = new Box(BoxLayout.X_AXIS);
//		buttonBox.add(this.showWorkspaceManagerButton);
//		buttonBox.add(this.hideWorkspaceManagerButton);
//		buttonBox.add(Box.createHorizontalStrut(10));
//		buttonBox.add(selectWorkspaceFolderButton);
		buttonBox.add(searchWorkspaceButton);
		buttonBox.add(Box.createHorizontalStrut(20));
		buttonBox.add(createNewSIPButton);
		buttonBox.add(createNewSIPFromTemplateButton);
		buttonBox.add(Box.createHorizontalStrut(10));
		buttonBox.add(openSIPButton);
		buttonBox.add(openSIPReadWriteNoFileOpsButton);
		buttonBox.add(openSIPReadOnlyButton);
		buttonBox.add(Box.createHorizontalStrut(10));
		buttonBox.add(openSIPInWorkspaceButton);
		buttonBox.add(openSIPInWorkspaceReadWriteNoFileOpsButton);
		buttonBox.add(openSIPInWorkspaceReadOnlyButton);
		buttonBox.add(Box.createHorizontalStrut(10));
		buttonBox.add(renameSIPButton);
		buttonBox.add(copySIPButton);
		buttonBox.add(deleteSIPButton);

		this.northPanel = new JPanel(new BorderLayout());
		this.northPanel.setBorder(new EmptyBorder(0, 10, 0, 10));
		this.northPanel.add(buttonBox, BorderLayout.WEST);
		this.northPanel.add(logoButton, BorderLayout.EAST);

		//	Footer:

		JLabel versionLabel = new JLabel(VersionNumber);
		versionLabel.addMouseListener(
				new MouseAdapter(){ @Override public void mouseClicked(MouseEvent e){ LauncherView.this.versionLabelClicked(e); }});

		GridBagPanel footerPanel = new GridBagPanel(new EmptyBorder(5, 10, 5, 10), new Insets(0, 0, 0, 5));
		footerPanel.add(this.footerTextField,			1,	1,   	GridBagConstraints.EAST,		GridBagConstraints.HORIZONTAL,	1, 0);
		footerPanel.add(versionLabel,					1,	2,   	GridBagConstraints.EAST);


		//	----------	Now arrange all in the main panel:

		this.workspaceManagerPanel = new JPanel(new BorderLayout());
		this.workspaceManagerPanel.setBorder(new EmptyBorder(0, 10, 0, 10));
		this.workspaceManagerPanel.add(new JScrollPane(this.sipTable), BorderLayout.CENTER);

		this.add(this.northPanel, BorderLayout.NORTH);
		this.add(this.workspaceManagerPanel, BorderLayout.CENTER);
		this.add(footerPanel, BorderLayout.SOUTH);

		this.setLocation(ScreenPosition);
		this.showWorkspaceManager(DoInitiallyOpenWorkspaceManager);

		this.enableOrDisableActions();
	}

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Parse the command line arguments.
	 */
	static protected CommandLine getCommandLine(String... args)
	{
		Options options = new Options();
		options.addOption("open", true, "(Optional) This SIP will be opened after startup");
		options.addOption("configDir", true, "(Optional) This directory is used as the configuration directory");
		options.addOption("help", false, "(Optional) What are the options? Display this text in the console");

		CommandLine commandLine = null;
		try
		{
			commandLine = new GnuParser().parse(options, args);
		}
		catch(ParseException ex)
		{
			System.err.println("Command line parsing failed: " + ex.getMessage());
			new HelpFormatter().printHelp("docuteam packer", options);
			System.exit(10);
			return null;
		}

		if (commandLine.hasOption("help"))
		{
			new HelpFormatter().printHelp("docuteam packer", options);
			System.exit(11);
			return null;
		}

		return commandLine;
	}


	/**
	 * Initialize the system.
	 */
	static protected void initialize(CommandLine commandLine)
	{
		JFrame splashWindow = new JFrame();
		splashWindow.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		splashWindow.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
		splashWindow.setUndecorated(true);
		splashWindow.add(new JLabel(VersionNumber, new ImageIcon("./resources/images/SplashScreen.png"), SwingConstants.CENTER));
		splashWindow.setSize(300, 100);
		splashWindow.setLocationRelativeTo(null);

		try
		{
			splashWindow.setVisible(true);

			//	60 seconds until a tooltip text disappears should be enough:
			ToolTipManager.sharedInstance().setDismissDelay(60000);
			Toolkit.getDefaultToolkit().setDynamicLayout(true);

			Logger.getLogger().info("Initializing docuteam packer:");

			String propertyFileName = DefaultPropertyFileName;

			//	If a config directory is specified, take these configurations from there (if those files ARE there - if not, use the default folder = "./config"):
			//		- log4j.xml
			//		- docuteamPacker.properties
			//		- levels.xml
			if (commandLine.hasOption("configDir"))
			{
				String configDir = commandLine.getOptionValue("configDir");
				Logger.getLogger().info("    configDir: " + configDir);

				//	Initialize log4j:
				String log4jConfigFile = configDir + "/log4j.xml";
				if (new File(log4jConfigFile).exists())
				{
					Logger.getLogger().info("    Init log4j from: " + log4jConfigFile);
					Logger.setConfigFile(log4jConfigFile);
				}

				//	Specify the location of the property file:
				String configFile = configDir + "/docuteamPacker.properties";
				if (new File(configFile).exists())
				{
					Logger.getLogger().info("    Init docuteam packer from: " + configFile);
					propertyFileName = configFile;
				}

				//	Specify the location of the levels file:
				String levelsConfigFile = configDir + "/levels.xml";
				if (new File(levelsConfigFile).exists())
				{
					Logger.getLogger().info("    Init levels from: " + levelsConfigFile);
					LevelsConfigFileName = levelsConfigFile;
				}
			}

			initialize(propertyFileName);
			SIPView.initialize();
		}
		finally
		{
			splashWindow.setVisible(false);
			splashWindow.dispose();
		}
	}


	//	This is called on startup only:
	static protected void initialize(String propertyFileName)
	{
		PropertyFile.initialize(propertyFileName);

		String language = PropertyFile.get("docuteamPacker.displayLanguage", null);
		if (language != null)
		{
			Locale.setDefault(new Locale(language));
		}
		IsDevelopingMode = "true".equalsIgnoreCase(PropertyFile.get("docuteamPacker.isDevMode", IsDevelopingMode.toString()));
		Logger.getLogger().info("    isDevMode: " + IsDevelopingMode);

		//	When in dev mode, let the system output go to the console, otherwise to the SystemOutView:
		if (!IsDevelopingMode)
		{
			SystemOutView.install();

			boolean doOpenSystemOutViewOnOutput = "true".equalsIgnoreCase(PropertyFile.get("docuteamPacker.openSystemOutViewOnOutput", "true"));
			Logger.getLogger().info("    openSystemOutViewOnOutput: " + doOpenSystemOutViewOnOutput);

			//	Switch on or off the automatic popping-up of the SystemOutView on output via property file:
			SystemOutView.setDoPopUpOnWrite(doOpenSystemOutViewOnOutput);
		}

		StringBuilder additionalInfoText = new StringBuilder("----------\n")
			.append(VersionProduct)
			.append(" ")
			.append(VersionNumber)
			.append(" (")
			.append(VersionLastChange)
			.append(")\n----------\n")
			.append(OperatingSystem.osName())
			.append(" ")
			.append(OperatingSystem.osVersion())
			.append(" (")
			.append(OperatingSystem.userLanguage())
			.append(")\n----------\n")
			.append(OperatingSystem.userName())
			.append("\n----------\n");
		for (String key: PropertyFile.getKeys())	additionalInfoText.append(key).append(" = ").append(PropertyFile.get(key)).append("\n");
		additionalInfoText
			.append("----------\n");
		SystemOutView.setAdditionalInfoText(additionalInfoText.toString());

		String displayLanguageFromPropertyFile = PropertyFile.get("docuteamPacker.displayLanguage", "");
		String usedDisplayLanguage =
			displayLanguageFromPropertyFile.isEmpty()
				? ch.docuteam.tools.os.OperatingSystem.userLanguage()
				: displayLanguageFromPropertyFile;
		I18N.initialize(usedDisplayLanguage, "translations.Translations");
		Logger.getLogger().info("    displayLanguage: " + usedDisplayLanguage);

		NewSIPDefaultsToZipped = "true".equalsIgnoreCase(PropertyFile.get("docuteamPacker.newSIPDefaultsToZipped", NewSIPDefaultsToZipped.toString()));
		Logger.getLogger().info("    newSIPDefaultsToZipped: " + NewSIPDefaultsToZipped);

		//	Initialize various directories:
		DataDirectory = FileUtil.asCanonicalFileName(PropertyFile.get("docuteamPacker.dataDir" + PropertyFilePathOSSuffix, DataDirectory, DataDirectory));
		Logger.getLogger().info("    dataDirectory: " + DataDirectory);

		SIPDirectory = FileUtil.asCanonicalFileName(PropertyFile.get("docuteamPacker.SIPDir" + PropertyFilePathOSSuffix, SIPDirectory, SIPDirectory));
		LastUsedOpenOrSaveDirectory = SIPDirectory;
		Logger.getLogger().info("    SIPDirectory: " + SIPDirectory);

		TemplateDirectory = FileUtil.asCanonicalFileName(PropertyFile.get("docuteamPacker.templateDir" + PropertyFilePathOSSuffix, TemplateDirectory, TemplateDirectory));
		Logger.getLogger().info("    templateDirectory: " + TemplateDirectory);

		Document.setBackupFolder(PropertyFile.get("docuteamPacker.backupDir" + PropertyFilePathOSSuffix, null, null));
		Logger.getLogger().info("    backupDirectory: " + Document.getBackupFolder());

		FileUtil.setTempFolder(PropertyFile.get("docuteamPacker.tempDir" + PropertyFilePathOSSuffix, OperatingSystem.javaTempDir() + "DocuteamPacker", OperatingSystem.javaTempDir() + "DocuteamPacker"));
		Logger.getLogger().info("    tempDirectory: " + FileUtil.getTempFolder());

		IngestFeedbackDirectory = PropertyFile.get("docuteamPacker.ingestFeedbackDir" + PropertyFilePathOSSuffix, null, null);
		if (IngestFeedbackDirectory != null)	IngestFeedbackDirectory = FileUtil.asCanonicalFileName(IngestFeedbackDirectory);
		Logger.getLogger().info("    ingestFeedbackDirectory: " + IngestFeedbackDirectory);

		//	Initialize LevelOfDescriptions:
		LevelOfDescription.setInitializationFilePath(LevelsConfigFileName);
		Logger.getLogger().info("    levels: " + LevelsConfigFileName);

		//	Initialize SA:
		SubmissionAgreement.initializeBaseURL(PropertyFile.get("docuteamPacker.SA.BASE.URL", ""));
		Logger.getLogger().info("    SABaseURL: " + SubmissionAgreement.getBaseURL());

		//	Update SAs from server on startup only if the property "docuteamPacker.SA.getSAsFromServerOnStartup" is "true":
		if ("true".equalsIgnoreCase(PropertyFile.get("docuteamPacker.SA.getSAsFromServerOnStartup", "false")))
		{
			Logger.getLogger().info("    getSAsFromServerOnStartup: true");

			try
			{
				List<String> saNamesToBeDeleted = updateSAsFromServer();

				if (!saNamesToBeDeleted.isEmpty() && !IsDevelopingMode)				//	When in dev mode, don't delete any SAs
				{
					//	Ask whether to delete the excessive SAs.
					//	Don't use "JOptionPane.showConfirmDialog(...)" because that way we can't explicitly specify the position of the dialog.

					StringBuilder confirmMessage = new StringBuilder(I18N.translate("QuestionDeleteExcessiveSAs") + "\n");
					for (String s: saNamesToBeDeleted)		confirmMessage.append("\n").append(s);

					JOptionPane optionPane = new JOptionPane(confirmMessage, JOptionPane.QUESTION_MESSAGE, JOptionPane.YES_NO_OPTION);
					JDialog dialog = optionPane.createDialog("Confirmation");
					dialog.pack();

					Dimension screenSize = java.awt.Toolkit.getDefaultToolkit().getScreenSize();
					Dimension dialogSize = dialog.getSize();
					int x = (screenSize.width - dialogSize.width)/2;
					int y = (screenSize.height/2 - dialogSize.height)/2;
					dialog.setLocation(x, y);
					dialog.setVisible(true);

					Object result = optionPane.getValue();
					if (result != null && ((Integer)result).intValue() == 0)
					{
						//	Delete the excessive SAs:
						for (String s: saNamesToBeDeleted)
						{
							Logger.getLogger().debug("Deleting SA: " + s);
							FileUtil.delete(SubmissionAgreement.getSAFolder() + "/" + s);
						}
					}
				}
			}
			catch (Exception ex)
			{
				Logger.getLogger().warn("Could not update SAs from Server: " + ex.getMessage());
			}
		}
		else
		{
			Logger.getLogger().info("    getSAsFromServerOnStartup: false");
		}

		//	Initialize OOConverter:
		String ooConverterInitRetries = PropertyFile.get("docuteamPacker.OOConverter.initializationRetries", "");
		if (!ooConverterInitRetries.isEmpty())		FileConverter.OOConverter.setNumberOfInitializationRetries(new Integer(ooConverterInitRetries));
		Logger.getLogger().info("    ooConverterInitRetries: " + FileConverter.OOConverter.getNumberOfInitializationRetries());

		FileConverter.OOConverter.initializeDontWait(PropertyFile.get("docuteamPacker.OOConverter.path" + PropertyFilePathOSSuffix, ""));
		Logger.getLogger().info("    ooConverterPath: " + FileConverter.OOConverter.getConverterPath());

		//	Initialize FilePreviewer:
		String filePreviewCacheSizeLimit = PropertyFile.get("docuteamPacker.filePreviewer.cacheSizeLimit", "");
		if (!filePreviewCacheSizeLimit.isEmpty())		FilePreviewer.setCacheSizeLimit(new Integer(filePreviewCacheSizeLimit));
		Logger.getLogger().info("    filePreviewCacheSizeLimit: " + FilePreviewer.getCacheSizeLimit());

		//	Initialize UIManager:
		try
		{
			if ("true".equalsIgnoreCase(PropertyFile.get("docuteamPacker.useSystemLookAndFeel", UseSystemLAF.toString())))
				UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());			//	com.sun.java.swing.plaf.windows.WindowsLookAndFeel on Windows, com.apple.laf.AquaLookAndFeel on OS X.
			else
				UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());		//	Switches to Metal (= Default cross platform laf)
		}
		catch (Exception e){}				//	Ignore it and set default LAF
		Logger.getLogger().info("    lookAndFeel: " + UIManager.getLookAndFeel().getName());
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	/**
	 * This is performed every time the application quits normally AND when it quits abnormally or the Java VM is terminated.
	 * This is a safety net to cleanup the workspace, delete all locks, and remove temporary files. Veto is not possible.
	 */
	static protected void shutdown()
	{
		Logger.getLogger().debug("Cleaning up...");

		try
		{
			for (SIPView sipView: LauncherView.openedSIPViews)
				if (sipView.getDocument() != null)		sipView.getDocument().unlockIfNecessary();

			FileConverter.OOConverter.disconnect();
			FilePreviewPanel.cleanupTemporaryFiles();
		}
		catch (Exception ex) {}		//	Ignore any exception

		Logger.getLogger().debug("Cleaning up done");
	}

	//	--------		Utilities			-------------------------------------------------------

	static protected List<String> updateSAsFromServer()
	{
		try
		{
			return SubmissionAgreement.updateSAsFromServer();
		}
		catch (MalformedURLException ex)
		{
			//	MainFrame might be null here (on application startup):
			JOptionPane.showMessageDialog(LauncherView, I18N.translate("MessageCantUpdateSAsFromServer"), I18N.translate("TitleCantUpdateSAsFromServer"), JOptionPane.ERROR_MESSAGE);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		return new ArrayList<String>();
	}

	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------

	/**
	 * This method has to be public because it is called by the OSXAdapter via reflection.
	 */
	public void openAboutWindow()
	{
		new AboutView(this);
	}


	/**
	 * The [Quit] button was clicked, or the [Window close] button was clicked, or <ctrl-q> or <cmd-q> was typed.
	 * First confirm. If confirmed and if the current document was modified, ask if to save the document.
	 * NOTES:
	 * 		This method has to be public because it is called by the OSXAdapter via reflection.
	 * 		This method has to return a boolean value which is interpreted by the OSXAdapter.
	 */
	public boolean requestQuit()
	{
		Logger.getLogger().debug("Requesting quit...");
		
		if (LauncherView.sipsWithSavingInProgress.size() > 0)
		{
			Logger.getLogger().debug("Quit stopped by SIPs currently being saved:\n" + LauncherView.sipsWithSavingInProgress);
			return false;
		}
		
		//	I have to use a copy of the openedSIPViews list otherwise I get a ConcurrentModificationException
		//		because the method "sipView.closeButtonClicked()" removes this sipView from this list.
		List<SIPView> openedSIPViews = new ArrayList<SIPView>(this.openedSIPViews);
		for (SIPView sipView: openedSIPViews)
		{
			sipView.toFront();

			//	If any of the sipViews cancelled, stop quitting:
			if (!sipView.closeButtonClicked())
			{
				Logger.getLogger().debug("Quit stopped by: '" + sipView.getSIPPath() + "'");
				return false;
			}
		}

		SearchView.closeAndDispose();
		this.setVisible(false);
		this.dispose();

		//	Set the tempFolder and all files and folders within to writable:
		FileUtil.setWritable(FileUtil.getTempFolder());

		System.exit(0);			//	This triggers the method shutdown()

		//	This method has to return a boolean value even though I quit the program before.
		return true;
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------

	/**
	 * Remove this sipView from the list of opened sipViews and refresh the SIP table.
	 * @param sipView
	 */
	public void unregister(SIPView sipView)
	{
		//	Remove the closed SIPView from the openedSIPViews list:
		this.openedSIPViews.remove(sipView);

		//	Refresh the SIP table:
		this.refreshSIPTable();
	}

	public KeyStroke getKeyStroke(int keyEvent, int modifiers)
	{
		return getKeyStroke(keyEvent, modifiers, keyEvent, modifiers);
	}

	public KeyStroke getKeyStroke(int macKeyEvent, int macModifiers, int pcKeyEvent, int pcModifiers)
	{
		return getKeyStroke(macKeyEvent, macModifiers, pcKeyEvent, pcModifiers, pcKeyEvent, pcModifiers);
	}

	public KeyStroke getKeyStroke(int macKeyEvent, int macModifiers, int winKeyEvent, int winModifiers, int linuxKeyEvent, int linuxModifiers)
	{
		if (OperatingSystem.isMacOSX())
		{
			return KeyStroke.getKeyStroke(macKeyEvent, macModifiers);
		}
		else if (OperatingSystem.isWindows())
		{
			return KeyStroke.getKeyStroke(winKeyEvent, winModifiers);
		}
		else if (OperatingSystem.isLinux())
		{
			return KeyStroke.getKeyStroke(linuxKeyEvent, linuxModifiers);
		}
		else
		{
			return KeyStroke.getKeyStroke(winKeyEvent, winModifiers);
		}
	}

	/**
	 * This document is just being saved. When wanting to open it, wait until saving is finished.
	 * @param document
	 */
	public void savingDocumentInProgress(Document document)
	{
		this.sipsWithSavingInProgress.add(document.getOriginalSIPFolder());
	}

	/**
	 * This document is finished being saved. When wanting to open it, now go on!
	 * @param document
	 */
	public void savingDocumentFinished(Document document)
	{
		this.sipsWithSavingInProgress.remove(document.getOriginalSIPFolder());
	}


	/**
	 * Refresh the SIP table. This is used when I know that the number of SIPs in the workspace was not changed.
	 */
	public void refreshSIPTable()
	{
		int sel = this.sipTable.getSelectedRow();
		this.getSIPTableModel().fireTableDataChanged();

		//	Try to retain the current selection index. If this fails, ignore it.
		try { this.sipTable.getSelectionModel().setSelectionInterval(sel, sel); }
		catch (Exception ex) {};
	}

	/**
	 * Reread the SIP table. The number of SIPs in the workspace might have changed. Try to retain the current selection index.
	 */
	public void rereadSIPTable()
	{
		int sel = this.sipTable.getSelectedRow();
		this.getSIPTableModel().setList();

		//	Try to retain the current selection index. If this fails, ignore it.
		try { this.sipTable.getSelectionModel().setSelectionInterval(sel, sel); }
		catch (Exception ex) {};
	}

	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	protected SIPTableModel getSIPTableModel()
	{
		return (SIPTableModel)this.sipTable.getModel();
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------

	protected void listSelectionChanged(ListSelectionEvent e)
	{
		this.selectedSIP =
				this.sipTable.getSelectedRow() == -1
						? null
						: this.getSIPTableModel().get(this.sipTable.convertRowIndexToModel(this.sipTable.getSelectedRow()));

		this.footerTextField.setText(this.selectedSIP == null? "": this.selectedSIP.getPath());
		this.enableOrDisableActions();
	}


	protected void listSelectionWasClicked(MouseEvent e)
	{
		if (e.getClickCount() == 2)
		{
			//	Double-Click:
			if ((e.getModifiersEx() & (InputEvent.ALT_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK)) == 0)
			{
				this.openSelectedSIPInWorkspace(Mode.ReadWrite);
			}
			else if ((e.getModifiersEx() & (InputEvent.ALT_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK)) == InputEvent.ALT_DOWN_MASK)
			{
				this.openSelectedSIPInWorkspace(Mode.ReadWriteNoFileOps);
			}
			else if ((e.getModifiersEx() & (InputEvent.SHIFT_DOWN_MASK | InputEvent.ALT_DOWN_MASK)) == InputEvent.SHIFT_DOWN_MASK)
			{
				this.openSelectedSIPInWorkspace(Mode.ReadOnly);
			}
		}
		else if (e.getButton() == MouseEvent.BUTTON3)
		{
			//	Right-Click:

			//	Find out which row was right-clicked:
			int rowNumber = this.sipTable.rowAtPoint(e.getPoint());

			//	Select this row:
			this.sipTable.getSelectionModel().setSelectionInterval(rowNumber, rowNumber);

			//	Show popup menu:
			this.popupMenu.show(this.sipTable, e.getX(), e.getY());
		}
		//	else: ignore. Single clicks are handled in listSelectionChanged().
	}


	protected void openDocuteamHomepage()
	{
		if (!java.awt.Desktop.isDesktopSupported())
		{
			System.err.println("Desktop is not supported");
			return;
		}

		java.awt.Desktop desktop = java.awt.Desktop.getDesktop();

		if (!desktop.isSupported(java.awt.Desktop.Action.BROWSE))
		{
			System.err.println("Desktop doesn't support the browse action");
			return;
		}

		try
		{
			desktop.browse(new java.net.URI("http://www.docuteam.ch"));
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
		}
	}


	protected void openHelpPage()
	{
		if (!java.awt.Desktop.isDesktopSupported())
		{
			System.err.println("Desktop is not supported");
			return;
		}

		java.awt.Desktop desktop = java.awt.Desktop.getDesktop();

		if (!desktop.isSupported(java.awt.Desktop.Action.BROWSE))
		{
			System.err.println("Desktop doesn't support the browse action");
			return;
		}

		try
		{
			desktop.browse(new java.net.URI(I18N.translate("HelpPageURL")));
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
		}
	}


	protected void openExceptionWindow()
	{
		SystemOutView.open();
	}


	protected void versionLabelClicked(MouseEvent e)
	{
		if ((e.getModifiers() & MouseEvent.META_MASK) != 0)
		{
			for (SIPView sipView: this.openedSIPViews)		System.out.println(sipView.sipPath);
		}
		else if ((e.getModifiers() & MouseEvent.ALT_MASK) != 0)
		{
			JDialog d = new JDialog(this, true);
			d.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			d.setTitle("I did it!");
			d.add(new JLabel(new ImageIcon("./resources/images/I.jpg")));
			d.setPreferredSize(new Dimension(100, 150));
			d.pack();
			d.setLocationRelativeTo(null);		//	Center on screen
			d.setVisible(true);
		}
	}


	protected void showWorkspaceManager(boolean doShow)
	{
		if (doShow)
		{
			this.rereadSIPTable();

			this.showWorkspaceManagerAction.setEnabled(false);
			this.hideWorkspaceManagerAction.setEnabled(true);

			this.workspaceManagerPanel.setVisible(true);
			this.showWorkspaceManagerButton.setVisible(false);
			this.hideWorkspaceManagerButton.setVisible(true);

			//	Remember the screen proportions before changing the view (except when the view is not yet visible):
			if (this.isVisible())		this.getSize(this.ScreenSizeWithoutWorkspaceManager);
			this.setPreferredSize(this.ScreenSizeWithWorkspaceManager);
			this.setResizable(true);
			this.pack();
		}
		else
		{
			this.sipTable.clearSelection();

			this.showWorkspaceManagerAction.setEnabled(true);
			this.hideWorkspaceManagerAction.setEnabled(false);

			this.workspaceManagerPanel.setVisible(false);
			this.showWorkspaceManagerButton.setVisible(true);
			this.hideWorkspaceManagerButton.setVisible(false);

			//	Remember the screen proportions before changing the view (except when the view is not yet visible):
			if (this.isVisible())		this.getSize(this.ScreenSizeWithWorkspaceManager);
			this.setPreferredSize(this.ScreenSizeWithoutWorkspaceManager);
			this.setResizable(true);
			this.pack();
		}
	}


	protected void createNewSIP()
	{
		CreateNewSIPDialog createNewSIPDialog = new CreateNewSIPDialog(this);
		if (!createNewSIPDialog.goButtonWasClicked)						return;

		String rootFolderName = createNewSIPDialog.rootFolderNameTextField.getText();
		String sourceFileOrFolderName = createNewSIPDialog.sourceFileOrFolderTextField.getText();
		final boolean createEmptySIP = createNewSIPDialog.selectSIPEmptyRadioButton.isSelected();
		String destinationFolderName = createNewSIPDialog.destinationFolderTextField.getText();
		String destinationFileName = createNewSIPDialog.destinationNameTextField.getText();
		boolean beZIP = createNewSIPDialog.beZIPCheckBox.isSelected();
		final Overview saOverview = (Overview)createNewSIPDialog.saComboBox.getSelectedItem();

		final String initialSourceFileOrFolder = createEmptySIP? rootFolderName: sourceFileOrFolderName;
		if (initialSourceFileOrFolder.isEmpty()
			|| destinationFileName.isEmpty())							return;

		if (beZIP)
		{
			if (!destinationFileName.toLowerCase().endsWith(".zip"))	destinationFileName += ".zip";
		}
		else
		{
			if (destinationFileName.toLowerCase().endsWith(".zip"))		destinationFileName = destinationFileName.substring(0, destinationFileName.length() - 4);
		}

		final String newSIPFileName = destinationFolderName + "/" + destinationFileName;

		new SwingWorker<Integer, Object>()
		{
			@Override public Integer doInBackground()
			{
				LauncherView.footerTextField.setText(I18N.translate("MessageFooterNewFile") + newSIPFileName + "...");
				SmallPeskyMessageWindow waitWindow = SmallPeskyMessageWindow.openBlocking(LauncherView.this, I18N.translate("MessageTempCreatingSIP"));

				Document document = null;
				try
				{
					LauncherView.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

					ExceptionCollector.clear();
					document = createEmptySIP
							? Document.createNewWithRootFolderName(newSIPFileName, initialSourceFileOrFolder, saOverview.saId, saOverview.dssId, Operator, waitWindow)
							: Document.createNewWithRoot          (newSIPFileName, initialSourceFileOrFolder, saOverview.saId, saOverview.dssId, Operator, waitWindow);

					LauncherView.footerTextField.setText(I18N.translate("MessageFooterNewFile") + newSIPFileName);

					if (!ExceptionCollector.isEmpty())
					{
						//	Close the waitWindow here otherwise it interferes with the MessageDialog:
						waitWindow.close();
						new ScrollableMessageDialog(LauncherView, I18N.translate("TitleWarningsOccurred"), ExceptionCollector.getAllDescriptions(), new ImageIcon("./resources/images/DocuteamPacker.png"));
					}
				}
				catch (java.lang.Exception ex)
				{
					ex.printStackTrace();
					document = null;

					LauncherView.footerTextField.setText(I18N.translate("MessageFooterCantCreateFile") + newSIPFileName);

					//	Close the waitWindow here otherwise it interferes with the MessageDialog:
					waitWindow.close();
					JOptionPane.showMessageDialog(LauncherView, ex.toString(), I18N.translate("TitleCantCreateSIP"), JOptionPane.ERROR_MESSAGE);
				}
				finally
				{
					//	At this point, document may or may not be null:
					if (document != null)
					{
						try
						{
							document.unlockIfNecessary();
							document.cleanupWorkingCopy();
						}
						catch (Exception e)
						{
							e.printStackTrace();
							JOptionPane.showMessageDialog(LauncherView, e.toString(), "Unable to cleanup working folder due to errors", JOptionPane.ERROR_MESSAGE);
						}

						LauncherView.this.rereadSIPTable();		//	Refresh the SIP list
						LauncherView.this.openSIP(newSIPFileName, Mode.ReadWrite);
					}

					waitWindow.close();		//	In case it was not closed yet...
					LauncherView.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				}

				return 0;
			}
		}.execute();
	}


	protected void createNewSIPFromTemplate()
	{
		CreateNewSIPFromTemplateDialog createNewSIPFromTemplateDialog = new CreateNewSIPFromTemplateDialog(this);
		if (!createNewSIPFromTemplateDialog.goButtonWasClicked)			return;

		final String templateName = (String)createNewSIPFromTemplateDialog.templateComboBox.getSelectedItem();
		String destinationFolderName = createNewSIPFromTemplateDialog.destinationFolderTextField.getText();
		String destinationFileName = createNewSIPFromTemplateDialog.destinationNameTextField.getText();
		boolean beZIP = createNewSIPFromTemplateDialog.beZIPCheckBox.isSelected();

		if (destinationFileName.isEmpty())								return;

		if (beZIP)
		{
			if (!destinationFileName.toLowerCase().endsWith(".zip"))	destinationFileName += ".zip";
		}
		else
		{
			if (destinationFileName.toLowerCase().endsWith(".zip"))		destinationFileName = destinationFileName.substring(0, destinationFileName.length() - 4);
		}

		final String newSIPFileName = destinationFolderName + "/" + destinationFileName;

		new SwingWorker<Integer, Object>()
		{
			@Override public Integer doInBackground()
			{
				LauncherView.footerTextField.setText(I18N.translate("MessageFooterNewFile") + newSIPFileName + "...");
				SmallPeskyMessageWindow waitWindow = SmallPeskyMessageWindow.openBlocking(LauncherView.this, I18N.translate("MessageTempCreatingSIP"));

				Document document = null;
				try
				{
					LauncherView.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

					ExceptionCollector.clear();
					document = Document.createNewFromTemplate(TemplateDirectory + "/" + templateName, newSIPFileName, Operator);

					LauncherView.footerTextField.setText(I18N.translate("MessageFooterNewFile") + newSIPFileName);

					if (!ExceptionCollector.isEmpty())
					{
						//	Close the waitWindow here otherwise it interferes with the MessageDialog:
						waitWindow.close();
						new ScrollableMessageDialog(LauncherView, I18N.translate("TitleWarningsOccurred"), ExceptionCollector.getAllDescriptions(), new ImageIcon("./resources/images/DocuteamPacker.png"));
					}
				}
				catch (java.lang.Exception ex)
				{
					ex.printStackTrace();
					document = null;

					LauncherView.footerTextField.setText(I18N.translate("MessageFooterCantCreateFile") + newSIPFileName);

					//	Close the waitWindow here otherwise it interferes with the MessageDialog:
					waitWindow.close();
					JOptionPane.showMessageDialog(LauncherView, ex.toString(), I18N.translate("TitleCantCreateSIP"), JOptionPane.ERROR_MESSAGE);
				}
				finally
				{
					//	At this point, document may or may not be null:
					if (document != null)
					{
						try
						{
							document.unlockIfNecessary();
							document.cleanupWorkingCopy();
						}
						catch (Exception e)
						{
							e.printStackTrace();
							JOptionPane.showMessageDialog(LauncherView, e.toString(), "Unable to cleanup working folder due to errors", JOptionPane.ERROR_MESSAGE);
						}

						LauncherView.this.rereadSIPTable();		//	Refresh the SIP list
						LauncherView.this.openSIP(newSIPFileName, Mode.ReadWrite);
					}

					waitWindow.close();		//	In case it was not closed yet...
					LauncherView.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				}

				return 0;
			}
		}.execute();
	}


	/**
	 * The [selectWorkspaceFolder] button was clicked:
	 */
	protected void selectWorkspaceFolder()
	{
		final JFileChooser fileChooser = new JFileChooser(SIPDirectory);
		fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
		fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

		if (fileChooser.showOpenDialog(this) != JFileChooser.APPROVE_OPTION)		return;		//	Cancel-button clicked

		SIPDirectory = FileUtil.asCanonicalFileName(fileChooser.getSelectedFile());
		this.setTitle(SIPDirectory);

		this.rereadSIPTable();
	}

	/**
	 * The [rereadWorkspaceFolder] button was clicked:
	 */
	protected void rereadWorkspaceFolder()
	{
		this.rereadSIPTable();
	}


	protected void openSearchView()
	{
		SearchView.open(this);
	}


	/**
	 * Rename the currently selected SIP. If it is locked by anybody (including the current user), don't rename.
	 */
	protected void renameSIP()
	{
		if (this.selectedSIP == null)								return;
		if (Document.isLocked(this.selectedSIP))					return;

		String title = I18N.translate("TitleRenameSIP");
		String message = I18N.translate("MessageEnterNewSIPName");
		String textFieldContent = this.selectedSIP.getName();
		int messageType = JOptionPane.QUESTION_MESSAGE;

		String newSIPName;
		Boolean ok;
		do
		{
			newSIPName = (String)JOptionPane.showInputDialog(this, message, title, messageType, null, null, textFieldContent);
			if ((newSIPName == null) || newSIPName.length() == 0)	return;

			ok = false;

			if (!FileUtil.isFileNameAllowed(newSIPName))
			{
				title = I18N.translate("TitleCantRenameSIP");
				message = I18N.translate("MessageBadLettersInFilename") + "\n" + I18N.translate("MessageEnterNewSIPName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = newSIPName;
				continue;
			}

			File newSIP = new File(this.selectedSIP.getParent() + "/" + newSIPName);
			if (newSIP.exists())
			{
				title = I18N.translate("TitleCantRenameSIP");
				message = I18N.translate("MessageSIPAlreadyExists") + "\n" + I18N.translate("MessageEnterNewSIPName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = newSIPName;
				continue;
			}

			try
			{
				FileUtil.renameTo(this.selectedSIP, new File(this.selectedSIP.getParent() + "/" + newSIPName));
			}
			catch (java.lang.Exception ex)
			{
				title = I18N.translate("TitleCantRenameSIP");
				message = ex.getMessage() + "\n" + I18N.translate("MessageEnterNewSIPName");
				messageType = JOptionPane.WARNING_MESSAGE;
				textFieldContent = this.selectedSIP.getName();
				continue;
			}

			ok = true;
		} while (!ok);

		this.rereadSIPTable();
	}



	/**
	 * Copy the currently selected SIP.
	 */
	protected void copySIP()
	{
		if (this.selectedSIP == null)					return;

		String title = I18N.translate("TitleCopySIP");
		String message = I18N.translate("MessageEnterNewSIPName");
		String textFieldContent = FileUtil.asFilePathWithoutExtension(this.selectedSIP.getName());

		String newItemName;
		Boolean ok;
		do
		{
			newItemName = CopySIPDialog.open(this, message, title, textFieldContent);
			if ((newItemName == null) || newItemName.length() == 0)		return;

			ok = false;

			if (!FileUtil.isFileNameAllowed(newItemName))
			{
				title = I18N.translate("TitleCantCopySIP");
				message = I18N.translate("MessageBadLettersInFilename") + "\n" + I18N.translate("MessageEnterNewSIPName");
				textFieldContent = FileUtil.asFilePathWithoutExtension(newItemName);
				continue;
			}

			File newSIP = new File(this.selectedSIP.getParent() + "/" + newItemName);
			if (newSIP.exists())
			{
				title = I18N.translate("TitleCantCopySIP");
				message = I18N.translate("MessageSIPAlreadyExists") + "\n" + I18N.translate("MessageEnterNewSIPName");
				textFieldContent = FileUtil.asFilePathWithoutExtension(newItemName);
				continue;
			}

			try
			{
				Document.copy(this.selectedSIP.getPath(), newSIP.getPath());
			}
			catch (java.lang.Exception ex)
			{
				title = I18N.translate("TitleCantCopySIP");
				message = ex.getMessage() + "\n" + I18N.translate("MessageEnterNewSIPName");
				textFieldContent = FileUtil.asFilePathWithoutExtension(newItemName);

				try { FileUtil.delete(newSIP); }
				catch (FileUtilExceptionListException ex1) {}

				continue;
			}

			ok = true;
		} while (!ok);

		this.rereadSIPTable();
	}


	/**
	 * Delete the currently selected SIP. If it is locked by anybody (including the current user), don't delete.
	 */
	protected void deleteSIP()
	{
		if (this.selectedSIP == null)					return;
		if (Document.isLocked(this.selectedSIP))		return;

		if (JOptionPane.showConfirmDialog(this, I18N.translate("QuestionDeleteSIP"), I18N.translate("TitleDeleteSIP"), JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION)		return;

		try
		{
			FileUtil.delete(this.selectedSIP);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		this.rereadSIPTable();
	}


	protected void checkIngestFeedback(boolean withFeedbackIfNothingFound)
	{
		Logger.getLogger().debug("Checking ingest feedback in folder: '" + IngestFeedbackDirectory + "'");

		try
		{
			List<String> feedbackFoundForSIPs = Document.checkIngestFeedback(SIPDirectory, SIPView.IngestSubmitDirectory, IngestFeedbackDirectory);

			Logger.getLogger().debug("Found ingest feedback: " + feedbackFoundForSIPs);

			if (feedbackFoundForSIPs.isEmpty())
			{
				if (withFeedbackIfNothingFound)		JOptionPane.showMessageDialog(this, I18N.translate("MessageCheckIngestFeedbackEmpty"), I18N.translate("HeaderCheckIngestFeedback"), JOptionPane.INFORMATION_MESSAGE);

				return;
			}

			StringBuilder feedbackFoundForSIPsMessage = new StringBuilder(I18N.translate("MessageCheckIngestFeedbackFound"));
			for (String sipName: feedbackFoundForSIPs)		feedbackFoundForSIPsMessage.append("\n").append(sipName);

			new ScrollableMessageDialog(this, I18N.translate("HeaderCheckIngestFeedback"), feedbackFoundForSIPsMessage.toString());

			//	If a SIPView on a feedbacked SIP is open, refresh this SIPView:
			for (String sipName: feedbackFoundForSIPs)
			{
				String sipPath = SIPDirectory + File.separator + sipName;

				for (SIPView openSIPView: this.openedSIPViews)
					if (openSIPView.getSIPPath().equals(sipPath))		openSIPView.read(sipPath, Mode.ReadOnly, null);
			}

			//	Refresh the SIP table:
			this.refreshSIPTable();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	/**
	 * Open a SIP on startup (if specified).
	 * Note: Must be public because it might be called from a subclass via the static variable "LauncherView".
	 */
	protected void openSIP(CommandLine commandLine)
	{
		//	If an "open" parameter is supplied in the command line, open this file:
		if (commandLine.hasOption("open"))
		{
			Logger.getLogger().debug("open: " + commandLine.getOptionValue("open"));
			this.openSIP(commandLine.getOptionValue("open"), Mode.ReadWrite);
		}
		else if (IsDevelopingMode)
		{
		//	ToDo: When starting in development mode, open this file:

//			Examples for Denis:
//
//			open();																				//	Empty
//			open("./files/ch-001194-4_55/mets modified.xml", false);							//	Big example
//			open("./files/ch-001194-4_459/mets modified.xml", false);							//	Small example
//			open("./files/DifferentFileTypes/mets.xml", false);									//	Many different file types
//			open("./files/DifferentFileTypes/mets.xml", true);									//	Many different file types, open read-only
//			this.openSIP("./files/DifferentLevels", false);										//	Different levels
//			open("./files/Icons/mets.xml", false);												//	Big example with many icons
//			open("./files/Locked/mets.xml", false);												//	Locked
//			open("./files/RestrictedAccess/mets.xml", false);									//	Restricted file access
//			open("/Users/denis/Desktop/SIP/mets.xml", false);									//	Temporary example
//			open("/Users/docuteam/Desktop/Created/Humor/mets.xml", false);						//	Temporary example

//			Examples for Andi:
//			...
		}
	}


	/**
	 * Ask for a path then open this SIP.
	 * @param mode
	 */
	protected void openSIP(Mode mode)
	{
		JFileChooser fileChooser = SIPFileChooser.getInstance();
		if (fileChooser.showOpenDialog(this) != JFileChooser.APPROVE_OPTION)		return;		//	Cancel-button clicked

		//	Remember the SIP's parent directory:
		LastUsedOpenOrSaveDirectory = FileUtil.asCanonicalFileName(fileChooser.getSelectedFile().getParent());

		//	Open this SIP:
		this.openSIP(fileChooser.getSelectedFile().getAbsolutePath(), mode);
	}


	/**
	 * Open the SIP denoted by sipPath. If it is locked by anybody except the current user, don't open.
	 * If this SIP is already open, bring this view to the front.
	 * @param mode
	 */
	protected void openSIP(String sipPath, Mode mode)
	{
		this.openSIP(sipPath, mode, null);
	}


	/**
	 * Open the SIP denoted by sipPath. If it is locked by anybody except the current user, open readOnly.
	 * If this SIP is already open, bring this view to the front.
	 * If treePath is not null, try to select the node at treePath.
	 * @param mode
	 */
	protected void openSIP(String sipPath, Mode mode, String admIdToSelect)
	{
		//	Don't allow opening document if it is just being saved:
		if (this.sipsWithSavingInProgress.contains(sipPath))
		{
			JOptionPane.showMessageDialog(this, I18N.translate("MessageCantOpenSIPWhileBeingSaved"), I18N.translate("TitleOpenSIP"), JOptionPane.INFORMATION_MESSAGE);
			return;
		}

		Logger.getLogger().debug("Opening SIP: " + sipPath);

		//	Already open? If yes, bring this view to the front:
		for (SIPView openSIPView: this.openedSIPViews)
			if (openSIPView.getSIPPath().equals(sipPath))
			{
				openSIPView.toFront();
				openSIPView.requestFocus();

				openSIPView.selectNode(admIdToSelect);

				return;
			}

		SIPView sipView = SIPView.open(this, sipPath, mode, admIdToSelect);

		//	Register this sipView:
		this.openedSIPViews.add(sipView);

		//	Update the sipTable to show that this SIP is now locked by the current user:
		this.getSIPTableModel().fireTableDataChanged();
	}


	/**
	 * Open the currently selected SIP. If it is locked by anybody except the current user, don't open.
	 * @param mode
	 */
	protected void openSelectedSIPInWorkspace(Mode mode)
	{
		if (this.selectedSIP == null)		return;

		this.openSIP(this.selectedSIP.getPath(), mode);
	}


	protected void openSIPInWorkspace(String sipName, Mode mode)
	{
		this.openSIP(SIPDirectory + File.separator + sipName, mode);
	}


	protected void openSIPInWorkspace(String sipName, Mode mode, String admIdToSelect)
	{
		this.openSIP(SIPDirectory + File.separator + sipName, mode, admIdToSelect);
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	protected void enableOrDisableActions()
	{
		if (this.selectedSIP == null)
		{
			//	Nothing selected:

			this.openSIPInWorkspaceAction.setEnabled(false);
			this.openSIPInWorkspaceReadWriteNoFileOpsAction.setEnabled(false);
			this.openSIPInWorkspaceReadOnlyAction.setEnabled(false);
			this.renameSIPInWorkspaceAction.setEnabled(false);
			this.copySIPInWorkspaceAction.setEnabled(false);
			this.deleteSIPInWorkspaceAction.setEnabled(false);
		}
		else
		{
			//	A SIP is selected:

			if (Document.isLockedBySomebodyElse(this.selectedSIP))
			{
				//	Somebody else locked this SIP, so I can only open it in read-only mode or copy it:
				this.openSIPInWorkspaceAction.setEnabled(false);
				this.openSIPInWorkspaceReadWriteNoFileOpsAction.setEnabled(false);
				this.openSIPInWorkspaceReadOnlyAction.setEnabled(true);
				this.renameSIPInWorkspaceAction.setEnabled(false);
				this.copySIPInWorkspaceAction.setEnabled(true);
				this.deleteSIPInWorkspaceAction.setEnabled(false);
			}
			else if (Document.isLocked(this.selectedSIP))
			{
				//	I locked this SIP myself, so I can't rename or delete it:
				this.openSIPInWorkspaceAction.setEnabled(true);
				this.openSIPInWorkspaceReadWriteNoFileOpsAction.setEnabled(true);
				this.openSIPInWorkspaceReadOnlyAction.setEnabled(true);
				this.renameSIPInWorkspaceAction.setEnabled(false);
				this.copySIPInWorkspaceAction.setEnabled(true);
				this.deleteSIPInWorkspaceAction.setEnabled(false);
			}
			else
			{
				//	No locks, all is allowed:
				this.openSIPInWorkspaceAction.setEnabled(true);
				this.openSIPInWorkspaceReadWriteNoFileOpsAction.setEnabled(true);
				this.openSIPInWorkspaceReadOnlyAction.setEnabled(true);
				this.renameSIPInWorkspaceAction.setEnabled(true);
				this.copySIPInWorkspaceAction.setEnabled(true);
				this.deleteSIPInWorkspaceAction.setEnabled(true);
			}
		}
	}

	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	static protected class SIPFileChooser extends JFileChooser
	{
		static private final ImageIcon			SIPFolderIcon = new ImageIcon("./resources/images/Packet.png");
		static private final SIPFileChooser		Singleton = new SIPFileChooser();


		static public SIPFileChooser getInstance()
		{
			return Singleton;
		}


		private SIPFileChooser()
		{
			super(LastUsedOpenOrSaveDirectory);

			this.setDialogType(JFileChooser.OPEN_DIALOG);
			this.setMultiSelectionEnabled(false);
			this.setDialogTitle(I18N.translate("TitleOpenSIP"));
			this.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
			this.setFileFilter(
					new javax.swing.filechooser.FileFilter()
					{
						@Override
						public boolean accept(File file)
						{
							//	No hidden files or folders:
							if (file.isHidden() || file.getName().startsWith("."))		return false;

							if (file.isFile())
							{
								//	Only zip files:
								if (file.getName().toLowerCase().endsWith(".zip"))		return true;
							}
							else
							{
								//	Only folders that contain a mets.xml	//	ToDo: (this doesn't work: then all folders are invisible!):
//								if (Document.isValidSIPFolder(file))					return true;
								return true;
							}

							return false;
						}

						@Override
						public String getDescription()
						{
							return "SIP Folder or ZIP-File";
						}
					});

			this.setFileView(
					new FileView()
					{
						/**
						 * Whether the directory is traversable or not. This might be
						 * useful, for example, if you want a directory to represent
						 * a compound document and don't want the user to descend into it.
						 */
						@Override
						public Boolean isTraversable(File folder)
						{
							if (folder == null || !folder.exists() || !folder.canRead())		return false;
							if (folder.isFile())												return false;

							return (!Document.isValidSIPFolder(folder));
						}

						/**
						 * The icon that represents this file in the <code>JFileChooser</code>.
						 */
						@Override
						public Icon getIcon(File file)
						{
							if (file.isFile())
							{
								if (file.getName().toLowerCase().endsWith(".zip"))		return SIPFolderIcon;
							}
							else
							{
								if (Document.isValidSIPFolder(file))					return SIPFolderIcon;
							}

							return null;
						}
					});
		}
	}



	static public class SIPTableModel extends AbstractTableModel
	{
		static private DecimalFormat SizeFormatter = (DecimalFormat)DecimalFormat.getInstance();

		static private java.io.FileFilter SIPFileFilter =
			new java.io.FileFilter()
			{
				@Override
				public boolean accept(File file)
				{
					//	No hidden files or folders:
					if (file.isHidden() || file.getName().startsWith("."))		return false;

					if (file.isFile())
					{
						//	Only zip files:
						if (file.getName().toLowerCase().endsWith(".zip"))		return true;
					}
					else
					{
						//	Only folders that contain a mets.xml file:
						if (Document.isValidSIPFolder(file))					return true;
					}

					return false;
				}
			};

		private File[]					sips = new File[] {};


		public void setList()
		{
			File sipDir = new File(SIPDirectory);
			if (!sipDir.exists())		sipDir.mkdirs();

			this.sips = sipDir.listFiles(SIPFileFilter);
			if (this.sips == null)		this.sips = new File[] {};
			Arrays.sort(this.sips);

			this.fireTableDataChanged();
		}


		public File get(int rowIndex)
		{
			return this.sips[rowIndex];
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		@Override
		public int getRowCount()
		{
			return this.sips.length;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		@Override
		public int getColumnCount()
		{
			return 5;
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getColumnName(int columnIndex)
		{
			switch (columnIndex)
			{
				case 0:		return I18N.translate("HeaderLauncherSIP");
				case 1:		return I18N.translate("HeaderLauncherLockedBy");
				case 2:		return I18N.translate("HeaderLauncherSize");
				case 3:		return I18N.translate("HeaderLauncherLastChangeDate");
				case 4:		return I18N.translate("HeaderLauncherIsReadOnly");
			}

			return null;
		}


		/**
		 *  Returns <code>Object.class</code> regardless of <code>columnIndex</code>.
		 *
		 *  @param columnIndex  the column being queried
		 *  @return the Object.class
		 */
		@Override
		public Class<?> getColumnClass(int columnIndex)
		{
			switch (columnIndex)
			{
				case 2:		return Long.class;
				case 3:		return Long.class;
				case 4:		return Boolean.class;
				default:	return super.getColumnClass(columnIndex);
			}
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		@Override
		public Object getValueAt(int rowIndex, int columnIndex)
		{
			File file = this.sips[rowIndex];
			String filePath = file.getPath();

			switch (columnIndex)
			{
				case 0:		return file.getName();
				case 1:		return Document.lockedByWhom(filePath);
				case 2:		return SizeFormatter.format((file.isFile() ? file.length() : FileUtils.sizeOfDirectory(file)) / 1024);
				case 3:		return (file.isFile()? file: new File(file.getPath() + File.separator + Document.DefaultMETSFileName)).lastModified();
				case 4:		return Document.isLockedBySomebodyElse(file);
			}

			return null;
		}
	}



	static protected class DateTableCellRenderer extends DefaultTableCellRenderer
	{
		private DateTableCellRenderer()
		{
			super();
			this.setHorizontalAlignment(DefaultTableCellRenderer.RIGHT);
		}


		@Override
		public void setValue(Object object)
		{
			if (object == null)		return;
			super.setValue(DateFormatter.getDateTimeString((Long)object, "dd.MM.yyyy HH:mm:ss"));
		}
	}



	static protected class CreateNewSIPDialog extends JDialog
	{
		protected JButton							selectSourceFileOrFolderButton;
		protected JButton							selectDestinationZIPOrFolderButton;
		protected JButton							selectDestinationIsWorkspaceButton;
		protected JButton							goButton;

		protected JRadioButton						selectSIPEmptyRadioButton;
		protected JRadioButton						selectSIPFromSourceFileOrFolderRadioButton;

		protected JTextField						rootFolderNameTextField;
		protected JTextField						sourceFileOrFolderTextField;
		protected JTextField						destinationFolderTextField;
		protected JTextField						destinationNameTextField;
		protected JCheckBox							beZIPCheckBox;
		protected JComboBox							saComboBox;

		protected JLabel							messageLabel;

		protected boolean							goButtonWasClicked = false;


		protected CreateNewSIPDialog(JFrame owner)
		{
			super(owner, I18N.translate("TitleCreateNewSIP"), true);

			this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			this.getRootPane().registerKeyboardAction(
					new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { CreateNewSIPDialog.this.close(); }},
					KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
					JComponent.WHEN_IN_FOCUSED_WINDOW);

			this.selectSIPEmptyRadioButton = new JRadioButton();
			this.selectSIPEmptyRadioButton.setSelected(true);
			this.selectSIPEmptyRadioButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.selectSIPRootRadioButtonClicked(); }});
			this.selectSIPFromSourceFileOrFolderRadioButton = new JRadioButton();
			this.selectSIPFromSourceFileOrFolderRadioButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.selectSIPRootRadioButtonClicked(); }});

			ButtonGroup radioButtonGroup = new ButtonGroup();
			radioButtonGroup.add(this.selectSIPEmptyRadioButton);
			radioButtonGroup.add(this.selectSIPFromSourceFileOrFolderRadioButton);

			this.rootFolderNameTextField = new JTextField("");
			this.rootFolderNameTextField.setToolTipText(I18N.translate("ToolTipRootFolderName"));
			this.rootFolderNameTextField.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.rootFolderNameTextFieldChanged(); }});
			this.rootFolderNameTextField.addFocusListener(
					new FocusAdapter(){ @Override public void focusLost(FocusEvent e)
						{ CreateNewSIPDialog.this.rootFolderNameTextFieldChanged(); }});

			this.selectSourceFileOrFolderButton = new JButton(new ImageIcon("./resources/images/OpenFolder.png"));
			this.selectSourceFileOrFolderButton.setToolTipText(I18N.translate("ToolTipSelectSource"));
			this.selectSourceFileOrFolderButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.selectSourceFolderButtonClicked(); }});

			this.sourceFileOrFolderTextField = new JTextField(new File(DataDirectory).getAbsolutePath());
			this.sourceFileOrFolderTextField.setToolTipText(I18N.translate("ToolTipSourceFileOrFolder"));
			this.sourceFileOrFolderTextField.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.sourceFolderTextFieldChanged(); }});
			this.sourceFileOrFolderTextField.addFocusListener(
					new FocusAdapter(){ @Override public void focusLost(FocusEvent e)
						{ CreateNewSIPDialog.this.sourceFolderTextFieldChanged(); }});

			this.selectDestinationZIPOrFolderButton = new JButton(new ImageIcon("./resources/images/OpenFolder.png"));
			this.selectDestinationZIPOrFolderButton.setToolTipText(I18N.translate("ToolTipSelectDestinationFolder"));
			this.selectDestinationZIPOrFolderButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.selectDestinationFolderButtonClicked(); }});

			this.selectDestinationIsWorkspaceButton = new JButton(new ImageIcon("./resources/images/Workspace.png"));
			this.selectDestinationIsWorkspaceButton.setToolTipText(I18N.translate("ToolTipSelectDestinationIsWorkspaceFolder"));
			this.selectDestinationIsWorkspaceButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.selectDestinationIsWorkspaceButtonClicked(); }});

			this.destinationFolderTextField = new JTextField(new File(LastUsedOpenOrSaveDirectory).getAbsolutePath());
			this.destinationFolderTextField.setToolTipText(I18N.translate("ToolTipDestinationFolder"));
			this.destinationFolderTextField.addFocusListener(
					new FocusAdapter()
					{ 
						@Override public void focusLost(FocusEvent e)
						{ 
							CreateNewSIPDialog.this.destinationFolderTextField
									.setText(CreateNewSIPDialog.this.destinationFolderTextField.getText().trim()); 
						}
					});

			this.destinationNameTextField = new JTextField(new File(DataDirectory).getName());
			this.destinationNameTextField.setToolTipText(I18N.translate("ToolTipDestinationName"));
			this.destinationNameTextField.addKeyListener(
					new KeyAdapter(){ @Override public void keyTyped(KeyEvent e)
						{ CreateNewSIPDialog.this.enableOrDisableButtonsAndFields(); }});
			this.destinationNameTextField.addFocusListener(
					new FocusAdapter()
					{ 
						@Override public void focusLost(FocusEvent e)
						{ 
							CreateNewSIPDialog.this.destinationNameTextField
									.setText(CreateNewSIPDialog.this.destinationNameTextField.getText().trim()); 
						}
					});

			this.beZIPCheckBox = new JCheckBox("ZIP", ch.docuteam.packer.gui.LauncherView.NewSIPDefaultsToZipped);
			this.beZIPCheckBox.setToolTipText(I18N.translate("ToolTipBeZIP"));

			this.saComboBox = new JComboBox(this.readSAOverviewFile().toArray());
			this.saComboBox.setToolTipText(I18N.translate("ToolTipSelectSA"));

			this.messageLabel = new JLabel();

			this.goButton = new JButton(new ImageIcon("./resources/images/Save.png"));
			this.goButton.setToolTipText(I18N.translate("ToolTipCreateNew"));
			this.goButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPDialog.this.goButtonClicked(); }});


			GridBagPanel gridBag = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(0, 5, 0, 0));
			gridBag.add(this.selectSIPEmptyRadioButton,								0,    0,   	GridBagConstraints.EAST);
			gridBag.add(new JLabel(I18N.translate("LabelNewSIPRootName")),			0,    1,   	GridBagConstraints.EAST);
			gridBag.add(this.rootFolderNameTextField,								0,    4,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(this.selectSIPFromSourceFileOrFolderRadioButton,			1,    0,   	GridBagConstraints.EAST);
			gridBag.add(new JLabel(I18N.translate("LabelNewSIPSource")),			1,    1,   	GridBagConstraints.EAST);
			gridBag.add(this.selectSourceFileOrFolderButton,						1,    3);
			gridBag.add(this.sourceFileOrFolderTextField,							1, 1, 4, 6,	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(new JLabel(" "), 											2,    2);
			gridBag.add(new JLabel(I18N.translate("LabelNewSIPDestination")),		3,    1,   	GridBagConstraints.EAST);
			gridBag.add(this.selectDestinationIsWorkspaceButton,					3,    2);
			gridBag.add(this.selectDestinationZIPOrFolderButton,					3,    3);
			gridBag.add(this.destinationFolderTextField,							3,    4,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	2, 0);
			gridBag.add(this.destinationNameTextField,								3,    5,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(this.beZIPCheckBox,											3,    6);
			gridBag.add(new JLabel(I18N.translate("LabelNewSIPSA")),				4,    1,   	GridBagConstraints.EAST);
			gridBag.add(this.saComboBox,											4, 4, 4, 6,	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(this.messageLabel,											5, 5, 0, 5,	GridBagConstraints.EAST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(this.goButton,												5,    6,   	GridBagConstraints.EAST);

			this.add(gridBag);

			this.setPreferredSize(new Dimension(800, 200));
			this.pack();
			this.setLocationRelativeTo(owner);

			this.rootFolderNameTextField.requestFocusInWindow();
			this.selectSIPRootRadioButtonClicked();

			this.setVisible(true);
		}


		protected void selectSIPRootRadioButtonClicked()
		{
			if (this.selectSIPEmptyRadioButton.isSelected())
				this.destinationNameTextField.setText(this.rootFolderNameTextField.getText());
			else
				this.destinationNameTextField.setText(new File(this.sourceFileOrFolderTextField.getText()).getName());

			this.enableOrDisableButtonsAndFields();
		}


		protected void selectSourceFolderButtonClicked()
		{
			JFileChooser fileChooser = new JFileChooser(this.sourceFileOrFolderTextField.getText());
			fileChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
			fileChooser.setDialogTitle(I18N.translate("TitleSelectSourceFileOrFolder"));
			fileChooser.setMultiSelectionEnabled(false);
			int result = fileChooser.showOpenDialog(this);
			if (result == JFileChooser.CANCEL_OPTION)		return;

			this.sourceFileOrFolderTextField.setText(fileChooser.getSelectedFile().getPath());
			this.sourceFolderTextFieldChanged();
		}


		protected void rootFolderNameTextFieldChanged()
		{
			this.rootFolderNameTextField.setText(this.rootFolderNameTextField.getText().trim());
			this.destinationNameTextField.setText(this.rootFolderNameTextField.getText());
			this.enableOrDisableButtonsAndFields();
		}


		protected void sourceFolderTextFieldChanged()
		{
			this.sourceFileOrFolderTextField.setText(this.sourceFileOrFolderTextField.getText().trim());
			this.destinationNameTextField.setText(new File(this.sourceFileOrFolderTextField.getText()).getName());
			this.enableOrDisableButtonsAndFields();
		}


		protected void selectDestinationFolderButtonClicked()
		{
			JFileChooser fileChooser = new JFileChooser(this.destinationFolderTextField.getText());
			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			fileChooser.setDialogTitle(I18N.translate("TitleSelectDestinationFolder"));
			fileChooser.setMultiSelectionEnabled(false);
			int result = fileChooser.showSaveDialog(this);
			if (result == JFileChooser.CANCEL_OPTION)		return;

			this.destinationFolderTextField.setText(fileChooser.getSelectedFile().getPath());
		}


		protected void selectDestinationIsWorkspaceButtonClicked()
		{
			this.destinationFolderTextField.setText(SIPDirectory);
		}


		protected void goButtonClicked()
		{
			String rootFolderName = this.rootFolderNameTextField.getText();
			String sourceFileOrFolder = this.sourceFileOrFolderTextField.getText();
			String destinationFolder = this.destinationFolderTextField.getText();
			String destinationName = this.destinationNameTextField.getText();

			//	Don't accept empty SIP name:
			if (destinationName.isEmpty())
			{
				GUIUtil.shake(this);
				this.messageLabel.setText(I18N.translate("MessageDestinationNameIsEmpty"));
				return;
			}

			if (this.selectSIPFromSourceFileOrFolderRadioButton.isSelected())
			{
				//	Don't accept empty source file or folder:
				if (sourceFileOrFolder.isEmpty())
				{
					GUIUtil.shake(this);
					this.messageLabel.setText(I18N.translate("MessageSourceFileOrFolderIsEmpty"));
					return;
				}

				//	If the destination folder lies within the source folder, show message and reject:
				if (destinationFolder.contains(sourceFileOrFolder))
				{
					GUIUtil.shake(this);
					this.messageLabel.setText(I18N.translate("MessageDestinationIsWithinSource"));
					return;
				}
			}
			else
			{
				//	Don't accept empty root folder name:
				if (rootFolderName.isEmpty())
				{
					GUIUtil.shake(this);
					this.messageLabel.setText(I18N.translate("MessageRootFolderNameIsEmpty"));
					return;
				}
			}

			if (this.beZIPCheckBox.isSelected())
			{
				if (!destinationName.toLowerCase().endsWith(".zip"))	destinationName += ".zip";
			}
			else
			{
				if (destinationName.toLowerCase().endsWith(".zip"))		destinationName = destinationName.substring(0, destinationName.length() - 4);
			}

			File destinationFile = new File(destinationFolder + "/" + destinationName);

			//	Don't accept if a SIP with this name already exists:
			if (destinationFile.exists())
			{
				GUIUtil.shake(this);
				this.messageLabel.setText(I18N.translate("MessageSIPExistsAlready"));
				return;
			}

			//	Remember the Data and SIP directories:
			DataDirectory = sourceFileOrFolder;
			LastUsedOpenOrSaveDirectory = destinationFolder;

			this.goButtonWasClicked = true;
			this.close();
		}


		protected void close()
		{
			this.setVisible(false);
			this.dispose();
		}

		protected List<Overview> readSAOverviewFile()
		{
			return SubmissionAgreement.getAllFinalOverviews();
		}


		protected void enableOrDisableButtonsAndFields()
		{
			if (this.selectSIPEmptyRadioButton.isSelected())
			{
				this.rootFolderNameTextField.setEnabled(true);
				this.selectSourceFileOrFolderButton.setEnabled(false);
				this.sourceFileOrFolderTextField.setEnabled(false);

				this.goButton.setEnabled(!(this.destinationNameTextField.getText().isEmpty() || this.rootFolderNameTextField.getText().isEmpty()));
			}
			else
			{
				this.rootFolderNameTextField.setEnabled(false);
				this.selectSourceFileOrFolderButton.setEnabled(true);
				this.sourceFileOrFolderTextField.setEnabled(true);

				this.goButton.setEnabled(!(this.destinationNameTextField.getText().isEmpty() || this.sourceFileOrFolderTextField.getText().isEmpty()));
			}
		}
	}



	static protected class CreateNewSIPFromTemplateDialog extends JDialog
	{
		protected JButton				selectDestinationZIPOrFolderButton;
		protected JButton				selectDestinationIsWorkspaceButton;
		protected JButton				goButton;

		protected JTextField			destinationFolderTextField;
		protected JTextField			destinationNameTextField;
		protected JComboBox				templateComboBox;
		protected JCheckBox				beZIPCheckBox;

		protected JLabel				messageLabel;

		protected boolean				goButtonWasClicked = false;


		protected CreateNewSIPFromTemplateDialog(JFrame owner)
		{
			super(owner, I18N.translate("TitleCreateNewSIPFromTemplate"), true);

			this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			this.getRootPane().registerKeyboardAction(
					new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { CreateNewSIPFromTemplateDialog.this.close(); }},
					KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
					JComponent.WHEN_IN_FOCUSED_WINDOW);

			this.templateComboBox = new JComboBox(this.getTemplateNames());
			this.templateComboBox.setToolTipText(I18N.translate("ToolTipSelectTemplate"));
			this.templateComboBox.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPFromTemplateDialog.this.templateSelected(); }});

			this.selectDestinationZIPOrFolderButton = new JButton(new ImageIcon("./resources/images/OpenFolder.png"));
			this.selectDestinationZIPOrFolderButton.setToolTipText(I18N.translate("ToolTipSelectDestinationFolder"));
			this.selectDestinationZIPOrFolderButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPFromTemplateDialog.this.selectDestinationFolderButtonClicked(); }});

			this.selectDestinationIsWorkspaceButton = new JButton(new ImageIcon("./resources/images/Workspace.png"));
			this.selectDestinationIsWorkspaceButton.setToolTipText(I18N.translate("ToolTipSelectDestinationIsWorkspaceFolder"));
			this.selectDestinationIsWorkspaceButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPFromTemplateDialog.this.selectDestinationIsWorkspaceButtonClicked(); }});

			this.destinationFolderTextField = new JTextField(new File(LastUsedOpenOrSaveDirectory).getAbsolutePath());
			this.destinationFolderTextField.setToolTipText(I18N.translate("ToolTipDestinationFolder"));

			this.destinationNameTextField = new JTextField((String)this.templateComboBox.getSelectedItem());
			this.destinationNameTextField.setToolTipText(I18N.translate("ToolTipDestinationName"));
			this.destinationNameTextField.addKeyListener(
					new KeyAdapter(){ @Override public void keyTyped(KeyEvent e)
						{ CreateNewSIPFromTemplateDialog.this.enableOrDisableButtonsAndFields(); }});

			this.beZIPCheckBox = new JCheckBox("ZIP", ch.docuteam.packer.gui.LauncherView.NewSIPDefaultsToZipped);
			this.beZIPCheckBox.setToolTipText(I18N.translate("ToolTipBeZIP"));

			this.messageLabel = new JLabel();

			this.goButton = new JButton(new ImageIcon("./resources/images/Save.png"));
			this.goButton.setToolTipText(I18N.translate("ToolTipCreateNew"));
			this.goButton.addActionListener(
					new ActionListener(){ @Override public void actionPerformed(ActionEvent e)
						{ CreateNewSIPFromTemplateDialog.this.goButtonClicked(); }});


			GridBagPanel gridBag = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(0, 5, 0, 0));
			gridBag.add(new JLabel(I18N.translate("LabelTemplate")),				1,    0,   	GridBagConstraints.EAST);
			gridBag.add(this.templateComboBox,										1,    3,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(new JLabel(I18N.translate("LabelNewSIPDestination")),		2,    0,   	GridBagConstraints.EAST);
			gridBag.add(this.selectDestinationIsWorkspaceButton,					2,    1);
			gridBag.add(this.selectDestinationZIPOrFolderButton,					2,    2);
			gridBag.add(this.destinationFolderTextField,							2,    3,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	2, 0);
			gridBag.add(this.destinationNameTextField,								2,    4,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(this.beZIPCheckBox,											2,    5);
			gridBag.add(this.messageLabel,											3, 3, 0, 4,	GridBagConstraints.EAST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(this.goButton,												3,    5,   	GridBagConstraints.EAST);

			this.add(gridBag);

			this.setPreferredSize(new Dimension(800, 150));
			this.pack();
			this.setLocationRelativeTo(owner);

			this.templateComboBox.requestFocusInWindow();
			this.enableOrDisableButtonsAndFields();

			this.setVisible(true);
		}


		protected void templateSelected()
		{
			this.destinationNameTextField.setText((String)this.templateComboBox.getSelectedItem());

			CreateNewSIPFromTemplateDialog.this.enableOrDisableButtonsAndFields();
		}


		protected void selectDestinationFolderButtonClicked()
		{
			JFileChooser fileChooser = new JFileChooser(this.destinationFolderTextField.getText());
			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			fileChooser.setDialogTitle(I18N.translate("TitleSelectDestinationFolder"));
			fileChooser.setMultiSelectionEnabled(false);
			int result = fileChooser.showSaveDialog(this);
			if (result == JFileChooser.CANCEL_OPTION)		return;

			this.destinationFolderTextField.setText(fileChooser.getSelectedFile().getPath());
		}


		protected void selectDestinationIsWorkspaceButtonClicked()
		{
			this.destinationFolderTextField.setText(SIPDirectory);
		}


		protected void goButtonClicked()
		{
			String templateName = (String)this.templateComboBox.getSelectedItem();
			String destinationFolder = this.destinationFolderTextField.getText();
			String destinationName = this.destinationNameTextField.getText();

			//	Don't accept empty template name:
			if (templateName == null)
			{
				GUIUtil.shake(this);
				this.messageLabel.setText(I18N.translate("MessageTemplateNameIsEmpty"));
				return;
			}

			//	Don't accept empty file name:
			if (destinationName.isEmpty())
			{
				GUIUtil.shake(this);
				this.messageLabel.setText(I18N.translate("MessageDestinationNameIsEmpty"));
				return;
			}

			if (this.beZIPCheckBox.isSelected())
			{
				if (!destinationName.toLowerCase().endsWith(".zip"))	destinationName += ".zip";
			}
			else
			{
				if (destinationName.toLowerCase().endsWith(".zip"))		destinationName = destinationName.substring(0, destinationName.length() - 4);
			}

			File destinationFile = new File(destinationFolder + "/" + destinationName);

			//	Don't accept if a SIP with this name already exists:
			if (destinationFile.exists())
			{
				GUIUtil.shake(this);
				this.messageLabel.setText(I18N.translate("MessageSIPExistsAlready"));
				return;
			}

			//	Remember the SIP directory:
			LastUsedOpenOrSaveDirectory = destinationFolder;

			this.goButtonWasClicked = true;
			this.close();
		}

		protected void close()
		{
			this.setVisible(false);
			this.dispose();
		}


		private String[] getTemplateNames()
		{
			File templateDirectory = new File(TemplateDirectory);
			if (!templateDirectory.exists() || templateDirectory.isFile())		return new String[]{};

			File[] templates = templateDirectory.listFiles((FilenameFilter)ch.docuteam.tools.file.FileFilter.VisibleAll);
			Vector<String> templateNames = new Vector<String>();
			for (File file: templates)		templateNames.add(file.getName());

			return templateNames.toArray(new String[]{});
		}


		protected void enableOrDisableButtonsAndFields()
		{
			 this.goButton.setEnabled(!this.destinationNameTextField.getText().isEmpty() && this.templateComboBox.getSelectedItem() != null);
		}
	}



	static protected class AboutView extends JDialog
	{
		protected JEditorPane		docuteamLink;
		protected JEditorPane		licenseLink;

		//	===========================================================================================
		//	========	Methods					=======================================================
		//	===========================================================================================

		//	========	Static Initializer		=======================================================

		//	========	Constructors Public		=======================================================

		/**
		 * This is for subclasses.
		 */
		protected AboutView(JFrame owner, String title)
		{
			super(owner, title, true);

			this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			this.getRootPane().registerKeyboardAction(
					new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { AboutView.this.close(); }},
					KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
					JComponent.WHEN_IN_FOCUSED_WINDOW);

			this.docuteamLink = new JEditorPane("text/html", "<span style='font-family:Arial'><a href='http://www.docuteam.ch'>Docuteam GmbH</a></span>");
			this.docuteamLink.setEditable(false);
			this.docuteamLink.setOpaque(false);
			this.docuteamLink.addHyperlinkListener(new HyperlinkListener()
			{ @Override public void hyperlinkUpdate(HyperlinkEvent hle) { if (HyperlinkEvent.EventType.ACTIVATED.equals(hle.getEventType())) { AboutView.this.openURL(hle.getURL()); }}});

			this.licenseLink = new JEditorPane("text/html", "<span style='font-family:Arial'><a href='http://www.gnu.org/licenses'>GNU General Public License</a></span>");
			this.licenseLink.setEditable(false);
			this.licenseLink.setOpaque(false);
			this.licenseLink.addHyperlinkListener(new HyperlinkListener()
			{ @Override public void hyperlinkUpdate(HyperlinkEvent hle) { if (HyperlinkEvent.EventType.ACTIVATED.equals(hle.getEventType())) { AboutView.this.openURL(hle.getURL()); }}});
		}


		protected AboutView(JFrame owner)
		{
			this(owner, I18N.translate("TitleAbout") + VersionProduct);

			GridBagPanel gridBagPanel1 = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(5, 5, 5, 5));
			gridBagPanel1.add(new JLabel(new ImageIcon("./resources/images/Logo_docuteam_packer.png")),
																							0, 0, 0, 2,	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel(VersionProduct),									1,    0,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(VersionNumber),									1,    1,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(VersionLastChange),								1,    2,   	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel(ch.docuteam.darc.admin.Version.Product),			2,    0,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(ch.docuteam.darc.admin.Version.Number),			2,    1,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(ch.docuteam.darc.admin.Version.LastChange),		2,    2,   	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel(ch.docuteam.converter.admin.Version.Product),		3,    0,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(ch.docuteam.converter.admin.Version.Number),		3,    1,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(ch.docuteam.converter.admin.Version.LastChange),	3,    2,   	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel(ch.docuteam.tools.admin.Version.Product),			4,    0,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(ch.docuteam.tools.admin.Version.Number),			4,    1,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(ch.docuteam.tools.admin.Version.LastChange),		4,    2,   	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel("Copyright (C) " + DateFormatter.getCurrentDateTimeString("yyyy") + " by: "),
																							5, 5, 0, 1,	GridBagConstraints.WEST);
			gridBagPanel1.add(this.docuteamLink,											5,    2,   	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel("License: "),										6,    0,   	GridBagConstraints.WEST);
			gridBagPanel1.add(this.licenseLink,												6, 6, 1, 2,	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel("OS: "),											7,    0,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(OperatingSystem.osName() + " " + OperatingSystem.osVersion()),
																							7, 7, 1, 2,	GridBagConstraints.EAST);
			gridBagPanel1.add(new JLabel("JVM: "),											8,    0,   	GridBagConstraints.WEST);
			gridBagPanel1.add(new JLabel(System.getProperty("java.vendor") + " " + System.getProperty("java.version")),
																							8, 8, 1, 2,	GridBagConstraints.EAST);

			GridBagPanel gridBagPanel2 = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(5, 5, 5, 5));
			gridBagPanel2.add(new JLabel("<html><b><u>OpenOffice Installations:</u></b></html>"),
																							0, 0, 0, 3,	GridBagConstraints.CENTER);
			gridBagPanel2.add(new JLabel("Local:"),											2,    2,   	GridBagConstraints.CENTER);
			gridBagPanel2.add(new JLabel("Remote:"),										2,    3,   	GridBagConstraints.CENTER);
			gridBagPanel2.add(new JLabel("Windows:"),										3,    1,   	GridBagConstraints.EAST);
			gridBagPanel2.add(new JLabel(FileConverter.OOConverter.isInstalledLocallyForWindows()? "X": ""),
																							3,    2,   	GridBagConstraints.CENTER);
			gridBagPanel2.add(new JLabel(FileConverter.OOConverter.getRemotePathForWindows()),
																							3,    3,   	GridBagConstraints.WEST,	GridBagConstraints.HORIZONTAL,	1, 0);
			gridBagPanel2.add(new JLabel("OS X:"),											4,    1,   	GridBagConstraints.EAST);
			gridBagPanel2.add(new JLabel(FileConverter.OOConverter.isInstalledLocallyForOSX()? "X": ""),
																							4,    2,   	GridBagConstraints.CENTER);
			gridBagPanel2.add(new JLabel(FileConverter.OOConverter.getRemotePathForOSX()),	4,    3,   	GridBagConstraints.WEST,	GridBagConstraints.HORIZONTAL,	1, 0);
			gridBagPanel2.add(new JLabel("Linux:"),											5,    1,   	GridBagConstraints.EAST);
			gridBagPanel2.add(new JLabel(FileConverter.OOConverter.isInstalledLocallyForLinux()? "X": ""),
																							5,    2,   	GridBagConstraints.CENTER);
			gridBagPanel2.add(new JLabel(FileConverter.OOConverter.getRemotePathForLinux()),
																							5,    3,   	GridBagConstraints.WEST,	GridBagConstraints.HORIZONTAL,	1, 0);

			Box box = new Box(BoxLayout.Y_AXIS);
			box.add(gridBagPanel1);
			box.add(gridBagPanel2);
			this.add(box);

			this.setPreferredSize(new Dimension(450, 500));
			this.setResizable(true);
			this.pack();
			this.setLocationRelativeTo(owner);
			this.setVisible(true);
		}


		protected void openURL(URL url)
		{
			try
			{
				Desktop.getDesktop().browse(url.toURI());
			}
			catch (java.lang.Exception ex)
			{
				ex.printStackTrace();
			}
		}


		protected void close()
		{
			this.setVisible(false);
			this.dispose();
		}
	}



	static protected class CopySIPDialog extends JDialog
	{
		protected JTextField			copyNameTextField;
		protected JCheckBox				beZIPCheckBox;
		protected JButton				goButton;

		protected boolean				goButtonWasClicked = false;


		static protected String open(JFrame owner, String message, String title, String textFieldContent)
		{
			CopySIPDialog dialog = new CopySIPDialog(owner, message, title, textFieldContent);

			if (!dialog.goButtonWasClicked)					return null;

			String name = dialog.copyNameTextField.getText();
			if (dialog.beZIPCheckBox.isSelected())
			{
				if (!name.toLowerCase().endsWith(".zip"))	name += ".zip";
			}
			else
			{
				if (name.toLowerCase().endsWith(".zip"))	name = FileUtil.asFilePathWithoutExtension(name);
			}

			return name;
		}


		private CopySIPDialog(JFrame owner, String message, String title, String textFieldContent)
		{
			super(owner, title, true);

			this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));
			this.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
			this.getRootPane().registerKeyboardAction(
					new AbstractAction() { @Override public void actionPerformed(ActionEvent e) { CopySIPDialog.this.close(); }},
					KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
					JComponent.WHEN_IN_FOCUSED_WINDOW);

			this.copyNameTextField = new JTextField(textFieldContent);
			this.copyNameTextField.addActionListener(
					new ActionListener() { @Override public void actionPerformed(ActionEvent e) { CopySIPDialog.this.goButtonWasClicked(); }});

			this.beZIPCheckBox = new JCheckBox("ZIP", ch.docuteam.packer.gui.LauncherView.NewSIPDefaultsToZipped);
			this.beZIPCheckBox.setToolTipText(I18N.translate("ToolTipBeZIP"));

			this.goButton = new JButton(new ImageIcon("./resources/images/Save.png"));
			this.goButton.addActionListener(
					new ActionListener() { @Override public void actionPerformed(ActionEvent e) { CopySIPDialog.this.goButtonWasClicked(); }});

			new JLabel(new ImageIcon("./resources/images/DocuteamPacker.png"));
			GridBagPanel gridBag = new GridBagPanel(new EmptyBorder(10, 10, 10, 10), new Insets(0, 5, 0, 5));
			gridBag.add(new JLabel(new ImageIcon("./resources/images/DocuteamPacker.png")),
																					0, 2, 0, 0,	GridBagConstraints.CENTER);
			gridBag.add(new MultiLineLabel(message, Component.LEFT_ALIGNMENT),		1,    1,   	GridBagConstraints.WEST);
			gridBag.add(this.copyNameTextField,										2,    1,   	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL,	1, 0);
			gridBag.add(this.beZIPCheckBox,											2,    2);
			gridBag.add(this.goButton,												3,    2,   	GridBagConstraints.EAST);

			this.add(gridBag);

			this.setPreferredSize(new Dimension(450, 150));
			this.pack();
			this.setLocationRelativeTo(owner);

			this.copyNameTextField.requestFocusInWindow();

			this.setVisible(true);
		}


		private void goButtonWasClicked()
		{
			this.goButtonWasClicked = true;
			this.close();
		}


		private void close()
		{
			this.setVisible(false);
			this.dispose();
		}

	}

}
