/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.sa.dss;

import java.io.FileNotFoundException;
import java.util.HashMap;
import java.util.Map;

import org.dom4j.Element;

import ch.docuteam.darc.sa.SubmissionAgreement;
import ch.docuteam.darc.sa.common.NodeAbstract;


/**
 * This class, used by <a href="../SubmissionAgreement.html">SubmissionAgreement</a>, represents a Submission Agreement's Data Submission Session.
 * It contains an ID, a title, a category, a record type, and an instance of <a href="./FileFormats.html">FileFormats</a>.
 *
 * @author denis
 *
 */
public class DataSubmissionSession extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String								dssID;

	private String								title;
	private DataSubmissionSessionCategory		category;
	private DataSubmissionSessionRecordType		recordTypes;
	private FileFormats							fileFormats;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	private DataSubmissionSession(SubmissionAgreement sa, Element element)
	{
		this.sa = sa;
		this.element = element;

		this.dssID = element.attributeValue("dssID");

		this.title = this.getElementText("SA:title", true);
		this.category = DataSubmissionSessionCategory.get(this.getElementText("SA:category", true));
		this.recordTypes = DataSubmissionSessionRecordType.get(this.getElementText("SA:recordTypes", true));

		this.fileFormats = FileFormats.parse(this.sa, this.element);
	}

	//	========	Static Public			=======================================================

	static public Map<String, DataSubmissionSession> parse(SubmissionAgreement sa, Element parent)
	{
		Map<String, DataSubmissionSession> dsss = new HashMap<String, DataSubmissionSession>();

		for (Object o: parent.selectNodes("./SA:dataSubmissionSession"))
		{
			DataSubmissionSession dss = new DataSubmissionSession(sa, (Element)o);
			dsss.put(dss.getDssID(), dss);
		}

		return dsss;
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getDssID()
	{
		return this.dssID;
	}

	public String getTitle()
	{
		return this.title;
	}

	public DataSubmissionSessionCategory getCategory()
	{
		return this.category;
	}

	public DataSubmissionSessionRecordType getRecordTypes()
	{
		return this.recordTypes;
	}

	public FileFormats getFileFormats()
	{
		return this.fileFormats;
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean allowsFile(String filePath) throws FileNotFoundException
	{
		// If no file formats are present, allow all files:
		if (this.fileFormats == null)		return true;

		return this.fileFormats.allowsFile(this.dssID, filePath);
	}

	public boolean allowsFileType(String filePath, String puid, String mimeType)
	{
		// If no file formats are present, allow all files:
		if (this.fileFormats == null)		return true;

		return this.fileFormats.allowsFileType(this.dssID, filePath, puid, mimeType);
	}

	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return new StringBuilder("[DSS:")
			.append(this.dssID + "/")
			.append(this.title + "/")
			.append(this.category + "/")
			.append(this.recordTypes + "/")
			.append(this.fileFormats + "]")
			.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
