/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import uk.gov.nationalarchives.droid.core.interfaces.IdentificationResult;
import ch.docuteam.tools.out.Logger;
import edu.harvard.hul.ois.jhove.*;

/**
 * @author denis
 *
 */
public class FileWithMetadata extends File
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String		Unknown = "(n/a)";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private IdentificationResult	DROIDData = null;
	private boolean					alreadyTriedToRetrieveDROID = false;

	private RepInfo					jhoveData = null;
	private boolean					alreadyTriedToRetrieveJhove = false;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	/**
	 * @param pathname
	 */
	public FileWithMetadata(String pathname)
	{
		super(pathname);
	}

	/**
	 * @param uri
	 */
	public FileWithMetadata(URI uri)
	{
		super(uri);
	}

	/**
	 * @param parent
	 * @param child
	 */
	public FileWithMetadata(String parent, String child)
	{
		super(parent, child);
	}

	/**
	 * @param parent
	 * @param child
	 */
	public FileWithMetadata(File parent, String child)
	{
		super(parent, child);
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getFormatName()
	{
		//	Note: formatName from droid takes precedence over formatName from jhove:
		String formatNameDroid = this.getFileFormatName_droid();
		return (Unknown.equals(formatNameDroid)? this.getFormat_jhove(): formatNameDroid);
	}

	public String getFormatName(Provider provider)
	{
		if (provider == Provider.DROID)				return this.getFileFormatName_droid();
		else if (provider == Provider.JHOVE)		return this.getFormat_jhove();

		return Unknown;
	}

	public String getFormatPronomID()
	{
		return this.getFileFormatPUID_droid();
	}

	public String getMimeType()
	{
		//	Note: mimeType from droid takes precedence over mimeType from jhove:
		String mimeTypeDroid = this.getMimeType_droid();
		return (Unknown.equals(mimeTypeDroid)? this.getMimeType_jhove(): mimeTypeDroid);
	}

	public String getMimeType(Provider provider)
	{
		if (provider == Provider.DROID)				return this.getMimeType_droid();
		else if (provider == Provider.JHOVE)		return this.getMimeType_jhove();

		return Unknown;
	}

	public int getWellFormed()
	{
		return this.getWellFormed_jhove();
	}

	public int getValid()
	{
		return this.getValid_jhove();
	}

	public boolean isConsistent()
	{
		return this.isConsistent_jhove();
	}

	public long getSize()
	{
		return this.getSize_jhove();
	}

	public Module getModule()
	{
		return this.getModule_jhove();
	}

	public List<String> getProfile()
	{
		return this.getProfile_jhove();
	}

	public String getChecksum()
	{
		return this.getChecksum(ChecksumType.MD5);
	}

	public String getChecksum(ChecksumType checksumType)
	{
		return this.getChecksum_jhove(checksumType);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	//	--------		JHove metaData		-------------------------------------------------------

	/**
	 * @return The property jHoveInfo, if it is already initialized. If not, initialize the property jHoveInfo and return it.
	 * It can however happen that it can't be initialized; in this case, return null.
	 */
	private RepInfo getJhoveMetadata()
	{
		Logger.getLogger().debug("Getting jhove Metadata...");

		if (this.jhoveData == null)		this.initializeJhoveMetadata();
		return this.jhoveData;
	}


	private void initializeJhoveMetadata()
	{
		//	If I already tried to get Jhove, don't try again:
		if (this.alreadyTriedToRetrieveJhove)
		{
			Logger.getLogger().debug("NOT retrying to retrieve jhove Metadata for file:" + this.getPath());
			return;
		}

		String filePath;
		try
		{
			filePath = this.getCanonicalPath();
		}
		catch (IOException ex)
		{
			ex.printStackTrace();
			return;
		}

		Logger.getLogger().debug("Retrieving jhove Metadata for file:" + filePath + "...");

		this.jhoveData = MetadataProviderJHOVE.getRepInfo(filePath);		//	May be null in case of an error
		if (this.jhoveData == null)		Logger.getLogger().debug("...NOK!");
		else						Logger.getLogger().debug("...OK");

		this.alreadyTriedToRetrieveJhove = true;
	}

	//	--------		Droid metaData		-------------------------------------------------------

	/**
	 * @return The property jHoveInfo, if it is already initialized. If not, initialize the property jHoveInfo and return it.
	 * It can however happen that it can't be initialized; in this case, return null.
	 */
	private IdentificationResult getDroidMetadata()
	{
		Logger.getLogger().debug("Getting droid Metadata...");

		if (this.DROIDData == null)		this.initializeDroidMetadata();
		return this.DROIDData;
	}


	private void initializeDroidMetadata()
	{
		//	If I already tried to retrieve DROID, don't try again:
		if (this.alreadyTriedToRetrieveDROID)
		{
			Logger.getLogger().debug("NOT retrying to retrieve droid Metadata for file:" + this.getPath());
			return;
		}

		String filePath;
		try
		{
			filePath = this.getCanonicalPath();
		}
		catch (IOException ex)
		{
			ex.printStackTrace();
			return;
		}

		Logger.getLogger().debug("Retrieving droid Metadata for file:" + filePath + "...");

		try
		{
			this.DROIDData = MetadataProviderDROID.getIdentificationResult(filePath);

			Logger.getLogger().debug("...OK");
		}
		catch (Exception ex)
		{
			Logger.getLogger().debug("...NOK!");
		}

		this.alreadyTriedToRetrieveDROID = true;
	}


	//	--------		Accessing			-------------------------------------------------------

	//	--------		Jhove metaData		-------------------------------------------------------

	private int getWellFormed_jhove()
	{
		if (this.getJhoveMetadata() == null)		return -1;

		return this.jhoveData.getWellFormed();
	}


	private int getValid_jhove()
	{
		if (this.getJhoveMetadata() == null)		return -1;

		return this.jhoveData.getValid();
	}


	private boolean isConsistent_jhove()
	{
		if (this.getJhoveMetadata() == null)		return false;

		return this.jhoveData.isConsistent();
	}


	private String getMimeType_jhove()
	{
		if (this.getJhoveMetadata() == null)		return Unknown;

		return this.jhoveData.getMimeType();
	}


	private String getFormat_jhove()
	{
		if (this.getJhoveMetadata() == null)		return Unknown;

		return this.jhoveData.getFormat();
	}


	private long getSize_jhove()
	{
		if (this.getJhoveMetadata() == null)		return -1;

		return this.jhoveData.getSize();
	}


	private Module getModule_jhove()
	{
		if (this.getJhoveMetadata() == null)		return null;

		return this.jhoveData.getModule();
	}


	@SuppressWarnings("unchecked")
	private List<String> getProfile_jhove()
	{
		if (this.getJhoveMetadata() == null)		return new ArrayList<String>();

		return this.jhoveData.getProfile();
	}


	private String getChecksum_jhove(ChecksumType checksumType)
	{
		if (this.getJhoveMetadata() == null)		return Unknown;

		for (Object o: this.jhoveData.getChecksum())
		{
			Checksum checksum = (Checksum)o;
			if (checksum.getType().equals(checksumType))	return checksum.getValue();
		}
		return Unknown;
	}


	//	--------		Droid metaData		-------------------------------------------------------

	private String getMimeType_droid()
	{
		if (this.getDroidMetadata() == null)		return Unknown;

		return this.DROIDData.getMimeType();
	}


	private String getFileFormatName_droid()
	{
		if (this.getDroidMetadata() == null)		return Unknown;

		return this.DROIDData.getName();
	}


	private String getFileFormatPUID_droid()
	{
		if (this.getDroidMetadata() == null)		return Unknown;

		return this.DROIDData.getPuid();
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	static public enum Provider
	{
		DROID, JHOVE;
	}

}
