/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;

import java.io.*;
import java.util.List;

import uk.gov.nationalarchives.droid.command.action.CommandExecutionException;
import uk.gov.nationalarchives.droid.command.container.Ole2ContainerContentIdentifier;
import uk.gov.nationalarchives.droid.command.container.ZipContainerContentIdentifier;
import uk.gov.nationalarchives.droid.container.*;
import uk.gov.nationalarchives.droid.container.ole2.Ole2IdentifierEngine;
import uk.gov.nationalarchives.droid.container.zip.ZipIdentifierEngine;
import uk.gov.nationalarchives.droid.core.BinarySignatureIdentifier_Extended;
import uk.gov.nationalarchives.droid.core.interfaces.*;
import uk.gov.nationalarchives.droid.core.interfaces.archive.IdentificationRequestFactory;
import uk.gov.nationalarchives.droid.core.interfaces.resource.FileSystemIdentificationRequest;
import uk.gov.nationalarchives.droid.core.interfaces.resource.RequestMetaData;
import uk.gov.nationalarchives.droid.core.signature.FileFormat;
import ch.docuteam.tools.file.exception.*;
import ch.docuteam.tools.out.Logger;

/**
 * This is an abstract class for getting file format information using DROID.
 * <br>
 * The DROID subsystem requires the signature file "config/DROID_SignatureFile_V66.xml" to be in the folder "config" in the working directory.
 *
 * @author denis
 *
 */
public abstract class MetadataProviderDROID
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String							DefaultSignatureFile = "config/DROID_SignatureFile_V82.xml";
	static private final String							DefaultContainerSignatureFile = "config/container-signature-20150327.xml";

	static private final int							DefaultExtensionUsage = 1;

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private String								SignatureFile = DefaultSignatureFile;
	static private String								ContainerSignatureFile = DefaultContainerSignatureFile;

	static private int									ExtensionUsage = DefaultExtensionUsage;

	static private BinarySignatureIdentifier_Extended	SignatureIdentificator = null;
	static private ContainerSignatureDefinitions		ContainerSignatureDefs = null;
	static private List<TriggerPuid>					ContainerSignatureTriggerPuids = null;

	private static Boolean								IsInitialized = false;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static public void setSignatureFile(String newSignatureFile)
	{
		SignatureFile = newSignatureFile;

		IsInitialized = false;
		//	I will (re-)initialize myself the next time I am used.
	}

	static public void setContainerSignatureFile(String newContainerSignatureFile)
	{
		ContainerSignatureFile = newContainerSignatureFile;

		IsInitialized = false;
		//	I will (re-)initialize myself the next time I am used.
	}

	/*
	 * set the usage of file extension matching:
	 * 1 = use tentative extension matching, i.e. only rely on extensions that do not have a defined signature
	 * 2 = use full extension matching, i.e. use extension even for file formats that have defined signatures
	 * anything else = disable extension matching altogether
	 */
	static public void setExtensionUsage(int levelOfExtensionUsage)
	{
		ExtensionUsage = levelOfExtensionUsage;
	}

	//	--------		Accessing			-------------------------------------------------------

	static public IdentificationResult getIdentificationResult(String filePath) throws DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException
	{
		List<IdentificationResult> resultList = getIdentificationResults(filePath);

		if (resultList == null || resultList.isEmpty())		throw new DROIDNoIdentificationFoundException(filePath);
		if (resultList.size() != 1)							throw new DROIDMultipleIdentificationsFoundException(filePath, resultList);

		return resultList.get(0);
	}

	//	The following are convenience methods (shortcuts for retrieving specific metadata directly):

	static public String getFileFormatPUID(String fileName) throws DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException
	{
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null)? null: result.getPuid();
	}

	static public String getMimeType(String fileName) throws DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException
	{
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null)? null: result.getMimeType();
	}

	static public String getFileFormatName(String fileName) throws DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException
	{
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null)? null: result.getName();
	}

	static public String getFileFormatVersion(String fileName) throws DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException
	{
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null)? null: result.getVersion();
	}

	static public String getFileFormatMethod(String fileName) throws DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException
	{
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null)? null: result.getMethod().getMethod();
	}

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static private void initializeIfNecessary() throws DROIDCouldNotInitializeException
	{
		if (IsInitialized)		return;

		try
		{
			Logger.getLogger().debug("Initializing DROID...");

			if (!new File(SignatureFile).exists())				throw new FileNotFoundException(SignatureFile);
			if (!new File(ContainerSignatureFile).exists())		throw new FileNotFoundException(ContainerSignatureFile);

			SignatureIdentificator = new BinarySignatureIdentifier_Extended();
			SignatureIdentificator.setSignatureFile(SignatureFile);
			SignatureIdentificator.init();

			ContainerSignatureDefs = new ContainerSignatureSaxParser().parse(new FileInputStream(ContainerSignatureFile));
			ContainerSignatureTriggerPuids = ContainerSignatureDefs.getTiggerPuids();

			IsInitialized = true;

			Logger.getLogger().debug("...OK");
		}
		catch (java.lang.Exception ex)
		{
			Logger.getLogger().debug("...NOK!");

			throw new DROIDCouldNotInitializeException(ex);
		}
	}

	//	--------		Calculating			-------------------------------------------------------

	static private List<IdentificationResult> getIdentificationResults(String filePath) throws DROIDCouldNotInitializeException, FileNotFoundException
	{
		initializeIfNecessary();

		if (!new File(filePath).exists())			throw new FileNotFoundException(filePath);

		File file = new File(filePath);
		RequestMetaData metadata = new RequestMetaData(file.length(), file.lastModified(), file.getName());
		RequestIdentifier id = new RequestIdentifier(file.toURI());
		IdentificationRequest request = new FileSystemIdentificationRequest(metadata, id);

		FileInputStream fis = null;
		try
		{
			fis = new FileInputStream(file);
			request.open(fis);

			// Identify the file. Try 3 different systems: first container, then binary (= signature), and finally file extension:
			//		(NOTE: To get the container identifications, I need the binary identifications)
			IdentificationResultCollection signatureResultCollection = SignatureIdentificator.matchBinarySignatures(request);
			IdentificationResultCollection containerResultCollection = getContainerResults(request, signatureResultCollection);

			IdentificationResultCollection finalResultCollection;
			if      (containerResultCollection.getResults().size() > 0)		finalResultCollection = containerResultCollection;
			else if (signatureResultCollection.getResults().size() > 0)		finalResultCollection = signatureResultCollection;
			else {
				switch (ExtensionUsage) {
				case 1:
					finalResultCollection = SignatureIdentificator.matchExtensions(request, false);
					break;
				case 2:
					finalResultCollection = SignatureIdentificator.matchExtensions(request, true);
					break;
				default:
					finalResultCollection = containerResultCollection;
				}
			}

			SignatureIdentificator.removeLowerPriorityHits(finalResultCollection);

			return finalResultCollection.getResults();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			return null;
		}
		finally
		{
			try
			{
				request.close();
				if (fis != null)	fis.close();
			}
			catch(IOException ex){};
		}
	}


	private static IdentificationResultCollection getContainerResults(IdentificationRequest request, IdentificationResultCollection results) throws CommandExecutionException
	{
		IdentificationResultCollection containerResults = new IdentificationResultCollection(request);

		for (IdentificationResult identResult: results.getResults())
		{
			String filePuid = identResult.getPuid();
			if (filePuid == null)			continue;

			TriggerPuid containerPuid = null;
			for (TriggerPuid tp: ContainerSignatureTriggerPuids)
			{
				if (tp.getPuid().equals(filePuid))
				{
					containerPuid = tp;
					break;
				}
			}
			if (containerPuid == null)		continue;

			IdentificationRequestFactory requestFactory = new ContainerFileIdentificationRequestFactory();
			String containerType = containerPuid.getContainerType();

			if ("OLE2".equals(containerType))
			{
				try
				{
					Ole2ContainerContentIdentifier ole2Identifier = new Ole2ContainerContentIdentifier();
					ole2Identifier.init(ContainerSignatureDefs, containerType);
					Ole2IdentifierEngine ole2IdentifierEngine = new Ole2IdentifierEngine();
					ole2IdentifierEngine.setRequestFactory(requestFactory);
					ole2Identifier.setIdentifierEngine(ole2IdentifierEngine);
					ole2Identifier.process(request.getSourceInputStream(), containerResults);
				}
				catch (IOException e)
				{
					e.printStackTrace();
				}
			}
			else if ("ZIP".equals(containerType))
			{
				try
				{
					ZipContainerContentIdentifier zipIdentifier = new ZipContainerContentIdentifier();
					zipIdentifier.init(ContainerSignatureDefs, containerType);
					ZipIdentifierEngine zipIdentifierEngine = new ZipIdentifierEngine();
					zipIdentifierEngine.setRequestFactory(requestFactory);
					zipIdentifier.setIdentifierEngine(zipIdentifierEngine);
					zipIdentifier.process(request.getSourceInputStream(), containerResults);
				}
				catch (IOException e)
				{
					e.printStackTrace();
				}
			}
			else
			{
				throw new CommandExecutionException("Unknown container type: " + containerPuid);
			}
		}

		IdentificationResultCollection finalContainerResults = new IdentificationResultCollection(request);
		for (IdentificationResult r: containerResults.getResults())
		{
			FileFormat ff = SignatureIdentificator.getFileFormatForPuid(r.getPuid());
			finalContainerResults.addResult(new MetadataProviderDROID_IdentificationResult(r, ff));
		}

		return finalContainerResults;
	}

}
