/**
 *	Copyright (C) 2011-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.out;



import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import ch.docuteam.tools.string.DateFormatter;

/**
 * This abstract class offers convenient functionality for writing text to a log file.
 * First the LogFile has to be initialized by supplying a LogFile name. After that, every call to write*(...) logs the supplied string to that file.
 * <br>
 * There are three interesting options to create such a file:
 * <br>
 * 1: initializeAppend(fileName): If the file exists, append every line to the file; if not, create the file
 * 2: initializeEmpty(fileName): 
 * 
 * @author denis
 *
 */
public abstract class LogFile
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	static final public String				CommentStartString = "#";
	
	//	========	Static Final Private	=======================================================

	static final private String				FileNameExtension = ".txt";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private String					fileName = null;
	
	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initialize the LogFile to log to the file with the supplied name. If a file with this name already exists, subsequent logs will be appended.
	 * @param initFileName The file name
	 */
	static public void initializeAppend(String initFileName)
	{
		initialize(initFileName, true);
	}


	/**
	 * Initialize the LogFile to log to the file with the supplied name. If a file with this name already exists, in will be deleted.
	 * @param initFileName The file name
	 */
	static public void initializeEmpty(String initFileName)
	{
		initialize(initFileName, false);
	}


	/**
	 * Initialize the LogFile to log to the file with the supplied name plus the current timestamp.
	 * Each call to this method creates a new log file with the supplied name plus the current timestamp.
	 * @param initFileName
	 */
	static public void initializeNewVersion(String initFileName)
	{
		String fileName;

		int suffixPos = initFileName.lastIndexOf(".");
		if (suffixPos == -1)
		{
			fileName = initFileName + "_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);
		}
		else
		{
			String suffix = initFileName.substring(suffixPos + 1);
			String rest = initFileName.substring(0, suffixPos);
			fileName = rest + "_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "." + suffix;
		}

		initializeEmpty(fileName);
	}
	
	
	//	--------		Writing				-------------------------------------------------------

	static public void writeln() throws IOException
	{
		write("\n");
	}
	
	
	static public void writeln(Object o) throws IOException
	{
		write("\n" + ((o == null)? "null": o.toString()));
	}
	
	
	/**
	 * Write an empty line to the log file.
	 */
	static public void write() throws IOException
	{
		write("");
	}


	/**
	 * Write the toString of the object to the log file.
	 */
	static public void write(Object o) throws IOException
	{
		write((o == null)? "null": o.toString());
	}


	//	--------		Accessing			-------------------------------------------------------

	static public String getFileName()
	{
		return fileName;
	}


	//	========	Static Private			=======================================================

	/**
	 * Initialize the LogFile to log to the file with the supplied name.
	 * @param initFileName The file name
	 * @param append If true, all subsequent logs will be appended in the case that the log file already existed. If false, the log file will be emptied first.
	 */
	static private void initialize(String initFileName, Boolean append)
	{
		fileName = initFileName;
		if (!fileName.endsWith(FileNameExtension))		fileName = fileName + FileNameExtension;

		FileWriter fileWriter = null;
		try
		{
			//	Create super-folders if necessary:
			new File(fileName).getParentFile().mkdirs();

			//	Create file for writing:
			fileWriter = new FileWriter(fileName, append);

			//	Write header:
			writeSeparatorLine();
			writeTimeStamp();
			writeSeparatorLine();
		}
		catch (IOException e)
		{
			e.printStackTrace();
			fileName = null;
		}
		finally
		{
			try { if (fileWriter != null) fileWriter.close(); } catch (IOException e1) { e1.printStackTrace(); }
		}
	}


	/**
	 * Write the string supplied to the log file.
	 * @param string The string to be logged
	 * @throws IOException Gets thrown if the LogFile is not initialized or if any other IOException occurs during writing to the file.
	 */
	static private void write(String string) throws IOException
	{
		if (fileName == null)		throw new IOException("Can't write to LogFile because the LogFile is not initialized.");

		//	Write to logFile, always append:
		FileWriter fileWriter = new FileWriter(fileName, true);
		try
		{
			fileWriter.append(string);
		}
		finally
		{
			fileWriter.close();
		}
	}
	
	/**
	 * Write a separator line to the log file.
	 */
	static private void writeSeparatorLine() throws IOException
	{
		writeln(CommentStartString + "================================================================================");
	}
	
	
	/**
	 * Write the current time stamp to the log file.
	 * @throws IOException
	 */
	static private void writeTimeStamp() throws IOException
	{
		writeln(CommentStartString + "==========\t\t" + DateFormatter.getCurrentDateTimeString(DateFormatter.German));
	}


}
