/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;

import static ch.docuteam.tools.ToolsConstants.*;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.zip.CRC32;

/**
 * This is an abstract class for calculating a file's checksum using the JHOVE Checksummer.
 * <br>
 * The JHOVE subsystem requires the configuration file "jhove.conf" to be in the folder "config" in the DOCUTOOLS_HOME folder.
 * @author denis
 * 
 */
public abstract class FileChecksumCalculator
{

	/**
	 * Given a file name, this method will return the calculated MD5 checksum of that file.<br>
	 * @return The checksum of the given file.
	 * @throws IOException 
	 * @throws FileNotFoundException 
	 * @throws NoSuchAlgorithmException 
	 */
	public static String getMD5(String filePath) throws FileNotFoundException, IOException, NoSuchAlgorithmException
	{
		return Algorithm.MD5.getDigest(filePath);
	}

	/**
	 * Given a file name, this method will return the calculated CRC32 checksum of that file.<br>
	 * @return The checksum of the given file.
	 * @throws IOException 
	 * @throws FileNotFoundException 
	 * @throws NoSuchAlgorithmException 
	 */
	public static String getCRC32(String filePath) throws FileNotFoundException, IOException, NoSuchAlgorithmException
	{
		return Algorithm.CRC32.getDigest(filePath);
	}

	/**
	 * Given a file name, this method will return the calculated SHA1 checksum of that file.<br>
	 * @return The SHA1 checksum of the given file.
	 * @throws IOException 
	 * @throws FileNotFoundException 
	 * @throws NoSuchAlgorithmException 
	 */
	public static String getSHA1(String filePath) throws FileNotFoundException, IOException, NoSuchAlgorithmException
	{
		return Algorithm.SHA1.getDigest(filePath);
	}
	
	/**
	 * Given a file name, this method will return the calculated SHA256 checksum of that file.<br>
	 * @return The SHA256 checksum of the given file.
	 * @throws IOException 
	 * @throws FileNotFoundException 
	 * @throws NoSuchAlgorithmException 
	 */
	public static String getSHA256(String filePath) throws FileNotFoundException, IOException, NoSuchAlgorithmException
	{
		return Algorithm.SHA256.getDigest(filePath);
	}

	/**
	 * Given a file name, this method will return the calculated SHA512 checksum of that file.<br>
	 * @return The SHA512 checksum of the given file.
	 * @throws IOException 
	 * @throws FileNotFoundException 
	 * @throws NoSuchAlgorithmException 
	 */
	public static String getSHA512(String filePath) throws FileNotFoundException, IOException, NoSuchAlgorithmException
	{
		return Algorithm.SHA512.getDigest(filePath);
	}


	public enum Algorithm
	{
		CRC32, MD5, SHA1, SHA256, SHA512;
		
		public static Algorithm lookup(String name) throws NoSuchAlgorithmException
		{
			Algorithm[] algorithms = Algorithm.values();
			for (Algorithm algorithm : algorithms)
			{
				if (algorithm.getCode().equals(name))
				{
					return algorithm;
				}
			}
			throw new NoSuchAlgorithmException();
		}
		
		public String getDigest(String filePath) throws IOException, NoSuchAlgorithmException
		{
			switch (this)
			{
			case CRC32:
			{
				return getChecksumZip(filePath);
			}
			default:
			{
				return getChecksum(filePath);
			}
			}
		}
		
		public String getCode()
		{
			switch (this)
			{
			case CRC32:
			{
				return "CRC-32";
			}
			case MD5:
			{
				return "MD5";
			}
			case SHA1:
			{
				return "SHA-1";
			}
			case SHA256:
			{
				return "SHA-256";
			}
			case SHA512:
			{
				return "SHA-512";
			}
			default:
			{
				throw new RuntimeException("Invalid algorithm");
			}
			}
		}
		
		private String getChecksumZip(String filePath) throws IOException {
			CRC32 checksum = new CRC32();

			String result = null;
			
			try(InputStream is = new FileInputStream(filePath)){
				byte[] bytes = new byte[BYTE_BUFFER_SIZE];
				int bytesRead;
				while((bytesRead = is.read(bytes)) != -1){
					checksum.update(bytes, 0, bytesRead);
				}
			}
			long value = checksum.getValue();
			// 8 digits, prefix with 0s if not long enough, X - print in upper case hexadecimal
			result = String.format("%08X", value).toLowerCase();
			return result;
		}
		
		private String getChecksum(String filePath) throws IOException, NoSuchAlgorithmException
		{
			MessageDigest md = MessageDigest.getInstance(getCode());
			
			String result = null;
			try(InputStream is = new FileInputStream(filePath)){
				byte[] bytes = new byte[BYTE_BUFFER_SIZE];
				int bytesRead;
				while((bytesRead = is.read(bytes)) != -1){
					md.update(bytes, 0, bytesRead);
				}
				byte[] digestBytes = md.digest();
				StringBuffer sb = new StringBuffer();
				for (byte b : digestBytes){
					sb.append(Integer.toString((b & 0xff) + 0x100, 16).substring(1));
				}
				result = sb.toString();
			}
			return result;
		}
	}


}
