/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;


import java.io.File;

/**
 * @author lem
 *
 */
public abstract class FileFilter extends javax.swing.filechooser.FileFilter implements java.io.FileFilter, java.io.FilenameFilter
{
	public static FileFilter	VisibleAll = new FileFilter()
	{
		
		@Override
		public boolean accept(File f)
		{
			return !(f.isHidden() || f.getName().startsWith("."));
		}

		@Override
		public String getDescription()
		{
			return "All visible files and directories";
		}
	};


	public static FileFilter	VisibleFiles = new FileFilter()
	{
		@Override
		public boolean accept(File f)
		{
			return f.isFile() && !(f.isHidden() || f.getName().startsWith("."));
		}

		@Override
		public String getDescription()
		{
			return "All visible files";
		}
	};


	public static FileFilter	VisibleDirectories = new FileFilter()
	{
		@Override
		public boolean accept(File f)
		{
			return f.isDirectory() && !(f.isHidden() || f.getName().startsWith("."));
		}

		@Override
		public String getDescription()
		{
			return "All visible directories";
		}
	};


	public static FileFilter	METSFiles = new FileFilter()
	{
		@Override
		public boolean accept(File f)
		{
			String fileNameLowerCase = f.getName().toLowerCase();
			return	!(f.isHidden() || fileNameLowerCase.startsWith("."))		//	No hidden files and directories
					&& (fileNameLowerCase.startsWith("mets") && fileNameLowerCase.endsWith(".xml"));
		}

		@Override
		public String getDescription()
		{
			return "METS XML-files";
		}
	};


	public static FileFilter	ZIPs = new FileFilter()
	{
		@Override
		public boolean accept(File f)
		{
			String fileName = f.getName();
			return	!(f.isHidden() || fileName.startsWith("."))		//	No hidden files and directories
					&& f.isFile()
					&& fileName.toLowerCase().endsWith(".zip");
		}

		@Override
		public String getDescription()
		{
			return "ZIP-files";
		}
	};


	public static FileFilter	METSFilesOrZIPs = new FileFilter()
	{
		@Override
		public boolean accept(File f)
		{
			String fileNameLowerCase = f.getName().toLowerCase();
			return	!(f.isHidden() || fileNameLowerCase.startsWith("."))		//	No hidden files and directories
					&& f.isFile()
					&& (fileNameLowerCase.endsWith(".zip")
						|| (fileNameLowerCase.startsWith("mets") && fileNameLowerCase.endsWith(".xml")));
		}

		@Override
		public String getDescription()
		{
			return "METS- or ZIP-files";
		}
	};


	/**
	 * Accept folders that contain the file mets.xml.
	 * @author denis
	 */
	public static FileFilter	SIPs = new FileFilter()
	{
		@Override
		public boolean accept(File file)
		{
			if (file.isHidden() || file.getName().startsWith("."))		return false;
			if (file.isFile())											return false;

			String[] metsFilesWithin = file.list();
			if (metsFilesWithin == null)								return false;

			for (String f: metsFilesWithin)
				if (f.equalsIgnoreCase("mets.xml"))						return true;

			return false;
		}

		@Override
		public String getDescription()
		{
			return "SIP Folders";
		}
	};


	/**
	 * Accept zip files and folders that contain the file mets.xml.
	 * @author denis
	 */
	public static FileFilter	SIPsOrZIPs = new FileFilter()
	{
		@Override
		public boolean accept(File file)
		{
			if (file.isHidden() || file.getName().startsWith("."))		return false;

			if (file.isFile())
			{
				if (file.getName().toLowerCase().endsWith(".zip"))		return true;
			}
			else
			{
				String[] metsFilesWithin = file.list();
				if (metsFilesWithin == null)							return false;

				for (String f: metsFilesWithin)
					if (f.equalsIgnoreCase("mets.xml"))					return true;
			}

			return false;
		}

		@Override
		public String getDescription()
		{
			return "SIP Folders or ZIP-Files";
		}
	};


	@Override
	public boolean accept(File dir, String name)
	{
		return this.accept(new File(dir, name));
	}

}
