/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.*;

import javax.swing.*;

import ch.docuteam.tools.os.SystemClipboard;
import ch.docuteam.tools.out.SystemOutRedirectionTarget;
import ch.docuteam.tools.out.SystemOutRedirector;

/**
 * The SystemOutView is a view that shows all system messages.
 * It can be configured to open on every new system message.
 * It is NOT a dialog so that it can stay open in the background.
 *
 * @author denis
 *
 */
public class SystemOutView extends JFrame implements SystemOutRedirectionTarget
{

	private static final String		ScreenTitle = "Console";
	private static Dimension		StartScreenSize = new Dimension(600, 400);


	private static SystemOutView	SoleInstance;

	/**
	 * To switch on or off that the SystemOutView automatically opens when an input occurs:
	 */
	private static boolean			DoPopUpOnWrite = true;

	private static String			AdditionalInfoText = "";


	private JTextArea				consoleMessagesTextArea;
	private Action					copyAction;


	public SystemOutView()
	{
		this("<html>An unexpected error occurred. Please copy this message to the system clipboard using the button on the left, then send it to the Dev Team (support@docuteam.ch) together with a brief description of what you did before this happened. Thank you.</html>");
	}


	/**
	 * The header text can be plain text or html code.
	 * @param headerText Text to be displayed at the top of the SystemOutView.
	 */
	public SystemOutView(String headerText)
	{
		super(ScreenTitle);

		this.consoleMessagesTextArea = new JTextArea();
		this.consoleMessagesTextArea.setEnabled(true);
		this.consoleMessagesTextArea.setEditable(false);
		this.consoleMessagesTextArea.setLineWrap(false);

		this.copyAction = new AbstractAction("Copy")
		{	@Override public void actionPerformed(ActionEvent e){ SystemOutView.this.copyAction(); }};
		this.copyAction.putValue(Action.SHORT_DESCRIPTION, "ctrl-c or cmd-c");

		this.consoleMessagesTextArea.getActionMap().put("Copy", this.copyAction);
		//	Bind this event to the ctrl-c keystoke:
		this.consoleMessagesTextArea.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_DOWN_MASK), "Copy");
		//	Bind this event to the cmd-c keystoke (for Macs):
		this.consoleMessagesTextArea.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.META_DOWN_MASK), "Copy");

		Box topRowBox = new Box(BoxLayout.X_AXIS);
		topRowBox.add(Box.createHorizontalStrut(10));
		topRowBox.add(new JButton(this.copyAction));
		topRowBox.add(Box.createHorizontalStrut(10));
		topRowBox.add(new JLabel(headerText));

		this.add(topRowBox, BorderLayout.NORTH);
		this.add(new JScrollPane(this.consoleMessagesTextArea), BorderLayout.CENTER);

		this.setPreferredSize(StartScreenSize);
		this.pack();
		this.setLocationRelativeTo(null);

		SoleInstance = this;
	}


	public static void install()
	{
		SystemOutRedirector.install(new SystemOutView());
	}


	public static void open()
	{
		if (SoleInstance == null)		SoleInstance = new SystemOutView();
		SoleInstance.setVisible(true);
	}


	public static void setDoPopUpOnWrite(boolean doPopUpOnWrite)
	{
		DoPopUpOnWrite = doPopUpOnWrite;
	}


	public static boolean doPopUpOnWrite()
	{
		return DoPopUpOnWrite;
	}


	public static void setAdditionalInfoText(String text)
	{
		AdditionalInfoText = (text == null)? "": text;
	}


	/* (non-Javadoc)
	 * @see doculib.util.SystemOutRedirectionTarget#write(java.lang.String)
	 */
	@Override
	public void write(String string)
	{
		this.consoleMessagesTextArea.append(string);

		if (DoPopUpOnWrite)		this.setVisible(true);
		else					this.toFront();
	}


	private void copyAction()
	{
		SystemClipboard.setText(AdditionalInfoText + this.consoleMessagesTextArea.getText());
	}


}
