/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.ldap;

import java.util.Hashtable;

import javax.naming.*;
import javax.naming.directory.*;

import ch.docuteam.tools.out.Logger;

/**
 * Simple client to access LDAP servers for search requests.
 * 
 * You have to give at least an URL where to find the LDAP server, optionally using authentication as well.
 * Initialize a connection using the create methods and do searches with the simple retrieve methods.
 * 
 * @author andreas
 *
 */
public abstract class LDAPClient
{

	private static final String				ContextFactoryClassName = "com.sun.jndi.ldap.LdapCtxFactory";


	private static String					LDAPUrl;
	private static String					SecurityPrincipal;
	private static String					SecurityCredentials;

	private static DirContext				LDAPContext;


	/**
	 * Main method for test purposes.
	 * 
	 * @param args[0]	URL of the LDAP server, e.g. ldap://ldap.example.org/
	 * @param args[1]	username for authentication
	 * @param args[2]	password for authentication
	 * @param args[3]	base DN for searches
	 * @param args[4]	filter expression, e.g. "(cn=*)"
	 * 
	 * @throws Exception 
	 */
	public static void main(String[] args) throws Exception
	{
		open(args[0], args[1], args[2]);
		System.out.println(retrieveFieldValueOfFirstHit(args[3], args[4], "cn"));
		System.out.println(retrieveFieldValueOfFirstHit(args[3], args[4], "department"));
		System.out.println(retrieveFieldValueOfFirstHit(args[3], args[4], "mail"));
	}


	/**
	 * Open a connection to a given LDAP url without authentication
	 * 
	 * @param ldapurl	String of the URL where to reach the LDAP server, e.g. ldap://ldap.example.org/
	 */
	public static DirContext open(String ldapurl) throws NamingException {
		LDAPUrl = ldapurl;
		
		return open();
	}

	/**
	 * Open a connection to a given LDAP url with authentication
	 * 
	 * @param ldapurl	String of the URL where to reach the LDAP server, e.g. ldap://ldap.example.org/
	 * @param username	Username to use for authentication
	 * @param password	Password to use for authentication
	 */
	public static DirContext open(String ldapurl, String username, String password) throws NamingException {
		LDAPUrl = ldapurl;
		SecurityPrincipal = username;
		SecurityCredentials = password;
		
		return open();
	}


	/**
	 * Gets the first value of a given LDAP attribute from a search result
	 * 
	 * @param baseDN	Distinguished name (DN) to use as base for the search
	 * @param filter	LDAP filter criteria according to official syntax, e.g. "(sn=*)"
	 * @param field		LDAP attribute of which the value is returned
	 */
	public static String retrieveFieldValueOfFirstHit(String baseDN, String filter, String field) throws NamingException
	{
		NamingEnumeration<SearchResult> results = retrieve(baseDN, filter);
		if (results == null)		throw new NullPointerException();

		return field + ": " + (String)results.next().getAttributes().get(field).get();
	}

	/**
	 * Gets the complete result of an empty search, i.e. "(cn=*)", on a given DN
	 * 
	 * @param baseDN	Distinguished name (DN) to use as base for the search
	 */
	public static NamingEnumeration<SearchResult> retrieve(String baseDN) throws NamingException
	{
		return retrieve(baseDN, "(cn=*)");
	}

	/**
	 * Gets the complete result of an given LDAP filter string
	 * 
	 * @param baseDN	Distinguished name (DN) to use as base for the search
	 * @param filter	LDAP filter criteria according to official syntax, e.g. "(sn=*)"
	 */
	public static NamingEnumeration<SearchResult> retrieve(String baseDN, String filter) throws NamingException
	{
		SearchControls controls = new SearchControls();
		controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
//		controls.setCountLimit(100);
		controls.setReturningObjFlag(true);							//	true or false makes no difference
//		controls.setReturningAttributes(new String[]{"sn", "givenName", "objectClass", "gecos"});		//	Attributes for People
//		controls.setReturningAttributes(new String[]{"cn", "gidNumber", "objectClass"});				//	Attributes for Groups

		NamingEnumeration<SearchResult> results = null;
//		try
//		{
			results = LDAPContext.search(baseDN, filter, controls);

			//	Lookup an object:
//				Object o = context.lookup("dc=example,dc=com");
//				System.out.println(o.getClass());

				//	Get attributes:
//				Attributes attrs = context.getAttributes("dc=example,dc=com");
//				System.out.println(attrs);

				//	Specify search attributes with values:
//				Attributes matchAttrs = new BasicAttributes(true);			//	true means: ignore case
//				matchAttrs.put(new BasicAttribute("sn", "Lins"));
//				matchAttrs.put(new BasicAttribute("mail"));
//				results = context.search("ou=People,dc=example,dc=com", matchAttrs);
//				results = context.search("ou=Group,dc=example,dc=com", matchAttrs);
//				results = context.search("dc=example,dc=com", matchAttrs);

				//	Specify returning attributes:
//				Attributes matchAttrs = new BasicAttributes(true);
//				matchAttrs.put(new BasicAttribute("sn", "Lins"));
//				matchAttrs.put(new BasicAttribute("mail"));
//				String[] returnAttrIds = {"sn", "telephonenumber", "mail", "whatever"};
//				results = context.search("ou=People,dc=example,dc=com", matchAttrs, returnAttrIds);

				//	Specify search controls and filters:
//				SearchControls controls = new SearchControls();
//				controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
//				controls.setReturningObjFlag(true);							//	true or false makes no difference
//				controls.setCountLimit(100);
//				controls.setReturningAttributes(new String[]{"sn", "givenName", "objectClass", "gecos"});
//				results = context.search("ou=People,dc=example,dc=com", "(sn=*)", controls);
//				results = context.search("ou=People,dc=example,dc=com", "(&(sn=L*)(objectClass=Person))", controls);		//	objectClass = person, posixGroup, *
//				results = context.search("ou=Group,dc=example,dc=com", "(cn=*)", controls);		//	objectClass = person, posixGroup, *
			return results;
	}

	/**
	 * Closes the current connection
	 * 
	 * @throws NamingException
	 */
	public static void close() throws NamingException {
		Logger.getLogger().debug("Closing");
		if (LDAPContext != null)		LDAPContext.close();
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	/**
	 * Print all search results to System.out.
	 * 
	 * @param results	results object of a LDAP search request.
	 * @throws NamingException
	 */
	public static void systemOut(NamingEnumeration<SearchResult> results) throws NamingException
	{
		if (results == null)		return;

		while (results.hasMore())
		{
			SearchResult searchResult = results.next();
			systemOut(searchResult);
		}
	}
	
	
	/**
	 * Print all attributes and all their values of a single search result to System.out.
	 * 
	 * @param searchResult	a single LDAP search result
	 * @throws NamingException
	 */
	public static void systemOut(SearchResult searchResult) throws NamingException
	{
		for (NamingEnumeration<? extends Attribute> attributes = searchResult.getAttributes().getAll(); attributes.hasMore();)
		{
			Attribute attribute = attributes.next();
			System.out.print(attribute.getID() + ": ");
		
			for (NamingEnumeration<?> values = attribute.getAll(); values.hasMore();)		System.out.print(" " + (String)values.next());

			System.out.println();
		}

		System.out.println("----------");
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	private static			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initializes the LDAP Context with the URL defined through on of the public "create"-methods.
	 */
	private static DirContext open() throws NamingException {
		Logger.getLogger().debug("Creating LDAP Connection: " + LDAPUrl);

		Hashtable<String, String> env = new Hashtable<String, String>();

		// Mandatory:
		env.put(Context.INITIAL_CONTEXT_FACTORY, ContextFactoryClassName);
		env.put(Context.PROVIDER_URL, LDAPUrl);

		// Optional:
		if (SecurityPrincipal!=null) {
			env.put(Context.SECURITY_AUTHENTICATION, "simple");
			env.put(Context.SECURITY_PRINCIPAL, SecurityPrincipal);
			env.put(Context.SECURITY_CREDENTIALS, SecurityCredentials);
		}

		return LDAPContext = new InitialDirContext(env);
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
