/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.bar;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.*;

import org.dom4j.*;
import org.dom4j.io.*;
import org.jdesktop.swingx.treetable.MutableTreeTableNode;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.mdconfig.LevelOfDescription;
import ch.docuteam.darc.mdconfig.MetadataElement;
import ch.docuteam.darc.mets.structmap.*;
import ch.docuteam.darc.util.XMLUtil;
import ch.docuteam.tools.exception.ExceptionCollector;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.Zipper;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.id.UniqueID;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.out.Tracer;
import ch.docuteam.tools.string.DateFormatter;
import ch.docuteam.tools.string.StringUtil;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import static ch.docuteam.darc.DarcConstants.*;

/**
 * @author denis
 *
 */
public class Document extends DocumentAbstract
{

	static private final String				BARNamespacePrefix = "BAR";
	//	TODO	allow namespace for version 3.12.x: http://bar.admin.ch/arelda/v3.13.2
	static private final String				BARNamespaceURI = "http://bar.admin.ch/arelda/v4";

	static private final String				RelativePathToMetadata_xml = "/header/metadata.xml";
	static private final String				RelativePathToContent = "/content";

	static private final String				BARFileNamePrefix = "SIP_" + DateFormatter.getCurrentDateTimeString("yyyyMMdd") + "_";


	/**
	 * @param args
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 */
	public static void main(String[] args) throws IOException, FileUtilExceptionListException
	{
		String workspace = OperatingSystem.userHome() + "Desktop/Workspace/";
		String sipPath = workspace + "SIPs";
		String barSipPath = workspace + "BAR-SIPs";

		//	I have to use the levels file that defines the BAR-levels:
		LevelOfDescription.setInitializationFilePath("config/levels_BAR.xml");

		try
		{
			//	ToDo:	----------	Create a BAR-SIP from an existing Matterhorn SIP:

//			String sipName = "SIP_20130307_Test1_Lem";
			String sipName = "SIP_20130307_Test1_Lem_LotsOfMetadata";

			String createdPath = createFromSIPIntoFolder(sipPath + "/" + sipName, barSipPath, "BARDocument.createFromSIPIntoFolder", new ObserverTracer());

			//	ToDo:	----------	Create a Matterhorn SIP from an existing BAR-SIP:

			//	This example contains 4 files that can't be identified by DROID:
//			String sipName = "SIP_20130307_Test1_Lem";
//			String sipName = "SIP_20130326_Test2";
//			String sipName = "SIP_20130402_M. Hess_1";
//			String sipName = "SIP_20070923_RIS_3Dossier";
//			String sipName = "SIP_20070923_RIS_3Dossier_plus";
//			String sipName = "SIP_20130402_Urs_1";
//			String sipName = "SIP_20130424_UVA_2";

//			String sipName = "SIP_20130307_Test1_Lem.zip";
//			String sipName = "SIP_20130326_Test2.zip";
//			String sipName = "SIP_20130402_M. Hess_1.zip";
//			String sipName = "SIP_20070923_RIS_3Dossier.zip";
//			String sipName = "SIP_20070923_RIS_3Dossier_plus.zip";
//			String sipName = "SIP_20130402_Urs_1.zip";
//			String sipName = "SIP_20130424_UVA_2.zip";

//			String createdPath = convertToSIPIntoFolder(barSipPath + "/" + sipName, sipPath, "BARDocument.convertToSIPIntoFolder", new ObserverTracer());

			Tracer.trace("Created: " + createdPath);
		}
		catch (Throwable x)
		{
			x.printStackTrace();
		}

		if (!ExceptionCollector.isEmpty())		Tracer.trace(ExceptionCollector.toStringAll());
	}

	static public String createFromSIPIntoFolder(String documentPath, String barSIPDestinationFolderPath) throws Exception
	{
		return createFromSIPIntoFolder(documentPath, barSIPDestinationFolderPath, "");
	}


	static public String createFromSIPIntoFolder(String documentPath, String barSIPDestinationFolderPath, String operatorName) throws Exception
	{
		return createFromSIPIntoFolder(documentPath, barSIPDestinationFolderPath, operatorName, null);
	}


	static public String createFromSIPIntoFolder(String documentPath, String barSIPDestinationFolderPath, String operatorName, Observer observer) throws Exception
	{
		String barSIPPath = barSIPDestinationFolderPath + "/" + BARFileNamePrefix + FileUtil.asFileNameWithoutExtension(documentPath);

		if (new File(barSIPPath).exists()) {
			throw new FileAlreadyExistsException(BARFileNamePrefix + FileUtil.asFileNameWithoutExtension(documentPath), barSIPDestinationFolderPath);
		}


		Logger.getLogger().info("Creating BAR-SIP: '" + barSIPPath + "'...");

		//	Create the required folder structure:
		FileUtil.copyToOverwriting("resources/templates/BARSIP", barSIPPath);

		//	Read the BAR Document template:
		SAXReader reader = new SAXReader();
		reader.setDocumentFactory(new BARSIPFactory());
		Document barSIPDocument = (Document)reader.read(new File(barSIPPath + RelativePathToMetadata_xml));
		barSIPDocument.addObserver(observer);

		//	This is the Matterhorn SIP to create the BAR SIP from:
		ch.docuteam.darc.mets.Document doc = ch.docuteam.darc.mets.Document.openReadOnly(documentPath, operatorName);
		try
		{
			Node insertPointAblieferungNode = barSIPDocument.selectSingleNode("/BAR:paket");
			Node insertPointInhaltsverzeichnisNode = barSIPDocument.selectSingleNode("/BAR:paket/BAR:inhaltsverzeichnis/BAR:ordner/BAR:name[text()='content']/..");
			File insertPointFolder = new File(barSIPPath + RelativePathToContent);
			barSIPDocument.insertRecursively(doc.getStructureMap().getRoot(), (Element)insertPointAblieferungNode, (Element)insertPointInhaltsverzeichnisNode, insertPointFolder);

			//	Now make "BAR" the default namespace:
			barSIPDocument.changeNamespace(Namespace.get(BARNamespacePrefix, BARNamespaceURI), Namespace.get(BARNamespaceURI));

			java.io.Writer oswriter = new java.io.OutputStreamWriter(new java.io.FileOutputStream(barSIPPath + RelativePathToMetadata_xml), "utf-8");
			XMLWriter writer = new XMLWriter(oswriter, new OutputFormat("	", false, "utf-8"));
			writer.write(barSIPDocument);
			writer.close();
		}
		finally
		{
			if (doc != null)		doc.cleanupWorkingCopy();
		}

		Logger.getLogger().info("Creating BAR-SIP: '" + barSIPPath + "'... done!");

		return barSIPPath;
	}



	static public String convertToSIPIntoFolder(String barSIPPath, String destinationFolderPath) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, DocumentException, FileUtilExceptionListException, DocumentIsReadOnlyException, java.lang.Exception
	{
		return convertToSIPIntoFolder(barSIPPath, destinationFolderPath, "");
	}


	static public String convertToSIPIntoFolder(String barSIPPath, String destinationFolderPath, String operatorName) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, DocumentException, FileUtilExceptionListException, DocumentIsReadOnlyException, java.lang.Exception
	{
		return convertToSIPIntoFolder(barSIPPath, destinationFolderPath, operatorName, null);
	}


	static public String convertToSIPIntoFolder(String barSIPPath, String destinationFolderPath, String operatorName, Observer observer) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, DocumentException, FileUtilExceptionListException, DocumentIsReadOnlyException, java.lang.Exception
	{
		String newZIPOrFolderFilePath = destinationFolderPath + "/" + FileUtil.asFileNameWithoutExtension(barSIPPath);

		if (new File(newZIPOrFolderFilePath).exists()) {
			throw new FileAlreadyExistsException(FileUtil.asFileNameWithoutExtension(barSIPPath), destinationFolderPath);
		}

		Logger.getLogger().info("Converting BAR-SIP to SIP: '" + newZIPOrFolderFilePath + "'...");

		ch.docuteam.darc.mets.Document document = null;
		Document barSIP = null;
		boolean isZIP = false;

		try
		{
			if (barSIPPath.toLowerCase().endsWith(ZIP_EXT))
			{
				isZIP = true;

				//	Unzip into temporary folder:
				String barSIPName = (new File(barSIPPath).getName());
				barSIPName = barSIPName.substring(0, barSIPName.length() - 4);
				String workingFolder = FileUtil.getTempFolder() + "/" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);

				Zipper.unzip(barSIPPath, workingFolder);
				new File(workingFolder).deleteOnExit();

				//	Set the (method-local) barSIPPath to the temporary working path:
				barSIPPath = workingFolder;

				//	Does this zip contain a folder with the BARSIP or the two folders "content" and "header"?
				List<File> zipContents = null;
				int counter = 0;
				do {
					zipContents = Arrays.asList(new File(barSIPPath).listFiles((java.io.FileFilter)ch.docuteam.tools.file.FileFilter.VisibleDirectories));
					if (zipContents.contains(new File(barSIPPath + "/content")) && zipContents.contains(new File(barSIPPath + "/header")))		break;

					//	Assume there is an (single) intermediate folder...
					File intermediateFolder= zipContents.get(0);
					barSIPPath += "/" + intermediateFolder.getName();
					counter++;
				} while (counter <= 2);
			}

			//	Read the BAR-SIP Document:
			SAXReader reader = new SAXReader();
			reader.setDocumentFactory(new BARSIPFactory());
			barSIP = (Document)reader.read(new File(barSIPPath + RelativePathToMetadata_xml));
			barSIP.filePath = barSIPPath;

			//	Create a new empty Matterhorn SIP:
			document = ch.docuteam.darc.mets.Document.createNewWithRootFolderName(newZIPOrFolderFilePath, "Paket", "sa_all-formats-01", "dss-01", operatorName, observer);

			//	Insert all BAR-SIP elements into the Matterhorn SIP:
			barSIP.insertPaketInto((NodeFolder)document.getStructureMap().getRoot());

			//	Save SIP to original folder:
			document.saveWithoutBackup();
		}
		finally
		{
			if (isZIP)
			{
				//	Cleanup working folder:
				FileUtil.deleteOnExit(new File(barSIPPath).getParentFile());
			}

			if (document != null)
			{
				//	Cleanup and remove lock-file:
				document.unlockIfNecessary();
				document.cleanupWorkingCopy();
			}
		}

		Logger.getLogger().info("Converting BAR-SIP to SIP: '" + newZIPOrFolderFilePath + "'... done!");

		return newZIPOrFolderFilePath;
	}


	/**
	 * Return the type of this document: METS, EAD, BAR, ...
	 */
	@Override
	public Type getDocumentType()
	{
		return Type.BAR;
	}

	private void insertRecursively(NodeAbstract node, Element insertPointAblieferungElement, Element insertPointInhaltsverzeichnisElement, File insertPointFolder) throws IOException, FileUtilExceptionListException
	{
		String level = node.getLevel().getName().toLowerCase();

		Logger.getLogger().debug("Inserting node: '" + node.getLabel() + "' with level: '" + level + "'");

		Element newAblieferungElement = null;
		if (level.equals("paket"))							newAblieferungElement = this.fillPaket(insertPointAblieferungElement, node);
		else if (level.equals("ablieferung"))				newAblieferungElement = this.createAblieferung(insertPointAblieferungElement, node);
		else if (level.equals("ordnungssystem"))			newAblieferungElement = this.createOrdnungssystem(insertPointAblieferungElement, node);
		else if (level.equals("ordnungssystemposition"))	newAblieferungElement = this.createOrdnungssystemposition(insertPointAblieferungElement, node);
		else if (level.equals("dossier"))					newAblieferungElement = this.createDossier(insertPointAblieferungElement, node);
		else if (level.equals("dokument"))					newAblieferungElement = this.createDokument(insertPointAblieferungElement, node);
		else if (level.equals("datei"))						newAblieferungElement = insertPointAblieferungElement;		//	Dont create any new ablieferungselement, just continue
		else if (level.equals("undefiniert"))				return;		//	ToDo: return?
		else												return;		//	ToDo: return?

		if (newAblieferungElement == null)					return;		//	ToDo: exception?

		this.createInhaltsverzeichnisElementAndFileOrFolder_ThenDiveIn(node, newAblieferungElement, insertPointInhaltsverzeichnisElement, insertPointFolder);

		this.cleanupIfNecessary(node, level, newAblieferungElement);
	}

	//	--------			Create BAR-SIP	-------------------------------------------------------

	private void createInhaltsverzeichnisElementAndFileOrFolder_ThenDiveIn(NodeAbstract node, Element newAblieferungElement, Element insertPointInhaltsverzeichnisElement, File insertPointFolder) throws IOException, FileUtilExceptionListException
	{
		//	Distinguish between file and folder:
		if (node.isFile())
		{
			NodeFile fileNode = (NodeFile)node;

			//	Insert new inhaltsverzeichnisElement for a file:
			String id = DateiId.getNext();
			this.createInhaltsverzeichnisElement(insertPointInhaltsverzeichnisElement, fileNode, id);

			//	Create a dateiRef to this inhaltsverzeichnisElement in the ablieferungElement:
			newAblieferungElement.addElement("BAR:dateiRef").addText(id);

			//	Copy file:
			try
			{
				FileUtil.copyToFolderOverwriting(node.getFile(), insertPointFolder, false);
			}
			catch (IOException ex)
			{
				ex.printStackTrace();
			}
			catch (FileUtilExceptionListException ex)
			{
				ex.printStackTrace();
			}
		}
		else //	Node is folder:
		{
			NodeFolder folderNode = (NodeFolder)node;

			//	Insert new inhaltsverzeichnisElement for a folder:
			Element newInhaltsverzeichnisElement = this.createInhaltsverzeichnisElement(insertPointInhaltsverzeichnisElement, folderNode);

			//	Create new folder:
			File newFolder = FileUtil.createFolderOverwriting(node.getFile().getName(), insertPointFolder);

			//	Dive in (recursion!):
			for (MutableTreeTableNode child: folderNode.getChildren())
				this.insertRecursively((NodeAbstract)child, newAblieferungElement, newInhaltsverzeichnisElement, newFolder);
		}
	}


	private Element fillPaket(Element paketElement, NodeAbstract node)
	{
		Logger.getLogger().debug("Filling paket for: '" + node.getLabel() + "'");
		this.distributeMessage("Filling paket for: '" + node.getLabel() + "'");

		this.addZusatzDatenElementFromDynamicMetadata(paketElement, node, "usage");
		this.addZusatzDatenElementFromUnconfiguredMetadata(paketElement, node);

		return paketElement;
	}


	private Element createAblieferung(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.getLogger().debug("Creating ablieferung for: '" + node.getLabel() + "'");
		this.distributeMessage("Creating ablieferung for: '" + node.getLabel() + "'");

		Element ablieferungElement = insertPointAblieferungElement.addElement("BAR:ablieferung");

		this.setAttribute(ablieferungElement, "xsi:type", "ablieferungFilesSIP");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferungstyp", node, "objectType");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferndeStelle", node, "accessNr");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:entstehungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:entstehungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferungsteile", node, "material");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:bemerkung", node, "comment");
		this.addZusatzDatenElementFromUnconfiguredMetadata(ablieferungElement, node);
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:ablieferungsnummer", node, "refCode");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:angebotsnummer", node, "refCodeAdmin");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:referenzBewertungsentscheid", node, "appraisalAndDestruction");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:referenzSchutzfristenFormular", node, "accessRestriction");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:schutzfristenkategorie", node, "accessPolicy");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:schutzfrist", node, "accessRestrictionPeriod");
		this.addElementAndSetTextFromDynamicMetadata(ablieferungElement, "BAR:provenienz/BAR:aktenbildnerName", node, "origination");

		return ablieferungElement;
	}


	private Element createOrdnungssystem(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.getLogger().debug("Creating ordnungssystem for: '" + node.getLabel() + "'");
		this.distributeMessage("Creating ordnungssystem for: '" + node.getLabel() + "'");

		Element ordnungssystemElement = insertPointAblieferungElement.addElement("BAR:ordnungssystem");

		this.addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:generation", node, "refCode");
		this.addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:anwendungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		this.addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:anwendungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		this.addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:mitbenutzung", node, "involved");
		this.addElementAndSetTextFromDynamicMetadata(ordnungssystemElement, "BAR:bemerkung", node, "comment");
		this.addZusatzDatenElementFromUnconfiguredMetadata(ordnungssystemElement, node);
		this.addElementAndSetText(ordnungssystemElement, "BAR:name", node.getUnitTitle());

		return ordnungssystemElement;
	}


	private Element createOrdnungssystemposition(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.getLogger().debug("Creating ordnungssystemposition for: '" + node.getLabel() + "'");
		this.distributeMessage("Creating ordnungssystemposition for: '" + node.getLabel() + "'");

		Element newOrdnungssystempositionElement = insertPointAblieferungElement.addElement("BAR:ordnungssystemposition");

		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:federfuehrendeOrganisationseinheit", node, "origination");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:klassifizierungskategorie", node, "accessRestrictionClassification");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:datenschutz", node, "accessRestrictionPrivacy");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:oeffentlichkeitsstatus", node, "accessRestrictionStatus");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:oeffentlichkeitsstatusBegruendung", node, "accessRestrictionStatusExplanation");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:sonstigeBestimmungen", node, "accessRestriction");
		this.addZusatzDatenElementFromUnconfiguredMetadata(newOrdnungssystempositionElement, node);
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:nummer", node, "refCode");
		this.addElementAndSetText(newOrdnungssystempositionElement, "BAR:titel", node.getUnitTitle());
		this.setAttributeFromDynamicMetadata(newOrdnungssystempositionElement, "id", node, "refCodeAdmin");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:schutzfristenkategorie", node, "accessPolicy");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:schutzfrist", node, "accessRestrictionPeriod");
		this.addElementAndSetTextFromDynamicMetadata(newOrdnungssystempositionElement, "BAR:schutzfristenBegruendung", node, "accessRestrictionExplanation");

		return newOrdnungssystempositionElement;
	}


	private Element createDossier(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.getLogger().debug("Creating dossier for: '" + node.getLabel() + "'");
		this.distributeMessage("Creating dossier for: '" + node.getLabel() + "'");

		Element newDossierElement = insertPointAblieferungElement.addElement("BAR:dossier");

		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:zusatzmerkmal", node, "archivalHistory");
		this.addElementAndSetText(newDossierElement, "BAR:titel", node.getUnitTitle());
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:inhalt", node, "abstract");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:formInhalt", node, "scopeContent");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:erscheinungsform", node, "characteristics");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:federfuehrendeOrganisationseinheit", node, "origination");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:entstehungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:entstehungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:entstehungszeitraumAnmerkung", node, "creationPeriodNotes");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:klassifizierungskategorie", node, "accessRestrictionClassification");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:datenschutz", node, "accessRestrictionPrivacy");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:oeffentlichkeitsstatus", node, "accessRestrictionStatus");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:oeffentlichkeitsstatusBegruendung", node, "accessRestrictionStatusExplanation");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:sonstigeBestimmungen", node, "accessRestriction");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:bemerkung", node, "comment");
		this.addZusatzDatenElementFromUnconfiguredMetadata(newDossierElement, node);
		this.setAttributeFromDynamicMetadata(newDossierElement, "id", node, "refCodeAdmin");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:aktenzeichen", node, "refCode");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:schutzfristenkategorie", node, "accessPolicy");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:schutzfrist", node, "accessRestrictionPeriod");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:schutzfristenBegruendung", node, "accessRestrictionExplanation");
		this.addElementAndSetTextFromDynamicMetadata(newDossierElement, "BAR:umfang", node, "material");

		return newDossierElement;
	}


	private Element createDokument(Element insertPointAblieferungElement, NodeAbstract node)
	{
		Logger.getLogger().debug("Creating dokument for: '" + node.getLabel() + "'");
		this.distributeMessage("Creating dokument for: '" + node.getLabel() + "'");

		Element newDokumentElement = insertPointAblieferungElement.addElement("BAR:dokument");

		this.addElementAndSetText(newDokumentElement, "BAR:titel", node.getLabel());
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:autor", node, "origination");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:erscheinungsform", node, "characteristics");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:dokumenttyp", node, "scopeContent");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:registrierdatum/BAR:datum", node, "creationPeriod");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:entstehungszeitraum/BAR:von/BAR:datum", node, "fromYear");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:entstehungszeitraum/BAR:bis/BAR:datum", node, "toYear");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:klassifizierungskategorie", node, "accessRestrictionClassification");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:datenschutz", node, "accessRestrictionPrivacy");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:oeffentlichkeitsstatus", node, "accessRestrictionStatus");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:oeffentlichkeitsstatusBegruendung", node, "accessRestrictionStatusExplanation");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:sonstigeBestimmungen", node, "accessRestriction");
		this.addElementAndSetTextFromDynamicMetadata(newDokumentElement, "BAR:bemerkung", node, "comment");
		this.addZusatzDatenElementFromUnconfiguredMetadata(newDokumentElement, node);
		this.setAttributeFromDynamicMetadata(newDokumentElement, "id", node, "refCodeAdmin");

		return newDokumentElement;
	}


	private Element createInhaltsverzeichnisElement(Element insertPointInhaltsverzeichnisElement, NodeFile fileNode, String id)
	{
		Logger.getLogger().debug("Creating inhaltsverzeichnis element for: '" + fileNode.getLabel() + "'");
		this.distributeMessage("Creating inhaltsverzeichnis element for: '" + fileNode.getLabel() + "'");

		String fileName = fileNode.getFile().getName();

		Element newInhaltsverzeichnisElement = insertPointInhaltsverzeichnisElement.addElement("BAR:datei");

		this.setAttribute(newInhaltsverzeichnisElement, "id", id);
		this.addElementAndSetText(newInhaltsverzeichnisElement, "BAR:name", fileName);
		this.addElementAndSetText(newInhaltsverzeichnisElement, "BAR:originalName", fileName);
		this.addElementAndSetText(newInhaltsverzeichnisElement, "BAR:pruefalgorithmus", fileNode.getChecksumType());
		this.addElementAndSetText(newInhaltsverzeichnisElement, "BAR:pruefsumme", fileNode.getChecksum());

		//	ToDo: Loop through other dynamic metadata?
//		addEigenschaft(newInhaltsverzeichnisElement, "Sonstiges", "Sonstiges");

		return newInhaltsverzeichnisElement;
	}


	private Element createInhaltsverzeichnisElement(Element insertPointInhaltsverzeichnisElement, NodeFolder folderNode)
	{
		Logger.getLogger().debug("Creating inhaltsverzeichnis element for: '" + folderNode.getLabel() + "'");
		this.distributeMessage("Creating inhaltsverzeichnis element for: '" + folderNode.getLabel() + "'");

		String fileName = folderNode.getFile().getName();

		Element newInhaltsverzeichnisElement = insertPointInhaltsverzeichnisElement.addElement("BAR:ordner");

		this.addElementAndSetText(newInhaltsverzeichnisElement, "BAR:name", fileName);
		this.addElementAndSetText(newInhaltsverzeichnisElement, "BAR:originalName", fileName);

		return newInhaltsverzeichnisElement;
	}


	@SuppressWarnings("unchecked")
	private void cleanupIfNecessary(NodeAbstract node, String level, Element newAblieferungElement)
	{
		if (level.equals("paket"))
		{
			Node zusatzDatenElement = newAblieferungElement.selectSingleNode("BAR:zusatzDaten");
			if (zusatzDatenElement != null)
			{
				//	Move umfangNode behind the "paketTyp" node:
				for (int i = 0; i < newAblieferungElement.content().size(); i++)
				{
					if ("paketTyp".equals(((Node)newAblieferungElement.content().get(i)).getName()))
					{
						zusatzDatenElement.detach();
						newAblieferungElement.content().add(i + 1, zusatzDatenElement);
						break;
					}
				}
			}
		}
		else if (level.equals("ablieferung"))				;
		else if (level.equals("ordnungssystem"))			;
		else if (level.equals("ordnungssystemposition"))	;
		else if (level.equals("dossier"))
		{
			Node umfangNode = newAblieferungElement.selectSingleNode("BAR:umfang");
			if (umfangNode != null)
			{
				//	Move umfangNode to the end of the newAblieferungElement, behind any possible dossiers:
				umfangNode.detach();
				newAblieferungElement.content().add(newAblieferungElement.content().size(), umfangNode);
			}
		}
		else if (level.equals("dokument"))					;
		else if (level.equals("datei"))						;
		else if (level.equals("undefiniert"))				;
		else												;
	}


	/**
	 * Change in all nodes the Namespace 'oldNS' to 'newNS'.
	 *
	 * @param oldNS
	 * @param newNS
	 * @throws Exception
	 */
	private void changeNamespace(Namespace oldNS, Namespace newNS) throws Exception
	{
		/**
		 * In DOM4J Navigation uses a Visitor pattern to traverse through the document, the same we will use to change the namespace.
		 * NOTE: This class exists only within this method as I don't anticipate any use outside of this scope.
		 */
		class NamespaceChangingVisitor extends VisitorSupport
		{
			private Namespace from;
			private Namespace to;

			public NamespaceChangingVisitor(Namespace from, Namespace to)
			{
				this.from = from;
				this.to = to;
			}

			@Override
			public void visit(Element node)
			{
				Namespace ns = node.getNamespace();

				if (ns.getURI().equals(this.from.getURI()))
				{
					QName newQName = new QName(node.getName(), this.to);
					node.setQName(newQName);
				}

				//	The ListIterator allows to remove elements from it during traversion:
				ListIterator<?> namespaces = node.additionalNamespaces().listIterator();
				while (namespaces.hasNext())
					if (((Namespace)namespaces.next()).getURI().equals(this.from.getURI()))		namespaces.remove();
			}
		}

		Logger.getLogger().debug("Make 'BAR' the default namespace...");
		this.distributeMessage("Make 'BAR' the default namespace...");

		this.accept(new NamespaceChangingVisitor(oldNS, newNS));
	}

	//	--------			Create SIP		-------------------------------------------------------

	private void insertPaketInto(NodeFolder rootFolder) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		Node paketNode = this.getRootElement();

		Logger.getLogger().debug("Inserting Paket");

		rootFolder.setLevel(this.levels.get("Paket"));

		//	Fill in data into rootFolder from rootNode:
		rootFolder.setDynamicMetadataValueForName("usage", this.getZusatzDaten(paketNode, "BAR:zusatzDaten"));		//	Key-Value Structure, converted to JSON-Format.

		this.insertAblieferungInto(rootFolder, paketNode);
	}


	private void insertAblieferungInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		Node ablieferungNode = parentNode.selectSingleNode("BAR:ablieferung");

		Logger.getLogger().debug("Inserting Ablieferung");

		NodeFolder ablieferungFolder = parentFolder.createNewFolder("Ablieferung");
		ablieferungFolder.setLevel(this.levels.get("Ablieferung"));

		//	Fill in data into ablieferungFolder from ablieferungNode:
		ablieferungFolder.setDynamicMetadataValueForName("accessNr", this.getText(ablieferungNode, "BAR:ablieferndeStelle"));
		ablieferungFolder.setDynamicMetadataValueForName("refCode", this.getText(ablieferungNode, "BAR:ablieferungsnummer"));
		ablieferungFolder.setDynamicMetadataValueForName("objectType", this.getText(ablieferungNode, "BAR:ablieferungstyp"));
		ablieferungFolder.setDynamicMetadataValueForName("material", this.getText(ablieferungNode, "BAR:ablieferungsteile"));
		ablieferungFolder.setDynamicMetadataValueForName("refCodeAdmin", this.getText(ablieferungNode, "BAR:angebotsnummer"));
		ablieferungFolder.setDynamicMetadataValueForName("fromYear", this.getText(ablieferungNode, "BAR:entstehungszeitraum/BAR:von/BAR:datum"));
		ablieferungFolder.setDynamicMetadataValueForName("toYear", this.getText(ablieferungNode, "BAR:entstehungszeitraum/BAR:bis/BAR:datum"));
		ablieferungFolder.setDynamicMetadataValueForName("comment", this.getText(ablieferungNode, "BAR:bemerkung"));
		ablieferungFolder.setDynamicMetadataValueForName("appraisalAndDestruction", this.getText(ablieferungNode, "BAR:referenzBewertungsentscheid"));
		ablieferungFolder.setDynamicMetadataValueForName("accessRestriction", this.getText(ablieferungNode, "BAR:referenzSchutzfristenFormular"));
		ablieferungFolder.setDynamicMetadataValueForName("accessPolicy", this.getText(ablieferungNode, "BAR:schutzfristenkategorie"));
		ablieferungFolder.setDynamicMetadataValueForName("accessRestrictionPeriod", this.getText(ablieferungNode, "BAR:schutzfrist"));
		ablieferungFolder.setDynamicMetadataValueForName("origination", this.getText(ablieferungNode, "BAR:provenienz/BAR:aktenbildnerName"));

		//	Dive in:
		this.insertOrdnungssystemInto(ablieferungFolder, ablieferungNode);
	}


	private void insertOrdnungssystemInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		Node osNode = parentNode.selectSingleNode("BAR:ordnungssystem");
		String label = this.getAsFileName(osNode, "BAR:name");

		Logger.getLogger().debug("Inserting Ordnungssystem " + label);

		NodeFolder osFolder = parentFolder.createNewFolder(label, true);
		osFolder.setLevel(this.levels.get("Ordnungssystem"));

		//	Fill in data into osFolder from osNode:
		osFolder.setDynamicMetadataValueForName("refCode", this.getText(osNode, "BAR:generation"));
		osFolder.setDynamicMetadataValueForName("fromYear", this.getText(osNode, "BAR:anwendungszeitraum/BAR:von/BAR:datum"));
		osFolder.setDynamicMetadataValueForName("toYear", this.getText(osNode, "BAR:anwendungszeitraum/BAR:bis/BAR:datum"));
		osFolder.setDynamicMetadataValueForName("involved", this.getText(osNode, "BAR:mitbenutzung"));
		osFolder.setDynamicMetadataValueForName("comment", this.getText(osNode, "BAR:bemerkung"));

		//	Dive in:
		this.insertOrdnungssystempositionenInto(osFolder, osNode);
	}


	private void insertOrdnungssystempositionenInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		List<?> osPositions = parentNode.selectNodes("BAR:ordnungssystemposition");
		for (Object o: osPositions)
		{
			Node osPosNode = (Node)o;
			String label = this.getAsFileName(osPosNode, "BAR:titel");

			Logger.getLogger().debug("Inserting Ordnungssystemposition " + label);

			NodeFolder osPosFolder = parentFolder.createNewFolder(label, true);
			osPosFolder.setLevel(this.levels.get("Ordnungssystemposition"));

			//	Fill in data into osPosFolder from osPosNode:
			osPosFolder.setDynamicMetadataValueForName("refCode", this.getText(osPosNode, "BAR:nummer"));
			osPosFolder.setDynamicMetadataValueForName("refCodeAdmin", ((Element)osPosNode).attributeValue("id"));
			osPosFolder.setDynamicMetadataValueForName("origination", this.getText(osPosNode, "BAR:federfuehrendeOrganisationseinheit"));
			osPosFolder.setDynamicMetadataValueForName("accessPolicy", this.getText(osPosNode, "BAR:schutzfristenkategorie"));
			osPosFolder.setDynamicMetadataValueForName("accessRestrictionPeriod", this.getText(osPosNode, "BAR:schutzfrist"));
			osPosFolder.setDynamicMetadataValueForName("accessRestrictionExplanation", this.getText(osPosNode, "BAR:schutzfristenBegruendung"));
			osPosFolder.setDynamicMetadataValueForName("accessRestrictionClassification", this.getText(osPosNode, "BAR:klassifizierungskategorie"));
			osPosFolder.setDynamicMetadataValueForName("accessRestrictionPrivacy", this.getText(osPosNode, "BAR:datenschutz"));
			osPosFolder.setDynamicMetadataValueForName("accessRestrictionStatus", this.getText(osPosNode, "BAR:oeffentlichkeitsstatus"));
			osPosFolder.setDynamicMetadataValueForName("accessRestrictionStatusExplanation", this.getText(osPosNode, "BAR:oeffentlichkeitsstatusBegruendung"));
			osPosFolder.setDynamicMetadataValueForName("accessRestriction", this.getText(osPosNode, "BAR:sonstigeBestimmungen"));

			//	Dive in:
			this.insertOrdnungssystempositionenInto(osPosFolder, osPosNode);		//	Recursion
			this.insertDossiersInto(osPosFolder, osPosNode);
		}
	}


	private void insertDossiersInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		List<?> dossiers = parentNode.selectNodes("BAR:dossier");
		for (Object o: dossiers)
		{
			Node dossierNode = (Node)o;
			String label = this.getAsFileName(dossierNode, "BAR:titel");

			Logger.getLogger().debug("Inserting Dossier " + label);

			NodeFolder dossierFolder = parentFolder.createNewFolder(label, true);
			dossierFolder.setLevel(this.levels.get("Dossier"));

			//	Fill in data into dossierFolder from dossierNode:
			dossierFolder.setDynamicMetadataValueForName("refCode", this.getText(dossierNode, "BAR:aktenzeichen"));
			dossierFolder.setDynamicMetadataValueForName("archivalHistory", this.getText(dossierNode, "BAR:zusatzmerkmal"));
			dossierFolder.setDynamicMetadataValueForName("abstract", this.getText(dossierNode, "BAR:inhalt"));
			dossierFolder.setDynamicMetadataValueForName("refCodeAdmin", ((Element)dossierNode).attributeValue("id"));
			dossierFolder.setDynamicMetadataValueForName("characteristics", this.getText(dossierNode, "BAR:erscheinungsform"));
			dossierFolder.setDynamicMetadataValueForName("scopeContent", this.getText(dossierNode, "BAR:formInhalt"));
			dossierFolder.setDynamicMetadataValueForName("material", this.getText(dossierNode, "BAR:umfang"));
			dossierFolder.setDynamicMetadataValueForName("origination", this.getText(dossierNode, "BAR:federfuehrendeOrganisationseinheit"));
			dossierFolder.setDynamicMetadataValueForName("fromYear", this.getText(dossierNode, "BAR:entstehungszeitraum/BAR:von/BAR:datum"));
			dossierFolder.setDynamicMetadataValueForName("toYear", this.getText(dossierNode, "BAR:entstehungszeitraum/BAR:bis/BAR:datum"));
			dossierFolder.setDynamicMetadataValueForName("creationPeriodNotes", this.getText(dossierNode, "BAR:entstehungszeitraumAnmerkung"));
			dossierFolder.setDynamicMetadataValueForName("comment", this.getText(dossierNode, "BAR:bemerkung"));
			dossierFolder.setDynamicMetadataValueForName("accessPolicy", this.getText(dossierNode, "BAR:schutzfristenkategorie"));
			dossierFolder.setDynamicMetadataValueForName("accessRestrictionPeriod", this.getText(dossierNode, "BAR:schutzfrist"));
			dossierFolder.setDynamicMetadataValueForName("accessRestrictionExplanation", this.getText(dossierNode, "BAR:schutzfristenBegruendung"));
			dossierFolder.setDynamicMetadataValueForName("accessRestrictionClassification", this.getText(dossierNode, "BAR:klassifizierungskategorie"));
			dossierFolder.setDynamicMetadataValueForName("accessRestrictionPrivacy", this.getText(dossierNode, "BAR:datenschutz"));
			dossierFolder.setDynamicMetadataValueForName("accessRestrictionStatus", this.getText(dossierNode, "BAR:oeffentlichkeitsstatus"));
			dossierFolder.setDynamicMetadataValueForName("accessRestrictionStatusExplanation", this.getText(dossierNode, "BAR:oeffentlichkeitsstatusBegruendung"));
			dossierFolder.setDynamicMetadataValueForName("accessRestriction", this.getText(dossierNode, "BAR:sonstigeBestimmungen"));

			//	Dive in:
			this.insertDossiersInto(dossierFolder, dossierNode);					//	Recursion
			this.insertDocumentsInto(dossierFolder, dossierNode);
			this.insertDateiRefsInto(dossierFolder, dossierNode);
		}
	}


	private void insertDocumentsInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, FolderNameIsEmptyException, FileUtilExceptionListException, MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException, LevelMetadataElementIsReadOnly, MetadataElementSetterPostActionException, MetadataElementAllowedValuesException
	{
		List<?> documents = parentNode.selectNodes("BAR:dokument");
		for (Object o: documents)
		{
			Node documentNode = (Node)o;
			String label = this.getAsFileName(documentNode, "BAR:titel");

			Logger.getLogger().debug("Inserting Document " + label);

			NodeFolder documentFolder = parentFolder.createNewFolder(label, true);
			documentFolder.setLevel(this.levels.get("Dokument"));

			//	Fill in data into documentFolder from documentNode:
			documentFolder.setDynamicMetadataValueForName("refCodeAdmin", ((Element)documentNode).attributeValue("id"));
			documentFolder.setDynamicMetadataValueForName("characteristics", this.getText(documentNode, "BAR:erscheinungsform"));
			documentFolder.setDynamicMetadataValueForName("scopeContent", this.getText(documentNode, "BAR:dokumenttyp"));
			documentFolder.setDynamicMetadataValueForName("origination", this.getText(documentNode, "BAR:autor"));
			documentFolder.setDynamicMetadataValueForName("fromYear", this.getText(documentNode, "BAR:entstehungszeitraum/BAR:von/BAR:datum"));
			documentFolder.setDynamicMetadataValueForName("toYear", this.getText(documentNode, "BAR:entstehungszeitraum/BAR:bis/BAR:datum"));
			documentFolder.setDynamicMetadataValueForName("creationPeriod", this.getText(documentNode, "BAR:registrierdatum/BAR:datum"));
			documentFolder.setDynamicMetadataValueForName("comment", this.getText(documentNode, "BAR:bemerkung"));
			documentFolder.setDynamicMetadataValueForName("accessRestrictionClassification", this.getText(documentNode, "BAR:klassifizierungskategorie"));
			documentFolder.setDynamicMetadataValueForName("accessRestrictionPrivacy", this.getText(documentNode, "BAR:datenschutz"));
			documentFolder.setDynamicMetadataValueForName("accessRestrictionStatus", this.getText(documentNode, "BAR:oeffentlichkeitsstatus"));
			documentFolder.setDynamicMetadataValueForName("accessRestrictionStatusExplanation", this.getText(documentNode, "BAR:oeffentlichkeitsstatusBegruendung"));
			documentFolder.setDynamicMetadataValueForName("accessRestriction", this.getText(documentNode, "BAR:sonstigeBestimmungen"));

			//	Dive in:
			this.insertDateiRefsInto(documentFolder, documentNode);
		}
	}


	private void insertDateiRefsInto(NodeFolder parentFolder, Node parentNode) throws FileAlreadyExistsException, IOException, FileOperationNotAllowedException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, FileUtilExceptionListException
	{
		List<?> fileRefs = parentNode.selectNodes("BAR:dateiRef");
		for (Object o: fileRefs)
		{
			Node fileRefNode = (Node)o;
			String relativeFilePath = this.getFilePath(fileRefNode);

			Logger.getLogger().debug("Inserting FileRef " + relativeFilePath);

			//	Insert file or folder, rename if necessary:
			NodeAbstract fileRefFile = parentFolder.insertFileOrFolder(this.filePath + relativeFilePath, true);

			fileRefFile.setLevel(this.levels.get("Datei"));

			//	Fill in data into fileRefFile from fileRefNode:
			//		(none)
		}
	}


	private String getFilePath(Node fileRefNode)
	{
		String fileRefId = fileRefNode.getText();
		Node fileNode = this.getRootElement().selectSingleNode("BAR:inhaltsverzeichnis//BAR:datei [@id='" + fileRefId + "']");

		Node parent = fileNode;
		String relativeFilePath = "";
		do
		{
			relativeFilePath = "/" + parent.selectSingleNode("BAR:name").getText() + relativeFilePath;
			parent = parent.getParent();
		}
		while (!(parent.selectSingleNode("BAR:name") == null));

		return relativeFilePath;
	}


	private void addElementAndSetTextFromDynamicMetadata(Element insertPoint, String newElementName, NodeAbstract node, String metadataName)
	{
		try
		{
			for (String text: node.getAllDynamicMetadataValuesForName(metadataName))
			{
				this.addElementAndSetText(insertPoint, newElementName, text);
			}
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	private void addElementAndSetText(Element insertPoint, String newElementName, String text)
	{
		if (text == null || text.isEmpty())		return;

		XMLUtil.add(insertPoint, newElementName, text);
	}


	@SuppressWarnings("unused")
	private void addEigenschaft(Element insertPoint, String eigenschaftName, String text)
	{
		Element newElement = XMLUtil.add(insertPoint, "BAR:eigenschaft", text);
		this.setAttribute(newElement, "name", eigenschaftName);
	}


	private void setAttributeFromDynamicMetadata(Element insertPoint, String attributeName, NodeAbstract node, String metadataName)
	{
		try
		{
			String text = node.getDynamicMetadataValueForName(metadataName);
			this.setAttribute(insertPoint, attributeName, text);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	private void setAttribute(Element insertPoint, String attributeName, String text)
	{
		if (text == null)		return;

		insertPoint.addAttribute(attributeName, text);
	}


	/**
	 * This is used only in "fillPaket"!
	 * @param insertPoint
	 * @param node
	 * @param metadataName
	 */
	@SuppressWarnings("unchecked")
	private void addZusatzDatenElementFromDynamicMetadata(Element insertPoint, NodeAbstract node, String metadataName)
	{
		List<String> jsonStrings;
		try
		{
			jsonStrings = node.getAllDynamicMetadataValuesForName(metadataName);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			return;
		}

		for (String jsonString: jsonStrings)
		{
			Map<String, String> keyValueMap;
			try
			{
				keyValueMap = new Gson().fromJson(jsonString, Map.class);
			}
			catch (JsonSyntaxException ex)
			{
				//	In the case the string is not in JSON-format, put the complete string as the content:
				keyValueMap = new HashMap<String, String>();
				keyValueMap.put(metadataName, jsonString);
			}

			if (keyValueMap.isEmpty())					continue;

			for (Map.Entry<String, String> entry: keyValueMap.entrySet())
			{
				String key = entry.getKey();
				String value = entry.getValue();
				if (value == null || value.isEmpty())	continue;

				this.addElementAndSetText(insertPoint, "BAR:zusatzDaten/BAR:merkmal[@name='" + key + "']", value);
			}
		}
	}


	private void addZusatzDatenElementFromUnconfiguredMetadata(Element insertPoint, NodeAbstract node)
	{
		//	Add the zusatzDatenElement only if it does not already exist:
		Element zusatzDatenElement = (Element)insertPoint.selectSingleNode("BAR:zusatzDaten");
		if (zusatzDatenElement == null)		zusatzDatenElement = insertPoint.addElement("BAR:zusatzDaten");

		//	This are the names of the metadata elements configured for this node's level:
		Set<String> configuredMetadata = node.getLevel().getDynamicMetadataElements().keySet();

		//	This loops through all available metadata element names:
		for (String accessorName: MetadataElement.getAll().keySet())
		{
			if (!configuredMetadata.contains(accessorName))
			{
				//	This is an unconfigured metadata element:

				List<String> values = null;
				try
				{
					values = node.getAllDynamicMetadataValuesForName_NoCheck(accessorName);
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}

				//	Ignore if empty:
				if (values == null || values.isEmpty())			continue;

				for (String value: values)
				{
					//	Ignore if empty:
					if (value == null || value.isEmpty())		continue;

					//	Add "merkmal" element:
					Element newElement = zusatzDatenElement.addElement("BAR:merkmal");
					newElement.addAttribute("name", accessorName);
					newElement.setText(value);
				}
			}
		}

		//	Remove this zusatzDatenElement if it is empty, i.e. when no merkmale were added:
		if (zusatzDatenElement.content().size() == 0)			zusatzDatenElement.detach();
	}



	/**
	 * Return the text of the node denoted by xPath relative to node.
	 * If node or the resulting xPath node is null, return null; otherwise return the node's text.
	 */
	private String getText(Node node, String xPath)
	{
		if (node == null)		return null;

		Node subNode = node.selectSingleNode(xPath);
		if (subNode == null)	return null;

		return subNode.getText();
	}


	private String getAsFileName(Node node, String xPath)
	{
		String label = this.getText(node, xPath);
		if (label == null || label.isEmpty()) {
			label = node.getName() + UniqueID.getXML();
		} else {
			label = FileUtil.asSafeFileName(label);
		}
		return label;
	}


	private String getZusatzDaten(Node node, String xPath)
	{
		if (node == null)		return null;

		Node subNode = node.selectSingleNode(xPath);
		if (subNode == null)	return null;

		Map<String, String> map = new TreeMap<String, String>();
		for (Object o: subNode.selectNodes("BAR:merkmal"))
		{
			Element itemElement = (Element)o;
			map.put(itemElement.attributeValue("name"), itemElement.getText());
		}

		return new Gson().toJson(map);
	}

	/**
	 * This Factory sets the default namespace for the BAR-SIP Document.
	 */
	static private class BARSIPFactory extends DocumentFactory
	{
		@Override
		public Document createDocument()
		{
			Map<String, String> namespaces = new TreeMap<String, String>();
			namespaces.put(BARNamespacePrefix, BARNamespaceURI);
			this.setXPathNamespaceURIs(namespaces);

			Document barSIP = new Document();
			barSIP.setDocumentFactory(this);

			return barSIP;
		}
	}

	static private class DateiId
	{
		static private int		DateiIdCounter = 0;

		static private String getNext()
		{
			return "d" + StringUtil.last("000000" + DateiIdCounter++, 6);
		}
	}


	static private class ObserverTracer implements Observer
	{
		@Override
		public void update(Observable o, Object arg)
		{
			Tracer.trace(arg);
		}
	}

}
