/**
 *	Copyright (C) 2015-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.ingest;

import static ch.docuteam.darc.DarcConstants.UNDERSCORE_ZIP;
import static ch.docuteam.darc.DarcConstants.ZIP_EXT;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import ch.docuteam.darc.mdconfig.LevelOfDescription;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.Zipper;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;

/**
 * This basic implementation of {@link ch.docuteam.darc.ingest.AIPCreator} will check some fundamental characteristics of the SIP and _move_ it to a directory
 * where it is expected to be picked up by subsequent processes. Only the {@link AIPCreatorBasic.submit(Document)} and
 * {@link AIPCreatorBasic.checkSubmission(Document)} methods are implemented.
 * 
 * @author Andreas Nef, Docuteam GmbH
 *
 */
public class AIPCreatorBasic implements AIPCreator {


	private String ingestSubmitDir;
	
	@Override
	public void initialize(Properties properties) throws Exception {
		String osSuffix = OperatingSystem.isWindows() ? "Win"
				: (OperatingSystem.isMacOSX() ? "OSX" : (OperatingSystem.isLinux() ? "Linux" : ""));
		ingestSubmitDir = properties.getProperty("AIPCreator.ingestSubmitDir." + osSuffix);
		if (ingestSubmitDir == null)
			throw new NullPointerException("Missing property: 'AIPCreator.ingestSubmitDir'");
	}

	/**
	 * This implementation will check that:
	 * - the SIP has no unsaved modifications
	 * - the target folder for the submit exists and doesn't already contain a SIP of the same name
	 * - the SIP is not locked or read-only
	 * - the nodes' file representations exist and are writable
	 * - mandatory metadata exist
	 * - all nodes have a defined level of description
	 */
	@Override
	public List<String> checkSubmission(Document document) {
		List<String> rejectMessages = new Vector<String>();

		// Unsaved changes?
		if (document.isModified()) {
			Logger.debug("Can't submit because SIP has unsaved changes");
			rejectMessages.add("MessageSubmitUnsavedChanges");
		}

		// Check if submit folder exists:
		if (!Files.exists(Paths.get(ingestSubmitDir))) {
			Logger.debug("Can't submit because submit folder doesn't exist '" + ingestSubmitDir + "'");
			rejectMessages.add("MessageSubmitFolderDoesNotExist '" + ingestSubmitDir + "'");
		}
		
		// Does it contain a SIP of the same name already?
		String submitSIPName = getSIPNameForSubmit(document);
		Path submitSIPPath = Paths.get(ingestSubmitDir, submitSIPName);
		if (Files.exists(submitSIPPath)) {
			Logger.debug("Can't submit because submit SIP exists already in submit folder '" + ingestSubmitDir + "'");
			rejectMessages.add("MessageSubmitSIPExistsAlreadyInSubmitFolder '" + submitSIPName + "'");
		}

		String sipName = document.getSIPName();
		Logger.info("Performing submit-check in SIP: '" + sipName + "'");

		// Is this document read-only? If yes, reject:
		if (document.isReadOnly()) {
			Logger.debug("Can't submit because SIP is read-only: '" + sipName + "'");
			rejectMessages.add("MessageSubmitSIPIsReadOnly '" + sipName + "'");
		}

		// Is this document locked? If yes, reject:
		if (document.isLocked()) {
			Logger.debug("Can't submit because SIP is locked: '" + sipName + "'");
			rejectMessages.add("MessageSubmitSIPIsLocked '" + sipName + "'");
		}

		// Check each node in tree:
		Logger.debug("Checking all nodes in SIP: '" + sipName + "'");
		for (String message : submitCheckRecursive(document.getStructureMap().getRoot())) {
			rejectMessages.add(message);
		}

		if (rejectMessages.isEmpty()) {
			Logger.debug("SIP is OK");
		} else {
			Logger.debug("Can't submit SIP because: " + rejectMessages);
		}

		return rejectMessages;
	}


	/**
	 * First check if submit can be issued. If yes, set all submit status
	 * entries that are "SubmitRequested" to "SubmitRequestPending", save and
	 * lock this document, then trigger the feeder-workflow "Ingest" (Here: by
	 * zipping the SIP into the submit folder).
	 * 
	 * @return a list of error messages. If this list is empty, the submit was
	 *         successful.
	 * @throws java.lang.Exception
	 */
	public List<String> submit(Document document) throws java.lang.Exception {
		// Check document before submitting. If there are messages, don't submit but return the messages:
		List<String> rejectMessages = checkSubmission(document);
		if (!rejectMessages.isEmpty()) {
			return rejectMessages;
		}

		try {
			Logger.getLogger().debug("Zipping to folder: '" + ingestSubmitDir + "' SIP: '" + document.getSIPName() + "'");

			if (document.isZIPFile()) {
				FileUtil.copyToOverwriting(document.getOriginalSIPFolder(),
						ingestSubmitDir + File.separator + getSIPNameForSubmit(document));
			} else {
				Zipper.zip(document.getOriginalSIPFolder(),
						ingestSubmitDir + File.separator + getSIPNameForSubmit(document), true);
			}
			
			document.unlockIfNecessary();
			// Delete SIP (Careful: working copy and original) 
			FileUtil.delete(document.getOriginalSIPFolder());
			document.cleanupWorkingCopy();
		} catch (IOException e) {
			Logger.getLogger().debug("Copying to submit folder failed", e);
			rejectMessages.add("Copying to submit folder failed: " + e.getMessage());
		}

		return rejectMessages;
	}


	@Override
	public List<String> checkIngestFeedback(String workspaceFolder) throws Exception {
		return null;
	}

	@Override
	public List<String> generateAIP(Document document) throws Exception {
		return null;
	}

	@Override
	public void confirmStorage(Document document) throws Exception {
		return;
	}


	/**
	 * Return the SIP fileName to be used when submitting this SIP. Since the
	 * submitted SIP will be zipped, it must have the ".zip" suffix. In
	 * addition, if the *original* SIP is zipped, the file name contains "_zip"
	 * for distinction; this is needed for the ingest feedback.
	 * 
	 * @return
	 */
	private String getSIPNameForSubmit(Document mets) {
		String zipFileNameForSubmit = FileUtil.asFileNameWithoutExtension(mets.getSIPName()) + UNDERSCORE_ZIP;
		String result = (mets.isZIPFile() ? zipFileNameForSubmit : mets.getSIPName()) + ZIP_EXT;
		return result;
	}

	private List<String> submitCheckRecursive(NodeAbstract node) {
		List<String> rejectMessages = new Vector<String>();

		// Check each node of the SIP if it could be submitted:
		for (NodeAbstract n : node.getWithDescendants())
			for (String message : this.submitCheck(n))
				rejectMessages.add(message);

		return rejectMessages;
	}

	private List<String> submitCheck(NodeAbstract node) {
		List<String> rejectMessages = new Vector<String>();
		String pathString = node.getPathString();

		Logger.info("Performing submit-check for node: '" + pathString + "' (" + node.getAdmId() + ")");

		// Check file permissions
		if (!node.fileExists())
			rejectMessages.add("MessageSubmitNodeFileDoesNotExist '" + pathString + "'");
		if (!node.canRead())
			rejectMessages.add("MessageSubmitNodeFileIsNotReadable '" + pathString + "'");
		if (!node.canWrite())
			rejectMessages.add("MessageSubmitNodeFileIsNotWritable '" + pathString + "'");

		// Don't allow undefined levels:
		if (node.getLevel().equals(LevelOfDescription.getUndefined()))
			rejectMessages.add("MessageSubmitNodeHasUndefinedLevel '" + pathString + "'");

		// Check mandatory metadata fields
		if (node.hasDynamicMetadataElementInstancesWhichAreMandatoryButNotSet())
			rejectMessages.add("MessageSubmitNodeMandatoryMetadataNotSet '" + pathString + "'");

		if (rejectMessages.isEmpty())
			Logger.info("Node is OK");
		else
			Logger.info("Can't submit node because: " + rejectMessages);

		return rejectMessages;
	}

}
