/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.amdsec;

import java.util.*;

import org.dom4j.Element;
import org.dom4j.Node;

import ch.docuteam.darc.mets.metshdr.Header;
import ch.docuteam.darc.premis.Event;
import ch.docuteam.darc.premis.Object;
import ch.docuteam.tools.exception.Exception;

/**
 * This class, used by the class <a href="./AMDSection.html">AMDSection</a>,
 * represents a METS digiprovMD entity. It contains a collection of
 * <a href="../../premis/Object.html">PREMIS Objects</a> and a collection of
 * <a href="../../premis/Event.html">PREMIS Events</a>.
 * <p>
 * This class inserts Exceptions into the
 * <a href="../../../docutools/exception/ExceptionCollector.html">
 * ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when no PREMIS:premis element in the DigiprovMD
 * element was found</li>
 * </ul>
 * 
 * @author denis
 */
public class DigiprovWithPremis extends DigiprovAbstract {

	private static final String XMLNS_PREMIS = "info:lc/xmlns/premis-v2";
	private static final String PREMIS_Version = "2.2";

	/**
	 * Sorted Map containing this digiprov's objects. Key is the Object's id,
	 * value is the Object itself.
	 */
	protected SortedMap<String, Object> objects = new TreeMap<String, Object>();

	/**
	 * Sorted Map containing this digiprov's events. Key is the Event's id,
	 * value is the Event itself.
	 */
	protected SortedMap<String, Event> events = new TreeMap<String, Event>();

	static {
		MDType = "PREMIS";
	}

	/**
	 * This constructor is used only when a new digiprov is created
	 * programmatically, for a NodeAbstract
	 */
	public DigiprovWithPremis(ch.docuteam.darc.mets.structmap.NodeAbstract node) {
		super(node);

		Element mdWrap = this.element.addElement("METS:mdWrap").addAttribute("MDTYPE", MDType);
		Element xmlData = mdWrap.addElement("METS:xmlData");
		xmlData.addElement("PREMIS:premis").addAttribute("xmlns:PREMIS", XMLNS_PREMIS).addAttribute("version",
				PREMIS_Version);

		// Create new object:
		Object newObject = this.addNewObject(node);

		// Create new event:
		this.addNewEventWithLinkObjectId(newObject.getId(), null, "Creation", "", "Success", "");
	}

	/**
	 * This constructor is used only when a new digiprov is created
	 * programmatically, for a metshdr.Header
	 */
	public DigiprovWithPremis(Header header) {
		super(header);

		Element mdWrap = this.element.addElement("METS:mdWrap").addAttribute("MDTYPE", MDType);
		Element xmlData = mdWrap.addElement("METS:xmlData");
		xmlData.addElement("PREMIS:premis").addAttribute("xmlns:PREMIS", XMLNS_PREMIS).addAttribute("version",
				PREMIS_Version);

		// Create new object:
		Object newObject = new Object(this,
				(Element) this.element.selectSingleNode("./METS:mdWrap[@MDTYPE='PREMIS']/METS:xmlData/PREMIS:premis"),
				header);
		this.objects.put(newObject.getId(), newObject);

		// DON'T create new event:
		// this.addNewEventWithLinkObjectId(newObject.getId(), "Creation", "",
		// "Success", "");
	}

	/**
	 * This constructor is used only when a METS-File is being read.
	 */
	private DigiprovWithPremis(AMDSection parent, Element digiprovElement) {
		super(parent, digiprovElement);

		Element premisElement = (Element) digiprovElement
				.selectSingleNode("./METS:mdWrap[@MDTYPE='PREMIS']/METS:xmlData/PREMIS:premis");
		if (premisElement == null) {
			Exception.remember("Couldn't locate node 'PREMIS:premis' in node 'METS:digiprovMD' with id=" + this.id);
			return;
		}

		for (Object object : Object.parse(this, premisElement))
			this.objects.put(object.getId(), object);
		for (Event event : Event.parse(this, premisElement))
			this.events.put(event.getId(), event);
	}

	/**
	 * This method is used only when a METS-File is being read. Create the list
	 * of Digiprovs out of the given parent AMDSection
	 * 
	 * @param parent
	 *            The parent (an AMDSection)
	 */
	static protected List<DigiprovWithPremis> parse(AMDSection parent) {
		List<DigiprovWithPremis> fileList = new ArrayList<DigiprovWithPremis>();

		for (java.lang.Object o : parent.getElement().selectNodes("./METS:digiprovMD"))
			fileList.add(new DigiprovWithPremis(parent, (Element) o));

		return fileList;
	}

	public Object getObject(String id) {
		return this.objects.get(id);
	}

	public List<Object> getObjects() {
		return new ArrayList<Object>(this.objects.values());
	}

	public Object getObject(Integer i) {
		return this.getObjects().get(i);
	}

	public Object getLastObject() {
		Integer objectCount = this.objects.size();
		if (objectCount <= 0)
			return null;

		return this.getObject(objectCount - 1);
	}

	public Event getEvent(String id) {
		return this.events.get(id);
	}

	public List<Event> getEvents() {
		return new ArrayList<Event>(this.events.values());
	}

	public Event getEvent(Integer i) {
		return this.getEvents().get(i);
	}

	public Event getLastEvent() {
		Integer eventCount = this.events.size();
		if (eventCount <= 0)
			return null;

		return this.getEvents().get(eventCount - 1);
	}

	/**
	 * Add a new Object to this digiprov.
	 */
	public Object addNewObject(ch.docuteam.darc.mets.structmap.NodeAbstract node) {
		Object newObject = new Object(this,
				(Element) this.element.selectSingleNode("./METS:mdWrap[@MDTYPE='PREMIS']/METS:xmlData/PREMIS:premis"),
				node);
		this.objects.put(newObject.getId(), newObject);

		this.document.setIsModified();

		return newObject;
	}

	/**
	 * Add a new Object with relationship information (linking to the event and
	 * to the object linked to that event) to this digiprov.
	 */
	public Object addNewObjectWithRelationship(ch.docuteam.darc.mets.structmap.NodeAbstract node, Event event) {
		Object newObject = this.addNewObject(node);
		newObject.addRelationships(event);
		return newObject;
	}

	/**
	 * Add a new Event to this digiprov with the current date as the event date
	 * 
	 * @param type
	 *            The Event type
	 * @param detail
	 *            The Event detail
	 * @param outcome
	 *            The Event outcome
	 * @param outcomeDetail
	 *            The Event outcome detail
	 * @return The added Event
	 */
	public Event addNewEvent(String type, String detail, String outcome, String outcomeDetail) {
		return this.addNewEvent(null, type, detail, outcome, outcomeDetail);
	}

	/**
	 * Add a new Event to this digiprov
	 * 
	 * @param date
	 *            The Event date/time, should be in format
	 *            "yyyy-MM-dd'T'HH:mm:ss"
	 * @param type
	 *            The Event type
	 * @param detail
	 *            The Event detail
	 * @param outcome
	 *            The Event outcome
	 * @param outcomeDetail
	 *            The Event outcome detail
	 * @return The added Event
	 */
	public Event addNewEvent(String date, String type, String detail, String outcome, String outcomeDetail) {
		return this.addNewEventWithLinkObjectId(this.getLastObject().getId(), date, type, detail, outcome,
				outcomeDetail);
	}

	/**
	 * Add a new Event to this digiprov at the 1. event position with the
	 * current date as event date. NOTE: This doesn't work correctly yet. The
	 * events objects are sorted according to their IDs, which is derived from
	 * the creation time. So the event dom4j ELEMENTS are being reordered within
	 * the mets file, but the event OBJECTS are still in their old order.
	 * 
	 * @param type
	 *            The Event type
	 * @param detail
	 *            The Event detail
	 * @param outcome
	 *            The Event outcome
	 * @param outcomeDetail
	 *            The Event outcome detail
	 * @return The added Event
	 */
	public Event addNewFirstEvent(String type, String detail, String outcome, String outcomeDetail) {
		return addNewFirstEvent(null, type, detail, outcome, outcomeDetail);
	}

	/**
	 * Add a new Event to this digiprov at the 1. event position. NOTE: This
	 * doesn't work correctly yet. The events objects are sorted according to
	 * their IDs, which is derived from the creation time. So the event dom4j
	 * ELEMENTS are being reordered within the mets file, but the event OBJECTS
	 * are still in their old order.
	 * 
	 * @param date
	 *            The Event date/time, should be in format
	 *            "yyyy-MM-dd'T'HH:mm:ss"
	 * @param type
	 *            The Event type
	 * @param detail
	 *            The Event detail
	 * @param outcome
	 *            The Event outcome
	 * @param outcomeDetail
	 *            The Event outcome detail
	 * @return The added Event
	 */
	@SuppressWarnings("unchecked")
	public Event addNewFirstEvent(String date, String type, String detail, String outcome, String outcomeDetail) {
		Event newEvent = this.addNewEvent(date, type, detail, outcome, outcomeDetail);

		// ToDo: Move this event to the 1. position of the events:
		Element parentElement = newEvent.getElement().getParent();

		int indexOfLastObject = -1;
		for (int i = parentElement.content().size() - 1; i >= 0; i--) {
			if (((Node) parentElement.content().get(i)).hasContent()
					&& "PREMIS:object".equals(((Element) parentElement.content().get(i)).getQualifiedName())) {
				indexOfLastObject = i;
				break;
			}
		}

		newEvent.getElement().detach();
		parentElement.content().add(indexOfLastObject + 1, newEvent.getElement());

		return newEvent;
	}

	@Override
	public String toString() {
		StringBuilder buf = new StringBuilder("[AMDSectionDigiprov:id=" + this.id);
		for (Object object : this.objects.values())
			buf.append("\n\t\t").append(object.toString());
		for (Event event : this.events.values())
			buf.append("\n\t\t").append(event.toString());
		buf.append("]");

		return buf.toString();
	}

	/**
	 * Create a new Event and add it to the Event map.
	 * 
	 * @return The added Event
	 */
	private Event addNewEventWithLinkObjectId(String linkObjectId, String date, String type, String detail,
			String outcome, String outcomeDetail) {
		Event newEvent = new Event(this,
				(Element) this.element.selectSingleNode("./METS:mdWrap[@MDTYPE='PREMIS']/METS:xmlData/PREMIS:premis"),
				linkObjectId, date, type, detail, outcome, outcomeDetail);
		this.events.put(newEvent.getId(), newEvent);

		this.document.setIsModified();

		return newEvent;
	}

}
