/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.ead;

import java.util.*;

import org.dom4j.*;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.mdconfig.LevelOfDescription;
import ch.docuteam.darc.mets.dmdsec.DMDSectionWithEAD;
import ch.docuteam.darc.mets.structmap.NodeAbstract.SubmitStatus;

/**
 * This abstract class is the superclass of {@link C} and {@link ArchDesc}. It
 * contains most of the dynamic metadata objects. Exempt are those that are
 * stored in the {@link DID} element. In addition, it contains one {@link DID}
 * and a List of {@link C} elements; so it is recursive.
 *
 * @author denis
 *
 */
public abstract class ComponentAbstract extends EADNodeAbstract {
	protected static final String Level = "otherlevel";
	protected static final String Role = "simple";

	protected static final String Xmlns_xlink = "http://www.w3.org/1999/xlink";

	protected static final String XPathPID = "EAD:dao[@xlink:role = '" + Role + "']";

	protected static final String XPathAccessPolicy = "EAD:accessrestrict[@type = 'accessPolicy']/EAD:p";
	protected static final String XPathAccessRestrictionClassification = "EAD:accessrestrict[@type = 'accessRestrictionsClassification']/EAD:p";
	protected static final String XPathAccessRestrictionExplanation = "EAD:accessrestrict[@type = 'accessRestrictionsExplanation']/EAD:p";
	protected static final String XPathAccessRestrictionPeriod = "EAD:accessrestrict[@type = 'accessRestrictionsPeriod']/EAD:p";
	protected static final String XPathAccessRestrictionPeriodBaseYear = "EAD:accessrestrict[@type = 'accessRestrictionsPeriodBaseYear']/EAD:p";
	protected static final String XPathAccessRestrictionPrivacy = "EAD:accessrestrict[@type = 'accessRestrictionsPrivacy']/EAD:p";
	protected static final String XPathAccessRestrictionStatus = "EAD:accessrestrict[@type = 'accessRestrictionsStatus']/EAD:p";
	protected static final String XPathAccessRestrictionStatusExplanation = "EAD:accessrestrict[@type = 'accessRestrictionsStatusExplanation']/EAD:p";
	protected static final String XPathAccessRestrictionIsPublic = "EAD:accessrestrict[@type = 'accessRestrictionsIsPublic']/EAD:p";
	protected static final String XPathAccessRestrictionIsMetadataPublic = "EAD:accessrestrict[@type = 'accessRestrictionsIsMetadataPublic']/EAD:p";
	protected static final String XPathAccessRestrictionClosureYear = "EAD:accessrestrict[@type = 'accessRestrictionsClosureYear']/EAD:p";
	protected static final String XPathRetentionPeriodBaseYear = "EAD:accessrestrict[@type = 'baseYear']/EAD:p";
	protected static final String XPathAccessRestriction = "EAD:accessrestrict[@type = 'restrictions']/EAD:p";
	protected static final String XPathRetentionPeriod = "EAD:accessrestrict[@type = 'retentionPeriod']/EAD:p";
	protected static final String XPathRetentionPolicy = "EAD:accessrestrict[@type = 'retentionPolicy']/EAD:p";
	protected static final String XPathUsagePermission = "EAD:accessrestrict[@type = 'usagePermission']/EAD:p";
	protected static final String XPathUsagePermissionExpiringDate = "EAD:accessrestrict[@type = 'usagePermissionExpiringDate']/EAD:p";

	protected static final String XPathAccruals = "EAD:accruals/EAD:p";
	protected static final String XPathAccessNr = "EAD:acqinfo/EAD:p";
	protected static final String XPathReproductions = "EAD:altformavail/EAD:p";
	protected static final String XPathAppraisalAndDestruction = "EAD:appraisal/EAD:p";
	protected static final String XPathAppraisalIsOnHold = "EAD:appraisal/EAD:note[@type = 'isOnHold']/EAD:p";
	protected static final String XPathAppraisalDateDisposed = "EAD:appraisal/EAD:note[@type = 'dateDisposed']/EAD:p";
	protected static final String XPathAppraisalRetentionPeriod = "EAD:appraisal/EAD:note[@type ='retentionPeriod']/EAD:p";
	protected static final String XPathArrangement = "EAD:arrangement/EAD:p";
	protected static final String XPathBibliography = "EAD:bibliography/EAD:p";
	protected static final String XPathDeathOfAuthor = "EAD:bioghist/EAD:note/EAD:p/EAD:date[@type = 'deathAuthor']";
	protected static final String XPathBiographicalHistory = "EAD:bioghist/EAD:p";
	protected static final String XPathLastname = "EAD:bioghist/EAD:note[@type = 'lastName']/EAD:p";
	protected static final String XPathFirstname = "EAD:bioghist/EAD:note[@type = 'firstName']/EAD:p";
	protected static final String XPathDateOfBirth = "EAD:bioghist/EAD:note[@type = 'dateBirth']/EAD:p";
	protected static final String XPathNationality = "EAD:bioghist/EAD:note[@type = 'nationality']/EAD:p";
	protected static final String XPathSex = "EAD:bioghist/EAD:note[@type = 'sex']/EAD:p";

	protected static final String XPathInstitution = "EAD:controlaccess/EAD:corpname[not(@role)]";
	protected static final String XPATH_INSTITUTION_CREATOR = "EAD:controlaccess/EAD:corpname[@role = 'creator']";
	protected static final String XPATH_INSTITUTION_INVOLVED = "EAD:controlaccess/EAD:corpname[@role = 'involved']";
	protected static final String XPathObjectType = "EAD:controlaccess/EAD:genreform";
	protected static final String XPATH_GEOG_NAME = "EAD:controlaccess/EAD:geogname[@role = 'general']";
	protected static final String XPathCompartment = "EAD:controlaccess/EAD:name[@role = 'compartment']";
	protected static final String XPathFundingSource = "EAD:controlaccess/EAD:name[@role = 'fundingSource']";
	protected static final String XPathInstitute = "EAD:controlaccess/EAD:name[@role = 'institute']";
	protected static final String XPathJournal = "EAD:controlaccess/EAD:name[@role = 'journal']";
	protected static final String XPathKeyword = "EAD:controlaccess/EAD:name[@role = 'keyword']";
	protected static final String XPathLocation = "EAD:controlaccess/EAD:name[@role = 'location']";
	protected static final String XPathMethod = "EAD:controlaccess/EAD:name[@role = 'method']";
	protected static final String XPATH_PLACE_OF_PUBLICATION = "EAD:controlaccess/EAD:name[@role = 'placeOfPublication']";
	protected static final String XPathPublisher = "EAD:controlaccess/EAD:name[@role = 'publisher']";
	protected static final String XPathStatus = "EAD:controlaccess/EAD:name[@role = 'status']";
	protected static final String XPathSubmitStatus = "EAD:controlaccess/EAD:name[@role = 'submitStatus']";
	protected static final String XPathUniversity = "EAD:controlaccess/EAD:name[@role = 'university']";
	protected static final String XPathAuthor = "EAD:controlaccess/EAD:persname[@role = 'author']";
	protected static final String XPATH_CREATOR = "EAD:controlaccess/EAD:persname[@role = 'creator']";
	protected static final String XPathInvolved = "EAD:controlaccess/EAD:persname[@role = 'involved']";
	protected static final String XPathResponsible = "EAD:controlaccess/EAD:persname[@role = 'responsible']";
	protected static final String XPathStaff = "EAD:controlaccess/EAD:persname[@role = 'staff']";
	protected static final String XPathSubject = "EAD:controlaccess/EAD:subject[not(@rules) and not(@role)]";
	protected static final String XPATH_SUBJECT_GENERAL = "EAD:controlaccess/EAD:subject[@rules = 'general' or @role='general']";
	protected static final String XPATH_UNIFORM_TITLE = "EAD:controlaccess/EAD:title[@type = 'uniform']";

	protected static final String XPathModeOfAcquisition = "EAD:custodhist/EAD:acqinfo/EAD:p";
	protected static final String XPathSourceType = "EAD:custodhist/EAD:note/EAD:p";
	protected static final String XPathArchivalHistory = "EAD:custodhist/EAD:p";

	protected static final String XPathComment = "EAD:note/EAD:p";

	protected static final String XPathDOIJournal = "EAD:odd[@type = 'doiJournal']/EAD:p";
	protected static final String XPATH_EDITION_STATEMENT = "EAD:odd[@type = 'edition']/EAD:p";
	protected static final String XPathProject = "EAD:odd[@type = 'project']/EAD:p";
	protected static final String XPathProjectAbbreviation = "EAD:odd[@type = 'projectAbbreviation']/EAD:p";
	protected static final String XPathProjectName = "EAD:odd[@type = 'projectName']/EAD:p";
	protected static final String XPathProjectTitle = "EAD:odd[@type = 'projectTitle']/EAD:p";
	protected static final String XPathUsage = "EAD:odd[@type = 'usage']/EAD:p";
	protected static final String XPathEvent = "EAD:odd[@type = 'event']/EAD:p";

	protected static final String XPathLocationOfOriginals = "EAD:originalsloc/EAD:p";
	protected static final String XPathFindingAids = "EAD:otherfindaid/EAD:p";
	protected static final String XPathCharacteristicsNote = "EAD:phystech/EAD:note/EAD:p";
	protected static final String XPathCharacteristics = "EAD:phystech/EAD:p";
	protected static final String XPathDigitization = "EAD:processinfo[@type = 'digitization']/EAD:p";
	protected static final String XPathDescriptionLevel = "EAD:processinfo[@type = 'level']/EAD:p";
	protected static final String XPathDescriptionLevelNotes = "EAD:processinfo[@type = 'levelNotes']/EAD:p";
	protected static final String XPathRevisions = "EAD:processinfo[@type = 'revisions']/EAD:p";
	protected static final String XPathDescriptionRules = "EAD:processinfo[@type = 'rules']/EAD:p";
	protected static final String XPathProcessInfoArchivist = "EAD:processinfo[@type = 'archivist']/EAD:p";
	protected static final String XPathProcessInfoDate= "EAD:processinfo[@type = 'date']/EAD:p";
	protected static final String XPathRelatedMaterial = "EAD:relatedmaterial/EAD:p";
	protected static final String XPathScopeContent = "EAD:scopecontent/EAD:p";
	protected static final String XPathConditionsOfReproductions = "EAD:userestrict/EAD:p";

	protected int depth = 0;

	protected LevelOfDescription otherLevel = LevelOfDescription.getUndefined();

	protected DID did;
	protected List<C> cs = new ArrayList<C>();

	/**
	 * pid is an optional value, containing the PIDs of this C or ArchDesc
	 * element.
	 */
	protected List<String> pid;

	/**
	 * submitStatus is an optional value, containing a SubmitStatus. Some
	 * submitStatuses disallow editing.
	 */
	protected SubmitStatus submitStatus;

	protected List<String> author;
	protected List<String> comment;

	protected List<String> accessNr;
	protected List<String> scopeContent;
	protected List<String> accessPolicy;
	protected List<String> accessRestriction;
	protected List<String> accessRestrictionExplanation;
	protected List<String> accessRestrictionPeriod;
	protected List<String> accessRestrictionPeriodBaseYear;
	protected List<String> accessRestrictionClassification;
	protected List<String> accessRestrictionPrivacy;
	protected List<String> accessRestrictionStatus;
	protected List<String> accessRestrictionStatusExplanation;
	protected List<String> accessRestrictionIsPublic;
	protected List<String> accessRestrictionIsMetadataPublic;
	protected List<String> accessRestrictionClosureYear;
	protected List<String> usagePermission;
	protected List<String> usagePermissionExpiringDate;
	protected List<String> retentionPolicy;
	protected List<String> retentionPeriodBaseYear;
	protected List<String> archivalHistory;
	protected List<String> sourceType;
	protected List<String> modeOfAcquisition;
	protected List<String> objectType;
	protected List<String> conditionsOfReproductions;
	protected List<String> characteristics;
	protected List<String> findingAids;
	protected List<String> locationOfOriginals;
	protected List<String> reproductions;
	protected List<String> relatedMaterial;
	protected List<String> institution;
	protected List<String> appraisalAndDestruction;
	protected List<String> appraisalIsOnHold;
	protected List<String> appraisalDateDisposed;
	protected List<String> appraisalRetentionPeriod;
	protected List<String> arrangement;
	protected List<String> bibliography;
	protected List<String> descriptionLevel;
	protected List<String> descriptionLevelNotes;
	protected List<String> descriptionRules;
	protected List<String> processInfoArchivist;
	protected List<String> processInfoDate;
	protected List<String> revisions;
	protected List<String> digitization;

	protected List<String> biographicalHistory;
	protected List<String> firstname;
	protected List<String> lastname;
	protected List<String> dateOfBirth;
	protected List<String> nationality;
	protected List<String> sex;
	protected List<String> deathOfAuthor;
	protected List<String> accruals;
	protected List<String> characteristicsNote;
	protected List<String> subject;

	protected List<String> responsible;
	protected List<String> involved;
	protected List<String> staff;
	protected List<String> projectName;
	protected List<String> projectTitle;
	protected List<String> project;
	protected List<String> projectAbbreviation;
	protected List<String> location;
	protected List<String> journal;
	protected List<String> university;
	protected List<String> institute;
	protected List<String> compartment;
	protected List<String> method;
	protected List<String> keyword;
	protected List<String> fundingSource;
	protected List<String> status;
	protected List<String> publisher;
	protected List<String> doiJournal;
	protected List<String> retentionPeriod;
	protected List<String> usage;
	protected List<String> event;

	protected List<String> creator;
	protected List<String> editionStatement;
	protected List<String> geogName;
	protected List<String> institutionCreator;
	protected List<String> institutionInvolved;
	protected List<String> placeOfPublication;
	protected List<String> subjectGeneral;
	protected List<String> uniformTitle;

	/**
	 * This constructor is used when an EAD-File or METS-file is being read.
	 */
	protected ComponentAbstract(NodeAbstract parent, Element element, int depth) {
		this(parent.getDocument(), element, depth);
		this.parent = parent;
	}

	/**
	 * This constructor is used when an EAD-File or METS-file is being read.
	 * NOTE: If I am an ArchDesc (without a parent node - my parent is a
	 * Document), this constructor is called directly, NOT via the constructor
	 * ComponentAbstract(NodeAbstract parent, Element element, int depth)!
	 */
	protected ComponentAbstract(DocumentAbstract document, Element element, int depth) {
		this.document = document;
		this.element = element;

		this.depth = depth;

		this.otherLevel = this.document.getLevels().get(element.attributeValue("otherlevel"));

		List<String> pids = new Vector<String>(3);
		for (Object e : element.selectNodes(XPathPID)) {
			pids.add(((Element) e).attributeValue("href"));
		}
		if (!pids.isEmpty()) {
			this.pid = pids;
		}

		this.submitStatus = SubmitStatus.named(this.getElementText(XPathSubmitStatus, false));

		author = getDynamicElementTexts(XPathAuthor);
		comment = getDynamicElementTexts(XPathComment);

		accessNr = getDynamicElementTexts(XPathAccessNr);
		scopeContent = getDynamicElementTexts(XPathScopeContent);
		accessPolicy = getDynamicElementTexts(XPathAccessPolicy);
		accessRestriction = getDynamicElementTexts(XPathAccessRestriction);
		accessRestrictionExplanation = getDynamicElementTexts(XPathAccessRestrictionExplanation);
		accessRestrictionPeriod = getDynamicElementTexts(XPathAccessRestrictionPeriod);
		accessRestrictionPeriodBaseYear = getDynamicElementTexts(XPathAccessRestrictionPeriodBaseYear);
		accessRestrictionClassification = getDynamicElementTexts(XPathAccessRestrictionClassification);
		accessRestrictionPrivacy = getDynamicElementTexts(XPathAccessRestrictionPrivacy);
		accessRestrictionStatus = getDynamicElementTexts(XPathAccessRestrictionStatus);
		accessRestrictionStatusExplanation = getDynamicElementTexts(XPathAccessRestrictionStatusExplanation);
		accessRestrictionIsPublic = getDynamicElementTexts(XPathAccessRestrictionIsPublic);
		accessRestrictionIsMetadataPublic = getDynamicElementTexts(XPathAccessRestrictionIsMetadataPublic);
		accessRestrictionClosureYear = getDynamicElementTexts(XPathAccessRestrictionClosureYear);
		usagePermission = getDynamicElementTexts(XPathUsagePermission);
		usagePermissionExpiringDate = getDynamicElementTexts(XPathUsagePermissionExpiringDate);
		retentionPolicy = getDynamicElementTexts(XPathRetentionPolicy);
		retentionPeriodBaseYear = getDynamicElementTexts(XPathRetentionPeriodBaseYear);
		archivalHistory = getDynamicElementTexts(XPathArchivalHistory);
		sourceType = getDynamicElementTexts(XPathSourceType);
		modeOfAcquisition = getDynamicElementTexts(XPathModeOfAcquisition);
		objectType = getDynamicElementTexts(XPathObjectType);
		conditionsOfReproductions = getDynamicElementTexts(XPathConditionsOfReproductions);
		characteristics = getDynamicElementTexts(XPathCharacteristics);
		findingAids = getDynamicElementTexts(XPathFindingAids);
		locationOfOriginals = getDynamicElementTexts(XPathLocationOfOriginals);
		reproductions = getDynamicElementTexts(XPathReproductions);
		relatedMaterial = getDynamicElementTexts(XPathRelatedMaterial);
		institution = getDynamicElementTexts(XPathInstitution);
		appraisalAndDestruction = getDynamicElementTexts(XPathAppraisalAndDestruction);
		appraisalIsOnHold= getDynamicElementTexts(XPathAppraisalIsOnHold);
		appraisalDateDisposed = getDynamicElementTexts(XPathAppraisalDateDisposed);
		appraisalRetentionPeriod = getDynamicElementTexts(XPathAppraisalRetentionPeriod);
		arrangement = getDynamicElementTexts(XPathArrangement);
		bibliography = getDynamicElementTexts(XPathBibliography);
		descriptionLevel = getDynamicElementTexts(XPathDescriptionLevel);
		descriptionLevelNotes = getDynamicElementTexts(XPathDescriptionLevelNotes);
		descriptionRules = getDynamicElementTexts(XPathDescriptionRules);
		processInfoArchivist = getDynamicElementTexts(XPathProcessInfoArchivist);
		processInfoDate = getDynamicElementTexts(XPathProcessInfoDate);
		revisions = getDynamicElementTexts(XPathRevisions);
		digitization = getDynamicElementTexts(XPathDigitization);

		biographicalHistory = getDynamicElementTexts(XPathBiographicalHistory);
		firstname = getDynamicElementTexts(XPathFirstname);
		lastname = getDynamicElementTexts(XPathLastname);
		dateOfBirth = getDynamicElementTexts(XPathDateOfBirth);
		nationality = getDynamicElementTexts(XPathNationality);
		sex = getDynamicElementTexts(XPathSex);
		deathOfAuthor = getDynamicElementTexts(XPathDeathOfAuthor);
		accruals = getDynamicElementTexts(XPathAccruals);
		characteristicsNote = getDynamicElementTexts(XPathCharacteristicsNote);
		subject = getDynamicElementTexts(XPathSubject);

		responsible = getDynamicElementTexts(XPathResponsible);
		involved = getDynamicElementTexts(XPathInvolved);
		staff = getDynamicElementTexts(XPathStaff);
		projectName = getDynamicElementTexts(XPathProjectName);
		projectTitle = getDynamicElementTexts(XPathProjectTitle);
		project = getDynamicElementTexts(XPathProject);
		projectAbbreviation = getDynamicElementTexts(XPathProjectAbbreviation);
		location = getDynamicElementTexts(XPathLocation);
		journal = getDynamicElementTexts(XPathJournal);
		university = getDynamicElementTexts(XPathUniversity);
		institute = getDynamicElementTexts(XPathInstitute);
		compartment = getDynamicElementTexts(XPathCompartment);
		method = getDynamicElementTexts(XPathMethod);
		keyword = getDynamicElementTexts(XPathKeyword);
		fundingSource = getDynamicElementTexts(XPathFundingSource);
		status = getDynamicElementTexts(XPathStatus);
		publisher = getDynamicElementTexts(XPathPublisher);
		doiJournal = getDynamicElementTexts(XPathDOIJournal);
		retentionPeriod = getDynamicElementTexts(XPathRetentionPeriod);
		usage = getDynamicElementTexts(XPathUsage);
		event = getDynamicElementTexts(XPathEvent);

		creator = getDynamicElementTexts(XPATH_CREATOR);
		editionStatement = getDynamicElementTexts(XPATH_EDITION_STATEMENT);
		geogName = getDynamicElementTexts(XPATH_GEOG_NAME);
		institutionCreator = getDynamicElementTexts(XPATH_INSTITUTION_CREATOR);
		institutionInvolved = getDynamicElementTexts(XPATH_INSTITUTION_INVOLVED);
		placeOfPublication = getDynamicElementTexts(XPATH_PLACE_OF_PUBLICATION);
		subjectGeneral = getDynamicElementTexts(XPATH_SUBJECT_GENERAL);
		uniformTitle = getDynamicElementTexts(XPATH_UNIFORM_TITLE);

		cs = C.parse(this, this.element, depth + 1);
		did = DID.parse(this, this.element);
	}

	/**
	 * This constructor is used when a new ComponentAbstract is created
	 * programmatically within an EAD Document:
	 */
	protected ComponentAbstract(ComponentAbstract parent, Element element, String title) {
		document = parent.getDocument();
		this.element = element;
		// 'level' is a mandatory attribute so I have to set it
		this.element.addAttribute("level", Level);

		depth = parent.depth + 1;
		did = new DID(this, title);

		document.setIsModified();
	}

	/**
	 * This constructor is used when a new ComponentAbstract is created
	 * programmatically within a METS Document:
	 */
	protected ComponentAbstract(DMDSectionWithEAD parent, Element element, String title) {
		document = parent.getDocument();
		this.element = element;
		// 'level' is a mandatory attribute so I have to set it
		this.element.addAttribute("level", Level).addAttribute("otherlevel", this.otherLevel.getName());

		depth = 0;
		did = new DID(this, title);

		document.setIsModified();
	}

	/**
	 * This is done on EAD export.
	 */
	protected void copyEADPropertiesFrom(ComponentAbstract source) {
		// Copy all EAD properties from the source component to me:
		setOtherLevel(source.getOtherLevel());
		setPID(source.getPID());
		setSubmitStatus(source.getSubmitStatus());

		setAuthor(source.getAuthor());
		setComment(source.getComment());

		setAccessNr(source.getAccessNr());
		setScopeContent(source.getScopeContent());
		setAccessPolicy(source.getAccessPolicy());
		setAccessRestriction(source.getAccessRestriction());
		setAccessRestrictionExplanation(source.getAccessRestrictionExplanation());
		setAccessRestrictionPeriod(source.getAccessRestrictionPeriod());
		setAccessRestrictionPeriodBaseYear(source.getAccessRestrictionPeriodBaseYear());
		setAccessRestrictionClassification(source.getAccessRestrictionClassification());
		setAccessRestrictionPrivacy(source.getAccessRestrictionPrivacy());
		setAccessRestrictionStatus(source.getAccessRestrictionStatus());
		setAccessRestrictionStatusExplanation(source.getAccessRestrictionStatusExplanation());
		setAccessRestrictionIsPublic(source.getAccessRestrictionIsPublic());
		setAccessRestrictionIsMetadataPublic(source.getAccessRestrictionIsMetadataPublic());
		setAccessRestrictionClosureYear(source.getAccessRestrictionClosureYear());
		setUsagePermission(source.getUsagePermission());
		setUsagePermissionExpiringDate(source.getUsagePermissionExpiringDate());
		setRetentionPolicy(source.getRetentionPolicy());
		setRetentionPeriodBaseYear(source.getRetentionPeriodBaseYear());
		setArchivalHistory(source.getArchivalHistory());
		setSourceType(source.getSourceType());
		setModeOfAcquisition(source.getModeOfAcquisition());
		setObjectType(source.getObjectType());
		setConditionsOfReproductions(source.getConditionsOfReproductions());
		setCharacteristics(source.getCharacteristics());
		setFindingAids(source.getFindingAids());
		setLocationOfOriginals(source.getLocationOfOriginals());
		setReproductions(source.getReproductions());
		setRelatedMaterial(source.getRelatedMaterial());
		setInstitution(source.getInstitution());
		setAppraisalAndDestruction(source.getAppraisalAndDestruction());
		setArrangement(source.getArrangement());
		setBibliography(source.getBibliography());
		setDescriptionLevel(source.getDescriptionLevel());
		setDescriptionLevelNotes(source.getDescriptionLevelNotes());
		setDescriptionRules(source.getDescriptionRules());
		setRevisions(source.getRevisions());
		setDigitization(source.getDigitization());

		setResponsible(source.getResponsible());
		setInvolved(source.getInvolved());
		setStaff(source.getStaff());
		setProjectName(source.getProjectName());
		setProjectTitle(source.getProjectTitle());
		setProject(source.getProject());
		setProjectAbbreviation(source.getProjectAbbreviation());
		setLocation(source.getLocation());
		setJournal(source.getJournal());
		setUniversity(source.getUniversity());
		setInstitute(source.getInstitute());
		setCompartment(source.getCompartment());
		setMethod(source.getMethod());
		setKeyword(source.getKeyword());
		setFundingSource(source.getFundingSource());
		setStatus(source.getStatus());
		setPublisher(source.getPublisher());
		setDoiJournal(source.getDoiJournal());
		setRetentionPeriod(source.getRetentionPeriod());
		setUsage(source.getUsage());
		setEvent(source.getEvent());

		setBiographicalHistory(source.getBiographicalHistory());
		setFirstname(source.getFirstname());
		setLastname(source.getLastname());
		setDateOfBirth(source.getDateOfBirth());
		setNationality(source.getNationality());
		setSex(source.getSex());
		setDeathOfAuthor(source.getDeathOfAuthor());
		setAccruals(source.getAccruals());
		setCharacteristicsNote(source.getCharacteristicsNote());
		setSubject(source.getSubject());

		// Data stored in my did element:
		setUnitTitle(source.getUnitTitle());
		setUnitTitleAdditional(source.getUnitTitleAdditional());
		setOrigination(source.getOrigination());
		setOriginationAgency(source.getOriginationAgency());
		setOriginationDepartment(source.getOriginationDepartment());
		setCreationPeriod(source.getCreationPeriod());
		setCreationPeriodNotes(source.getCreationPeriodNotes());
		setFrom(source.getFrom());
		setFromYear(source.getFromYear());
		setTo(source.getTo());
		setToYear(source.getToYear());
		setRelationPeriod(source.getRelationPeriod());
		setLanguage(source.getLanguage());
		setLanguageNotes(source.getLanguageNotes());
		setRefCode(source.getRefCode());
		setRefCodeOld(source.getRefCodeOld());
		setRefCodeAdmin(source.getRefCodeAdmin());
		setRefCodeIsVisible(source.getRefCodeIsVisible());
		setRefCodeSeparator(source.getRefCodeSeparator());
		setAccessionNumber(source.getAccessionNumber());
		setExtent(source.getExtent());
		setExtentUnit(source.getExtentUnit());
		setExtentNote(source.getExtentNote());
		setMaterial(source.getMaterial());

		setYear(source.getYear());
		setAbstract(source.getAbstract());

		setCartographicMaterial(source.getCartographicMaterial());
		setCodeLocation(source.getCodeLocation());
		setCreator(source.getCreator());
		setDimensions(source.getDimensions());
		setDOI(source.getDOI());
		setEditionStatement(source.getEditionStatement());
		setFullLocation(source.getFullLocation());
		setGeogName(source.getGeogName());
		setInstitutionCreator(source.getInstitutionCreator());
		setInstitutionInvolved(source.getInstitutionInvolved());
		setISBN(source.getISBN());
		setISSN(source.getISSN());
		setPlaceOfPublication(source.getPlaceOfPublication());
		setSubjectGeneral(source.getSubjectGeneral());
		setUniformTitle(source.getUniformTitle());
		setUnitTitleVarying(source.getUnitTitleVarying());
		setFormat(source.getFormat());
		setSize(source.getSize());
		setDate(source.getDate());
		setUnitTitleOriginal(source.getUnitTitleOriginal());
	}

	public DID getDid() {
		return did;
	}

	public List<C> getCs() {
		return cs;
	}

	public C getC(int i) {
		return cs.get(i);
	}

	public String getOtherLevelName() {
		return otherLevel.getName();
	}

	public void setOtherLevelName(String newLevel) {
		setOtherLevel(document.getLevels().get(newLevel));
	}

	public LevelOfDescription getOtherLevel() {
		return otherLevel;
	}

	public void setOtherLevel(LevelOfDescription level) {
		this.otherLevel = level;
		// .addAttribute("level", Level) This attribute is already set because
		// it is mandatory
		element.addAttribute("otherlevel", otherLevel.getName());

		document.setIsModified();
	}

	public String getUnitTitle() {
		return did.getUnitTitle();
	}

	public void setUnitTitle(String unitTitle) {
		did.setUnitTitle(unitTitle);
	}

	/**
	 * Return the submit status. If it is null, return
	 * SubmitStatus.SubmitUndefined.
	 * 
	 * @return
	 */
	public SubmitStatus getSubmitStatus() {
		if (submitStatus == null)
			return SubmitStatus.SubmitUndefined;

		return submitStatus;
	}

	public void setSubmitStatus(SubmitStatus submitStatus) {
		if (submitStatus == null) {
			submitStatus = SubmitStatus.SubmitUndefined;
		}

		this.submitStatus = submitStatus;

		Node submitStatusNode = element.selectSingleNode(XPathSubmitStatus);
		if (submitStatusNode == null) {
			Element controlAccessNode = (Element) element.selectSingleNode("EAD:controlaccess");
			if (controlAccessNode == null)
				controlAccessNode = element.addElement("EAD:controlaccess");

			submitStatusNode = controlAccessNode.addElement("EAD:name").addAttribute("role", "submitStatus");
		}
		submitStatusNode.setText(this.submitStatus.name());

		document.setIsModified();
	}

	public List<String> getPID() {
		return pid;
	}

	/**
	 * Dynamic Metadata: special way to set the values - don't use the standard
	 * way with setDynamicElementTexts(value, XPath);
	 */
	public void setPID(List<String> values) {
		pid = values;

		// First delete all elements (if any)...:
		for (Object node : element.selectNodes(XPathPID))
			((Node) node).detach();

		// ... then set them (if not null):
		if (values != null) {
			for (String pid : values) {
				element.addElement("EAD:dao").addAttribute(new QName("role", new Namespace("xlink", Xmlns_xlink)), Role)
						.addAttribute(new QName("href", new Namespace("xlink", Xmlns_xlink)), pid);
			}
		}

		document.setIsModified();
	}

	public List<String> getUnitTitleAdditional() {
		return did.getUnitTitleAdditional();
	}

	public void setUnitTitleAdditional(List<String> s) {
		did.setUnitTitleAdditional(s);
	}

	public List<String> getAuthor() {
		return author;
	}

	public void setAuthor(List<String> s) {
		this.author = s;

		setDynamicElementTexts(this.author, XPathAuthor);
		if (this.author == null) {
			deleteParentElementIfEmpty(XPathAuthor);
		}

		document.setIsModified();
	}

	public List<String> getComment() {
		return this.comment;
	}

	public void setComment(List<String> values) {
		this.comment = values;

		setDynamicElementTexts(this.comment, XPathComment);
		if (this.comment == null)
			deleteParentElementIfEmpty(XPathComment);

		document.setIsModified();
	}

	public List<String> getAccessNr() {
		return this.accessNr;
	}

	public void setAccessNr(List<String> values) {
		this.accessNr = values;

		setDynamicElementTexts(this.accessNr, XPathAccessNr);
		if (this.accessNr == null) {
			deleteParentElementIfEmpty(XPathAccessNr);
		}

		document.setIsModified();
	}

	public List<String> getOrigination() {
		return did.getOrigination();
	}

	public void setOrigination(List<String> s) {
		did.setOrigination(s);
	}

	public List<String> getOriginationAgency() {
		return did.getOriginationAgency();
	}

	public void setOriginationAgency(List<String> s) {
		did.setOriginationAgency(s);
	}

	public List<String> getOriginationDepartment() {
		return did.getOriginationDepartment();
	}

	public void setOriginationDepartment(List<String> s) {
		did.setOriginationDepartment(s);
	}

	public List<String> getCreationPeriod() {
		return did.getCreationPeriod();
	}

	public void setCreationPeriod(List<String> s) {
		did.setCreationPeriod(s);
	}

	public List<String> getCreationPeriodNotes() {
		return did.getCreationPeriodNotes();
	}

	public void setCreationPeriodNotes(List<String> s) {
		did.setCreationPeriodNotes(s);
	}

	public List<String> getFromYear() {
		return did.getFromYear();
	}

	public void setFromYear(List<String> s) {
		did.setFromYear(s);
	}

	public List<String> getToYear() {
		return did.getToYear();
	}

	public void setToYear(List<String> s) {
		did.setToYear(s);
	}

	public List<String> getFrom() {
		return did.getFrom();
	}

	public void setFrom(List<String> from) {
		did.setFrom(from);
	}

	public List<String> getTo() {
		return did.getTo();
	}

	public void setTo(List<String> to) {
		did.setTo(to);
	}

	public List<String> getRelationPeriod() {
		return did.getRelationPeriod();
	}

	public void setRelationPeriod(List<String> s) {
		did.setRelationPeriod(s);
	}

	public List<String> getScopeContent() {
		return this.scopeContent;
	}

	public void setScopeContent(List<String> values) {
		this.scopeContent = values;

		setDynamicElementTexts(this.scopeContent, XPathScopeContent);
		if (this.scopeContent == null) {
			deleteParentElementIfEmpty(XPathScopeContent);
		}

		document.setIsModified();
	}

	public List<String> getLanguage() {
		return did.getLanguage();
	}

	public void setLanguage(List<String> s) {
		did.setLanguage(s);
	}

	public List<String> getLanguageNotes() {
		return did.getLanguageNotes();
	}

	public void setLanguageNotes(List<String> s) {
		did.setLanguageNotes(s);
	}

	public List<String> getAccessPolicy() {
		return accessPolicy;
	}

	public void setAccessPolicy(List<String> values) {
		this.accessPolicy = values;

		setDynamicElementTexts(this.accessPolicy, XPathAccessPolicy);
		if (this.accessPolicy == null) {
			deleteParentElementIfEmpty(XPathAccessPolicy);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestriction() {
		return accessRestriction;
	}

	public void setAccessRestriction(List<String> values) {
		this.accessRestriction = values;

		setDynamicElementTexts(this.accessRestriction, XPathAccessRestriction);
		if (this.accessRestriction == null) {
			deleteParentElementIfEmpty(XPathAccessRestriction);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionExplanation() {
		return accessRestrictionExplanation;
	}

	public void setAccessRestrictionExplanation(List<String> values) {
		this.accessRestrictionExplanation = values;

		setDynamicElementTexts(this.accessRestrictionExplanation, XPathAccessRestrictionExplanation);
		if (this.accessRestrictionExplanation == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionExplanation);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionPeriod() {
		return accessRestrictionPeriod;
	}

	public void setAccessRestrictionPeriod(List<String> values) {
		this.accessRestrictionPeriod = values;

		setDynamicElementTexts(this.accessRestrictionPeriod, XPathAccessRestrictionPeriod);
		if (this.accessRestrictionPeriod == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionPeriod);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionPeriodBaseYear() {
		return accessRestrictionPeriodBaseYear;
	}

	public void setAccessRestrictionPeriodBaseYear(List<String> values) {
		this.accessRestrictionPeriodBaseYear = values;

		setDynamicElementTexts(this.accessRestrictionPeriodBaseYear, XPathAccessRestrictionPeriodBaseYear);
		if (this.accessRestrictionPeriodBaseYear == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionPeriodBaseYear);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionClassification() {
		return accessRestrictionClassification;
	}

	public void setAccessRestrictionClassification(List<String> values) {
		this.accessRestrictionClassification = values;

		setDynamicElementTexts(this.accessRestrictionClassification, XPathAccessRestrictionClassification);
		if (this.accessRestrictionClassification == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionClassification);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionPrivacy() {
		return accessRestrictionPrivacy;
	}

	public void setAccessRestrictionPrivacy(List<String> values) {
		this.accessRestrictionPrivacy = values;

		setDynamicElementTexts(this.accessRestrictionPrivacy, XPathAccessRestrictionPrivacy);
		if (this.accessRestrictionPrivacy == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionPrivacy);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionStatus() {
		return this.accessRestrictionStatus;
	}

	public void setAccessRestrictionStatus(List<String> values) {
		this.accessRestrictionStatus = values;

		setDynamicElementTexts(this.accessRestrictionStatus, XPathAccessRestrictionStatus);
		if (this.accessRestrictionStatus == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionStatus);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionStatusExplanation() {
		return this.accessRestrictionStatusExplanation;
	}

	public void setAccessRestrictionStatusExplanation(List<String> values) {
		this.accessRestrictionStatusExplanation = values;

		setDynamicElementTexts(this.accessRestrictionStatusExplanation, XPathAccessRestrictionStatusExplanation);
		if (this.accessRestrictionStatusExplanation == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionStatusExplanation);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionIsPublic() {
		return this.accessRestrictionIsPublic;
	}

	public void setAccessRestrictionIsPublic(List<String> values) {
		this.accessRestrictionIsPublic = values;

		setDynamicElementTexts(this.accessRestrictionIsPublic, XPathAccessRestrictionIsPublic);
		if (this.accessRestrictionIsPublic == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionIsPublic);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionIsMetadataPublic() {
		return this.accessRestrictionIsMetadataPublic;
	}

	public void setAccessRestrictionIsMetadataPublic(List<String> values) {
		this.accessRestrictionIsMetadataPublic = values;

		setDynamicElementTexts(this.accessRestrictionIsMetadataPublic, XPathAccessRestrictionIsMetadataPublic);
		if (this.accessRestrictionIsMetadataPublic == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionIsMetadataPublic);
		}

		document.setIsModified();
	}

	public List<String> getAccessRestrictionClosureYear() {
		return this.accessRestrictionClosureYear;
	}

	public void setAccessRestrictionClosureYear(List<String> values) {
		this.accessRestrictionClosureYear = values;

		setDynamicElementTexts(this.accessRestrictionClosureYear, XPathAccessRestrictionClosureYear);
		if (this.accessRestrictionClosureYear == null) {
			deleteParentElementIfEmpty(XPathAccessRestrictionClosureYear);
		}

		document.setIsModified();
	}

	public List<String> getUsagePermission() {
		return this.usagePermission;
	}

	public void setUsagePermission(List<String> values) {
		this.usagePermission = values;

		setDynamicElementTexts(this.usagePermission, XPathUsagePermission);
		if (this.usagePermission == null) {
			deleteParentElementIfEmpty(XPathUsagePermission);
		}

		document.setIsModified();
	}

	public List<String> getUsagePermissionExpiringDate() {
		return this.usagePermissionExpiringDate;
	}

	public void setUsagePermissionExpiringDate(List<String> values) {
		this.usagePermissionExpiringDate = values;

		setDynamicElementTexts(this.usagePermissionExpiringDate, XPathUsagePermissionExpiringDate);
		if (this.usagePermissionExpiringDate == null) {
			deleteParentElementIfEmpty(XPathUsagePermissionExpiringDate);
		}

		document.setIsModified();
	}

	public List<String> getRetentionPolicy() {
		return this.retentionPolicy;
	}

	public void setRetentionPolicy(List<String> retentionPolicy) {
		this.retentionPolicy = retentionPolicy;

		setDynamicElementTexts(this.retentionPolicy, XPathRetentionPolicy);
		if (this.retentionPolicy == null) {
			deleteParentElementIfEmpty(XPathRetentionPolicy);
		}

		document.setIsModified();
	}

	public List<String> getRetentionPeriodBaseYear() {
		return this.retentionPeriodBaseYear;
	}

	public void setRetentionPeriodBaseYear(List<String> retentionPeriodBaseYear) {
		this.retentionPeriodBaseYear = retentionPeriodBaseYear;

		setDynamicElementTexts(this.retentionPeriodBaseYear, XPathRetentionPeriodBaseYear);
		if (this.retentionPeriodBaseYear == null) {
			deleteParentElementIfEmpty(XPathRetentionPeriodBaseYear);
		}

		document.setIsModified();
	}

	public List<String> getArchivalHistory() {
		return archivalHistory;
	}

	public void setArchivalHistory(List<String> values) {
		this.archivalHistory = values;

		setDynamicElementTexts(this.archivalHistory, XPathArchivalHistory);
		if (this.archivalHistory == null) {
			deleteParentElementIfEmpty(XPathArchivalHistory);
		}

		document.setIsModified();
	}

	public List<String> getSourceType() {
		return this.sourceType;
	}

	public void setSourceType(List<String> values) {
		this.sourceType = values;

		setDynamicElementTexts(this.sourceType, XPathSourceType);
		if (this.sourceType == null) {
			deleteParentElementIfEmpty(XPathSourceType);
		}

		document.setIsModified();
	}

	public List<String> getModeOfAcquisition() {
		return this.modeOfAcquisition;
	}

	public void setModeOfAcquisition(List<String> values) {
		this.modeOfAcquisition = values;

		setDynamicElementTexts(this.modeOfAcquisition, XPathModeOfAcquisition);
		if (this.modeOfAcquisition == null) {
			deleteParentElementIfEmpty(XPathModeOfAcquisition);
		}

		document.setIsModified();
	}

	public List<String> getObjectType() {
		return this.objectType;
	}

	public void setObjectType(List<String> objectType) {
		this.objectType = objectType;

		setDynamicElementTexts(this.objectType, XPathObjectType);
		if (this.objectType == null) {
			deleteParentElementIfEmpty(XPathObjectType);
		}

		document.setIsModified();
	}

	public List<String> getRefCode() {
		return did.getRefCode();
	}

	public void setRefCode(List<String> refCode) {
		did.setRefCode(refCode);
	}

	public List<String> getRefCodeOld() {
		return did.getRefCodeOld();
	}

	public void setRefCodeOld(List<String> refCodeOld) {
		did.setRefCodeOld(refCodeOld);
	}

	public List<String> getRefCodeAdmin() {
		return did.getRefCodeAdmin();
	}

	public void setRefCodeAdmin(List<String> refCodeAdmin) {
		did.setRefCodeAdmin(refCodeAdmin);
	}

	public List<String> getRefCodeIsVisible() {
		return did.getRefCodeIsVisible();
	}

	public void setRefCodeIsVisible(List<String> refCodeIsVisible) {
		did.setRefCodeIsVisible(refCodeIsVisible);
	}

	public List<String> getRefCodeSeparator() {
		return did.getRefCodeSeparator();
	}

	public void setRefCodeSeparator(List<String> refCodeSeparator) {
		did.setRefCodeSeparator(refCodeSeparator);
	}

	public List<String> getAccessionNumber() {
		return did.getAccessionNumber();
	}

	public void setAccessionNumber(List<String> accessionNumber) {
		did.setAccessionNumber(accessionNumber);
	}

	public List<String> getExtent() {
		return did.getExtent();
	}

	public void setExtent(List<String> extent) {
		did.setExtent(extent);
	}

	public List<String> getExtentNote() {
		return did.getExtentNote();
	}

	public void setExtentNote(List<String> extentNote) {
		did.setExtentNote(extentNote);
	}

	public List<String> getExtentUnit() {
		return did.getExtentUnit();
	}

	public void setExtentUnit(List<String> extentUnit) {
		did.setExtentUnit(extentUnit);
	}

	public List<String> getMaterial() {
		return did.getMaterial();
	}

	public void setMaterial(List<String> material) {
		did.setMaterial(material);
	}

	public List<String> getConditionsOfReproductions() {
		return this.conditionsOfReproductions;
	}

	public void setConditionsOfReproductions(List<String> values) {
		this.conditionsOfReproductions = values;

		setDynamicElementTexts(this.conditionsOfReproductions, XPathConditionsOfReproductions);
		if (this.conditionsOfReproductions == null) {
			deleteParentElementIfEmpty(XPathConditionsOfReproductions);
		}

		document.setIsModified();
	}

	public List<String> getCharacteristics() {
		return this.characteristics;
	}

	public void setCharacteristics(List<String> values) {
		this.characteristics = values;

		setDynamicElementTexts(this.characteristics, XPathCharacteristics);
		if (this.characteristics == null) {
			deleteParentElementIfEmpty(XPathCharacteristics);
		}

		document.setIsModified();
	}

	public List<String> getFindingAids() {
		return this.findingAids;
	}

	public void setFindingAids(List<String> values) {
		this.findingAids = values;

		setDynamicElementTexts(this.findingAids, XPathFindingAids);
		if (this.findingAids == null) {
			deleteParentElementIfEmpty(XPathFindingAids);
		}

		document.setIsModified();
	}

	public List<String> getLocationOfOriginals() {
		return this.locationOfOriginals;
	}

	public void setLocationOfOriginals(List<String> values) {
		this.locationOfOriginals = values;

		setDynamicElementTexts(this.locationOfOriginals, XPathLocationOfOriginals);
		if (this.locationOfOriginals == null) {
			deleteParentElementIfEmpty(XPathLocationOfOriginals);
		}

		document.setIsModified();
	}

	public List<String> getReproductions() {
		return this.reproductions;
	}

	public void setReproductions(List<String> values) {
		this.reproductions = values;

		setDynamicElementTexts(this.reproductions, XPathReproductions);
		if (this.reproductions == null) {
			deleteParentElementIfEmpty(XPathReproductions);
		}

		document.setIsModified();
	}

	public List<String> getRelatedMaterial() {
		return this.relatedMaterial;
	}

	public void setRelatedMaterial(List<String> values) {
		this.relatedMaterial = values;

		setDynamicElementTexts(this.relatedMaterial, XPathRelatedMaterial);
		if (this.relatedMaterial == null) {
			deleteParentElementIfEmpty(XPathRelatedMaterial);
		}

		document.setIsModified();
	}

	public List<String> getInstitution() {
		return this.institution;
	}

	public void setInstitution(List<String> institution) {
		this.institution = institution;

		setDynamicElementTexts(this.institution, XPathInstitution);
		if (this.institution == null) {
			deleteParentElementIfEmpty(XPathInstitution);
		}

		document.setIsModified();
	}

	public List<String> getAppraisalAndDestruction() {
		return this.appraisalAndDestruction;
	}

	public void setAppraisalAndDestruction(List<String> values) {
		this.appraisalAndDestruction = values;

		setDynamicElementTexts(this.appraisalAndDestruction, XPathAppraisalAndDestruction);
		if (this.appraisalAndDestruction == null) {
			deleteParentElementIfEmpty(XPathAppraisalAndDestruction);
		}

		document.setIsModified();
	}

	public List<String> getAppraisalIsOnHold() {
		return this.appraisalIsOnHold;
	}

	public void setAppraisalIsOnHold(List<String> values) {
		this.appraisalIsOnHold = values;

		setDynamicElementTexts(this.appraisalIsOnHold, XPathAppraisalIsOnHold);
		if (this.appraisalIsOnHold == null) {
			deleteParentElementIfEmpty(XPathAppraisalIsOnHold);
		}

		document.setIsModified();
	}

	public List<String> getAppraisalDateDisposed() {
		return this.appraisalDateDisposed;
	}

	public void setAppraisalDateDisposed(List<String> values) {
		this.appraisalDateDisposed = values;

		setDynamicElementTexts(this.appraisalDateDisposed, XPathAppraisalDateDisposed);
		if (this.appraisalDateDisposed == null) {
			deleteParentElementIfEmpty(XPathAppraisalDateDisposed);
		}

		document.setIsModified();
	}

	public List<String> getAppraisalRetentionPeriod() {
		return this.appraisalRetentionPeriod;
	}

	public void setAppraisalRetentionPeriod(List<String> values) {
		this.appraisalRetentionPeriod = values;

		setDynamicElementTexts(this.appraisalRetentionPeriod, XPathAppraisalRetentionPeriod);
		if (this.appraisalRetentionPeriod == null) {
			deleteParentElementIfEmpty(XPathAppraisalRetentionPeriod);
		}

		document.setIsModified();
	}

	public List<String> getArrangement() {
		return this.arrangement;
	}

	public void setArrangement(List<String> values) {
		this.arrangement = values;

		setDynamicElementTexts(this.arrangement, XPathArrangement);
		if (this.arrangement == null) {
			deleteParentElementIfEmpty(XPathArrangement);
		}

		document.setIsModified();
	}

	public List<String> getBibliography() {
		return this.bibliography;
	}

	public void setBibliography(List<String> values) {
		this.bibliography = values;

		setDynamicElementTexts(this.bibliography, XPathBibliography);
		if (this.bibliography == null) {
			deleteParentElementIfEmpty(XPathBibliography);
		}

		document.setIsModified();
	}

	public List<String> getDescriptionLevel() {
		return this.descriptionLevel;
	}

	public void setDescriptionLevel(List<String> descriptionLevel) {
		this.descriptionLevel = descriptionLevel;

		setDynamicElementTexts(this.descriptionLevel, XPathDescriptionLevel);
		if (this.descriptionLevel == null) {
			deleteParentElementIfEmpty(XPathDescriptionLevel);
		}

		document.setIsModified();
	}

	public List<String> getDescriptionLevelNotes() {
		return this.descriptionLevelNotes;
	}

	public void setDescriptionLevelNotes(List<String> values) {
		this.descriptionLevelNotes = values;

		setDynamicElementTexts(this.descriptionLevelNotes, XPathDescriptionLevelNotes);
		if (this.descriptionLevelNotes == null) {
			deleteParentElementIfEmpty(XPathDescriptionLevelNotes);
		}

		document.setIsModified();
	}

	public List<String> getDescriptionRules() {
		return this.descriptionRules;
	}

	public void setDescriptionRules(List<String> values) {
		this.descriptionRules = values;

		setDynamicElementTexts(this.descriptionRules, XPathDescriptionRules);
		if (this.descriptionRules == null) {
			deleteParentElementIfEmpty(XPathDescriptionRules);
		}

		document.setIsModified();
	}

	public List<String> getProcessInfoArchivist() {
		return this.processInfoArchivist;
	}

	public void setProcessInfoArchivist(List<String> values) {
		this.processInfoArchivist = values;

		setDynamicElementTexts(this.processInfoArchivist, XPathProcessInfoArchivist);
		if (this.processInfoArchivist == null) {
			deleteParentElementIfEmpty(XPathProcessInfoArchivist);
		}

		document.setIsModified();
	}

	public List<String> getProcessInfoDate() {
		return this.processInfoDate;
	}

	public void setProcessInfoDate(List<String> values) {
		this.processInfoDate = values;

		setDynamicElementTexts(this.processInfoDate, XPathProcessInfoDate);
		if (this.processInfoDate == null) {
			deleteParentElementIfEmpty(XPathProcessInfoDate);
		}

		document.setIsModified();
	}

	public List<String> getRevisions() {
		return this.revisions;
	}

	public void setRevisions(List<String> revisions) {
		this.revisions = revisions;

		setDynamicElementTexts(this.revisions, XPathRevisions);
		if (this.revisions == null) {
			deleteParentElementIfEmpty(XPathRevisions);
		}

		document.setIsModified();
	}

	public List<String> getDigitization() {
		return this.digitization;
	}

	public void setDigitization(List<String> value) {
		this.digitization = value;

		setDynamicElementTexts(this.digitization, XPathDigitization);
		if (this.digitization == null) {
			deleteParentElementIfEmpty(XPathDigitization);
		}

		document.setIsModified();
	}

	// -------- ETH Forschungsdaten:

	public List<String> getResponsible() {
		return this.responsible;
	}

	public void setResponsible(List<String> responsible) {
		this.responsible = responsible;

		setDynamicElementTexts(this.responsible, XPathResponsible);
		if (this.responsible == null) {
			deleteParentElementIfEmpty(XPathResponsible);
		}

		document.setIsModified();
	}

	public List<String> getInvolved() {
		return this.involved;
	}

	public void setInvolved(List<String> involved) {
		this.involved = involved;

		setDynamicElementTexts(this.involved, XPathInvolved);
		if (this.involved == null) {
			deleteParentElementIfEmpty(XPathInvolved);
		}

		document.setIsModified();
	}

	public List<String> getStaff() {
		return this.staff;
	}

	public void setStaff(List<String> staff) {
		this.staff = staff;

		setDynamicElementTexts(this.staff, XPathStaff);
		if (this.staff == null) {
			deleteParentElementIfEmpty(XPathStaff);
		}

		document.setIsModified();
	}

	public List<String> getProjectName() {
		return this.projectName;
	}

	public void setProjectName(List<String> projectName) {
		this.projectName = projectName;

		setDynamicElementTexts(this.projectName, XPathProjectName);
		if (this.projectName == null) {
			deleteParentElementIfEmpty(XPathProjectName);
		}

		document.setIsModified();
	}

	public List<String> getProjectTitle() {
		return this.projectTitle;
	}

	public void setProjectTitle(List<String> projectTitle) {
		this.projectTitle = projectTitle;

		setDynamicElementTexts(this.projectTitle, XPathProjectTitle);
		if (this.projectTitle == null) {
			deleteParentElementIfEmpty(XPathProjectTitle);
		}

		document.setIsModified();
	}

	public List<String> getProject() {
		return this.project;
	}

	public void setProject(List<String> project) {
		this.project = project;

		setDynamicElementTexts(this.project, XPathProject);
		if (this.project == null) {
			deleteParentElementIfEmpty(XPathProject);
		}

		document.setIsModified();
	}

	public List<String> getProjectAbbreviation() {
		return this.projectAbbreviation;
	}

	public void setProjectAbbreviation(List<String> projectAbbreviation) {
		this.projectAbbreviation = projectAbbreviation;

		setDynamicElementTexts(this.projectAbbreviation, XPathProjectAbbreviation);
		if (this.projectAbbreviation == null) {
			deleteParentElementIfEmpty(XPathProjectAbbreviation);
		}

		document.setIsModified();
	}

	public List<String> getLocation() {
		return this.location;
	}

	public void setLocation(List<String> location) {
		this.location = location;

		setDynamicElementTexts(this.location, XPathLocation);
		if (this.location == null) {
			deleteParentElementIfEmpty(XPathLocation);
		}

		document.setIsModified();
	}

	public List<String> getJournal() {
		return this.journal;
	}

	public void setJournal(List<String> journal) {
		this.journal = journal;

		setDynamicElementTexts(this.journal, XPathJournal);
		if (this.journal == null) {
			deleteParentElementIfEmpty(XPathJournal);
		}

		document.setIsModified();
	}

	public List<String> getUniversity() {
		return this.university;
	}

	public void setUniversity(List<String> university) {
		this.university = university;

		setDynamicElementTexts(this.university, XPathUniversity);
		if (this.university == null) {
			deleteParentElementIfEmpty(XPathUniversity);
		}

		document.setIsModified();
	}

	public List<String> getInstitute() {
		return this.institute;
	}

	public void setInstitute(List<String> institute) {
		this.institute = institute;

		setDynamicElementTexts(this.institute, XPathInstitute);
		if (this.institute == null) {
			deleteParentElementIfEmpty(XPathInstitute);
		}

		document.setIsModified();
	}

	public List<String> getCompartment() {
		return this.compartment;
	}

	public void setCompartment(List<String> compartment) {
		this.compartment = compartment;

		setDynamicElementTexts(this.compartment, XPathCompartment);
		if (this.compartment == null) {
			deleteParentElementIfEmpty(XPathCompartment);
		}

		document.setIsModified();
	}

	public List<String> getMethod() {
		return this.method;
	}

	public void setMethod(List<String> method) {
		this.method = method;

		setDynamicElementTexts(this.method, XPathMethod);
		if (this.method == null) {
			deleteParentElementIfEmpty(XPathMethod);
		}

		document.setIsModified();
	}

	public List<String> getYear() {
		return did.getYear();
	}

	public void setYear(List<String> year) {
		did.setYear(year);
	}

	public List<String> getKeyword() {
		return this.keyword;
	}

	public void setKeyword(List<String> keyword) {
		this.keyword = keyword;

		setDynamicElementTexts(this.keyword, XPathKeyword);
		if (this.keyword == null) {
			deleteParentElementIfEmpty(XPathKeyword);
		}

		document.setIsModified();
	}

	public List<String> getFundingSource() {
		return this.fundingSource;
	}

	public void setFundingSource(List<String> fundingSource) {
		this.fundingSource = fundingSource;

		setDynamicElementTexts(this.fundingSource, XPathFundingSource);
		if (this.fundingSource == null) {
			deleteParentElementIfEmpty(XPathFundingSource);
		}

		document.setIsModified();
	}

	public List<String> getStatus() {
		return this.status;
	}

	public void setStatus(List<String> status) {
		this.status = status;

		setDynamicElementTexts(this.status, XPathStatus);
		if (this.status == null) {
			deleteParentElementIfEmpty(XPathStatus);
		}

		document.setIsModified();
	}

	public List<String> getPublisher() {
		return publisher;
	}

	public void setPublisher(List<String> publisher) {
		this.publisher = publisher;

		setDynamicElementTexts(this.publisher, XPathPublisher);
		if (this.publisher == null) {
			deleteParentElementIfEmpty(XPathPublisher);
		}

		document.setIsModified();
	}

	public List<String> getAbstract() {
		return did.getAbstract();
	}

	public void setAbstract(List<String> abstract_) {
		did.setAbstract(abstract_);
	}

	public List<String> getDoiJournal() {
		return this.doiJournal;
	}

	public void setDoiJournal(List<String> doiJournal) {
		this.doiJournal = doiJournal;

		setDynamicElementTexts(this.doiJournal, XPathDOIJournal);
		if (this.doiJournal == null) {
			deleteParentElementIfEmpty(XPathDOIJournal);
		}

		document.setIsModified();
	}

	public List<String> getRetentionPeriod() {
		return this.retentionPeriod;
	}

	public void setRetentionPeriod(List<String> retentionPeriod) {
		this.retentionPeriod = retentionPeriod;

		setDynamicElementTexts(this.retentionPeriod, XPathRetentionPeriod);
		if (this.retentionPeriod == null) {
			deleteParentElementIfEmpty(XPathRetentionPeriod);
		}

		document.setIsModified();
	}

	public List<String> getUsage() {
		return this.usage;
	}

	public void setUsage(List<String> usage) {
		this.usage = usage;

		setDynamicElementTexts(this.usage, XPathUsage);
		if (this.usage == null) {
			deleteParentElementIfEmpty(XPathUsage);
		}

		document.setIsModified();
	}

	public List<String> getEvent() {
		return this.event;
	}

	public void setEvent(List<String> event) {
		this.event = event;

		setDynamicElementTexts(this.event, XPathEvent);
		if (this.event == null) {
			deleteParentElementIfEmpty(XPathEvent);
		}

		document.setIsModified();
	}

	public List<String> getBiographicalHistory() {
		return this.biographicalHistory;
	}

	public void setBiographicalHistory(List<String> values) {
		this.biographicalHistory = values;

		setDynamicElementTexts(this.biographicalHistory, XPathBiographicalHistory);
		if (this.biographicalHistory == null) {
			deleteParentElementIfEmpty(XPathBiographicalHistory);
		}

		document.setIsModified();
	}

	public List<String> getFirstname() {
		return this.firstname;
	}

	public void setFirstname(List<String> values) {
		this.firstname = values;

		setDynamicElementTexts(this.firstname, XPathFirstname);
		if (this.firstname == null) {
			deleteParentElementIfEmpty(XPathFirstname);
		}

		document.setIsModified();
	}

	public List<String> getLastname() {
		return this.lastname;
	}

	public void setLastname(List<String> values) {
		this.lastname = values;

		setDynamicElementTexts(this.lastname, XPathLastname);
		if (this.lastname == null) {
			deleteParentElementIfEmpty(XPathLastname);
		}

		document.setIsModified();
	}

	public List<String> getDateOfBirth() {
		return this.dateOfBirth;
	}

	public void setDateOfBirth(List<String> values) {
		this.dateOfBirth = values;

		setDynamicElementTexts(this.dateOfBirth, XPathDateOfBirth);
		if (this.dateOfBirth == null) {
			deleteParentElementIfEmpty(XPathDateOfBirth);
		}

		document.setIsModified();
	}

	public List<String> getNationality() {
		return this.nationality;
	}

	public void setNationality(List<String> values) {
		this.nationality = values;

		setDynamicElementTexts(this.nationality, XPathNationality);
		if (this.nationality == null) {
			deleteParentElementIfEmpty(XPathNationality);
		}

		document.setIsModified();
	}

	public List<String> getSex() {
		return this.sex;
	}

	public void setSex(List<String> values) {
		this.sex = values;

		setDynamicElementTexts(this.sex, XPathSex);
		if (this.sex == null) {
			deleteParentElementIfEmpty(XPathSex);
		}

		document.setIsModified();
	}

	public List<String> getDeathOfAuthor() {
		return this.deathOfAuthor;
	}

	public void setDeathOfAuthor(List<String> values) {
		this.deathOfAuthor = values;

		setDynamicElementTexts(this.deathOfAuthor, XPathDeathOfAuthor);
		if (this.deathOfAuthor == null) {
			deleteParentElementIfEmpty(XPathDeathOfAuthor);
		}

		document.setIsModified();
	}

	public List<String> getAccruals() {
		return this.accruals;
	}

	public void setAccruals(List<String> values) {
		this.accruals = values;

		setDynamicElementTexts(this.accruals, XPathAccruals);
		if (this.accruals == null) {
			deleteParentElementIfEmpty(XPathAccruals);
		}

		document.setIsModified();
	}

	public List<String> getCharacteristicsNote() {
		return this.characteristicsNote;
	}

	public void setCharacteristicsNote(List<String> values) {
		this.characteristicsNote = values;

		setDynamicElementTexts(this.characteristicsNote, XPathCharacteristicsNote);
		if (this.characteristicsNote == null) {
			deleteParentElementIfEmpty(XPathCharacteristicsNote);
		}

		document.setIsModified();
	}

	public List<String> getSubject() {
		return this.subject;
	}

	public void setSubject(List<String> values) {
		this.subject = values;

		setDynamicElementTexts(this.subject, XPathSubject);
		if (this.subject == null) {
			deleteParentElementIfEmpty(XPathSubject);
		}

		document.setIsModified();
	}
	
	public List<String> getCartographicMaterial() {
		return did.getCartographicMaterial();
	}


	public void setCartographicMaterial(List<String> cartographicMaterial) {
		did.setCartographicMaterial(cartographicMaterial);
	}


	public List<String> getCodeLocation() {
		return did.getCodeLocation();
	}


	public void setCodeLocation(List<String> codeLocation) {
		did.setCodeLocation(codeLocation);
	}


	public List<String> getCreator() {
		return creator;
	}


	public void setCreator(List<String> creator) {
		this.creator = creator;
		setDynamicElementTexts(this.creator, XPATH_CREATOR);
		if (this.creator == null) {
			deleteParentElementIfEmpty(XPATH_CREATOR);
		}
		document.setIsModified();
	}


	public List<String> getDimensions() {
		return did.getDimensions();
	}


	public void setDimensions(List<String> dimensions) {
		did.setDimensions(dimensions);
	}


	public List<String> getDOI() {
		return did.getDOI();
	}


	public void setDOI(List<String> doi) {
		did.setDOI(doi);
	}


	public List<String> getEditionStatement() {
		return editionStatement;
	}


	public void setEditionStatement(List<String> editionStatement) {
		this.editionStatement = editionStatement;
		setDynamicElementTexts(this.editionStatement, XPATH_EDITION_STATEMENT);
		if (this.editionStatement == null) {
			deleteParentElementIfEmpty(XPATH_EDITION_STATEMENT);
		}
		document.setIsModified();
	}


	public List<String> getFullLocation() {
		return did.getFullLocation();
	}


	public void setFullLocation(List<String> fullLocation) {
		did.setFullLocation(fullLocation);
	}


	public List<String> getGeogName() {
		return geogName;
	}


	public void setGeogName(List<String> geogName) {
		this.geogName = geogName;
		setDynamicElementTexts(this.geogName, XPATH_GEOG_NAME);
		if (this.geogName == null) {
			deleteParentElementIfEmpty(XPATH_GEOG_NAME);
		}
		document.setIsModified();
	}


	public List<String> getInstitutionCreator() {
		return institutionCreator;
	}


	public void setInstitutionCreator(List<String> institutionCreator) {
		this.institutionCreator = institutionCreator;
		setDynamicElementTexts(this.institutionCreator, XPATH_INSTITUTION_CREATOR);
		if (this.institutionCreator == null) {
			deleteParentElementIfEmpty(XPATH_INSTITUTION_CREATOR);
		}
		document.setIsModified();
	}


	public List<String> getInstitutionInvolved() {
		return institutionInvolved;
	}


	public void setInstitutionInvolved(List<String> institutionInvolved) {
		this.institutionInvolved = institutionInvolved;
		setDynamicElementTexts(this.institutionInvolved, XPATH_INSTITUTION_INVOLVED);
		if (this.institutionInvolved == null) {
			deleteParentElementIfEmpty(XPATH_INSTITUTION_INVOLVED);
		}
		document.setIsModified();
	}


	public List<String> getISBN() {
		return did.getISBN();
	}


	public void setISBN(List<String> isbn) {
		did.setISBN(isbn);
	}


	public List<String> getISSN() {
		return did.getISSN();
	}


	public void setISSN(List<String> issn) {
		did.setISSN(issn);
	}


	public List<String> getPlaceOfPublication() {
		return placeOfPublication;
	}


	public void setPlaceOfPublication(List<String> placeOfPublication) {
		this.placeOfPublication = placeOfPublication;
		setDynamicElementTexts(this.placeOfPublication, XPATH_PLACE_OF_PUBLICATION);
		if (this.placeOfPublication == null) {
			deleteParentElementIfEmpty(XPATH_PLACE_OF_PUBLICATION);
		}
		document.setIsModified();
	}


	public List<String> getSubjectGeneral() {
		return subjectGeneral;
	}


	public void setSubjectGeneral(List<String> subjectGeneral) {
		this.subjectGeneral = subjectGeneral;
		setDynamicElementTexts(this.subjectGeneral, XPATH_SUBJECT_GENERAL);
		if (this.subjectGeneral == null) {
			deleteParentElementIfEmpty(XPATH_SUBJECT_GENERAL);
		}
		document.setIsModified();
	}


	public List<String> getUniformTitle() {
		return uniformTitle;
	}


	public void setUniformTitle(List<String> uniformTitle) {
		this.uniformTitle = uniformTitle;
		setDynamicElementTexts(this.uniformTitle, XPATH_UNIFORM_TITLE);
		if (this.uniformTitle == null) {
			deleteParentElementIfEmpty(XPATH_UNIFORM_TITLE);
		}
		document.setIsModified();
	}


	public List<String> getUnitTitleVarying() {
		return did.getUnitTitleVarying();
	}


	public void setUnitTitleVarying(List<String> unitTitleVarying) {
		did.setUnitTitleVarying(unitTitleVarying);
	}
	
	
	public List<String> getFormat() {
		return did.getFormat();
	}


	public void setFormat(List<String> format) {
		did.setFormat(format);
	}


	public List<String> getSize() {
		return did.getSize();
	}


	public void setSize(List<String> size) {
		did.setSize(size);
	}


	public List<String> getDate() {
		return did.getDate();
	}


	public void setDate(List<String> date) {
		did.setDate(date);
	}


	public List<String> getUnitTitleOriginal() {
		return did.getUnitTitleOriginal();
	}


	public void setUnitTitleOriginal(List<String> unitTitleOriginal) {
		did.setUnitTitleOriginal(unitTitleOriginal);
	}
	

	public ComponentAbstract searchUnitTitle(String title) {
		if (did.getUnitTitle().equals(title)) {
			return this;
		}

		for (C c : this.cs) {
			ComponentAbstract found = c.searchUnitTitle(title); // Recursion!
			if (found != null) {
				return found;
			}
		}

		return null;
	}

	/**
	 * This is done on EAD export.
	 */
	public abstract void fillEADElementsFromNodeRecursively(ch.docuteam.darc.mets.structmap.NodeAbstract node);

	protected String toString(String name) {
		StringBuilder stringBuilder = new StringBuilder();

		// Indent:
		for (int i = 0; i < this.depth; i++) {
			stringBuilder.append("\t");
		}

		stringBuilder.append("[" + name + "(").append(this.depth).append("):'").append(this.otherLevel).append("':")
				.append(did);

		for (C c : this.cs) {
			stringBuilder.append("\n").append(c);
		}

		return stringBuilder.append("]").toString();
	}

	protected String toStringShort(String name) {
		return new StringBuilder().append("[" + name + "(").append(this.depth).append("):'").append(this.otherLevel)
				.append("':").append(did).append("]").toString();
	}

}
