/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.ead;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.Element;

import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.util.XMLUtil;

/**
 * This class occurs in <a href="./ArchDesc.html">ArchDesc</a> and <a href="./C.html">C</a> objects.
 * It contains many of the dynamic metadata objects.
 *
 * @author denis
 *
 */
public class DID extends EADNodeAbstract {

	static private final String XPATH_UnitTitle = "EAD:unittitle[@label = 'main']";

	static private final String XPATH_UnitTitleAdditional = "EAD:unittitle[@label = 'additional']";
	private static final String XPATH_UNIT_TITLE_VARYING = "EAD:unittitle[@label = 'varying']";
	private static final String XPATH_UNIT_TITLE_ORIGINAL = "EAD:unittitle[@label = 'original']";

	static private final String XPATH_CreationPeriod = "EAD:unitdate[@label = 'creationPeriod']";
	static private final String XPATH_CreationPeriodNotes = "EAD:unitdate[@label = 'creationPeriodNotes']";
	static private final String XPATH_Year = "EAD:unitdate[@label = 'year']";
	static private final String XPATH_FromYear = "EAD:unitdate[@label = 'fromYear']";
	static private final String XPATH_ToYear = "EAD:unitdate[@label = 'toYear']";
	private static final String XPATH_FROM = "EAD:unitdate[@label = 'from']";
	private static final String XPATH_TO = "EAD:unitdate[@label = 'to']";
	static private final String XPATH_RelationPeriod = "EAD:unitdate[@label = 'relationPeriod']";
	private static final String XPATH_DATE = "EAD:unitdate[@label = 'date']";

	static private final String XPATH_Language = "EAD:langmaterial/EAD:language";
	static private final String XPATH_LanguageNotes = "EAD:langmaterial";

	static private final String XPATH_RefCode = "EAD:unitid[@type = 'refCode']";
	static private final String XPATH_RefCodeOld = "EAD:unitid[@type = 'refCodeOld']";
	static private final String XPATH_RefCodeAdmin = "EAD:unitid[@type = 'refCodeAdmin']";
	static private final String XPATH_AccessionNumber = "EAD:unitid[@type = 'accession']";
	private static final String XPATH_DOI = "EAD:unitid[@type = 'otherStandardIdentifier']";
	private static final String XPATH_ISBN = "EAD:unitid[@type = 'isbn']";
	private static final String XPATH_ISSN = "EAD:unitid[@type = 'issn']";
	private static final String XPATH_RefCodeSeparator= "EAD:unitid[@type = 'separator']";
	private static final String XPATH_RefCodeIsVisible = "EAD:unitid[@type = 'isVisible']";
	private static final String XPATH_CARTOGRAPHIC_MATERIAL = "EAD:materialspec[@label = 'cartographic']";
	private static final String XPATH_DIMENSIONS = "EAD:physdesc/EAD:dimensions";
	static private final String XPATH_Extent = "EAD:physdesc/EAD:extent[not(@type)]";
	static private final String XPATH_ExtentNote = "EAD:physdesc/EAD:extent[@type = 'note']";
	private static final String XPATH_ExtentUnit = "EAD:physdesc/EAD:extent[not(@type)]/@unit";
	static private final String XPATH_Material = "EAD:physdesc/EAD:physfacet";
	private static final String XPATH_FORMAT = "EAD:physdesc[@label = 'format']";
	private static final String XPATH_SIZE = "EAD:physdesc[@label = 'size']";
	private static final String XPATH_CODE_LOCATION = "EAD:physloc[@label = 'code']";
	private static final String XPATH_FULL_LOCATION = "EAD:physloc[@label = 'fullLocation']";

	static private final String XPATH_Origination = "EAD:origination[not(@label)]";
	static private final String XPATH_OriginationAgency = "EAD:origination[@label = 'agency']";
	static private final String XPATH_OriginationDepartment = "EAD:origination[@label = 'department']";
	static private final String XPATH_Abstract = "EAD:abstract";

	private String unitTitle;

	private List<String> unitTitleAdditional;
	private List<String> origination;
	private List<String> originationAgency;
	private List<String> originationDepartment;
	private List<String> creationPeriod;
	private List<String> creationPeriodNotes;
	private List<String> fromYear;
	private List<String> toYear;
	private List<String> from;
	private List<String> to;
	private List<String> relationPeriod;
	private List<String> language;
	private List<String> languageNotes;
	private List<String> refCode;
	private List<String> refCodeOld;
	private List<String> refCodeAdmin;
	private List<String> refCodeSeparator;
	private List<String> refCodeIsVisible;
	private List<String> accessionNumber;
	private List<String> extent;
	private List<String> extentNote;
	private List<String> extentUnit;
	private List<String> material;

	private List<String> year;
	private List<String> abstract_;

	private List<String> cartographicMaterial;
	private List<String> codeLocation;
	private List<String> dimensions;
	private List<String> doi;
	private List<String> fullLocation;
	private List<String> isbn;
	private List<String> issn;
	private List<String> unitTitleVarying;
	private List<String> format;
	private List<String> size;
	private List<String> date;
	private List<String> unitTitleOriginal;
	

	/**
	 * This constructor is used only when an EAD-File is being read.
	 */
	private DID(NodeAbstract parent, Element element) {
		this.document = parent.getDocument();
		this.parent = parent;
		this.element = element;

		unitTitle = getElementText(XPATH_UnitTitle, true);

		unitTitleAdditional = getDynamicElementTexts(XPATH_UnitTitleAdditional);
		origination = getDynamicElementTexts(XPATH_Origination);
		originationAgency = getDynamicElementTexts(XPATH_OriginationAgency);
		originationDepartment = getDynamicElementTexts(XPATH_OriginationDepartment);
		creationPeriod = getDynamicElementTexts(XPATH_CreationPeriod);
		creationPeriodNotes = getDynamicElementTexts(XPATH_CreationPeriodNotes);
		fromYear = getDynamicElementTexts(XPATH_FromYear);
		toYear = getDynamicElementTexts(XPATH_ToYear);
		from = getDynamicElementTexts(XPATH_FROM);
		to = getDynamicElementTexts(XPATH_TO);
		relationPeriod = getDynamicElementTexts(XPATH_RelationPeriod);
		language = getDynamicElementTexts(XPATH_Language);
		Element languageNotesElement = ((Element) element.selectSingleNode(XPATH_LanguageNotes));
		if (languageNotesElement != null) {
			languageNotes = new ArrayList<String>(1);
			languageNotes.add(languageNotesElement.attributeValue("label"));
		}
		refCode = getDynamicElementTexts(XPATH_RefCode);
		refCodeOld = getDynamicElementTexts(XPATH_RefCodeOld);
		refCodeAdmin = getDynamicElementTexts(XPATH_RefCodeAdmin);
		refCodeIsVisible = getDynamicElementTexts(XPATH_RefCodeIsVisible);
		refCodeSeparator = getDynamicElementTexts(XPATH_RefCodeSeparator);
		accessionNumber = getDynamicElementTexts(XPATH_AccessionNumber);
		extent = getDynamicElementTexts(XPATH_Extent);
		extentNote = getDynamicElementTexts(XPATH_ExtentNote);
		extentUnit = getDynamicElementTexts(XPATH_ExtentUnit);
		material = getDynamicElementTexts(XPATH_Material);

		year = getDynamicElementTexts(XPATH_Year);
		abstract_ = getDynamicElementTexts(XPATH_Abstract);
		
		cartographicMaterial = getDynamicElementTexts(XPATH_CARTOGRAPHIC_MATERIAL);
		codeLocation = getDynamicElementTexts(XPATH_CODE_LOCATION);
		dimensions = getDynamicElementTexts(XPATH_DIMENSIONS);
		doi = getDynamicElementTexts(XPATH_DOI);
		fullLocation = getDynamicElementTexts(XPATH_FULL_LOCATION);
		isbn = getDynamicElementTexts(XPATH_ISBN);
		issn = getDynamicElementTexts(XPATH_ISSN);
		unitTitleVarying = getDynamicElementTexts(XPATH_UNIT_TITLE_VARYING);
		format = getDynamicElementTexts(XPATH_FORMAT);
		size = getDynamicElementTexts(XPATH_SIZE);
		date = getDynamicElementTexts(XPATH_DATE);
		unitTitleOriginal = getDynamicElementTexts(XPATH_UNIT_TITLE_ORIGINAL);
		
	}


	/**
	 * This constructor is used when a new DID is created programmatically.
	 */
	protected DID(ComponentAbstract parent, String title)
	{
		this.document = parent.getDocument();

		this.unitTitle = title;

		//	create dom4j element:
		this.element = parent.getElement().addElement("EAD:did");
		this.element.addElement("EAD:unittitle")
						.addAttribute("label", "main")
						.addText(title);

		document.setIsModified();
	}

	/**
	 * This method is used only when an EAD-File is being read.
	 */
	static DID parse(NodeAbstract parent, Element element)
	{
		return new DID(parent, (Element)element.selectSingleNode("./EAD:did"));
	}

	public String getUnitTitle()
	{
		return this.unitTitle;
	}

	public void setUnitTitle(String unitTitle)
	{
		this.unitTitle = unitTitle;

		this.element.selectSingleNode(XPATH_UnitTitle).setText(this.unitTitle);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getUnitTitleAdditional()
	{
		return this.unitTitleAdditional;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setUnitTitleAdditional(List<String> values)
	{
		this.unitTitleAdditional = values;

		setDynamicElementTexts(this.unitTitleAdditional, XPATH_UnitTitleAdditional);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getOrigination()
	{
		return this.origination;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setOrigination(List<String> values)
	{
		this.origination = values;

		setDynamicElementTexts(this.origination, XPATH_Origination);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getOriginationAgency()
	{
		return this.originationAgency;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setOriginationAgency(List<String> values)
	{
		this.originationAgency = values;

		setDynamicElementTexts(this.originationAgency, XPATH_OriginationAgency);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getOriginationDepartment()
	{
		return this.originationDepartment;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setOriginationDepartment(List<String> values)
	{
		this.originationDepartment = values;

		setDynamicElementTexts(this.originationDepartment, XPATH_OriginationDepartment);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getCreationPeriod()
	{
		return this.creationPeriod;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setCreationPeriod(List<String> values)
	{
		this.creationPeriod = values;

		setDynamicElementTexts(this.creationPeriod, XPATH_CreationPeriod);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getCreationPeriodNotes()
	{
		return this.creationPeriodNotes;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setCreationPeriodNotes(List<String> values)
	{
		this.creationPeriodNotes = values;

		setDynamicElementTexts(this.creationPeriodNotes, XPATH_CreationPeriodNotes);

		document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getFromYear()
	{
		return this.fromYear;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setFromYear(List<String> values)
	{
		this.fromYear = values;

		setDynamicElementTexts(this.fromYear, XPATH_FromYear);

		document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getToYear()
	{
		return toYear;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setToYear(List<String> values)
	{
		this.toYear = values;

		setDynamicElementTexts(this.toYear, XPATH_ToYear);

		document.setIsModified();
	}
	
	public List<String> getFrom() {
		return from;
	}


	public void setFrom(List<String> from) {
		this.from = from;
		setDynamicElementTexts(this.from, XPATH_FROM);
		document.setIsModified();
	}


	public List<String> getTo() {
		return to;
	}


	public void setTo(List<String> to) {
		this.to = to;
		setDynamicElementTexts(this.to, XPATH_TO);
		document.setIsModified();
	}


	public List<String> getCartographicMaterial() {
		return cartographicMaterial;
	}


	public void setCartographicMaterial(List<String> cartographicMaterial) {
		this.cartographicMaterial = cartographicMaterial;
		setDynamicElementTexts(this.cartographicMaterial, XPATH_CARTOGRAPHIC_MATERIAL);
		document.setIsModified();
	}


	public List<String> getCodeLocation() {
		return codeLocation;
	}


	public void setCodeLocation(List<String> codeLocation) {
		this.codeLocation = codeLocation;
		setDynamicElementTexts(this.codeLocation, XPATH_CODE_LOCATION);
		document.setIsModified();
	}


	public List<String> getDimensions() {
		return dimensions;
	}


	public void setDimensions(List<String> dimensions) {
		this.dimensions = dimensions;
		setDynamicElementTexts(this.dimensions, XPATH_DIMENSIONS);
		document.setIsModified();
	}


	public List<String> getDOI() {
		return doi;
	}


	public void setDOI(List<String> doi) {
		this.doi = doi;
		setDynamicElementTexts(this.doi, XPATH_DOI);
		document.setIsModified();
	}


	public List<String> getFullLocation() {
		return fullLocation;
	}


	public void setFullLocation(List<String> fullLocation) {
		this.fullLocation = fullLocation;
		setDynamicElementTexts(this.fullLocation, XPATH_FULL_LOCATION);
		document.setIsModified();
	}


	public List<String> getISBN() {
		return isbn;
	}


	public void setISBN(List<String> isbn) {
		this.isbn = isbn;
		setDynamicElementTexts(this.isbn, XPATH_ISBN);
		document.setIsModified();
	}


	public List<String> getISSN() {
		return issn;
	}


	public void setISSN(List<String> issn) {
		this.issn = issn;
		setDynamicElementTexts(this.issn, XPATH_ISSN);
		document.setIsModified();
	}


	public List<String> getUnitTitleVarying() {
		return unitTitleVarying;
	}


	public void setUnitTitleVarying(List<String> unitTitleVarying) {
		this.unitTitleVarying = unitTitleVarying;
		setDynamicElementTexts(this.unitTitleVarying, XPATH_UNIT_TITLE_VARYING);
		document.setIsModified();
	}
	

	public List<String> getFormat() {
		return format;
	}


	public void setFormat(List<String> format) {
		this.format = format;
		setDynamicElementTexts(this.format, XPATH_FORMAT);
		document.setIsModified();
	}


	public List<String> getSize() {
		return size;
	}


	public void setSize(List<String> size) {
		this.size = size;
		setDynamicElementTexts(this.size, XPATH_SIZE);
		document.setIsModified();
	}


	public List<String> getDate() {
		return date;
	}


	public void setDate(List<String> date) {
		this.date = date;
		setDynamicElementTexts(this.date, XPATH_DATE);
		document.setIsModified();
	}


	public List<String> getUnitTitleOriginal() {
		return unitTitleOriginal;
	}


	public void setUnitTitleOriginal(List<String> unitTitleOriginal) {
		this.unitTitleOriginal = unitTitleOriginal;
		setDynamicElementTexts(this.unitTitleOriginal, XPATH_UNIT_TITLE_ORIGINAL);
		document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getRelationPeriod()
	{
		return this.relationPeriod;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRelationPeriod(List<String> values)
	{
		this.relationPeriod = values;

		setDynamicElementTexts(this.relationPeriod, XPATH_RelationPeriod);

		document.setIsModified();
	}


	/**
	 * Dynamic Metadata
	 */
	public List<String> getLanguage()
	{
		return this.language;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setLanguage(List<String> values)
	{
		this.language = values;

		setDynamicElementTexts(this.language, XPATH_Language);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getLanguageNotes()
	{
		return this.languageNotes;
	}

	/**
	 * Dynamic Metadata: special way to set the values - don't use the standard way with setDynamicElementTexts(value, XPATH_);
	 */
	public void setLanguageNotes(List<String> values)
	{
		this.languageNotes = values;

		Element base = (Element)this.element.selectSingleNode(XPATH_LanguageNotes);
		if (base == null)				base = this.element.addElement(XPATH_LanguageNotes);

		if (this.languageNotes != null && this.languageNotes.size() != 0)
		{
			//	Just take the 1st element and ignore all others (anyway there should never be more than one):
			base.addAttribute("label", this.languageNotes.get(0));
		}
		else
		{
			base.addAttribute("label", null);
		}

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getExtent()
	{
		return this.extent;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setExtent(List<String> values)
	{
		this.extent = values;

		setDynamicElementTexts(this.extent, XPATH_Extent);
		if (this.extent == null && this.extentUnit == null) {
			deleteParentElementIfEmpty(XPATH_Extent);
		}

		document.setIsModified();
	}
	
	public List<String> getExtentNote() {
		return extentNote;
	}
	
	public void setExtentNote(List<String> values)
	{
		extentNote = values;

		setDynamicElementTexts(extentNote, XPATH_ExtentNote);
		if (extentNote == null) {
			deleteParentElementIfEmpty(XPATH_ExtentNote);
		}

		document.setIsModified();
	}
	
	public List<String> getExtentUnit() {
		return extentUnit;
	}
	
	/**
	 * This setter is special as the XPath refers to an attribute instead of an
	 * element, and the element in question is used as another dynamic metadata
	 * item.
	 * 
	 * @param values
	 */
	public void setExtentUnit(List<String> values)
	{
		extentUnit = values;

		//	Collect the actual elements
		String parentXPath = XMLUtil.asParentPath(XPATH_ExtentUnit);
		ArrayList<Element> base = new ArrayList<Element>();
		base.addAll(this.element.selectNodes(parentXPath));
		String genericElementXPath = parentXPath.replaceAll("\\[not\\(@\\w+\\)\\]", "");

		int counter = 0;

		//	add values
		if (extentUnit != null) {
			for (String unit: extentUnit) {
				//	add element for the attribute if it doesn't exist yet
				if (base.size() <= counter) {
					Element e = XMLUtil.add(this.element, genericElementXPath);
					base.add(e);
				}
				
				//	add the attribute
				base.get(counter).addAttribute("unit", unit);
				
				counter++;
			}
		}
		
		//	clean any additional attributes/elements if empty
		while (counter < base.size()) {
			base.get(counter).addAttribute("unit", null);
			deleteElementIfEmpty(XPATH_ExtentUnit, base.get(counter));
		}

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getMaterial()
	{
		return this.material;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setMaterial(List<String> values)
	{
		this.material = values;

		setDynamicElementTexts(this.material, XPATH_Material);
		if (this.material == null) {
			deleteParentElementIfEmpty(XPATH_Material);
		}

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCode()
	{
		return this.refCode;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCode(List<String> values)
	{
		this.refCode = values;

		setDynamicElementTexts(this.refCode, XPATH_RefCode);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCodeOld()
	{
		return this.refCodeOld;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCodeOld(List<String> values)
	{
		this.refCodeOld = values;

		setDynamicElementTexts(this.refCodeOld, XPATH_RefCodeOld);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCodeAdmin()
	{
		return this.refCodeAdmin;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCodeAdmin(List<String> values)
	{
		this.refCodeAdmin = values;

		setDynamicElementTexts(this.refCodeAdmin, XPATH_RefCodeAdmin);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCodeIsVisible()
	{
		return this.refCodeIsVisible;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCodeIsVisible(List<String> values)
	{
		this.refCodeIsVisible = values;

		setDynamicElementTexts(this.refCodeIsVisible, XPATH_RefCodeIsVisible);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getRefCodeSeparator()
	{
		return this.refCodeSeparator;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setRefCodeSeparator(List<String> values)
	{
		this.refCodeSeparator = values;

		setDynamicElementTexts(this.refCodeSeparator, XPATH_RefCodeSeparator);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAccessionNumber()
	{
		return this.accessionNumber;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAccessionNumber(List<String> values)
	{
		this.accessionNumber = values;

		setDynamicElementTexts(this.accessionNumber, XPATH_AccessionNumber);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getYear()
	{
		return this.year;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setYear(List<String> year)
	{
		this.year = year;

		setDynamicElementTexts(this.year, XPATH_Year);

		document.setIsModified();
	}

	/**
	 * Dynamic Metadata
	 */
	public List<String> getAbstract()
	{
		return this.abstract_;
	}

	/**
	 * Dynamic Metadata
	 */
	public void setAbstract(List<String> abstract_)
	{
		this.abstract_ = abstract_;

		setDynamicElementTexts(this.abstract_, XPATH_Abstract);

		document.setIsModified();
	}

	@Override
	public String toString()
	{
		return new StringBuilder("[did:'")
			.append(this.unitTitle)
			.append("']")
			.toString();
	}

}
