/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.ingest;

import java.util.List;
import java.util.Properties;

import ch.docuteam.darc.mets.Document;

/**
 * This class is an abstract static wrapper around the interface "AIPCreator".
 * Initialize it with the instance, class, or class name of the implementation.
 * After that, calling the static methods initialize(Properties),
 * checkSubmission(Document), generateAIP(Document), or confirmStorage(Document)
 * just forwards the calls to the implementation instance.
 *
 * @author denis
 *
 */
public abstract class AIPCreatorProxy {

	static private AIPCreator AIPCreatorInstance;

	/**
	 * Initialize this proxy with a class name. If className is null or empty,
	 * ignore it silently.
	 * 
	 * @param implClassName
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 * @throws ClassNotFoundException
	 */
	static public void initializeImpl(String implClassName)
			throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		if (implClassName == null || implClassName.isEmpty())
			return;

		initializeImpl(Class.forName(implClassName));
	}

	/**
	 * Initialize this proxy with a class:
	 * 
	 * @param implClass
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 */
	static public void initializeImpl(Class<?> implClass) throws InstantiationException, IllegalAccessException {
		initializeImpl((AIPCreator) implClass.newInstance());
	}

	/**
	 * Initialize this proxy with an implementation of AIPCreator:
	 * 
	 * @param implInstance
	 */
	static public void initializeImpl(AIPCreator implInstance) {
		AIPCreatorInstance = implInstance;
	}

	static public AIPCreator getAIPCreator() {
		return AIPCreatorInstance;
	}

	static public boolean isUsable() {
		return AIPCreatorInstance != null;
	}

	/**
	 * Initialize the implementation.
	 * 
	 * @param properties
	 *            A set of properties to initialize the implementation.
	 * @throws Exception
	 *             In case some mandatory properties are missing.
	 */
	static public void initialize(Properties properties) throws Exception {
		if (AIPCreatorInstance == null)
			throw new NullPointerException("Implementation of interface 'AIPCreator' is not set");

		AIPCreatorInstance.initialize(properties);
	}

	/**
	 * Check the document if an AIP could be generated.
	 * 
	 * @param document
	 *            The document to be tested
	 * @return A list of error messages. If this list is empty, the AIP could be
	 *         created.
	 * @throws ClassNotFoundException
	 */
	static public List<String> checkSubmission(Document document) {
		if (AIPCreatorInstance == null)
			throw new NullPointerException("Implementation of interface 'AIPCreator' is not set");

		return AIPCreatorInstance.checkSubmission(document);
	}

	/**
	 * Actually submit the SIP.
	 * 
	 * @param document
	 *            The document to create the AIP from.
	 * @param destination
	 *            The place where the created AIP should be copied to.
	 * @return Whatever the generator wants to return.
	 * @throws Exception
	 */
	static public List<String> submit(Document document) throws Exception {
		if (AIPCreatorInstance == null)
			throw new NullPointerException("Implementation of interface 'AIPCreator' is not set");

		return AIPCreatorInstance.submit(document);
	}

	/**
	 * Check for ingest feedback.
	 * 
	 * @param document
	 *            The document to check for update.
	 * @return List of messages the implementation wants to return.
	 * @throws Exception
	 */
	static public List<String> checkIngestFeedback(String workspaceFolder) throws Exception {
		if (AIPCreatorInstance == null)
			throw new NullPointerException("Implementation of interface 'AIPCreator' is not set");

		return AIPCreatorInstance.checkIngestFeedback(workspaceFolder);
	}

	/**
	 * Actually generate the AIP out of the Document.
	 * 
	 * @param document
	 *            The document to create the AIP from.
	 * @param destination
	 *            The place where the created AIP should be copied to.
	 * @return Whatever the generator wants to return.
	 * @throws Exception
	 */
	static public List<String> generateAIP(Document document) throws Exception {
		if (AIPCreatorInstance == null)
			throw new NullPointerException("Implementation of interface 'AIPCreator' is not set");

		return AIPCreatorInstance.generateAIP(document);
	}

	/**
	 * Get feedback about the stored AIP. The returning object is completly
	 * undefined for now.
	 * 
	 * @param document
	 * @param destination
	 * @return
	 * @throws Exception
	 */
	static public void confirmStorage(Document document) throws Exception {
		if (AIPCreatorInstance == null)
			throw new NullPointerException("Implementation of interface 'AIPCreator' is not set");

		AIPCreatorInstance.confirmStorage(document);
	}

}
