/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jdesktop.swingx.treetable.MutableTreeTableNode;

import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.mets.structmap.NodeFolder;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;

/**
 * Provides methods to customize defaultExpressions
 * Use by adding an attribute defaultExpression to a MetadataElement element in levels.xml like:
 * <code>defaultExpression="ch.docuteam.darc.mdconfig.MetadataElementDefaultExpressions.getUniqueNumericValueWithParentsPrefix( object1, object2 )"</code>
 * where object1 and object2 are arbitrary objects.
 * 
 * @author christian
 *
 */
public abstract class MetadataElementDefaultExpressions {

	// the following regex matches strings like:
	// - 2015_02_01_Titel_Redaktionsangaben
	// - 2015_02_Titel_Redaktionsangaben
	// - 2015_Titel_Redaktionsangaben
	// - 2015_02_01_Titel
	// - 2015_02_01
	// - 2015
	private static final String DATE_NAME_REGEX = "(?<date>(?<year>^\\d{4})((_(?<month>\\d{2}))(_(?<day>\\d{2}))?)?)(_|$)(?<title>[^_]+)?(_(?<notes>.+))?";
	private static final String INPUT_DATE_FORMAT = "yyyy_MM_dd";
	private static final String OUTPUT_DATE_FORMAT = "d.M.yyyy";
	
	/**
	 * Calculate a unique value by prepending the parents value.
	 * Start with numbering if no siblings exist 
	 * Else take highest value among siblings and increment it
	 * 
	 * @param context the current NodeAbstract
	 * @param metadataElement the current MetadataElement
	 * @return computed unique value among siblings by using the parents element value and checking the siblings, computing a unique value
	 * @throws Exception
	 */
	public static String getUniqueNumericValueWithParentsPrefix(NodeAbstract context, MetadataElement metadataElement)
			throws Exception {
		String returnValue = null;
		if (context.isRoot()) {
			// start numbering in root node
			returnValue = "1";
		} else {
			// get the parent node to obtain its value as prefix for the
			// childrens values
			NodeFolder parent = (NodeFolder) context.getParent();
			String prefix = parent.getDynamicMetadataValueForName(metadataElement.getAccessorName()) + "/";
			for (MutableTreeTableNode child : ((NodeFolder) context.getParent()).getChildren()) {
				// Don't compare with itself:
				if (child == context) {
					continue;
				}

				if (returnValue == null) {
					// set value to current nodes value if not yet set
					returnValue = ((NodeAbstract) child)
							.getDynamicMetadataValueForName(metadataElement.getAccessorName());

				} else {
					// get current childs value and compare with existing value
					// in return value. Take the upper value
					String value = ((NodeAbstract) child)
							.getDynamicMetadataValueForName(metadataElement.getAccessorName());
					int comparison = value.compareTo(returnValue);
					returnValue = comparison > 0 ? value : returnValue;
				}
			}
			if (returnValue == null) {
				// value is not yet set, so start numbering with prefix
				returnValue = prefix + "1";
			} else {
				// prepend prefix if not yet present and increment the value
				if (!returnValue.startsWith(prefix)) {
					returnValue = prefix + returnValue;
				}
				returnValue = StringUtil.increment(returnValue);
			}
		}
		return returnValue;
	}

	/**
	 * Expect the first part of the filename to be a date in the format YYYY[_MM[_DD]] and return European formatted date.
	 * 
	 * @param context the current NodeAbstract
	 * @return date in format d.M.yyyy
	 * @throws Exception
	 */
	public static String extractDateFromFilename(NodeAbstract context) throws Exception {
		String result = null;
		String fileName = context.getLabel();

		if (fileName != null) {
			Pattern p = Pattern.compile(DATE_NAME_REGEX);
			Matcher m = p.matcher(fileName);

			if (m.find()) {
				String datePart = m.group("date");
				int dateLength = datePart.length();
				String year = m.group("year");
				String month = null;
				if (m.group("month") != null) {
					// remove the first 0 of month if there is one
					month = m.group("month").replaceFirst("^0", "");
				}

				// the switch is needed for the cases when day or month info is
				// missing from the date and then it can't be parsed to
				// LocalDate
				switch (dateLength) {
				case 10:
					DateTimeFormatter inputFormat = DateTimeFormatter.ofPattern(INPUT_DATE_FORMAT);
					try {
						LocalDate date = LocalDate.parse(datePart, inputFormat);
						DateTimeFormatter outputFormatter = DateTimeFormatter.ofPattern(OUTPUT_DATE_FORMAT);
						result = date.format(outputFormatter);
						if (!date.format(inputFormat).equals(datePart)) {
							Logger.getLogger().warn("The date string '" + datePart + "' will be parsed as '" + result + "'");
						}
					} catch (DateTimeParseException e) {
						if (Integer.parseInt(month) < 13) {
							result = month + "." + year;
						} else {
							result = year;
						}
						Logger.getLogger().warn("The date string '" + datePart + "'  will be parsed as '" + result + "'");
					}
					break;
				case 7:
					if (Integer.parseInt(month) < 13) {
						result = month + "." + year;
					} else {
						Logger.getLogger().warn("The month " + month + " is not valid and only the year " + year + " will be used.");
						result = year;
					}
					break;
				case 4:
					result = datePart;
					break;
				}

			}

		}
		
		if (result == null) {
			throw new Exception("No date for filename " + fileName);
		}
		return result;
	}

	/**
	 * Filter the title of the filename YYYY[_MM[_DD]]_title[_notes].
	 * 
	 * @param context the current NodeAbstract
	 * @return title part of the filename
	 * @throws Exception
	 */
	public static String extractTitleFromFilename(NodeAbstract context) throws Exception {
		return extractPartOfFileName(context, "title");
	}
	
	/**
	 * Filter the notes of the filename YYYY[_MM[_DD]]_title[_notes].
	 * 
	 * @param context the current NodeAbstract
	 * @return title part of the filename
	 * @throws Exception
	 */
	public static String extractNotesFromFilename(NodeAbstract context) throws Exception {
		return extractPartOfFileName(context, "notes");
	}

	/**
	 * Helper method to parse a part of the title that follows after the
	 * date (i.e. either title or notes, YYYY[_MM[_DD]]_title[_notes])
	 * 
	 * @param context
	 * @param partLabel
	 * @return
	 */
	private static String extractPartOfFileName(NodeAbstract context, String partLabel) {
		String result = null;
		String fileName = FileUtil.asFileNameWithoutExtension(context.getLabel());

		if (fileName != null) {
			Pattern p = Pattern.compile(DATE_NAME_REGEX);
			Matcher m = p.matcher(fileName);

			if (m.find()) {
				result = m.group(partLabel);
			}
		}
		return result;
	}
}
