/**
 *	Copyright (C) 2014-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import ch.docuteam.darc.exceptions.MetadataElementValidatorException;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.mets.structmap.NodeFolder;
import ch.docuteam.tools.translations.I18N;

/**
 * A MetadataElementValidator that checks if a given date range is within the
 * ancestors range and comprises the descendants date ranges".
 *
 * @author christian
 *
 */
public class MetadataElementValidatorDateHierarchyRangeCH implements MetadataElementValidator {

	@Override
	public void check(String value, NodeAbstract context, MetadataElement metadataElement) throws MetadataElementValidatorException 
	{
		//	There is no value set, so return
		if (value == null || value.isEmpty())	return;

		//	Check for the locale's value of "no date"
		if (value.equals(I18N.translate_NoCheck("NoDate")))	return;

		//	Check the date range value as such, using the respective validator
		new MetadataElementValidatorDateRangeCH().check(value, context, metadataElement);

		//	Split the date range to single dates
		String[] nodeValues = value.split(" *- *");

		//	Calculate the full date range by expanding not complete dates to full dates
		//	Complete the lower part by adding missing date parts, e.g. "12.2000" to "01.12.2000"
		//	Complete the upper part by adding missing date parts, e.g. "12.2000" to "31.12.2000"
		Calendar[] values = getValueRange(nodeValues);
		
		//	Check if values of children exceed new value
		Calendar[] limits = null;
		try
		{
			//	Calculate the minimal lower bound and the maximal upper bound among children (further descendants were checked when child values were set).
			limits = getChildrenValues(context, metadataElement);
		}
		catch (MetadataElementValidatorException e)
		{
			throw new MetadataElementValidatorException(e.getLocalizedMessage());
		}

		//	Limits can be null or an array of length == 2; it's ok if limits is null or an array with length == 2 and both elements are null.
		if (limits != null && (limits[0] != null || limits[1] != null))
		{
			//	Compare the limits with those calculated from children values and throw exception if children values exceed the values range.
			if ((limits[0] != null && values[0].after(limits[0])) || (limits[1] != null && values[1].before(limits[1])))
			{
				throw new MetadataElementValidatorException(I18N.translate_NoCheck("MessageValidatorDateHierarchyRangeCHNotValid", value));
			}
		}

		// If context is root, there are no further parents, so omit check
		if (!context.isRoot()) 
		{
			String[] parentValues = null;
			try
			{
				//	Calculate the minimal lower bound and the maximal upper bound of the parent.
				parentValues = getParentValues(context, metadataElement);
				if (parentValues == null || parentValues.length == 0) return;
			}
			catch (Exception e)
			{
				throw new MetadataElementValidatorException(e.getLocalizedMessage());
			}
			//	Calculate the full date range of the parents value by expanding not complete dates to full dates
			//	Complete the lower part by adding missing date parts, e.g. "12.2000" to "01.12.2000"
			//	Complete the upper part by adding missing date parts, e.g. "12.2000" to "31.12.2000"
			limits = getValueRange(parentValues);

			//	Compare if the values of the context node are within those of the parents node
			if (values[0].before(limits[0]) || values[1].after(limits[1]))
			{
				throw new MetadataElementValidatorException(I18N.translate_NoCheck("MessageValidatorDateHierarchyRangeCHNotValid", value));
			}
		}
	}

	/**
	 * Computes the date range as a calendar array containing the lower and the upper bound of the range. 
	 * If the given dateRange has incomplete dates they will be completed with maximal or minimal values respective.
	 * @param dateRange
	 * @return Calendar[]
	 * @throws MetadataElementValidatorException
	 */
	private Calendar[] getValueRange(String[] dateRange) throws MetadataElementValidatorException
	{
		Calendar[] range = new Calendar[2];
		
		switch (dateRange.length)
		{
		case 1:
		{
			//	Use the same dateRange value for calculate lower and upper bound
			range[0] = getLowerValue(dateRange[0]);
			range[1] = getUpperValue(dateRange[0]);
			break;
		}
		case 2:
		{
			//	Use both given values as lower and upper bounds
			range[0] = getLowerValue(dateRange[0]);
			range[1] = getUpperValue(dateRange[1]);
			break;
		}
		default:
		{
			// It is expected to get exactly one or two dates else throw exception
			throw new MetadataElementValidatorException(I18N.translate_NoCheck("MessageValidatorDateHierarchyRangeCHInvalidNumberOfValues", dateRange.length));
		}
		}
		return range;
	}

	/**
	 * Searches the node tree for the lowest lower and the highest upper bound within children
	 * @param context
	 * @param metadataElement
	 * @return lowest lower and highest upper limit of date range within children or <code>null</code> if none has been found
	 * @throws MetadataElementValidatorException 
	 */
	private Calendar[] getChildrenValues(NodeAbstract context, MetadataElement metadataElement) throws MetadataElementValidatorException
	{
		if (!context.isFolder()) return null;
		
		Calendar[] limits = new Calendar[2];

		for (NodeAbstract child: ((NodeFolder)context).getDescendants())
		{
			String value = null;
			try {
				value = child.getDynamicMetadataValueForName(metadataElement.getAccessorName());
			} catch (Exception e) { 
				//	Ignore exceptions here, in which case this child does not have a value that needs to be considered for the check
			}
			if (value != null && !value.isEmpty())
			{
				String[] values = value.split(" *- *");
				if (values.length == 0) continue;
				
				Calendar lowerLimit = getLowerValue(values[0]);
				if (limits[0] == null || limits[0].after(lowerLimit))
					limits[0] = lowerLimit;

				Calendar upperLimit = getUpperValue(values.length > 1 ? values[1] : values[0]);
				if (limits[1] == null || limits[1].before(upperLimit))
					limits[1] = upperLimit;
			}
		}
		return limits;
	}

	/**
	 * Searches the node tree for an ancestor with given date range
	 * @param context
	 * @param metadataElement
	 * @return lower and upper limit of date range or <code>null</code> if none has been found
	 * @throws Exception
	 */
	private String[] getParentValues(NodeAbstract context, MetadataElement metadataElement) throws Exception
	{
		NodeFolder parent = (NodeFolder) context.getParent();
		String value = parent.getDynamicMetadataValueForName(metadataElement.getAccessorName());
		if (value == null || value.isEmpty())
		{
			if (parent.isRoot())
			{
				return null;
			}
			else
			{
				return getParentValues(parent, metadataElement);
			}
		}
		else
		{
			if (value.equals(I18N.translate_NoCheck("NoDate")))	return null;
			return value.split(" *- *");
		}
	}

	/**
	 * Builds a calendar from string type date with minimal value for not given date parts
	 * @param value
	 * @return Calendar
	 * @throws MetadataElementValidatorException
	 */
	private Calendar getLowerValue(String value) throws MetadataElementValidatorException
	{
		Calendar lowerValue = Calendar.getInstance();
		try
		{
			Date date = getDateFormat(value).parse(value);
			lowerValue.setTime(date);
			switch(value.length())
			{
			case 4:
			{
				//	Only the year is given, add lowest possible month and day
				lowerValue.set(Calendar.MONTH, lowerValue.getMinimum(Calendar.MONTH));
				lowerValue.set(Calendar.DAY_OF_MONTH, lowerValue.getMinimum(Calendar.DAY_OF_MONTH));
				break;
			}
			case 7:
			{
				//	Only month and year are given, add lowest possible day
				lowerValue.set(Calendar.DAY_OF_MONTH, lowerValue.getMinimum(Calendar.DAY_OF_MONTH));
				break;
			}
			}
		}
		catch(ParseException e)
		{
			throw new MetadataElementValidatorException(I18N.translate_NoCheck("MessageValidatorDateCHCantConvert", value));
		}
		return lowerValue;
	}
	
	/**
	 * Builds a calendar from string type date with maximal value for not given date parts
	 * @param value
	 * @return Calendar
	 * @throws MetadataElementValidatorException
	 */
	private Calendar getUpperValue(String value) throws MetadataElementValidatorException
	{
		Calendar upperValue = Calendar.getInstance();
		try
		{
			Date date = getDateFormat(value).parse(value);
			upperValue.setTime(date);
			switch(value.length())
			{
			case 4:
			{
				//	Only the year is given, add highest possible month and day
				upperValue.set(Calendar.MONTH, upperValue.getMaximum(Calendar.MONTH));
				upperValue.set(Calendar.DAY_OF_MONTH, upperValue.getMaximum(Calendar.DAY_OF_MONTH));
				break;
			}
			case 7:
			{
				//	Year and month is given, add highest possible day in month
				upperValue.set(Calendar.DAY_OF_MONTH, upperValue.getMaximum(Calendar.DAY_OF_MONTH));
				break;
			}
			}
		}
		catch(ParseException e)
		{
			throw new MetadataElementValidatorException(I18N.translate_NoCheck("MessageValidatorDateCHCantConvert", value));
		}
		return upperValue;
	}
	
	/**
	 * Return the date format for a given date string
	 * @param date
	 * @return DateFormat with specified format
	 * @throws MetadataElementValidatorException
	 */
	private DateFormat getDateFormat(String date) throws MetadataElementValidatorException
	{
		
		DateFormat df;
		switch (date.length())
		{
		case 4:
			df = new SimpleDateFormat("yyyy");
			break;
		case 6: case 7:
			df = new SimpleDateFormat("M.yyyy");
			break;
		case 8: case 9: case 10:
			df = new SimpleDateFormat("d.M.yyyy");
			break;
		default:
			//	given the use of MetadataElementValidatorDateCH above, the code should never reach this point
			throw new MetadataElementValidatorException(I18N.translate_NoCheck("MessageValidatorDateCHCantConvert", date));
		}
		df.setLenient(false);
		return df;
	}
}
