package ch.docuteam.converter;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.ConnectException;

import javax.swing.SwingWorker;

import com.artofsolving.jodconverter.DocumentConverter;
import com.artofsolving.jodconverter.XmlDocumentFormatRegistry;
import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeConnection;
import com.artofsolving.jodconverter.openoffice.connection.SocketOpenOfficeConnection;
import com.artofsolving.jodconverter.openoffice.converter.OpenOfficeDocumentConverter;

import ch.docuteam.converter.exceptions.OOConverterShallNotBeUsedException;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;

//To start the OpenOffice service headless:
// Windows:
// soffice.exe -headless -nofirststartwizard
// -accept="socket,host=localhost,port=8100;urp;StarOffice.Service"
// OS X:
// /Applications/OpenOffice.org.app/Contents/MacOS/soffice.bin -headless
// -nofirststartwizard
// -accept="socket,host=localhost,port=8100;urp;StarOffice.Service" &
/**
 * This class is a wrapper around the JODConverter class, which in turn is a
 * wrapper around the OpenOffice Document Converter class. This class is used by
 * the FileConverter reflectively or directly via the main() method, hence it
 * seems to the compiler as if it is unused. The only method used from the
 * outside is static public void main().
 */

public class OOConverter {

	// Several different default values:
	static private final String DefaultPathBase = "apps";
	static private final String DefaultPathMac = DefaultPathBase + "/OpenOffice.org.app";
	static private final String DefaultPathWindows = DefaultPathBase + "/OpenOffice.org 3";
	static private final String DefaultPathLinux = DefaultPathBase + "/openoffice4";
	static private final String PathRelativeMac = "/Contents/MacOS/soffice.bin";
	static private final String PathRelativeWindows = "/program/soffice.exe";
	static private final String PathRelativeLinux = "/program/soffice";

	static private final String DefaultHost = "localhost";
	static private final int DefaultPort = 8100;

	static private final String DefaultJODDocumentFormatsFile = "config/document-formats.xml";

	static private final int DefaultNumberOfInitializationRetries = 20;

	static private String Path = OperatingSystem.isWindows() ? DefaultPathWindows
			: (OperatingSystem.isMacOSX() ? DefaultPathMac : DefaultPathLinux);
	static private String Host = DefaultHost;
	static private int Port = DefaultPort;

	static private String JODDocumentFormatsFile = DefaultJODDocumentFormatsFile;

	static private boolean ShallNotBeUsed = false;
	static private boolean IsInitializing = false;
	static private int NumberOfInitializationRetries = DefaultNumberOfInitializationRetries;

	static private OpenOfficeConnection Connection;
	static private DocumentConverter Converter;
	static private Process OOProcess;

	/**
	 * This main method is the gateway to run the converter. It's NOT just for
	 * tests! Don't delete it!
	 * 
	 * @throws OOConverterShallNotBeUsedException
	 */
	static public void main(String... args)
			throws IOException, FileUtilExceptionListException, OOConverterShallNotBeUsedException {
		if (args.length != 2) {
			System.err.println("ERROR: Wrong number of arguments.");
			System.err.println("");
			System.err.println("Usage: FileConverter$OOConverter [path/to/]sourceFile [path/to/]destinationFile");
			return;
		}

		convert2PDF(args[0], args[1]);
	}

	/**
	 * This must be called BEFORE the OOConverter is initialized!
	 */
	public static void setJODDocumentsFormatFile(String newJODDocumentFormatsFile) {
		JODDocumentFormatsFile = newJODDocumentFormatsFile;
	}

	/**
	 * This must be called BEFORE the OOConverter is initialized!
	 */
	public static void setNumberOfInitializationRetries(Integer newNumberOfInitializationRetries) {
		if (newNumberOfInitializationRetries == null)
			return;
		if (newNumberOfInitializationRetries <= 0)
			return;

		NumberOfInitializationRetries = newNumberOfInitializationRetries;
	}

	static public Integer getNumberOfInitializationRetries() {
		return NumberOfInitializationRetries;
	}

	static public String getConverterPath() {
		return Path;
	}

	static public boolean shallNotBeUsed() {
		return ShallNotBeUsed;
	}

	/**
	 * Initialize in a separate thread.
	 */
	static public void initializeDontWait(String ooConverterPath, String ooConverterHost, Integer ooConverterPort) {
		if (!(ooConverterHost == null || ooConverterHost.isEmpty()))
			Host = ooConverterHost;
		if (!(ooConverterPort == null))
			Port = ooConverterPort;
		initializeDontWait(ooConverterPath);
	}

	/**
	 * Initialize in a separate thread.
	 */
	static public void initializeDontWait(String ooConverterPath) {
		if (!(ooConverterPath == null || ooConverterPath.isEmpty()))
			Path = FileUtil.asCanonicalFileName(ooConverterPath);
		initializeDontWait();
	}

	/**
	 * Initialize in a separate thread.
	 */
	static public void initializeDontWait() {
		new SwingWorker<Integer, Object>() {
			@Override
			public Integer doInBackground() {
				initialize();

				return 0;
			}
		}.execute();
	}

	static public void initialize(String ooConverterPath, String ooConverterHost, Integer ooConverterPort) {
		if (!(ooConverterHost == null || ooConverterHost.isEmpty()))
			Host = ooConverterHost;
		if (!(ooConverterPort == null))
			Port = ooConverterPort;
		initialize(ooConverterPath);
	}

	static public void initialize(String ooConverterPath) {
		if (!(ooConverterPath == null || ooConverterPath.isEmpty()))
			Path = FileUtil.asCanonicalFileName(ooConverterPath);
		initialize();
	}

	static public void initialize() {
		// Maybe the Converter exists already?
		if (Converter != null)
			return;

		// Converter does not exist. But maybe the OO service is just being
		// initialized in another thread? If yes, wait until initialization
		// is finished:
		while (IsInitializing) {
			Logger.getLogger().debug("Waiting for initialization to finish...");
			try {
				Thread.sleep(500);
			} catch (InterruptedException x) {
			}
		}

		// Maybe the Converter exists now?
		if (Converter != null)
			return;

		// Try to create the converter - if it succeeded, the OO Service was
		// already running:
		Converter = createConverter();
		if (Converter != null)
			return;

		String ooConverterCompletePath = Path;
		if (OperatingSystem.isWindows())
			ooConverterCompletePath += PathRelativeWindows;
		else if (OperatingSystem.isMacOSX())
			ooConverterCompletePath += PathRelativeMac;
		else if (OperatingSystem.isLinux())
			ooConverterCompletePath += PathRelativeLinux;

		Logger.getLogger().debug("Starting OO Service on host:'" + Host + "' port:'" + Port + "' path:'"
				+ ooConverterCompletePath + "'");

		// The OO Service is not running, so start it:
		try {
			// This is when initializing in an own thread:
			IsInitializing = true;

			try {
				// Is the OO application where it's supposed to be?
				if (!new File(ooConverterCompletePath).exists()) {
					if (Path.startsWith(DefaultPathBase)) {
						Logger.getLogger().info(
								"Could not find the default OO application for converting files in folder: " + Path);

						// The OO path is the defaultPath but OO is not
						// there - so I suppose that OO shall not be used at
						// all:
						ShallNotBeUsed = true;
						return;
					}

					System.out.println("I was told that the OO application is right there.");
					System.out.println("Maybe this location is wrong or the OO application is not installed there?");

					throw new FileNotFoundException(ooConverterCompletePath);
				}

				// Start the OOProcess:
				OOProcess = Runtime.getRuntime()
						.exec(new String[] { ooConverterCompletePath, "-headless", "-nofirststartwizard",
								"-accept=socket,host=" + Host + ",port=" + Port + ";urp;StarOffice.Service" });
			} catch (IOException x) {
				x.printStackTrace();
				return;
			}

			// Now that the OOProcess is running, create the Converter,
			// retry several times, wait 1 sec between each try:
			for (int i = NumberOfInitializationRetries; i > 0; i--) {
				Logger.getLogger().debug(i + ":");

				Converter = createConverter();
				if (Converter != null) {
					Logger.getLogger().debug("OOConverter established");
					break;
				}

				// Wait a sec and then retry:
				try {
					Thread.sleep(1000);
				} catch (InterruptedException x) {
				}
			}
		} finally {
			IsInitializing = false;
		}
	}

	static public void disconnect() {
		if (Connection != null) {
			Logger.getLogger().debug("Disconnecting OOConverter...");

			// Sometimes, connection.disconnect() throws a
			// NullPointerException for no obvious reason:
			try {
				Connection.disconnect();
			} catch (NullPointerException x) {
			} finally {
				Connection = null;
			}
		}

		if (OOProcess != null) {
			Logger.getLogger().debug("Destroying OOProcess...");

			OOProcess.destroy();

			// Wait a sec (otherwise it won't work):
			try {
				Thread.sleep(1000);
			} catch (InterruptedException x) {
			}
		}

		// Clean-up thoroughly:
		Path = OperatingSystem.isWindows() ? DefaultPathWindows
				: (OperatingSystem.isMacOSX() ? DefaultPathMac : DefaultPathLinux);
		Host = DefaultHost;
		Port = DefaultPort;

		JODDocumentFormatsFile = DefaultJODDocumentFormatsFile;

		NumberOfInitializationRetries = DefaultNumberOfInitializationRetries;

		Converter = null;
		ShallNotBeUsed = false;
	}

	static public void convert2PDF(String sourceFileName, String destinationFileName)
			throws IOException, FileUtilExceptionListException, OOConverterShallNotBeUsedException {
		initializeIfNecessary();
		if (ShallNotBeUsed)
			throw new OOConverterShallNotBeUsedException();

		Logger.getLogger().info("Converting file: '" + sourceFileName + "' to PDF: '" + destinationFileName + "'");

		File sourceFile = new File(sourceFileName);
		File destinationFile = new File(destinationFileName);

		if (!sourceFile.exists())
			throw new FileNotFoundException(sourceFileName);
		FileUtil.createFolderMerging(destinationFile.getParentFile());
		FileUtil.delete(destinationFile);

		try {
			Converter.convert(sourceFile, destinationFile);
		} catch (NullPointerException x) {
			Logger.getLogger().error("File conversion failed - maybe the OpenOffice service is not running?");
			throw x;
		}
	}

	static public boolean isInstalledLocallyForWindows() {
		return new File(DefaultPathWindows).exists();
	}

	static public boolean isInstalledLocallyForOSX() {
		return new File(DefaultPathMac).exists();
	}

	static public boolean isInstalledLocallyForLinux() {
		return new File(DefaultPathLinux).exists();
	}

	/**
	 * Return the remote path for Windows if the current OS is Windows and the
	 * path is not the local default path. Otherwise return an empty string.
	 * 
	 * @return
	 */
	static public String getRemotePathForWindows() {
		return OperatingSystem.isWindows() && !Path.equals(DefaultPathWindows) ? Path : "";
	}

	/**
	 * Return the remote path for OSX if the current OS is OSX and the path is
	 * not the local default path. Otherwise return an empty string.
	 * 
	 * @return
	 */
	static public String getRemotePathForOSX() {
		return OperatingSystem.isMacOSX() && !Path.equals(DefaultPathMac) ? Path : "";
	}

	/**
	 * Return the remote path for Linux if the current OS is Linux and the path
	 * is not the local default path. Otherwise return an empty string.
	 * 
	 * @return
	 */
	static public String getRemotePathForLinux() {
		return OperatingSystem.isLinux() && !Path.equals(DefaultPathLinux) ? Path : "";
	}

	static private void initializeIfNecessary() {
		if (Converter == null)
			initialize();
	}

	static private DocumentConverter createConverter() {
		DocumentConverter converter = null;

		try {
			Logger.getLogger().debug("Connecting to OO Service...");

			(Connection = new SocketOpenOfficeConnection(Port)).connect();
			try {
				Logger.getLogger().debug(
						"Trying to initialize OO Service with initialization file '" + JODDocumentFormatsFile + "'");
				converter = new OpenOfficeDocumentConverter(Connection,
						new XmlDocumentFormatRegistry(new FileInputStream(JODDocumentFormatsFile)));
			} catch (FileNotFoundException e) {
				Logger.getLogger()
						.debug("Initialization file '" + JODDocumentFormatsFile + "' not found, using defaults");
				converter = new OpenOfficeDocumentConverter(Connection);
			}

			Logger.getLogger().debug("... OO Service connected");
		} catch (ConnectException e) {
			Logger.getLogger().debug(e.toString());
		}

		return converter;
	}

}
