/**
 *	Copyright (C) 2014-2015 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.converter;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.URL;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

import ch.docuteam.converter.exceptions.PDFToolsConverterBadVMArgumentException;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.out.Logger;
import ch.snb.pdfgenerator.SNBPDFGenerator;

/**
 * This class is a wrapper around the SNBPDFGenerator. It is used by the
 * FileConverter reflectively or directly via the main() method, hence it seems
 * to the compiler as if it is unused. The only method used from the outside is
 * static public void main().
 */
public abstract class SNBPDFConverter
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static final private String				DefaultServiceURL = "http://pdfgeneratortest/SNBPDFGenerator.asmx";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private SNBPDFGenerator			Generator;
	static private String					ServiceURL	= DefaultServiceURL;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * This main method is the gateway to run the converter. It's NOT just for tests! Don't delete it!
	 * @throws MalformedURLException 
	 */
	static public void main(String... args) throws FileNotFoundException, MalformedURLException, IOException
	{
		if (args.length != 2 && args.length != 5)
		{
			System.err.println("ERROR: Wrong number of arguments.");
			System.err.println("");
			System.err.println("Usage: SNBPDFGenerator [path/to/]sourceFile [path/to/]destinationFile [pdfSettings securitySettings fileTypeSettings]");
			return;
		}

		if (args.length == 2)			convert2PDF(args[0], args[1]);
		else if (args.length == 5)		convert2PDF(args[0], args[1], args[2], args[3], args[4]);
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initialize the generator using the default service url.
	 * @throws SocketException
	 * @throws PDFToolsConverterBadVMArgumentException
	 * @throws MalformedURLException 
	 */
	static public void initialize() throws MalformedURLException
	{
		initialize(ServiceURL);
	}


	/**
	 * Initialize remote pdf generator.
	 * @param url The service URL for connecting to the pdf generator.
	 * @throws MalformedURLException 
	 */
	static public void initialize(String url) throws MalformedURLException
	{
		Logger.debug("Initializing service at: '" + url + "'");
		Generator = new ch.snb.pdfgenerator.SNBPDFGenerator(new URL(url));
	}


	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	static public void convert2PDF(String sourceFilePath, String destinationFilePath) throws FileNotFoundException, MalformedURLException, IOException
	{
		convert2PDF(sourceFilePath, destinationFilePath, "", "", "");
	}


	/**
	 * 
	 * @param sourceFilePath
	 * @param destinationFilePath
	 * @param pdfSettings defaults to "Standard_SNB"
	 * @param securitySettings defaults to "No Security"
	 * @param fileTypeSettings defaults to "Standard"
	 * @throws FileNotFoundException
	 * @throws MalformedURLException
	 * @throws IOException
	 */
	static public void convert2PDF(String sourceFilePath, String destinationFilePath, String pdfSettings, String securitySettings, String fileTypeSettings) throws FileNotFoundException, MalformedURLException, IOException
	{
		initializeIfNecessary();
		if (Generator == null)
		{
			Logger.error("PDF generator can not be initialized");
			return;
		}

		Logger.info("Converting: '" + sourceFilePath + "' to: '" + destinationFilePath + "'");

		FileInputStream fisSource = new FileInputStream(sourceFilePath);
		Logger.debug("Parameters:\tFileType = " + FileUtil.asFileNameExtension(sourceFilePath) + ", pdfSettings = " + pdfSettings + ", securitySettings = " + securitySettings + ", fileTypeSettings = " + fileTypeSettings);
		byte[] output = Generator.getSNBPDFGeneratorSoap().createPdfUsingCustomSettings(IOUtils.toByteArray(fisSource), FileUtil.asFileNameExtension(sourceFilePath), pdfSettings, securitySettings, fileTypeSettings);
		Logger.debug("Output length: " + output.length);
		fisSource.close();
		FileUtils.writeByteArrayToFile(new File(destinationFilePath), output);
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static private void initializeIfNecessary() throws MalformedURLException
	{
		if (Generator == null)		initialize();
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}