/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.db;


import java.sql.*;

import ch.docuteam.tools.out.Logger;

/**
 * @author denis
 *
 */
public abstract class MySQLJDBCConnection
{

	//		DB Property File Keys:
	
	public static final String		PropertyNameDBHost = "DBHost";
	public static final String		PropertyNameDBPort = "DBPort";
	public static final String		PropertyNameDBName = "DBName";
	public static final String		PropertyNameDBUserName = "DBUserName";
	public static final String		PropertyNameDBUserPassword = "DBUserPassword";

	//		DB SSL Property File Keys:
	
	public static final String		PropertyNameSSLKeyStore = "javax.net.ssl.keyStore";
	public static final String		PropertyNameSSLKeyStorePassword = "javax.net.ssl.keyStorePassword";
	public static final String		PropertyNameSSLTrustStore = "javax.net.ssl.trustStore";
	public static final String		PropertyNameSSLTrustStorePassword = "javax.net.ssl.trustStorePassword";


	private static final String		DBConnectionPrefix = "jdbc:mysql://";
	private static final String		DBConnectionSSLPostfix = "?verifyServerCertificate=true&useSSL=true&requireSSL=true";


	private static String			ConnectionString;
	private static String			UserName;
	private static String			Password;
	
	private static Connection		Connection;
	
	
	/**
	 * 	Create the JDBCConnection
	 */
	public static Connection create(String dbHost, String dbPort, String dbName, String userName, String password) throws SQLException
	{
		ConnectionString = DBConnectionPrefix + dbHost + ":" + dbPort + "/" + dbName;
		UserName = userName;
		Password = password;

		return create();
	}

	
	/**
	 * 	Create the JDBCConnection using a SSL
	 */
	public static Connection createUsingSSL(String dbHost, String dbPort, String dbName, String userName, String password, String sslKeyStore, String sslKeyStorePassword, String sslTrustStore, String sslTrustStorePassword) throws SQLException
	{
		System.setProperty(PropertyNameSSLKeyStore, sslKeyStore);
		System.setProperty(PropertyNameSSLKeyStorePassword, sslKeyStorePassword);
		System.setProperty(PropertyNameSSLTrustStore, sslTrustStore);
		System.setProperty(PropertyNameSSLTrustStorePassword, sslTrustStorePassword);

		ConnectionString = DBConnectionPrefix + dbHost + ":" + dbPort + "/" + dbName + DBConnectionSSLPostfix;
		UserName = userName;
		Password = password;

		return create();
	}
	

	public static Connection getConnection()
	{
		return Connection;
	}
	
	
	public static ResultSet executeQuery(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.getLogger().info("\t" + query);
		return Connection.createStatement().executeQuery(query);
	}
	
	
	public static int executeUpdate(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.getLogger().info("\t" + query);
		int rowCount = Connection.createStatement().executeUpdate(query);
		Logger.getLogger().info("\t-> " + rowCount);
		return rowCount;
	}
	
	public static ResultSet executeInsert(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.getLogger().info("\t" + query);
		Statement statement = Connection.createStatement();
		int rowCount = statement.executeUpdate(query, new String[] {"id"});
		Logger.getLogger().info("\t-> " + rowCount);
		return statement.getGeneratedKeys();
	}
	
	
	public static PreparedStatement prepareStatement(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.getLogger().info("\t" + query);
		return Connection.prepareStatement(query, new String[] {"id"});
	}
	
	/**
	 * Close the connection
	 * @throws SQLException
	 */
	public static void close() throws SQLException
	{
		Logger.getLogger().debug("Closing");
		if (Connection != null)		Connection.close();
	}

	private static Connection create() throws SQLException
	{
		Logger.getLogger().debug("Creating: " + ConnectionString);
		return Connection = DriverManager.getConnection(ConnectionString, UserName, Password);
	}
	

	private static Connection reOpenIfNecessary() throws SQLException
	{
		if ((Connection == null) || !Connection.isValid(0))		create();
		return Connection;
	}

}
