/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;
/**
 *
 */

import static ch.docuteam.tools.ToolsConstants.*;
import java.io.*;
import java.nio.charset.Charset;
import java.util.*;

import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.out.Logger;
import de.schlichtherle.truezip.zip.*;

/**
 * This abstract class zips and unzips files and directories.
 *
 * @author denis
 */
public abstract class Zipper
{

	private static final Charset DefaultZipReadCharset = Charset.forName(System.getProperty("zip.encoding", Charset.defaultCharset().name()));

	private static Charset ZipReadCharset = DefaultZipReadCharset;
	
	public static Charset getZipReadCharset() {
		return ZipReadCharset;
	}


	public static void setZipReadCharset(Charset zipReadCharset) {
		ZipReadCharset = zipReadCharset;
	}


	public static void setZipReadCharset(String zipReadCharset) {
		ZipReadCharset = Charset.forName(zipReadCharset);
	}


	/**
	 * This single-argument method zips whatever is in the source path (a single file or directory name) and creates a .zip-file with the same name as the source file (with '.zip' appended).
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	public static File zip(String sourcePath) throws IOException
	{
		return zip(sourcePath, sourcePath + ".zip");
	}


	/**
	 * This two-argument method zips whatever is in the source path (a single file or directory name) and creates the .zip-file as specified in the second parameter (with '.zip' appended if necessary).
	 * In case of a directory, all contained files and directories are zipped too.
	 * In case the boolean parameter 'skipTopFolder' is true AND the top element is a directory, zipping excludes the top folder and zips only the content of it.
	 * @return The created zip-file
	 */
	public static File zip(String sourcePath, String zipFilePath, boolean skipTopFolder) throws IOException
	{
		if (!skipTopFolder)			return zip(sourcePath, zipFilePath);

		//	Does source file exist?:
		File sourceFile = new File(sourcePath);
		if (!sourceFile.exists())
		{
			Logger.getLogger().error("Source file for zipping doesn't exist:" + sourcePath);
			throw new FileNotFoundException(sourcePath);
		}

		if (sourceFile.isFile())	return zip(sourcePath, zipFilePath);

		//	Construct the list of file names within the sourcePath (which is a folder):
		List<String> subFileNames = new ArrayList<String>();
		for (String fileName: sourceFile.list())		subFileNames.add(sourcePath + "/" + fileName);

		return zip(subFileNames.toArray(new String[]{}), zipFilePath);
	}


	/**
	 * This two-argument method zips whatever is in the source path (a single file or directory name) and creates the .zip-file as specified in the second parameter (with '.zip' appended if necessary).
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	public static File zip(String sourcePath, String zipFilePath) throws IOException
	{
		Logger.getLogger().info("Zipping: " + sourcePath + " Into:" + zipFilePath);

		//	Does source file exist?:
		File sourceFile = new File(sourcePath);
		if (!sourceFile.exists())
		{
			Logger.getLogger().error("Source file for zipping doesn't exist:" + sourcePath);
			throw new FileNotFoundException(sourcePath);
		}

		String zipRootDirectory = sourceFile.getParent();

		//	Add ".zip" to destination file name if necessary:
		if (!(zipFilePath.endsWith(".zip") || zipFilePath.endsWith(".ZIP"))){
			zipFilePath += ".zip";
		}

		//	Create destination directories if necessary:
		new File(zipFilePath).getParentFile().mkdirs();

		//	Zip it:
		ZipOutputStream zipOutStream = new ZipOutputStream(new FileOutputStream(zipFilePath));
		try
		{
			zip(zipRootDirectory, sourceFile, zipOutStream);
		}
		finally
		{
			zipOutStream.close();
		}

		return new File(zipFilePath);
	}


	/**
	 * This three-argument method zips whatever is in the source path (a single file or a directory) and creates the .zip-file as specified in the second argument (with '.zip' appended if necessary),
	 * but INSIDE the zip-file, the root file or directory is renamed as specified in the 3rd parameter.
	 * When extracting this zip-file, the resulting file or directory then has this name.
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	public static File zip(String sourcePath, String zipFilePath, String newName) throws IOException
	{
		Logger.getLogger().info("Zipping: " + sourcePath + " Into:" + zipFilePath + " Renaming to:" + newName);

		//	Does source file exist?:
		File sourceFile = new File(sourcePath);
		if (!sourceFile.exists())
		{
			Logger.getLogger().error("Source file for zipping doesn't exist:" + sourcePath);
			throw new FileNotFoundException(sourcePath);
		}

		//	Rename source File, zip it, then rename it back:
		String newSourcePath = sourceFile.getParent() + "/" + newName;
		File newSourceFile = new File(newSourcePath);

		try
		{
			if (!sourceFile.renameTo(newSourceFile))		throw new IOException("Could not rename File '" + sourcePath + "' to: '" + newSourcePath + "'");
			return zip(newSourcePath, zipFilePath);
		}
		finally
		{
			if (!newSourceFile.renameTo(sourceFile))		throw new IOException("Could not rename File '" + newSourcePath + "' back to: '" + sourcePath + "'");
		}
	}


	/**
	 * This multi-argument method zips whatever is in the source paths (an array of file or directory names) and creates the .zip-file as specified in the second parameter (with '.zip' appended if necessary).
	 * IMPORTANT NOTE: All source paths must be located within the same common directory!
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	public static File zip(String[] sourcePaths, String zipFilePath) throws IOException
	{
		Logger.getLogger().info("Zipping: " + Arrays.toString(sourcePaths) + " Into:" + zipFilePath);

		if (sourcePaths.length == 0)
		{
			Logger.getLogger().info("Nothing to zip found");
			return null;
		}

		String zipRootDirectory = new File(sourcePaths[0]).getParent();

		//	Add ".zip" to destination file name if necessary:
		if (!(zipFilePath.endsWith(".zip") || zipFilePath.endsWith(".ZIP")))		zipFilePath += ".zip";

		//	Create destination directories if necessary:
		new File(zipFilePath).getParentFile().mkdirs();

		//	Zip it:
		ZipOutputStream zipOutStream = new ZipOutputStream(new FileOutputStream(zipFilePath));
		try
		{
			for (String sourcePath: sourcePaths)
			{
				File sourceFile = new File(sourcePath);
				if (sourceFile.exists())
				{
					zip(zipRootDirectory, new File(sourcePath), zipOutStream);
				}
				else
				{
					Logger.getLogger().error("Source file for zipping doesn't exist:" + sourcePath);
					throw new FileNotFoundException(sourcePath);
				}
			}
		}
		finally
		{
			zipOutStream.close();
		}

		return new File(zipFilePath);
	}



	/**
	 * Unzip the zipFile to the same folder where the zipFile is.
	 */
	public static void unzip(String zipFilePath) throws IOException
	{
		String destPath = ".";

		Integer lastIndexOfFileSeparator = zipFilePath.lastIndexOf("/");
		if (lastIndexOfFileSeparator != -1)		destPath = zipFilePath.substring(0, lastIndexOfFileSeparator);

		unzip(zipFilePath, destPath);
	}


	/**
	 * Unzip the zipFile to the destination folder. Non-existing folders are generated if necessary.
	 */
	public static void unzip(String zipFilePath, String destPath) throws IOException
	{
		Logger.getLogger().info("Unzipping: " + zipFilePath + " Into:" + destPath);

		//	Does source file exist?:
		if (!new File(zipFilePath).exists())
		{
			Logger.getLogger().error("Source file for unzipping doesn't exist:" + zipFilePath);
			throw new FileNotFoundException(zipFilePath);
		}

		ZipFile zipFile = new ZipFile(zipFilePath, ZipReadCharset);
		try
		{
			for (ZipEntry entry: Collections.list(zipFile.entries()))
			{
				unzipEntry(zipFile, entry, destPath);
			}
		}
		finally
		{
			zipFile.close();
		}
	}



	private static void zip(String zipRootDir, File sourceFile, ZipOutputStream zipOutStream) throws IOException {
		if (sourceFile.isDirectory()) {
			zipDirectory(zipRootDir, sourceFile, zipOutStream);
		} else {
			zipFile(zipRootDir, sourceFile, zipOutStream);
		}
	}


	private static void zipDirectory(String zipRootDir, File sourceDir, ZipOutputStream zipOutputStream)
			throws IOException {
		Logger.getLogger().debug("Zipping Folder: " + sourceDir.getPath());

		// If no read access on the source directory is granted to the current
		// user, skip it:
		if (!sourceDir.canRead()) {
			Logger.getLogger().warn("No read access for '" + OperatingSystem.userName() + "' on folder: '"
					+ sourceDir.getPath() + "', skipping.");
			return;
		}

		//	Empty folder have to be created explicitly (those others are generated automatically):
		if (sourceDir.listFiles().length == 0)
		{
			//	... but remove the zipRootDir from the filePath beforehand:
			//	NOTE: I have to use "File.separator" here because both file names are in the platform-dependent form
			String relativeSourceDirName =
				((zipRootDir == null)
					? sourceDir.getPath()
					: sourceDir.getPath().replace(zipRootDir + File.separator, "")
				) + "/";	//	<- NOTE: This has to be "/", NOT File.separator!

			//	Special case (Thanx to Andi for finding this!):
			//	When the source contains a drive specification ("A:\"), remove the colon:
			if (relativeSourceDirName.charAt(1) == ':')		relativeSourceDirName = relativeSourceDirName.replace(":", "");

			//	Replace all backslashes by "/", otherwise the created zip file is not portable from Win to OSX:
			relativeSourceDirName = relativeSourceDirName.replace('\\', '/');

			zipOutputStream.putNextEntry(new ZipEntry(relativeSourceDirName));
		}

		// Now recurse into the directory:
		for (File file : sourceDir.listFiles()) {
			zip(zipRootDir, file, zipOutputStream);
		}
	}


	private static void zipFile(String zipRootDir, File sourceFile, ZipOutputStream zipOutputStream) throws IOException
	{
		Logger.getLogger().debug("Zipping File:   " + sourceFile.getPath());

		//	If no read access on the source file is granted to the current user, skip it:
		if (!sourceFile.canRead())
		{
			Logger.getLogger().warn("No read access for '" + OperatingSystem.userName() + "' on file: '" + sourceFile.getPath() + "', skipping.");
			return;
		}

		//	First create the file entry in the zipOutputStream...
		//	... but remove the zipRootDir beforehand:
		//	NOTE: I have to use "File.separator" here because both file names are in the platform-dependent form
		String relativeSourceFileName =
			(zipRootDir == null)
				? sourceFile.getPath()
				: sourceFile.getPath().replace(zipRootDir + File.separator, "");

		//	Special case (Thanx to Andi for finding this!):
		//	When the source contains a drive specification ("A:\"), remove the colon:
		if (relativeSourceFileName.charAt(1) == ':')	relativeSourceFileName = relativeSourceFileName.replace(":", "");

		//	Replace all backslashes by "/", otherwise the created zip file is not portable from Win to OSX:
		relativeSourceFileName = relativeSourceFileName.replace('\\', '/');

		zipOutputStream.putNextEntry(new ZipEntry(relativeSourceFileName));

		//	... then fill the entry with the file content:
		BufferedInputStream fileInputStream = new BufferedInputStream(new FileInputStream(sourceFile), BYTE_BUFFER_SIZE);

		try {
			byte data[] = new byte[BYTE_BUFFER_SIZE];
			for (int i; (i = fileInputStream.read(data, 0, BYTE_BUFFER_SIZE)) > 0;)
				zipOutputStream.write(data, 0, i);
		} finally {
			zipOutputStream.closeEntry();
			fileInputStream.close();
		}
	}



	private static void unzipEntry(ZipFile zipFile, ZipEntry zipEntry, String destDir) throws IOException
	{
		File file = new File(destDir, zipEntry.getName());

		if (zipEntry.isDirectory())
		{
			Logger.getLogger().debug("Unzipping Folder: " + zipEntry.getName());
			file.mkdirs();
		}
		else
		{
			Logger.getLogger().debug("Unzipping File:   " + zipEntry.getName());

			new File(file.getParent()).mkdirs();

			InputStream zipEntryInputStream = zipFile.getInputStream(zipEntry);
			OutputStream fileOutputStream = new FileOutputStream(file);

			try
			{
				byte data[] = new byte[BYTE_BUFFER_SIZE];
				for (int i; (i = zipEntryInputStream.read(data)) != -1;)		fileOutputStream.write(data, 0, i);
			}
			finally
			{
				fileOutputStream.close();
				zipEntryInputStream.close();
			}
		}
	}

}
