/**
 *	Copyright (C) 2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file.ssh;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;

import ch.docuteam.tools.out.Logger;

/**
 * Takes care of establishing and closing a SSH session.
 * 
 * @author iliya
 *
 */
public class SSHSession {

	private String user;
	private String host;
	private int port;
	private String privateKeyFile;
	private String knownHostsFile;
	private Session session;
	
	private JSch jsch;

	/**
	 * Initialize the secure session parameters
	 * @param user
	 * @param host
	 * @param port
	 * @param privateKeyFile
	 * @param knownHostsFile
	 */
	public SSHSession(String user, String host, int port, String privateKeyFile, String knownHostsFile) {
		this.user = user;
		this.host = host;
		this.port = port;
		this.privateKeyFile = privateKeyFile;
		this.knownHostsFile = knownHostsFile;
	}

	/**
	 * Opens a secure session
	 * @return
	 * @throws JSchException
	 * @throws FileNotFoundException
	 */
	public Session connect() throws JSchException, FileNotFoundException {
		if (session == null) {
			Logger.getLogger().debug("Creating new SSH Session");
			session = getJsch().getSession(user, host, port);
		}
		if (!session.isConnected()) {
			Logger.getLogger().debug("Connecting SSH Session");
			session.connect();
		}
		Logger.getLogger().debug("Host connected");
		return session;
	}

	/**
	 * Disconnects the session to the remote host.
	 */
	public void disconnect() {
		if (session != null && session.isConnected()) {
			session.disconnect();
			Logger.getLogger().debug("Host Session disconnected");
		}
	}

	private JSch getJsch() throws JSchException, FileNotFoundException {
		if (jsch == null) {
			jsch = new JSch();
			try (FileInputStream hostsStream = new FileInputStream(knownHostsFile)) {
				jsch.setKnownHosts(hostsStream);
				jsch.addIdentity(privateKeyFile);
			} catch (FileNotFoundException e) {
				Logger.error("Could not find known hosts file!", e);
				throw e;
			} catch (IOException e) {
				// Ignore, exception when closing the stream to the known hosts file
				Logger.warn("Could not properly close the known hosts file at '" + knownHostsFile + "'");
			} catch (JSchException e) {
				Logger.error("Could not initialize a secure channel!", e);
				throw e;
			}
		}
		return jsch;
	}

}
