/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.common;

import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.regex.MatchResult;

import org.dom4j.Element;
import org.dom4j.Node;

import ch.docuteam.docutools.exception.Exception;
import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.string.StringUtil;


/**
 * Abstract superclass of all elements that are contained in a <a href="../mets/Document.html">Document</a>.
 * All these elements have these four properties in common:
 * <ul>
 * <li>document: The METS document</li>
 * <li>parent: The node containing me</li>
 * <li>element: My dom4j xpath element</li>
 * <li>id: (optional) An identification string</li>
 * </ul>
 * This class inserts Exceptions into the <a href="../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when a mandatory element is missing</li>
 * </ul>
 * @author denis
 *
 */
public abstract class NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected DocumentAbstract			document;
	protected NodeAbstract				parent;
	protected Element					element;
	protected String					id;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public DocumentAbstract getDocument()
	{
		return this.document;
	}

	public NodeAbstract getParent()
	{
		return this.parent;
	}

	public Element getElement()
	{
		return this.element;
	}

	public String getId()
	{
		return this.id;
	}


	public String getElementText(String xPath, Boolean isMandatory)
	{
		Node subNode = this.element.selectSingleNode(xPath);
		if (isMandatory && (subNode == null))
			Exception.remember("Mandatory element missing in File: '" + this.document.getName() + "':\n\t'" + xPath + "'\nin location: '" + this.element.getPath() + "'\nID: " + this.id);

		return ((subNode == null)? "": subNode.getText());
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	//	Set the id. Protected because only subclasses may set the id.
	protected void setId(String id)
	{
		this.id = id;
		this.element.addAttribute("ID", this.id);
	}

	//	--------		Accessing Dynamic	-------------------------------------------------------

	/**
	 *
	 * @param xPath
	 * @return
	 */
	protected List<String> getDynamicElementTexts(String xPath)
	{
		@SuppressWarnings("unchecked")
		List<Node> nodes = this.element.selectNodes(xPath);
		if (nodes.isEmpty())		return null;

		List<String> results = new ArrayList<String>(nodes.size());
		for (Node node: nodes)		results.add(node.getText());

		return results;
	}


	protected void setDynamicElementTexts(List<String> values, String xPath)
	{
		//	First delete all elements...:
		for (Object node: this.element.selectNodes(xPath))		((Node)node).detach();

		//	... then set them (if not null):
		if (values != null)
		{
			//	Create all intermediary elements if necessary:
			Element parentElement = this.element;

			List<String> xPathElements = StringUtil.split(xPath, "/");
			for (int i= 0; i < xPathElements.size() - 1; i++)
			{
				String xPathElement = xPathElements.get(i);
				if (xPathElement.equals("."))	continue;

				Element newElement = (Element)parentElement.selectSingleNode(xPathElement);
				if (newElement == null)			newElement = this.addElement(parentElement, xPathElement);

				parentElement = newElement;
			}

			//	Add the last element to the last intermediary element (or to my own element, if no intermediary elements exist).
			//	Multiply this as often as necessary.
			for (String s: values)
			{
				this.addElement(parentElement, xPathElements.get(xPathElements.size() - 1)).setText(s == null? "": s);
			}
		}
	}


	/**
	 * Add to the parent element a new element according to the xPath string.
	 * If the xPath string contains an attribute spec (but no more than one for now!), add this attribute to the newly created element.
	 */
	private Element addElement(Element parentElement, String xPathElement)
	{
		Element newElement = null;

		if (xPathElement.contains("[@"))
		{
			//	xPathElement contains an attribute:
			Scanner s = new Scanner(xPathElement);
			try
			{
				s.findInLine("(.+)\\[@(\\w*) *= *'(\\w*)'\\]");
				MatchResult result = s.match();

				newElement = parentElement.addElement(result.group(1));
				if (result.group(2) != null && result.group(3) != null)		newElement.addAttribute(result.group(2), result.group(3));
			}
			catch (IllegalStateException ex)
			{
				Logger.error("Bad syntax in " + xPathElement);
			}
			finally
			{
				s.close();
			}
		}
		else
		{
			newElement = parentElement.addElement(xPathElement);
		}

		return newElement;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
