/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.dc;

import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.QName;

import ch.docuteam.docudarc.common.NodeAbstract;


/**
 * This class, used by <a href="./OAI_DC.html">OAI_DC</a>, represents a DC element.
 * It contains just a text (= the content) and the language.
 *
 * @author denis
 *
 */
public class DC extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String		DCNamespacePrefix = "DC";
	static private final String		DCNamespaceURI = "http://purl.org/dc/elements/1.1/";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String					language;
	private String					text;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	/**
	 * Create a new DC from element, and set the Namespace Prefix to uppercase, if necessary.
	 * If move is true, relocate the element from it's original parent to the new one.
	 * This happens when no intermediary OAI_DC element was present and had to be inserted.
	 */
	private DC(OAI_DC parent, Element element, boolean move)
	{
		//	Move means that the element has to be relocated from it's original parent to the new one:
		if (move)
		{
			element.detach();
			parent.getElement().add(element);
		}

		this.document = parent.getDocument();
		this.parent = parent;
		this.element = element;

		this.language = this.element.attributeValue("lang");
		this.text = this.element.getText();

		//	Set the Namespace prefix to be uppercase for sure:
		if (!this.element.getNamespacePrefix().equals(DCNamespacePrefix))		this.element.setQName(new QName(this.element.getName(), new Namespace(DCNamespacePrefix, DCNamespaceURI)));
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	static String getNamespacePrefix()
	{
		return DCNamespacePrefix;
	}

	//	========	Static Private			=======================================================

	/**
	 * This method is used only when a METS-File is being read.
	 * If move is true, relocate the DC elements from their original parent to me.
	 * This happens when no intermediary OAI_DC element was present and I had to be inserted.
	 * @param parent
	 */
	static void parse(OAI_DC parent, Element element, boolean move)
	{
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":contributor"))			parent.getContributors().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":coverage"))			parent.getCoverages().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":creator"))				parent.getCreators().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":date"))				parent.getDates().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":description"))			parent.getDescriptions().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":format"))				parent.getFormats().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":identifier"))			parent.getIdentifiers().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":language"))			parent.getLanguages().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":publisher"))			parent.getPublishers().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":relation"))			parent.getRelations().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":rights"))				parent.getRights().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":source"))				parent.getSources().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":subject"))				parent.getSubjects().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":title"))				parent.getTitles().add(new DC(parent, (Element)o, move));
		for (Object o: element.selectNodes(".//" + DCNamespacePrefix + ":type"))				parent.getTypes().add(new DC(parent, (Element)o, move));
	}


	/**
	 * This method is used only when a OAI_DC is created programmatically out of a DC-Document.
	 */
	static void create(OAI_DC parent, Document oaiDcDocument)
	{
		//	Get the namespace-prefix from the oaiDcDocument (we are flexible!):
		String dcNamespacePrefixUsedInFile = oaiDcDocument.getRootElement().getNamespaceForURI(DCNamespaceURI).getPrefix();
		if (dcNamespacePrefixUsedInFile == null || dcNamespacePrefixUsedInFile.isEmpty())			return;

		//	Are there any DC Elements at all? If no, return:
		if (oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":*").size() == 0)		return;

		//	Now move the DC Elements from their existing position to the new OAI_DC Element:
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":contributor"))		parent.getContributors().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":coverage"))		parent.getCoverages().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":creator"))			parent.getCreators().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":date"))			parent.getDates().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":description"))		parent.getDescriptions().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":format"))			parent.getFormats().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":identifier"))		parent.getIdentifiers().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":language"))		parent.getLanguages().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":publisher"))		parent.getPublishers().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":relation"))		parent.getRelations().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":rights"))			parent.getRights().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":source"))			parent.getSources().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":subject"))			parent.getSubjects().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":title"))			parent.getTitles().add(new DC(parent, (Element)o, true));
		for (Object o: oaiDcDocument.selectNodes(".//" + dcNamespacePrefixUsedInFile + ":type"))			parent.getTypes().add(new DC(parent, (Element)o, true));
	}

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * @return the language
	 */
	public String getLanguage()
	{
		return this.language;
	}

	/**
	 * @return the text
	 */
	public String getText()
	{
		return this.text;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return
			new StringBuilder("[DC:")
				.append((this.language == null)? "": this.language + "/")
				.append(this.text)
				.append("]")
				.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
