/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.ead;

import java.io.*;
import java.util.Map;
import java.util.TreeMap;

import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.*;

import ch.docuteam.docudarc.common.DocumentAbstract;
import ch.docuteam.docutools.exception.ExceptionCollector;
import ch.docuteam.docutools.exception.ExceptionCollectorException;


/**
 * An instance of this class is the java-equivalent of an EAD file.
 * It contains
 * <ul>
 * <li><a href="./Header.html">Header</a></li>
 * <li><a href="./ArchDesc.html">ArchDesc</a></li>
 * </ul>
 *
 * @author denis
 *
 */
public class Document extends DocumentAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static protected final String	DefaultEADTemplateFile = "resources/templates/ead.xml";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static protected String			EADTemplateFile = DefaultEADTemplateFile;

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected Header				eadHeader;
	protected ArchDesc				archDesc;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	static public Document createEmpty() throws FileNotFoundException, DocumentException, ExceptionCollectorException
	{
		return read(EADTemplateFile);
	}


	static public Document read(String filePath) throws FileNotFoundException, DocumentException, ExceptionCollectorException
	{
		if (!new File(filePath).exists())		throw new FileNotFoundException(filePath);

		SAXReader reader = new SAXReader();
		reader.setDocumentFactory(new DocumentFactory());
		Document ead = (Document)reader.read(new File(filePath));

		ExceptionCollector.clear();
		ead.parse(filePath);
		if (!ExceptionCollector.isEmpty())		throw new ExceptionCollectorException();

		return ead;
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	static public void setEADTemplateFile(String newEADTemplateFile)
	{
		EADTemplateFile = newEADTemplateFile;
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public Header getEADHeader()
	{
		return this.eadHeader;
	}

	public ArchDesc getArchDesc()
	{
		return this.archDesc;
	}

	public void setId(String id)
	{
		this.eadHeader.setEadId(id);
	}

	public String getId()
	{
		return this.eadHeader.getEadId();
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------

	/**
	 * Return the type of this document: METS, EAD, ...
	 */
	@Override
	public Type getDocumentType()
	{
		return Type.EAD;
	}

	//	--------		Business Ops		-------------------------------------------------------

	public ComponentAbstract searchUnitTitle(String title)
	{
		return this.archDesc.searchUnitTitle(title);
	}

	//	--------		Persistence			-------------------------------------------------------

	public void saveAs(String filePath) throws IOException
	{
		this.filePath = filePath;
		new File(this.filePath).getParentFile().mkdirs();

		this.save();
	}


	public void save() throws IOException
	{
		Writer oswriter = new java.io.OutputStreamWriter(new FileOutputStream(this.filePath), "utf-8");
		OutputFormat.createPrettyPrint();
		OutputFormat outformat = new OutputFormat("	", false, "utf-8");
		XMLWriter writer = new XMLWriter(oswriter, outformat);
		writer.write(this);
		writer.flush();
		writer.close();

		this.isModified = false;
	}

	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return new StringBuilder("\n====\tEAD-Document on '")
			.append(this.filePath)
			.append("'")
			.append(this.isModified? "*": "")
			.append("\t====\n\t")
			.append(this.eadHeader)
			.append("\n")
			.append(this.archDesc)
			.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	private Document parse(String filePath)
	{
		this.filePath = filePath;

		Element root = this.getRootElement();
		this.eadHeader = Header.parse(this, root);
		this.archDesc = ArchDesc.parse(this, root);

		return this;
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	/**
	 * This class is needed within the class <a href="./Document.html">Document</a> to make the SAXReader generate an instance of the Document class.
	 *
	 * @author denis
	 */
	static private class DocumentFactory extends org.dom4j.DocumentFactory
	{
		// Factory methods
		@Override
		public Document createDocument()
		{
			Map<String, String> namespaces = new TreeMap<String, String>();
			namespaces.put("EAD", "urn:isbn:1-931666-22-9");
			namespaces.put("xlink", "http://www.w3.org/1999/xlink");
			this.setXPathNamespaceURIs(namespaces);

			ch.docuteam.docudarc.ead.Document ead = new ch.docuteam.docudarc.ead.Document();
			ead.setDocumentFactory(this);

			return ead;
		}

	}

}
