/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.ead;

import java.lang.reflect.InvocationTargetException;

import bsh.EvalError;
import ch.docuteam.docudarc.exceptions.MetadataElementValidatorException;
import ch.docuteam.docudarc.mdconfig.LevelMetadataElement;
import ch.docuteam.docudarc.mets.structmap.NodeAbstract;
import ch.docuteam.docutools.string.ToolTipText;

/**
 * This class represents concrete instances of <a href="./LevelMetadataElement.html">LevelMetadataElements</a>.
 * Since an instance of LevelMetadataElement can be "repeating", instances of this class are the repetitions, together with their corresponding values.
 *
 * @author denis
 *
 */
public class MetadataElementInstance implements ToolTipText
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private LevelMetadataElement		levelMetadataElement;
	private NodeAbstract				node;
	private int							i;
	private String						value;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	public MetadataElementInstance(LevelMetadataElement lme, NodeAbstract node, String value, int i)
	{
		this.levelMetadataElement = lme;
		this.node = node;
		this.i = i;
		this.value = value;
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	public String getName()
	{
		return this.levelMetadataElement.getId();
	}

	public String getValue()
	{
		return this.value;
	}

	public void setValue(String value) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException
	{
		//	Avoid overwriting a value with the same value:
		if (this.value == null && value == null)			return;
		if (this.value == null && value.isEmpty())			return;
		if (value != null && value.equals(this.value))		return;

		this.levelMetadataElement.setValueInNode(this.i, value, this.node);
		this.value = this.levelMetadataElement.getValueFromNode(this.i, this.node);
	}

	/**
	 * Set my value to the default value as defined in my LevelMetadataElement, but only if my value is not yet set (i.e. is null or empty).
	 * If it is already set, leave it as it is.
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementValidatorException
	 * @throws EvalError
	 */
	public void initializeValueIfNecessary() throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException, EvalError
	{
		if (this.value == null || this.value.isEmpty())
		{
			String defaultValue = this.getLevelMetadataElement().getDefaultValueFromNode(this.node);
			if (defaultValue != null)		this.setValue(defaultValue);
		}
	}

	public int getIndex()
	{
		return this.i;
	}

	public LevelMetadataElement getLevelMetadataElement()
	{
		return this.levelMetadataElement;
	}

	public String getAttributesString()
	{
		StringBuilder s = new StringBuilder();
		s.append(this.isMandatory()? (this.value == null || this.value.isEmpty()? "!": "*"): (this.isAlwaysDisplayed()?"O": " "));
		s.append(this.isRepeatable()? "+": " ");
		s.append(this.isReadOnly()? "X": " ");

		return s.toString();
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean isMandatory()
	{
		return this.levelMetadataElement.isMandatory();
	}

	public boolean isAlwaysDisplayed()
	{
		return this.levelMetadataElement.isAlwaysDisplayed();
	}

	public boolean isRepeatable()
	{
		return this.levelMetadataElement.isRepeatable();
	}

	public boolean isReadOnly()
	{
		return this.levelMetadataElement.isReadOnly();
	}

	public boolean keepInTemplate()
	{
		return this.levelMetadataElement.keepInTemplate();
	}

	/**
	 * This implementation is awkward! I have to omit "!this.value.isEmpty()", otherwise I can't insert new empty MetadataElements.
	 * @return
	 */
	public boolean isToBeDisplayed()
	{
		return (this.isMandatoryOrSet() || this.levelMetadataElement.isAlwaysDisplayed());
	}

	/**
	 * This implementation is awkward! I have to omit "!this.value.isEmpty()", otherwise I can't insert new empty MetadataElements.
	 * @return
	 */
	public boolean isMandatoryOrSet()
	{
		return (this.levelMetadataElement.isMandatory() || this.value != null);
	}

	/**
	 * This implementation is awkward! I have to insert "this.value.isEmpty()", otherwise I can't display empty mandatory elements.
	 * @return
	 */
	public boolean isMandatoryButNotSet()
	{
		return (this.levelMetadataElement.isMandatory() && (this.value == null || this.value.isEmpty()));
	}

	/**
	 * Return those elements which are mandatory or empty, but not undefined.
	 * @return
	 */
	public boolean isMandatoryOrEmpty()
	{
		return (this.levelMetadataElement.isMandatory() || (this.value != null && this.value.isEmpty()));
	}

	public boolean canBeDeleted() throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		return this.levelMetadataElement.canDeleteOneInstanceFromNode(this.node);
	}

	//	--------		Interface			-------------------------------------------------------

	@Override
	public String getToolTipText()
	{
		return this.levelMetadataElement.getMetadataElement().getToolTipText();
	}

	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder builder = new StringBuilder();
		builder
			.append("MetadataElement [")
			.append(this.getAttributesString())
			.append(this.levelMetadataElement.isRepeatable()? this.i: " ")
			.append(this.levelMetadataElement.getId())
			.append("='")
			.append(this.value)
			.append("']");
		return builder.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
