/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.mdconfig;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import bsh.EvalError;
import ch.docuteam.docudarc.exceptions.MetadataElementCantAddException;
import ch.docuteam.docudarc.exceptions.MetadataElementCantDeleteException;
import ch.docuteam.docudarc.exceptions.MetadataElementValidatorException;
import ch.docuteam.docudarc.mets.structmap.NodeAbstract;
import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.translations.I18N;


/**
 * A LevelMetadataElement is a link between a <a href="LevelOfDescription.html">LevelOfDescription</a> and a <a href="MetadataElement.html">MetadataElement</a>
 * and adds additional properties to this relationship:
 * <ul>
 * <li>isMandatory: a boolean property, whether or not this metadata element is mandatory for this level. Mandatory elements must not be empty</li>
 * <li>isAlwaysDisplayed: a boolean property, whether or not this metadata element is always displayed for this level in the GUI</li>
 * <li>isRepeatable: a boolean property, whether or not this metadata element can occur more than once for this level</li>
 * <li>isReadOnly: a boolean property, whether or not this metadata element can be manually changed for this level</li>
 * <li>keepInTemplate: a boolean property, whether or not this metadata element value remains for this level, when a template is created out of this SIP</li>
 * <li>displayRows: an integer property denoting the number of rows for displaying this metadata element in the GUI for this level</li>
 * </ul>
 * @author denis
 *
 */
public class LevelMetadataElement
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private MetadataElement					metadataElement;

	private boolean							isMandatory;			//	Note: isMandatory takes precedence over isAlwaysDisplayed!
	private boolean							isAlwaysDisplayed;		//	Note: isAlwaysDisplayed is WEAKER than isMandatory!
	private boolean							isRepeatable;
	private boolean							isReadOnly;
	private boolean							keepInTemplate;
	private int								displayRows;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	LevelMetadataElement(String metadataElementName, String isMandatory, String isAlwaysDisplayed, String isRepeatable, String isReadOnly, String keepInTemplate, String displayRows)
	{
		this.metadataElement = MetadataElement.get(metadataElementName);
		if (this.metadataElement == null)		throw new NullPointerException("MetadataElement '" + metadataElementName + "' is undefined");

		this.isMandatory = new Boolean(isMandatory);
		this.isAlwaysDisplayed = new Boolean(isAlwaysDisplayed);
		this.isRepeatable = new Boolean(isRepeatable);
		this.isReadOnly = new Boolean(isReadOnly);
		this.keepInTemplate = new Boolean(keepInTemplate);
		try
		{
			this.displayRows = new Integer(displayRows);
		}
		catch(NumberFormatException x)
		{
			this.displayRows = 0;
		}

		Logger.debug("Created: " + this.toStringLong());
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	public String getId()
	{
		return this.metadataElement.getId();
	}

	public MetadataElement getMetadataElement()
	{
		return this.metadataElement;
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean isMandatory()
	{
		return this.isMandatory;
	}

	public boolean isAlwaysDisplayed()
	{
		return this.isAlwaysDisplayed;
	}

	public boolean isRepeatable()
	{
		return this.isRepeatable;
	}

	public boolean isReadOnly()
	{
		return this.isReadOnly;
	}

	public boolean keepInTemplate()
	{
		return this.keepInTemplate;
	}

	public int getDisplayRows()
	{
		return this.displayRows;
	}

	//	--------		Inquiring calculated-------------------------------------------------------

	/**
	 * Return true if an instance of this MD Element can be added to this node, false otherwise.
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	public boolean canAddOneInstanceToNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		//	Repeatable: can of course be added:
		if (this.isRepeatable)											return true;

		//	NOT repeatable and (mandatory or always displayed), hence it is already there and therefore can NOT be added:
		if (this.isMandatory || this.isAlwaysDisplayed)					return false;

		//	NOT repeatable, not mandatory, not always displayed, not yet set: can be added:
		if (this.metadataElement.getValueFromNode(node) == null)		return true;

		return false;
	}

	/**
	 * Return true if an instance of this MD Element can be deleted from this node, false otherwise.
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	public boolean canDeleteOneInstanceFromNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		List<String> values = this.metadataElement.getValueFromNode(node);

		if (this.isRepeatable)
		{
			//	A repeatable item can be removed except it is mandatory or always displayed, and has exacatly one value left:
			if (   (values != null && values.size() <= 1)
				&& (this.isMandatory || this.isAlwaysDisplayed))		return false;
		}
		else
		{
			//	Can only be removed if this element is not repeatable, not mandatory, and not always displayed:
			if (this.isMandatory || this.isAlwaysDisplayed)				return false;
		}

		return true;
	}

	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * Send the getter method to the passed C object c and return all elements.
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	public List<String> getAllValuesFromNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		List<String> value = this.metadataElement.getValueFromNode(node);
		return (value == null? new ArrayList<String>(): value);
	}

	/**
	 * Send the getter method to the passed C object c and return the i-th element.
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IllegalArgumentException
	 */
	public String getValueFromNode(int i, NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		List<String> value = this.metadataElement.getValueFromNode(node);
		return (value == null? null: value.get(i));
	}


	/**
	 * Send the setter method to the passed C object c with the parameter 'value' at index position i.
	 * @throws MetadataElementValidatorException
	 */
	public void setValueInNode(int i, String newValue, NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementValidatorException
	{
		String errorMessage = this.validateValueAgainstNode(newValue, node);
		if (errorMessage != null)	throw new MetadataElementValidatorException(errorMessage);

		List<String> values = this.metadataElement.getValueFromNode(node);

		//	If the existing value was cleared or never set, it is null:
		if (values == null)			values = new ArrayList<String>(1);

		//	In the case that the 0th value is set initially:
		if (values.size() == 0)		values.add(newValue);
		else						values.set(i, newValue);

		//	If now the only element in value is null, set the whole value to null:
		if (values.size() == 1
		&&  newValue == null)		values = null;

		this.metadataElement.setValueInNode(values, node);
	}


	/**
	 * This is used when switching the node's LevelOfDescription.
	 * In this case, some dynamic metadata values (those that do not appear in the new level) must be cleared out.
	 * @param node
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public void clearValueInNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		this.metadataElement.setValueInNode(null, node);
	}


	/**
	 * Return the default value, which comes from a String in the levels config file.
	 * This string can be undefined, then the default value is null.
	 * This string can be a java expression, which will be executed. The current AbstractNode is available using the variable "object".
	 * @param o
	 * @return
	 * @throws EvalError
	 */
	public String getDefaultValueFromNode(NodeAbstract node) throws EvalError
	{
		return this.getMetadataElement().getDefaultValueFromNode(node);
	}


	/**
	 * Evaluate the String 'value' in the context of the Node 'context'.
	 * @param value The value to be tested
	 * @param context This is the context of the value (i.e. the container object). It may be an arbitraty object or null, the validator must handle it.
	 * @return Null if the value is OK, an error message otherwise.
	 */
	public String validateValueAgainstNode(String value, NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		return this.getMetadataElement().validateValueAgainstNode(value, node);
	}


	public List<ch.docuteam.docudarc.ead.MetadataElementInstance> getMetadataElementInstancesOfNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
	{
		ArrayList<ch.docuteam.docudarc.ead.MetadataElementInstance> mdElementInstances = new ArrayList<ch.docuteam.docudarc.ead.MetadataElementInstance>();

		if (this.isRepeatable)
		{
			List<String> values = this.metadataElement.getValueFromNode(node);
			if (values != null)
			{
				int i = 0;
				for(String value: values)	mdElementInstances.add(new ch.docuteam.docudarc.ead.MetadataElementInstance(this, node, value, i++));
			}
		}
		else
		{
			String value = this.getValueFromNode(0, node);
			mdElementInstances.add(new ch.docuteam.docudarc.ead.MetadataElementInstance(this, node, value, 0));
		}

		return mdElementInstances;
	}


	/**
	 * Add an metadata element to the node.
	 * If the default value is specified, preset the value with this, otherwise with the empty string.
	 * @param node
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementCantAddException
	 * @throws EvalError
	 */
	public void addMetadataElementInstanceToNode(NodeAbstract node) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementCantAddException, EvalError
	{
		if (!this.canAddOneInstanceToNode(node))	throw new MetadataElementCantAddException(this.getId());

		List<String> values = this.metadataElement.getValueFromNode(node);

		if (this.isRepeatable)
		{
			if (values == null)						values = new ArrayList<String>();							//	First repeatable element
		}
		else
		{
			//	Is not repeatable:
			if (values != null)						throw new MetadataElementCantAddException(this.getId());	//	Is already set
			values = new ArrayList<String>();
		}

		//	Initialize the value with the default value if specified, otherwise with the empty string:
		String defaultValue = this.getDefaultValueFromNode(node);
		values.add(defaultValue != null? defaultValue: "");

		this.metadataElement.setValueInNode(values, node);
	}


	public void deleteMetadataElementInstanceFromNode(NodeAbstract node, ch.docuteam.docudarc.ead.MetadataElementInstance mde) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementCantDeleteException
	{
		this.deleteMetadataElementInstanceFromNode(node, mde.getIndex());
	}

	public void deleteMetadataElementInstanceFromNode(NodeAbstract node, int i) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException, MetadataElementCantDeleteException
	{
		if (!this.canDeleteOneInstanceFromNode(node))		throw new MetadataElementCantDeleteException(this.getId());

		List<String> values = this.metadataElement.getValueFromNode(node);

		if (this.isRepeatable)
		{
			if (this.isMandatory && values.size() <= 1)		throw new MetadataElementCantDeleteException(this.getId());

			values.remove(i);
			if (values.isEmpty())	values = null;
		}
		else
		{
			if (this.isMandatory)							throw new MetadataElementCantDeleteException(this.getId());

			values = null;
		}

		this.metadataElement.setValueInNode(values, node);
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return I18N.translate(this.metadataElement.getAccessorName());
	}

	public String toStringLong()
	{
		return "[LevelMetadataElement: " + this.metadataElement + " " + this.isMandatory + " " + this.isRepeatable + " " + this.isReadOnly + " " + this.keepInTemplate + " " + this.displayRows+ "]";
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
